/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.ping.ejb.session;

import java.rmi.RemoteException;
import java.security.Principal;
import java.util.ArrayList;
import javax.ejb.EJBException;
import javax.ejb.SessionBean;
import javax.ejb.SessionContext;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.HinemosUnknownException;
import com.clustercontrol.MonitorDuplicateException;
import com.clustercontrol.MonitorNotFoundException;
import com.clustercontrol.bean.HinemosModuleConstant;
import com.clustercontrol.monitor.run.bean.MonitorInfo;
import com.clustercontrol.ping.factory.AddMonitorPing;
import com.clustercontrol.ping.factory.DeleteMonitorPing;
import com.clustercontrol.ping.factory.ModifyMonitorPing;
import com.clustercontrol.ping.factory.SelectMonitorPing;

/**
 * PING監視を制御するSession Bean<BR>
 * 
 * @ejb.bean name="MonitorPingController"
 *           jndi-name="MonitorPingController"
 *           type="Stateless"
 *           transaction-type="Container"
 * 
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=MonitorPingInfoLocal"
 * 
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Home"
 *     
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="LocalHome"
 *     
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Local"
 * 
 */
public abstract class MonitorPingControllerBean implements SessionBean {
	
	protected static Log m_log = LogFactory.getLog( MonitorPingControllerBean.class );
	
	@SuppressWarnings("unused")
	private SessionContext m_context;
	
	public void setSessionContext(SessionContext ctx) throws EJBException, RemoteException {
		m_context = ctx;
	}
	
	/**
	 * ping監視情報をマネージャに登録します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="PingMonitorWrite"
     *     method-intf="Remote"
	 * 
	 * @param info 通知情報
	 * @return 登録に成功した場合、true
	 * @throws MonitorDuplicateException 
	 * @throws HinemosUnknownException 
	 * @see  com.clustercontrol.monitor.run.bean.MonitorInfo
	 */
	public boolean addPing(MonitorInfo info) throws MonitorDuplicateException, HinemosUnknownException {
		
		Principal pri = m_context.getCallerPrincipal();

		// ping監視情報を登録
		AddMonitorPing ping = new AddMonitorPing();
		boolean flag;
		try {
			flag = ping.add(info, pri.getName());
		} catch (MonitorDuplicateException e) {
			m_log.warn("addPing() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (Exception e) {
			m_log.error("addPing() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw new HinemosUnknownException(e.getMessage(), e);
		}
		return flag;
	}
	
	/**
	 * マネージャのping監視情報を更新します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="PingMonitorWrite"
     *     method-intf="Remote"
	 * 
	 * @param info 通知情報
	 * @return 変更に成功した場合、true
	 * @throws MonitorNotFoundException 
	 * @throws HinemosUnknownException
	 * @see  com.clustercontrol.monitor.run.bean.MonitorInfo
	 */
	public boolean modifyPing(MonitorInfo info) throws MonitorNotFoundException, HinemosUnknownException {
		
		Principal pri = m_context.getCallerPrincipal();

		// ping監視情報を更新
		ModifyMonitorPing ping = new ModifyMonitorPing();
		boolean flag;
		try {
			flag = ping.modify(info, pri.getName());
		} catch (MonitorNotFoundException e) {
			m_log.warn("modifyPing() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (Exception e) {
			m_log.error("modifyPing() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw new HinemosUnknownException(e.getMessage(), e);
		}
		return flag;
	}
	
	/**
	 * ping監視情報をマネージャから削除します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="PingMonitorWrite"
     *     method-intf="Remote"
	 * 
	 * @param monitorId 監視項目ID
	 * @return 削除に成功した場合、true
	 * @throws MonitorNotFoundException
	 * @throws HinemosUnknownException 
	 */
	public boolean deletePing(String monitorId) throws MonitorNotFoundException, HinemosUnknownException {

		// ping監視情報を削除
		DeleteMonitorPing ping = new DeleteMonitorPing();
		boolean flag;
		try {
			flag = ping.delete(HinemosModuleConstant.MONITOR_PING, monitorId);
		} catch (MonitorNotFoundException e) {
			m_log.warn("deletePing() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (Exception e) {
			m_log.error("deletePing() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw new HinemosUnknownException(e.getMessage(), e);
		}
		return flag;
	}
	
	/**
	 * ping監視情報を取得します。<BR>
	 * 
	 * 引数のmonitorIdに対応するPING監視設定情報を取得します。
	 * 
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="PingMonitorRead"
     *     method-intf="Remote"
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @param monitorId 監視項目ID
	 * @return ping監視情報
	 * @throws MonitorNotFoundException
	 * @throws HinemosUnknownException
	 * @see  com.clustercontrol.monitor.run.bean.MonitorInfo
	 */
	public MonitorInfo getPing(String monitorId) throws MonitorNotFoundException, HinemosUnknownException {

		// ping監視情報を取得
		SelectMonitorPing ping = new SelectMonitorPing();
		MonitorInfo info = null;
		
		try {
			info = ping.getMonitor(HinemosModuleConstant.MONITOR_PING, monitorId);
		} catch (MonitorNotFoundException e) {
			throw e;
		} catch (Exception e) {
			m_log.error("getPing " + e.getMessage(), e);
			throw new HinemosUnknownException("getPing " + e.getMessage(), e);
		}

		return info;
	}

	/**
	 * ping監視一覧リストを取得します。<BR>
	 * 
	 * 戻り値はArrayListのArrayListであり値は以下のように格納される。
	 * 
	 * <Pre>
	 * 
			ArrayList info = new ArrayList();
			info.add(monitor.getMonitorId());
			info.add(monitor.getDescription());
			info.add(facilityPath);
			info.add(MonitorTypeConstant.typeToString(monitor.getMonitorType().intValue()));
			info.add(monitor.getCalendarId());
			info.add(RunIntervalConstant.typeToString(monitor.getRunInterval().intValue()));
			info.add(schedule.getValid(monitor.getMonitorTypeId(), monitor.getMonitorId()));
			info.add(monitor.getRegUser());
			info.add(monitor.getRegDate() == null ? null:new Date(monitor.getRegDate().getTime()));
			info.add(monitor.getUpdateUser());
			info.add(monitor.getUpdateDate() == null ? null:new Date(monitor.getUpdateDate().getTime()));
			list.add(info);
	 * </Pre>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="PingMonitorRead"
     *     method-intf="Remote"
     *     
     * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @return Objectの2次元配列
	 * @throws MonitorNotFoundException
	 * @throws HinemosUnknownException 
	 */
	public ArrayList<MonitorInfo> getPingList() throws MonitorNotFoundException, HinemosUnknownException {

		// ping監視一覧を取得
		SelectMonitorPing ping = new SelectMonitorPing();
		ArrayList<MonitorInfo> list = null;
		try {
			list = ping.getMonitorList(HinemosModuleConstant.MONITOR_PING);
		} catch (MonitorNotFoundException e) {
			throw e;
		} catch (Exception e) {
			m_log.error("getPingList " + e.getMessage(), e);
			throw new HinemosUnknownException(e.getMessage(), e);
		}
		return list;
	}
	
}
