/*

Copyright (C) 2006 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

 */

package com.clustercontrol.performance.jmx;

import java.text.ParseException;

import javax.naming.CompositeName;
import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.Name;
import javax.naming.NamingException;
import javax.rmi.PortableRemoteObject;

import org.jboss.naming.NonSerializableFactory;
import org.jboss.system.ServiceMBeanSupport;
import org.quartz.CronTrigger;
import org.quartz.JobDetail;
import org.quartz.Scheduler;
import org.quartz.SchedulerException;
import org.quartz.impl.StdScheduler;

import com.clustercontrol.performance.bean.QuartzConstant;
import com.clustercontrol.performance.ejb.session.CollectorControllerBean;
import com.clustercontrol.performance.job.RepositoryCheckJob;

/**
 * JBoss起動時に能管理機能の実績収集、リソース監視機能用性能値収集を再実行するためのJMXサービスを実装したクラス
 * 
 * @jmx.mbean
 *     name="user:service=PerformanceService,name=PerformanceService"
 *     description="PerformanceService MBean"
 *     extends="org.jboss.system.ServiceMBean"
 * 
 * @jboss.service servicefile="performance"
 */
public class PerformanceService extends ServiceMBeanSupport implements PerformanceServiceMBean{
	private String jndiName = "PerformanceService";  // JNDIに登録する際の名前

	private String quartzJndiName = "QuartzRAM";  // Quartzをlookupする際のJNDI名

	private String checkInterval = "15 */1 * * * ? *"; // チェックジョブの起動条件
	
	private final String m_checkJobName = "REPOSITORY_CHECK_JOB";
	private final String m_checkJobGroup = "PERF_COLLECT";
	
	private PerformanceRestartManager manager;

	/**
	 * PerformanceServiceクラスのインスタンスを生成します。
	 *
	 */
	public PerformanceService(){
	}

	/**
	 * 自身が登録されているJNDI名を取得します。
	 * 
	 * @jmx.managed-attribute
	 * @return JNDI名
	 */
	public String getJndiName() {
		return jndiName;
	}

	/**
	 * JNDI名を設定します。
	 * 
	 * @jmx.managed-attribute
	 * 
	 * @param jndiName JNDI名
	 * @throws NamingException 
	 * @throws NamingException
	 */
	public void setJndiName(String jndiName) throws NamingException {
		log.info("setJndiName() SnmpService : " + jndiName);

		String oldName = this.jndiName;
		this.jndiName = jndiName;

		if (super.getState() == STARTED) {
			try {
				unbind(oldName);
			} catch (NamingException e){
				NamingException ne = new NamingException("Failed to unbind");
				ne.setRootCause(e);
				log.error(ne);
			}

			try{
				rebind();
			} catch (NamingException e){
				NamingException ne = new NamingException("Failed to rebind");
				ne.setRootCause(e);
				log.error(ne);
			}
		}
	}

	/**
	 * QuartzのサービスのJNDI名を返します。
	 * @return QuartzサービスのJNDI名
	 *
	 * @jmx.managed-attribute
	 */
	public String getQuartzJndiName() {
		return quartzJndiName;
	}

	/**
	 * QuartzのサービスのJNDI名を設定します。
	 * @param jndiName QuartzサービスのJNDI名
	 * 
	 * @jmx.managed-attribute
	 */
	public void setQuartzJndiName(String jndiName) {
		this.quartzJndiName = jndiName;
	}
	
	/**
	 * リポジトリチェックジョブの実行スケジュール設定を返します。
	 * @return チェックジョブの実行スケジュール
	 *
	 * @jmx.managed-attribute
	 */
	public String getRepositoryCheckInterval() {
		return checkInterval;
	}

	/**
	 * リポジトリチェックジョブの実行スケジュールを設定します。
	 * @param jndiName チェックジョブの実行スケジュール
	 * 
	 * @jmx.managed-attribute
	 */
	public void setRepositoryCheckInterval(String checkInterval) {
		this.checkInterval = checkInterval;
	}
	
	/**
	 * サービス名を取得します。
	 * 
	 * @jmx.managed-attribute
	 */
	@Override
	public String getName() {
		return "PerformanceService(" + jndiName + ")";
	}

	/**
	 * DBに登録されている収集で、ステータスが実行中の収集を再開します。
	 * 
	 * @jmx.managed-operation
	 */
	public void restartCollector(){
		manager.restartAll();
	}

	/**
	 * サービス生成時にJMXマイクロカーネルから呼ばれます。
	 */
	@Override
	public void createService() {
		log.info("Create PerformanceService(" + jndiName + ")");

		manager = new PerformanceRestartManager();

		log.info("Created PerformanceService(" + jndiName + ")");
	}

	/**
	 * サービス開始時にJMXマイクロカーネルから呼ばれます。
	 */
	@Override
	public void startService() throws NamingException {
		log.info("Start PerformanceService(" + jndiName + ")");
		// 性能管理機能の全ての収集を再開する
		manager.restartAll();
		
		// 定期チェック用のジョブをスケジューリングする
		scheduleCheckJob();

		rebind();
		log.info("Started PerformanceService(" + jndiName + ")");
	}

	/**
	 * サービスを停止時にJMXマイクロカーネルから呼ばれます。
	 */
	@Override
	public void stopService() throws NamingException {
		log.info("Stop PerformanceService(" + jndiName + ")");
		unbind(jndiName);

		// テーブルを全て削除する
//		snmpSharedTable.deleteTableAll();

		log.info("Stoped PerformanceService(" + jndiName + ")");
	}

	/**
	 * サービスを削除時にJMXマイクロカーネルから呼ばれます。
	 */
	@Override
	public void destroyService() throws Exception {
		log.info("Destroy PerformanceService(" + jndiName + ")");

		manager = null;

		log.info("Destroyed PerformanceService(" + jndiName + ")");
	}

	/**
	 * 
	 * @param rootCtx
	 * @param name
	 * @return
	 * @throws NamingException
	 */
	private static Context createContext(Context rootCtx, Name name) throws NamingException {
		Context subctx = rootCtx;

		for (int n = 0; n < name.size(); n++) {
			String atom = name.get(n);

			try {
				Object obj = subctx.lookup(atom);
				subctx = (Context) obj;
			} catch (NamingException e) {
				// 存在しない場合は、サブコンテキストを生成
				subctx = subctx.createSubcontext(atom);
			}
		}

		return subctx;
	}

	/**
	 * JNDIサービスにPerformanceRestartManagerクラスのインスタンスをバインドします。
	 * @throws NamingException
	 */
	private void rebind() throws NamingException {
		InitialContext rootCtx = new InitialContext();

		Name fullName = rootCtx.getNameParser("").parse(jndiName);
		Name parentName = fullName;
		if(fullName.size() > 1){
			parentName = fullName.getPrefix(fullName.size()-1);
		} else {
			parentName = new CompositeName();
		}

		Context parentCtx = createContext(rootCtx, parentName);
		Name atomName = fullName.getSuffix(fullName.size()-1);
		String atomStirng = atomName.get(0);

		NonSerializableFactory.rebind(parentCtx, atomStirng, manager);
	}

	/**
	 * JNDIサービスにバインドされているインスタンスをアンバインドします。
	 * @param jndiName JNDI名
	 * @throws NamingException
	 */
	private void unbind(String jndiName) throws NamingException {
		InitialContext rootCtx = null;

		try {
			rootCtx = new InitialContext();

			Name fullName = rootCtx.getNameParser("").parse(jndiName);
			Name atomName = fullName.getSuffix(fullName.size() - 1);
			String atom = atomName.get(0);

			rootCtx.unbind(jndiName);
			NonSerializableFactory.unbind(atom);
		} finally {
			if(rootCtx != null) { 
				rootCtx.close(); 
			}
		}
	}

	/**
	 * 性能管理機能で利用するQuartzのJNID名を返します。
	 * @jmx.managed-attribute
	 */
	public String getQuartzName() {
		return QuartzConstant.getQuartzName();
	}

	/**
	 * 性能管理機能で利用するQuartzのJNID名を設定します。
	 * @jmx.managed-attribute
	 */
	public void setQuartzName(String quartzName) {
		QuartzConstant.setQuartzName(quartzName);
	}

	/**
	 * 計算処理実行開始を設定するための、SNMPポーリングからのディレイタイムを返します。
	 * 対象は、収集間隔1分未満のもの
	 * 
	 * @jmx.managed-attribute
	 */
	public int getDelayTimeUnder1min() {
		return QuartzConstant.getDelayTimeUnder1min();
	}

	/**
	 * 計算処理実行開始を設定するための、SNMPポーリングからのディレイタイムを設定します。
	 * 対象は、収集間隔1分未満のもの
	 * 
	 * @jmx.managed-attribute
	 */
	public void setDelayTimeUnder1min(int delayTimeUnder1min) {
		QuartzConstant.setDelayTimeUnder1min(delayTimeUnder1min);
	}

	/**
	 * 計算処理実行開始を設定するための、SNMPポーリングからのディレイタイムを返します。
	 * 対象は、収集間隔1分以上のもの
	 * 
	 * @jmx.managed-attribute
	 */
	public int getDelayTimeOver1min() {
		return QuartzConstant.getDelayTimeOver1min();
	}

	/**
	 * 計算処理実行開始を設定するための、SNMPポーリングからのディレイタイムを設定します。
	 * 対象は、収集間隔1分以上のもの
	 *  
	 * @jmx.managed-attribute
	 */	
	public void setDelayTimeOver1min(int delayTimeOver1min) {
		QuartzConstant.setDelayTimeOver1min(delayTimeOver1min);
	}

	/**
	 * 実績収集で1行あたりに必要な物理サイズ容量（Byte）のうち収集項目ID分を除いたサイズを返します。
	 * サイズ見積り時に使われます。
	 * 
	 * @jmx.managed-attribute
	 */
	public long getFixRowSize() {
		return CollectorControllerBean.getFixRowSize();
	}

	/**
	 * 実績収集で1行あたりに必要な物理サイズ容量（Byte）のうち収集項目ID分を除いたサイズを設定します。
	 * サイズ見積り時に使われます。
	 *  
	 * @jmx.managed-attribute
	 */
	public void setFixRowSize(long sizePerRow) {
		CollectorControllerBean.setFixRowSize(sizePerRow);
	}

	private void scheduleCheckJob(){
		// 新規にジョブを生成
		JobDetail job = new JobDetail(m_checkJobName, m_checkJobGroup, RepositoryCheckJob.class);
		//ジョブ完了時に削除されないようにする
		job.setDurability(true);

		//JobDetailに変数を設定
		job.getJobDataMap().put("jndiName", jndiName);

		//CronTriggerを作成
		CronTrigger cronTrigger = new CronTrigger(m_checkJobName, m_checkJobGroup);

		//起動失敗した場合は、次の起動予定時刻をセットするように設定
		cronTrigger.setMisfireInstruction(CronTrigger.MISFIRE_INSTRUCTION_DO_NOTHING);

		try {
			cronTrigger.setCronExpression(checkInterval);
		} catch (ParseException e) {
			log.error(e.getMessage(), e);
		}

		//QuartzのSchedulerをルックアップ
		try {
			InitialContext ctx = new InitialContext();
			Object obj = ctx.lookup(quartzJndiName);
			Scheduler scheduler = (Scheduler)PortableRemoteObject.narrow(obj, StdScheduler.class);

			// 既に登録されているジョブを削除する(登録されていない場合は何もおこらない)
			scheduler.deleteJob(m_checkJobName, m_checkJobGroup);

			// ジョブを登録する
			scheduler.scheduleJob(job, cronTrigger);
		} catch (NamingException e) {
			log.error(e.getMessage(), e);
		} catch (SchedulerException e) {
			log.error(e.getMessage(), e);
		}
	}
}
