/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.notify.factory;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;

import javax.ejb.CreateException;
import javax.ejb.FinderException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.NotifyNotFoundException;
import com.clustercontrol.bean.HinemosModuleConstant;
import com.clustercontrol.bean.YesNoConstant;
import com.clustercontrol.notify.bean.NotifyEventInfo;
import com.clustercontrol.notify.bean.NotifyInfo;
import com.clustercontrol.notify.bean.NotifyInfoDetail;
import com.clustercontrol.notify.bean.NotifyJobInfo;
import com.clustercontrol.notify.bean.NotifyLogEscalateInfo;
import com.clustercontrol.notify.bean.NotifyMailInfo;
import com.clustercontrol.notify.bean.NotifyStatusInfo;
import com.clustercontrol.notify.bean.NotifyTypeConstant;
import com.clustercontrol.notify.ejb.entity.NotifyCommandInfoLocal;
import com.clustercontrol.notify.ejb.entity.NotifyEventInfoLocal;
import com.clustercontrol.notify.ejb.entity.NotifyInfoLocal;
import com.clustercontrol.notify.ejb.entity.NotifyInfoUtil;
import com.clustercontrol.notify.ejb.entity.NotifyJobInfoLocal;
import com.clustercontrol.notify.ejb.entity.NotifyLogEscalateInfoLocal;
import com.clustercontrol.notify.ejb.entity.NotifyMailInfoLocal;
import com.clustercontrol.notify.ejb.entity.NotifyStatusInfoLocal;
import com.clustercontrol.notify.util.NotifyInfoCache;
import com.clustercontrol.repository.ejb.session.RepositoryControllerLocal;
import com.clustercontrol.repository.ejb.session.RepositoryControllerUtil;
import com.clustercontrol.util.apllog.AplLogger;

/**
 * 通知情報を検索するクラスです。
 *
 * @version 3.0.0
 * @since 1.0.0
 */
public class SelectNotify {
	
	/** ログ出力のインスタンス。 */
	protected static Log m_log = LogFactory.getLog( SelectNotify.class );
	
	/**
	 * 通知情報を返します。
	 * 
	 * @param notifyId 取得対象の通知ID
	 * @return 通知情報
	 * @throws FinderException
	 * @throws NamingException
	 * 
	 * @see com.clustercontrol.notify.ejb.entity.NotifyInfoBean
	 * @see com.clustercontrol.notify.ejb.entity.NotifyEventInfoBean
	 */
	public NotifyInfo getNotify(String notifyId) throws NotifyNotFoundException, NamingException {

		
			// 通知情報を取得
			NotifyInfoLocal notify = null;
			try{
				notify = NotifyInfoUtil.getLocalHome().findByPrimaryKey(notifyId);
			} catch (FinderException e){
				AplLogger apllog = new AplLogger("NOTIFY", "notify");
				String[] args = { notifyId };
				apllog.put("SYS", "003", args);
				m_log.debug("delete():" + e.getMessage());
		        throw new NotifyNotFoundException(e.getMessage(), e);
			}
			
			return getNotify(notify);
			
		
		
	}
	
	public NotifyInfo getNotify(NotifyInfoLocal notify) throws NamingException {
		
		NotifyInfo bean = null;
		String notifyId = notify.getNotifyId();
		
		ArrayList<NotifyInfoDetail> notifyEvents = getNotifyDetails(notifyId,notify);
		
		bean = new NotifyInfo(
				notify.getNotifyId(),
				NotifyInfoCache.getDescription(notifyId),
				NotifyInfoCache.getNotifyType(notifyId),
				NotifyInfoCache.getInitialCount(notifyId),
				NotifyInfoCache.getRenotifyType(notifyId),
				NotifyInfoCache.getRenotifyPeriod(notifyId),
				NotifyInfoCache.getRegDate(notifyId),
				NotifyInfoCache.getUpdateDate(notifyId),
				NotifyInfoCache.getRegUser(notifyId),
				NotifyInfoCache.getUpdateUser(notifyId),
				NotifyInfoCache.getVaildFlg(notifyId),
				notifyEvents
		);
		return bean;
	}
	
	/**
	 * 通知詳細情報を取得します。
	 * 
	 * @param notifyId
	 * @param notify
	 * @return
	 */
	public ArrayList<NotifyInfoDetail> getNotifyDetails(String notifyId, NotifyInfoLocal notify)	{

		
		ArrayList<NotifyInfoDetail> notifyEvents = new ArrayList<NotifyInfoDetail>();
	
		try {
			RepositoryControllerLocal repository = RepositoryControllerUtil.getLocalHome().create();
			
			if(notify.getNotifyType() ==  NotifyTypeConstant.TYPE_STATUS ){
				Collection<NotifyStatusInfoLocal> ct = notify.getNotifyStatusInfo();
				
			    Iterator<NotifyStatusInfoLocal> itr = ct.iterator();
			    NotifyStatusInfoLocal local = null;
			    while(itr.hasNext()){
			        local = itr.next();
			        NotifyStatusInfo status = new NotifyStatusInfo(
			        		notifyId,
							local.getPriority(),
							local.getValidFlg(),
							local.getStatusInvalidFlg(),
			        		local.getStatusUpdatePriority(),
			        		local.getStatusValidPeriod());
			        		
			        notifyEvents.add(status);
			    }
			
			}else if(notify.getNotifyType() == NotifyTypeConstant.TYPE_EVENT){
				
				Collection<NotifyEventInfoLocal> ct = notify.getNotifyEventInfo();
				
			    Iterator<NotifyEventInfoLocal> itr = ct.iterator();
			    NotifyEventInfoLocal local = null;
			    while(itr.hasNext()){
			        local = itr.next();
			        NotifyEventInfo event = new NotifyEventInfo(
			        		notifyId,
							local.getPriority(),
							local.getValidFlg(),
							local.getEventNormalState());
			        
			        notifyEvents.add(event);
			    }
			
			}else if(notify.getNotifyType() == NotifyTypeConstant.TYPE_MAIL){
				Collection<NotifyMailInfoLocal> ct = notify.getNotifyMailInfo();
				
			    Iterator<NotifyMailInfoLocal> itr = ct.iterator();
			    NotifyMailInfoLocal local = null;
			    while(itr.hasNext()){
			        local = itr.next();
			        NotifyMailInfo status = new NotifyMailInfo(
			        		notifyId,
							local.getPriority(),
							local.getMailTemplateId(),
							local.getValidFlg(),
							local.getMailAddress());
			        		
			        notifyEvents.add(status);
			    }
			
			}else if(notify.getNotifyType() == NotifyTypeConstant.TYPE_JOB){
				Collection<NotifyJobInfoLocal> ct = notify.getNotifyJobInfo();
				
			    Iterator<NotifyJobInfoLocal> itr = ct.iterator();
			    NotifyJobInfoLocal local = null;
			    while(itr.hasNext()){
			        local = itr.next();
			        
			        String facilityPath = repository.getFacilityPath(local.getJobExecFacility(), null);
			        
			        NotifyJobInfo status = new NotifyJobInfo(
			        		notifyId,
							local.getPriority(),
							local.getValidFlg(),
							local.getJobunitId(),
							local.getJobId(),
							local.getJobFailurePriority(),
			        		local.getJobExecFacilityFlg(),
			        		local.getJobExecFacility(),
			        		facilityPath);
							
			        notifyEvents.add(status);
			    }
			    
			}else if(notify.getNotifyType() == NotifyTypeConstant.TYPE_LOG_ESCALATE){
			
				Collection<NotifyLogEscalateInfoLocal> ct = notify.getNotifyLogEscalateInfo();
				
				Iterator<NotifyLogEscalateInfoLocal> itr = ct.iterator();
				NotifyLogEscalateInfoLocal local = null;
				while(itr.hasNext()){
					local = itr.next();
					
					String facilityPath = repository.getFacilityPath(local.getEscalateFacility(), null);
					
					NotifyLogEscalateInfo status = new NotifyLogEscalateInfo(
							notifyId,
							local.getPriority(),
							local.getValidFlg(),
							local.getEscalateMessage(),
							local.getSyslogPriority(),
							local.getSyslogFacility(),
							local.getEscalateFacilityFlg(),
							local.getEscalateFacility(),
							facilityPath,
							local.getEscalatePort());
							
					notifyEvents.add(status);
				}
			}else if(notify.getNotifyType() == NotifyTypeConstant.TYPE_COMMAND){
				
				Collection<NotifyCommandInfoLocal> ct = notify.getNotifyCommandInfo();
				
				for (NotifyCommandInfoLocal local : ct){
					notifyEvents.add(local.getData());
				}
			}
		}
		catch (NamingException e) {
			m_log.error("getNotifyDetails() : " + e.getMessage(), e);
		}
		catch (CreateException e) {
			m_log.error("getNotifyDetails() : " + e.getMessage(), e);
		}
	
	return notifyEvents;
}
	
	/**
	 * 通知ID一覧を返します。
	 * <p>
	 * 通知IDの昇順に並んだ通知ID一覧を返します。
	 * 
	 * @return 通知ID一覧
	 * @throws FinderException
	 * @throws NamingException
	 * 
	 * @see com.clustercontrol.notify.ejb.entity.NotifyInfoBean
	 */
	public ArrayList getNotifyIdList() throws NotifyNotFoundException, NamingException {

		// 戻り値の初期化
		ArrayList<String> list = new ArrayList<String>();
		
		try 
		{
			// 通知情報一覧を取得
			Collection ct = NotifyInfoUtil.getLocalHome().findAllOrderByNotifyId();
			
			Iterator<NotifyInfoLocal> itr = ct.iterator();
			while(itr.hasNext())
			{
				NotifyInfoLocal notify = itr.next();
				list.add(notify.getNotifyId());
			}
		} catch (FinderException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
            apllog.put("SYS", "005");
            m_log.debug("getNotifyList():" + e.getMessage());
			throw new NotifyNotFoundException(e.getMessage(),e);
		} catch (NamingException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
            apllog.put("SYS", "005");
            m_log.debug("getNotifyList():" + e.getMessage());
			throw e;
		}
		return list;
	}

	/**
	 * 通知情報一覧を返します(障害検知用通知を除く)。
	 * <p>
	 * <ol>
	 * <li>通知IDの昇順に並んだ全ての通知情報を取得します。</li>
	 * <li>１通知情報をテーブルのカラム順（{@link com.clustercontrol.notify.bean.NotifyTableDefine}）に、リスト（{@link ArrayList}）にセットします。</li>
	 * <li>この１通知情報を保持するリストを、通知情報一覧を保持するリスト（{@link ArrayList}）に格納し返します。<BR>
	 *  <dl>
	 *  <dt>通知情報一覧（Objectの2次元配列）</dt>
	 *  <dd>{ 通知情報1 {カラム1の値, カラム2の値, … }, 通知情報2{カラム1の値, カラム2の値, …}, … }</dd>
	 *  </dl>
	 * </li>
	 * </ol>
	 * 
	 * @return 通知情報一覧（Objectの2次元配列）
	 * @throws CreateException
	 * @throws FinderException
	 * @throws NamingException
	 * 
	 * @see com.clustercontrol.notify.bean.NotifyTableDefine
	 * @see #collectionToArray(Collection)
	 */
	public ArrayList<NotifyInfoLocal> getNotifyList() throws CreateException, NotifyNotFoundException, NamingException {

		ArrayList<NotifyInfoLocal> list = null;
		try 
		{
			// 通知情報一覧を取得
			Collection<NotifyInfoLocal> ct = NotifyInfoUtil.getLocalHome().findAllOrderByNotifyId();
			Collection<NotifyInfoLocal> rtn = new ArrayList<NotifyInfoLocal>();
			
			// システム機能（障害検知、セルフチェック）の通知IDを除く
			for(NotifyInfoLocal notify : ct){
				String regex1 = "^" + HinemosModuleConstant.SYSYTEM_TROUBLE_DETECTION + ".*";
				String regex2 = "^" + HinemosModuleConstant.SYSYTEM_SELFCHECK + ".*";
					
				// FIXME : 文字列の比較による障害検知用通知IDの特定方法を変更する
				// 結果用Collectionから削除対象以外のものをリストに追加する
				if(! notify.getNotifyId().matches(regex1) && ! notify.getNotifyId().matches(regex2)){
					rtn.add(notify);
				}
			}
			
			// 2次元配列に変換
			list = this.collectionToArray(rtn);
			
		} catch (CreateException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
			apllog.put("SYS", "006");
			m_log.debug("getNotifyList():" + e.getMessage());
			throw e;
		} catch (FinderException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
			apllog.put("SYS", "006");
			m_log.debug("getNotifyList():" + e.getMessage());
			throw new NotifyNotFoundException(e.getMessage(),e);
		} catch (NamingException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
			apllog.put("SYS", "006");
			m_log.debug("getNotifyList():" + e.getMessage());
			throw e;
		}
		return list;
	}
	
	/**
	 * 通知情報マップを返します。
	 * <p>
	 * <ol>
	 * <li>全ての通知情報を取得します。</li>
	 * <li>通知情報マップに、通知情報をセットします。</li>
	 *  <dl>
	 *   <dt>通知情報マップ</dt>
	 *   <dd>キー：通知ID</dd>
	 *   <dd>値　：通知情報</dd>
	 *  </dl>
	 * </li>
	 * </ol>
	 * 
	 * @return 通知情報マップ
	 * @throws FinderException
	 * @throws NamingException
	 */
	public HashMap<String, NotifyInfo> getNotifyMap() throws NotifyNotFoundException, NamingException {

		HashMap<String, NotifyInfo> map = new HashMap<String, NotifyInfo>();
		try 
		{
			// 通知情報一覧を取得
			Collection ct = NotifyInfoUtil.getLocalHome().findAll();
			
			Iterator itr = ct.iterator();
			NotifyInfo bean = null;
			NotifyInfoLocal notify = null;

			while(itr.hasNext()){
				
				notify = (NotifyInfoLocal)itr.next();
			    bean = getNotify(notify);
				
		        map.put(notify.getNotifyId(), bean);
			}
			
		} catch (FinderException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
            apllog.put("SYS", "006");
            m_log.debug("getNotifyMap():" + e.getMessage());
			throw new NotifyNotFoundException(e.getMessage(),e);
		} catch (NamingException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
            apllog.put("SYS", "006");
            m_log.debug("getNotifyMap():" + e.getMessage());
			throw e;
		}
		return map;
	}

	/**
	 * 障害検知用通知情報一覧を返します。
	 * <p>
	 * <ol>
	 * <li>通知IDの昇順に並んだ全ての通知情報を取得します。</li>
	 * <li>１通知情報をテーブルのカラム順（{@link com.clustercontrol.notify.bean.NotifyTableDefine}）に、リスト（{@link ArrayList}）にセットします。</li>
	 * <li>この１通知情報を保持するリストを、通知情報一覧を保持するリスト（{@link ArrayList}）に格納し返します。<BR>
	 *  <dl>
	 *  <dt>通知情報一覧（Objectの2次元配列）</dt>
	 *  <dd>{ 通知情報1 {カラム1の値, カラム2の値, … }, 通知情報2{カラム1の値, カラム2の値, …}, … }</dd>
	 *  </dl>
	 * </li>
	 * </ol>
	 * 
	 * @return 通知情報一覧（Objectの2次元配列）
	 * @throws CreateException
	 * @throws FinderException
	 * @throws NamingException
	 * 
	 * @see com.clustercontrol.notify.bean.NotifyTableDefine
	 * @see #collectionToArray(Collection)
	 */
	public ArrayList<NotifyInfoLocal> getTroubleDetectionNotifyList() throws CreateException, NotifyNotFoundException, NamingException {

		ArrayList<NotifyInfoLocal> list = null;
		try 
		{
			// 通知情報一覧を取得
			Collection<NotifyInfoLocal> ct = NotifyInfoUtil.getLocalHome().findAllOrderByNotifyId();
			Collection<NotifyInfoLocal> rtn = new ArrayList<NotifyInfoLocal>();
			
			if(ct.size() != 0) {
			
				Iterator<NotifyInfoLocal> itr = ct.iterator();
				
				// 障害検知用の通知IDのみを取得
				while(itr.hasNext()){
					
					NotifyInfoLocal notify = itr.next();
					String regex = "^" + HinemosModuleConstant.SYSYTEM_TROUBLE_DETECTION + ".*";
					
					// FIXME : 文字列の比較による障害検知用通知IDの特定方法を変更する
					// 結果用Collectionから対象のCollectionのみリストに加える
					if(notify.getNotifyId().matches(regex)){
						rtn.add(notify);
					}
				}
			}
			
			// 2次元配列に変換
			list = this.collectionToArray(rtn);
			
		} catch (CreateException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
            apllog.put("SYS", "006");
            m_log.debug("getNotifyList():" + e.getMessage());
			throw e;
		} catch (FinderException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
            apllog.put("SYS", "006");
            m_log.debug("getNotifyList():" + e.getMessage());
			throw new NotifyNotFoundException(e.getMessage(),e);
		} catch (NamingException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
            apllog.put("SYS", "006");
            m_log.debug("getNotifyList():" + e.getMessage());
			throw e;
		}
		return list;
	}
	
	/**
	 * 障害検知用通知情報マップを返します。
	 * <p>
	 * <ol>
	 * <li>障害検知用の通知情報を取得します。</li>
	 * <li>通知情報マップに、通知情報をセットします。</li>
	 *  <dl>
	 *   <dt>通知情報マップ</dt>
	 *   <dd>キー：通知ID</dd>
	 *   <dd>値　：通知情報</dd>
	 *  </dl>
	 * </li>
	 * </ol>
	 * 
	 * @return 通知情報マップ
	 * @throws FinderException
	 * @throws NamingException
	 */
	public HashMap<String, NotifyInfo> getTroubleDetectionNotifyMap() throws NotifyNotFoundException, NamingException {

		HashMap<String, NotifyInfo> map = new HashMap<String, NotifyInfo>();
		try 
		{
			// 通知情報一覧を取得
			Collection<NotifyInfoLocal> ct = NotifyInfoUtil.getLocalHome().findByUseType(HinemosModuleConstant.SYSYTEM_TROUBLE_DETECTION+"%");
			
			Iterator<NotifyInfoLocal> itr = ct.iterator();
			NotifyInfo bean = null;
			NotifyInfoLocal notify = null;

			while(itr.hasNext()){
				
				notify = itr.next();
				bean = getNotify(notify);
				
				map.put(notify.getNotifyId(), bean);
			}
			
		} catch (FinderException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
			apllog.put("SYS", "006");
			m_log.debug("getNotifyMap():" + e.getMessage());
			throw new NotifyNotFoundException(e.getMessage(),e);
		} catch (NamingException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
			apllog.put("SYS", "006");
			m_log.debug("getTroubleDetectionNotifyMap():" + e.getMessage());
			throw e;
		}
		return map;
	}
	
	
	/**
	 * セルフチェック用通知情報一覧を返します。
	 * <p>
	 * <ol>
	 * <li>通知IDの昇順に並んだ全ての通知情報を取得します。</li>
	 * <li>１通知情報をテーブルのカラム順（{@link com.clustercontrol.notify.bean.NotifyTableDefine}）に、リスト（{@link ArrayList}）にセットします。</li>
	 * <li>この１通知情報を保持するリストを、通知情報一覧を保持するリスト（{@link ArrayList}）に格納し返します。<BR>
	 *  <dl>
	 *  <dt>通知情報一覧（Objectの2次元配列）</dt>
	 *  <dd>{ 通知情報1 {カラム1の値, カラム2の値, … }, 通知情報2{カラム1の値, カラム2の値, …}, … }</dd>
	 *  </dl>
	 * </li>
	 * </ol>
	 * 
	 * @return 通知情報一覧（Objectの2次元配列）
	 * @throws CreateException
	 * @throws FinderException
	 * @throws NamingException
	 * 
	 * @see com.clustercontrol.notify.bean.NotifyTableDefine
	 * @see #collectionToArray(Collection)
	 */
	public ArrayList<NotifyInfoLocal> getSelfCheckNotifyList() throws CreateException, NotifyNotFoundException, NamingException {

		ArrayList<NotifyInfoLocal> list = null;
		try 
		{
			// 通知情報一覧を取得
			Collection<NotifyInfoLocal> ct = NotifyInfoUtil.getLocalHome().findAllOrderByNotifyId();
			Collection<NotifyInfoLocal> rtn = new ArrayList<NotifyInfoLocal>();
			
			if(ct.size() != 0) {
			
				Iterator<NotifyInfoLocal> itr = ct.iterator();
				
				// セルフチェック用の通知IDのみを取得
				while(itr.hasNext()){
					
					NotifyInfoLocal notify = itr.next();
					String regex = "^" + HinemosModuleConstant.SYSYTEM_SELFCHECK+ ".*";
					
					// FIXME : 文字列の比較による障害検知用通知IDの特定方法を変更する
					// 結果用Collectionから対象のCollectionのみリストに加える
					if(notify.getNotifyId().matches(regex)){
						rtn.add(notify);
					}
				}
			}
			
			// 2次元配列に変換
			list = this.collectionToArray(rtn);
			
		} catch (CreateException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
            apllog.put("SYS", "006");
            m_log.debug("getNotifyList():" + e.getMessage());
			throw e;
		} catch (FinderException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
            apllog.put("SYS", "006");
            m_log.debug("getNotifyList():" + e.getMessage());
			throw new NotifyNotFoundException(e.getMessage(),e);
		} catch (NamingException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
            apllog.put("SYS", "006");
            m_log.debug("getNotifyList():" + e.getMessage());
			throw e;
		}
		return list;
	}
	
	/**
	 * DBより取得した通知情報のObjectの2次元配列を返します。
	 * <p>
	 * <ol>
	 * <li>通知情報をテーブルのカラム順（{@link com.clustercontrol.notify.bean.NotifyTableDefine}）に、リスト（{@link ArrayList}）にセットします。</li>
	 * <li>1つの通知情報を保持するリストを、通知情報一覧を保持するリスト（{@link ArrayList}）に格納します。
	 * <p>
	 *  <dl>
	 *  <dt>通知情報一覧（Objectの2次元配列）</dt>
	 *  <dd>{ 通知情報1 {カラム1の値, カラム2の値, … }, 通知情報2{カラム1の値, カラム2の値, …}, … }</dd>
	 *  </dl>
	 * </li>
	 * </ol>
	 * 
	 * @param ct 通知情報のコレクション
	 * @return 通知情報一覧（Objectの2次元配列）
	 * 
	 * @see com.clustercontrol.notify.bean.NotifyTableDefine
	 */
	@SuppressWarnings("unchecked")
	private ArrayList collectionToArray(Collection ct) throws CreateException, NamingException{
		
		ArrayList list = new ArrayList();
		Iterator itr = ct.iterator();
		while(itr.hasNext())
		{
			NotifyInfoLocal notify = (NotifyInfoLocal)itr.next();
			
			String notifyId = notify.getNotifyId();
			ArrayList info = new ArrayList();
			info.add(YesNoConstant.BOOLEAN_NO);
			info.add(NotifyInfoCache.getVaildFlg(notifyId).intValue());
			info.add(notifyId);
			info.add(NotifyInfoCache.getDescription(notifyId));
			info.add(NotifyInfoCache.getNotifyType(notifyId));
			info.add(NotifyInfoCache.getRegUser(notifyId));
			Date regDate = NotifyInfoCache.getRegDate(notifyId);
			info.add(regDate == null ? null:new Date(regDate.getTime()));
			info.add(NotifyInfoCache.getUpdateUser(notifyId));
			Date updateDate = NotifyInfoCache.getUpdateDate(notifyId);
			info.add(updateDate == null ? null:new Date(updateDate.getTime()));
			list.add(info);
		}
		return list;
	}
}
