/*
 
Copyright (C) 2007 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.maintenance.ejb.session;

import java.rmi.RemoteException;
import java.security.Principal;
import java.sql.SQLException;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Date;
import javax.ejb.CreateException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;
import javax.ejb.SessionBean;
import javax.ejb.SessionContext;
import javax.naming.InitialContext;
import javax.naming.NamingException;
import javax.transaction.HeuristicMixedException;
import javax.transaction.HeuristicRollbackException;
import javax.transaction.InvalidTransactionException;
import javax.transaction.NotSupportedException;
import javax.transaction.RollbackException;
import javax.transaction.SystemException;
import javax.transaction.Transaction;
import javax.transaction.TransactionManager;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.quartz.SchedulerException;

import com.clustercontrol.CalendarNotFountException;
import com.clustercontrol.HinemosUnknownException;
import com.clustercontrol.NotifyNotFoundException;
import com.clustercontrol.calendar.ejb.session.CalendarControllerLocal;
import com.clustercontrol.calendar.ejb.session.CalendarControllerUtil;
import com.clustercontrol.maintenance.bean.MaintenanceInfoDataImp;
import com.clustercontrol.maintenance.bean.ScheduleInfo;
import com.clustercontrol.maintenance.factory.AddMaintenance;
import com.clustercontrol.maintenance.factory.DeleteMaintenance;
import com.clustercontrol.maintenance.factory.MaintenanceCrun;
import com.clustercontrol.maintenance.factory.MaintenanceEvent;
import com.clustercontrol.maintenance.factory.MaintenanceJob;
import com.clustercontrol.maintenance.factory.MaintenanceVM;
import com.clustercontrol.maintenance.factory.ModifyMaintenance;
import com.clustercontrol.maintenance.factory.ModifySchedule;
import com.clustercontrol.maintenance.factory.OperationMaintenance;
import com.clustercontrol.maintenance.factory.SelectMaintenanceInfo;
import com.clustercontrol.maintenance.factory.SelectMaintenanceKeepMst;
import com.clustercontrol.maintenance.factory.SelectMaintenanceTypeMst;
import com.clustercontrol.maintenance.factory.SelectSchedule;
import com.clustercontrol.util.apllog.AplLogger;

/**
 * 
 * メンテナンス機能を管理する Session Bean です。<BR>
 * 
 * @ejb.bean name="MaintenanceController"
 *           display-name="MaintenanceController"
 *           description="MaintenanceController"
 *           jndi-name="MaintenanceController"
 *           type="Stateless"
 *           view-type="remote"
 * 
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=MaintenanceInfoLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=MaintenanceKeepMstLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=MaintenanceTypeMstLocal"
 * 
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Home"
 *     
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="LocalHome"
 *     
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Local"
 *           
 */
public abstract class MaintenanceControllerBean implements SessionBean {

   protected static Log m_log = LogFactory.getLog( MaintenanceControllerBean.class );

	/** コンテキスト情報 */
	@SuppressWarnings("unused")
	private SessionContext m_context;
	
	public MaintenanceControllerBean() {
		super();
	}
	
	/**
	 * メンテナンス情報を追加します。
	 * 
	 * @throws CreateException 
	 * @throws NamingException 
	 * @throws SystemException 
	 * @throws IllegalStateException 
	 * @throws NotSupportedException 
	 * @throws RollbackException 
	 * @throws HeuristicMixedException 
	 * @throws HeuristicRollbackException 
	 * @throws SchedulerException 
	 * @throws ParseException 
	 * @throws RemoteException 
	 * @throws HinemosUnknownException 
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="MaintenanceWrite"
     *     method-intf="Remote"
	 * 
	 */
	public void addMaintenance(MaintenanceInfoDataImp data, ScheduleInfo schedule) throws CreateException, NamingException, IllegalStateException, SystemException, NotSupportedException, RollbackException, HeuristicMixedException, HeuristicRollbackException, SchedulerException, ParseException, RemoteException, HinemosUnknownException {
		m_log.debug("addMaintenance");
		
		Principal pri = m_context.getCallerPrincipal();
		
		TransactionManager tm = null;
		Transaction oldTx = null;
		boolean result = false;
		
		try {
			
			// TransactionManagerを取得
			InitialContext context = new InitialContext();
			tm = (TransactionManager)context.lookup("java:/TransactionManager");
			
			try {
				// 実行中のスレッドがトランザクションに関連付けられている場合は、トランザクションを退避 
				if(tm.getTransaction() != null){
					oldTx = tm.suspend();
				}
				
					// トランザクション開始
					if(tm.getTransaction() == null){
						tm.begin();	
					}
					
					// メンテナンス情報を登録
					AddMaintenance add = new AddMaintenance();
					add.addMaintenance(data, pri.getName());
					// Quartzへ登録
					addSchedule(schedule);
					
					result = true;
					
					
					if(result){
						tm.commit();
					}
			} catch (SecurityException e) {
				m_log.warn("addMaintenance() : " + e.getClass().getSimpleName() +
						", " + e.getMessage());
				m_context.setRollbackOnly();
				throw e;
			} catch (SystemException e) {
				m_log.warn("addMaintenance() : " + e.getClass().getSimpleName() +
						", " + e.getMessage());
				m_context.setRollbackOnly();
				throw e;
			} catch (NotSupportedException e) {
				m_log.warn("addMaintenance() : " + e.getClass().getSimpleName() +
						", " + e.getMessage());
				m_context.setRollbackOnly();
				throw e;
			} catch (RollbackException e) {
				m_log.warn("addMaintenance() : " + e.getClass().getSimpleName() +
						", " + e.getMessage());
				m_context.setRollbackOnly();
				throw e;
			} catch (HeuristicMixedException e) {
				m_log.warn("addMaintenance() : " + e.getClass().getSimpleName() +
						", " + e.getMessage());
				m_context.setRollbackOnly();
				throw e;
			} catch (HeuristicRollbackException e) {
				m_log.warn("addMaintenance() : " + e.getClass().getSimpleName() +
						", " + e.getMessage());
				m_context.setRollbackOnly();
				throw e;
			} catch (HinemosUnknownException e) {
				m_log.warn("addMaintenance() : " + e.getClass().getSimpleName() +
						", " + e.getMessage());
				m_context.setRollbackOnly();
				throw e;
			} finally {
				if(tm.getTransaction() != null){
					if(!result){
						tm.rollback();
					}
				}
			}
				
		} catch (CreateException e) {
			m_log.warn("addMaintenance() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (NamingException e) {
			m_log.warn("addMaintenance() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (SchedulerException e) {
			m_log.warn("addMaintenance() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (ParseException e) {
			m_log.warn("addMaintenance() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (RemoteException e) {
			m_log.warn("addMaintenance() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} finally {
			
			// 一時停止していたトランザクションを再開
			if(oldTx != null){
				try{
					tm.resume(oldTx);
				
				} catch(InvalidTransactionException e){
					throw e;
				} catch(IllegalStateException e){
					throw e;
				} catch(SystemException e){
					throw e;
				}
			}
		}
		
	}
	
	/**
	 * メンテナンス情報を変更します。
	 * 
	 * @throws FinderException 
	 * @throws NamingException 
	 * @throws SystemException 
	 * @throws NotSupportedException 
	 * @throws RollbackException 
	 * @throws HeuristicMixedException 
	 * @throws HeuristicRollbackException 
	 * @throws SchedulerException 
	 * @throws ParseException 
	 * @throws RemoteException 
	 * @throws HinemosUnknownException 
	 * @throws NotifyNotFoundException 
	 * @throws RemoveException 
	 * @throws CreateException 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="MaintenanceWrite"
     *     method-intf="Remote"
	 * 
	 */
	public void modifyMaintenance(MaintenanceInfoDataImp data, ScheduleInfo schedule) throws FinderException, NamingException, SystemException, NotSupportedException, RollbackException, HeuristicMixedException, HeuristicRollbackException, SchedulerException, ParseException, RemoteException, NotifyNotFoundException, HinemosUnknownException {
		m_log.debug("modifyMaintenance");
		
		Principal pri = m_context.getCallerPrincipal();
		
		TransactionManager tm = null;
		Transaction oldTx = null;
		boolean result = false;
		
		try {
			
			// TransactionManagerを取得
			InitialContext context = new InitialContext();
			tm = (TransactionManager)context.lookup("java:/TransactionManager");
			
			try {
				// 実行中のスレッドがトランザクションに関連付けられている場合は、トランザクションを退避 
				if(tm.getTransaction() != null){
					oldTx = tm.suspend();
				}
				
					// トランザクション開始
					if(tm.getTransaction() == null){
						tm.begin();	
					}
					
					// メンテナンス情報を登録
					ModifyMaintenance modify = new ModifyMaintenance();
					modify.modifyMaintenance(data, pri.getName());
					// Quartzへ登録
					modifySchedule(schedule);
					
					result = true;
					
					
					if(result){
						tm.commit();
					}
			} catch (SecurityException e) {
				m_log.warn("modifyMaintenance() : " + e.getClass().getSimpleName() +
						", " + e.getMessage());
				m_context.setRollbackOnly();
				throw e;
			} catch (SystemException e) {
				m_log.warn("modifyMaintenance() : " + e.getClass().getSimpleName() +
						", " + e.getMessage());
				m_context.setRollbackOnly();
				throw e;
			} catch (NotSupportedException e) {
				m_log.warn("modifyMaintenance() : " + e.getClass().getSimpleName() +
						", " + e.getMessage());
				m_context.setRollbackOnly();
				throw e;
			} catch (RollbackException e) {
				m_log.warn("modifyMaintenance() : " + e.getClass().getSimpleName() +
						", " + e.getMessage());
				m_context.setRollbackOnly();
				throw e;
			} catch (HeuristicMixedException e) {
				m_log.warn("modifyMaintenance() : " + e.getClass().getSimpleName() +
						", " + e.getMessage());
				m_context.setRollbackOnly();
				throw e;
			} catch (HeuristicRollbackException e) {
				m_log.warn("modifyMaintenance() : " + e.getClass().getSimpleName() +
						", " + e.getMessage());
				m_context.setRollbackOnly();
				throw e;
			} catch (CreateException e) {
				m_log.warn("modifyMaintenance() : " + e.getClass().getSimpleName() +
						", " + e.getMessage());
				m_context.setRollbackOnly();
				e.printStackTrace();
			} catch (RemoveException e) {
				m_log.warn("modifyMaintenance() : " + e.getClass().getSimpleName() +
						", " + e.getMessage());
				m_context.setRollbackOnly();
				e.printStackTrace();
			} catch (NotifyNotFoundException e) {
				m_log.warn("modifyMaintenance() : " + e.getClass().getSimpleName() +
						", " + e.getMessage());
				m_context.setRollbackOnly();
				throw e;
			} catch (HinemosUnknownException e) {
				m_log.warn("modifyMaintenance() : " + e.getClass().getSimpleName() +
						", " + e.getMessage());
				m_context.setRollbackOnly();
				throw e;
			} catch (FinderException e) {
				m_log.warn("modifyMaintenance() : " + e.getClass().getSimpleName() +
						", " + e.getMessage());
				m_context.setRollbackOnly();
				throw e;
			} finally {
				try {
					if(tm.getTransaction() != null){
						if(!result){
							tm.rollback();
						}
					}
				} catch (IllegalStateException e) {
					m_log.warn("modifyMaintenance() : " + e.getClass().getSimpleName() +
							", " + e.getMessage());
					m_context.setRollbackOnly();
					throw e;
				} catch (SecurityException e) {
					m_log.warn("modifyMaintenance() : " + e.getClass().getSimpleName() +
							", " + e.getMessage());
					m_context.setRollbackOnly();
					throw e;
				} catch (SystemException e) {
					m_log.warn("modifyMaintenance() : " + e.getClass().getSimpleName() +
							", " + e.getMessage());
					m_context.setRollbackOnly();
					throw e;
				}
			}
				
		} catch (NamingException e) {
			m_log.warn("modifyMaintenance() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (SchedulerException e) {
			m_log.warn("modifyMaintenance() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (ParseException e) {
			m_log.warn("modifyMaintenance() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (RemoteException e) {
			m_log.warn("modifyMaintenance() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} finally {
			
			// 一時停止していたトランザクションを再開
			if(oldTx != null){
				try{
					tm.resume(oldTx);
				
				} catch(InvalidTransactionException e){
					m_log.warn("modifyMaintenance() : " + e.getClass().getSimpleName() +
							", " + e.getMessage());
					m_context.setRollbackOnly();
					throw e;
				} catch(IllegalStateException e){
					m_log.warn("modifyMaintenance() : " + e.getClass().getSimpleName() +
							", " + e.getMessage());
					m_context.setRollbackOnly();
					throw e;
				} catch(SystemException e){
					m_log.warn("modifyMaintenance() : " + e.getClass().getSimpleName() +
							", " + e.getMessage());
					m_context.setRollbackOnly();
					throw e;
				}
			}
		}
		
	}
	
	/**
	 * メンテナンス情報を削除します。
	 * 
	 * @throws RemoveException 
	 * @throws NamingException 
	 * @throws SystemException 
	 * @throws NotSupportedException 
	 * @throws RollbackException 
	 * @throws HeuristicMixedException 
	 * @throws HeuristicRollbackException 
	 * @throws SchedulerException 
	 * @throws RemoteException 
	 * @throws CreateException 
	 * @throws FinderException 
	 * @throws EJBException 
	 * @throws HinemosUnknownException 
	 * @throws NotifyNotFoundException 
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="MaintenanceWrite"
     *     method-intf="Remote"
	 * 
	 */
	public void deleteMaintenance(String maintenanceId) throws RemoveException, NamingException, SystemException, NotSupportedException, RollbackException, HeuristicMixedException, HeuristicRollbackException, SchedulerException, RemoteException, EJBException, NotifyNotFoundException, HinemosUnknownException{
		m_log.debug("deleteMaintenance");
		
		TransactionManager tm = null;
		Transaction oldTx = null;
		boolean result = false;
		
		try {
			
			// TransactionManagerを取得
			InitialContext context = new InitialContext();
			tm = (TransactionManager)context.lookup("java:/TransactionManager");
			
			try {
				// 実行中のスレッドがトランザクションに関連付けられている場合は、トランザクションを退避 
				if(tm.getTransaction() != null){
					oldTx = tm.suspend();
				}
				
					// トランザクション開始
					if(tm.getTransaction() == null){
						tm.begin();	
					}
					
					// メンテナンス情報を削除
					DeleteMaintenance delete = new DeleteMaintenance();
					delete.deleteMaintenance(maintenanceId);
					
					deleteSchedule(maintenanceId);

					result = true;
					
					
					if(result){
						tm.commit();
					}
			} catch (SecurityException e) {
				m_log.warn("deleteMaintenance() : " + e.getClass().getSimpleName() +
						", " + e.getMessage());
				m_context.setRollbackOnly();
				throw e;
			} catch (SystemException e) {
				m_log.warn("deleteMaintenance() : " + e.getClass().getSimpleName() +
						", " + e.getMessage());
				m_context.setRollbackOnly();
				throw e;
			} catch (NotSupportedException e) {
				m_log.warn("deleteMaintenance() : " + e.getClass().getSimpleName() +
						", " + e.getMessage());
				m_context.setRollbackOnly();
				throw e;
			} catch (RollbackException e) {
				m_log.warn("deleteMaintenance() : " + e.getClass().getSimpleName() +
						", " + e.getMessage());
				m_context.setRollbackOnly();
				throw e;
			} catch (HeuristicMixedException e) {
				m_log.warn("deleteMaintenance() : " + e.getClass().getSimpleName() +
						", " + e.getMessage());
				m_context.setRollbackOnly();
				throw e;
			} catch (HeuristicRollbackException e) {
				m_log.warn("deleteMaintenance() : " + e.getClass().getSimpleName() +
						", " + e.getMessage());
				m_context.setRollbackOnly();
				throw e;
			} catch (FinderException e) {
				m_log.warn("deleteMaintenance() : " + e.getClass().getSimpleName() +
						", " + e.getMessage());
				m_context.setRollbackOnly();
				e.printStackTrace();
			} catch (CreateException e) {
				m_log.warn("deleteMaintenance() : " + e.getClass().getSimpleName() +
						", " + e.getMessage());
				m_context.setRollbackOnly();
				e.printStackTrace();
			} catch (EJBException e) {
				m_log.warn("deleteMaintenance() : " + e.getClass().getSimpleName() +
						", " + e.getMessage());
				m_context.setRollbackOnly();
				throw e;
			} catch (NotifyNotFoundException e) {
				m_log.warn("deleteMaintenance() : " + e.getClass().getSimpleName() +
						", " + e.getMessage());
				m_context.setRollbackOnly();
				throw e;
			} catch (HinemosUnknownException e) {
				m_log.warn("deleteMaintenance() : " + e.getClass().getSimpleName() +
						", " + e.getMessage());
				m_context.setRollbackOnly();
				throw e;
			} catch (RemoveException e) {
				m_log.warn("deleteMaintenance() : " + e.getClass().getSimpleName() +
						", " + e.getMessage());
				m_context.setRollbackOnly();
				throw e;
			} finally {
				try {
					if(tm.getTransaction() != null){
						if(!result){
							tm.rollback();
						}
					}
				} catch (IllegalStateException e) {
					m_log.warn("deleteMaintenance() : " + e.getClass().getSimpleName() +
							", " + e.getMessage());
					m_context.setRollbackOnly();
					throw e;
				} catch (SecurityException e) {
					m_log.warn("deleteMaintenance() : " + e.getClass().getSimpleName() +
							", " + e.getMessage());
					m_context.setRollbackOnly();
					throw e;
				} catch (SystemException e) {
					m_log.warn("deleteMaintenance() : " + e.getClass().getSimpleName() +
							", " + e.getMessage());
					m_context.setRollbackOnly();
					throw e;
				}
			}
				
		} catch (NamingException e) {
			m_log.warn("deleteMaintenance() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (SchedulerException e) {
			m_log.warn("deleteMaintenance() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (RemoteException e) {
			m_log.warn("deleteMaintenance() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} finally {
			
			// 一時停止していたトランザクションを再開
			if(oldTx != null){
				try{
					tm.resume(oldTx);
				} catch(InvalidTransactionException e){
					m_log.warn("deleteMaintenance() : " + e.getClass().getSimpleName() +
							", " + e.getMessage());
					m_context.setRollbackOnly();
					throw e;
				} catch(IllegalStateException e){
					m_log.warn("deleteMaintenance() : " + e.getClass().getSimpleName() +
							", " + e.getMessage());
					m_context.setRollbackOnly();
					throw e;
				} catch(SystemException e){
					m_log.warn("deleteMaintenance() : " + e.getClass().getSimpleName() +
							", " + e.getMessage());
					m_context.setRollbackOnly();
					throw e;
				}
			}
		}
		
		
	}
	
	
	
	/**
	 * イベントログを削除します。
	 * 
	 * @param keep_id 保存期間
	 * @param status 削除対象のステータス(true=全イベント、false=確認済みイベント)
	 * @return 削除件数
	 * 
	 * @throws CreateException 
	 * @throws NamingException 
	 * @throws FinderException 
	 * @throws SQLException
	 * @throws RemoveException
	 * @throws HinemosUnknownException 
	 * @throws NotifyNotFoundException 
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.transaction
	 *     type="NotSupported"
	 *     
	 * @ejb.permission
     *     unchecked="true"
     *     method-intf="Remote"
     *     
	 */
	public int deleteEventLog(String keep_id, boolean status)  throws FinderException, NamingException, CreateException, SQLException, RemoveException, HinemosUnknownException, NotifyNotFoundException{
		m_log.debug("deleteEventLog() : keep_id = " + keep_id + ", status = " + status);
		MaintenanceEvent event = new MaintenanceEvent();
		
		int ret = 0;
		
		try{
			ret = event.delete(keep_id, status);
		}catch(NamingException e){
			m_log.warn("deleteEventLog() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}catch(CreateException e){
			m_log.warn("deleteEventLog() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}catch(SQLException e){
			m_log.warn("deleteEventLog() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}catch(RemoveException e){
			m_log.warn("deleteEventLog() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}catch(HinemosUnknownException e){
			m_log.warn("deleteEventLog() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}catch(NotifyNotFoundException e){
			m_log.warn("deleteEventLog() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (FinderException e) {
			m_log.warn("deleteEventLog() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}
		
		return ret;
	}

	
	/**
	 * ジョブ実行履歴を削除します。
	 * 
	 * @param keep_id 保存期間
	 * @param status 削除対象のステータス(true=全履歴、false=実行状態が「終了」または「変更済み」の履歴)
	 * @return 削除件数
	 * 
	 * @throws CreateException 
	 * @throws NamingException 
	 * @throws FinderException 
	 * @throws SQLException
	 * @throws RemoveException 
	 * @throws HinemosUnknownException 
	 * @throws NotifyNotFoundException 
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.transaction
	 *     type="NotSupported"
	 *     
	 * @ejb.permission
     *     unchecked="true"
     *     method-intf="Remote"
     *     
	 */
	public int deleteJobHistory(String keep_id, boolean status) throws FinderException, NamingException, CreateException, SQLException, RemoveException, HinemosUnknownException, NotifyNotFoundException {
		m_log.debug("deleteJobHistory() : keep_id = " + keep_id + ", status = " + status);
		MaintenanceJob job = new MaintenanceJob();
		
		int ret = 0;
		
		try{
			ret = job.delete(keep_id, status);
		}catch(NamingException e){
			m_log.warn("deleteEventLog() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}catch(CreateException e){
			m_log.warn("deleteEventLog() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}catch(SQLException e){
			m_log.warn("deleteEventLog() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}catch(RemoveException e){
			m_log.warn("deleteEventLog() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}catch(HinemosUnknownException e){
			m_log.warn("deleteEventLog() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}catch(NotifyNotFoundException e){
			m_log.warn("deleteEventLog() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (FinderException e) {
			m_log.warn("deleteEventLog() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}
		
		return ret;
	}
	
	
	/**
	 * 
	 * 一括制御履歴を削除します。
	 * 
	 * @param keep_id 保存期間
	 * @param status 削除対象のステータス(true=全履歴、false=実行状態が「終了」の履歴)
	 * @return 削除件数
	 * 
	 * @throws CreateException 
	 * @throws NamingException 
	 * @throws FinderException 
	 * @throws SQLException 
	 * @throws RemoveException 
	 * @throws HinemosUnknownException 
	 * @throws NotifyNotFoundException 
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.transaction
	 *     type="NotSupported"
	 *     
	 * @ejb.permission
     *     unchecked="true"
     *     method-intf="Remote"
     *     
	 */	
	public int deleteCollectiveRunHistory(String keep_id, boolean status)  throws FinderException, NamingException, CreateException, SQLException, RemoveException, HinemosUnknownException, NotifyNotFoundException{
		m_log.debug("deleteCollectiveRunHistory() : keep_id = " + keep_id + ", status = " + status);
		MaintenanceCrun crun = new MaintenanceCrun();
		
		int ret = 0;
		
		try{
			ret = crun.delete(keep_id, status);
		}catch(NamingException e){
			m_log.warn("deleteEventLog() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}catch(CreateException e){
			m_log.warn("deleteEventLog() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}catch(SQLException e){
			m_log.warn("deleteEventLog() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}catch(RemoveException e){
			m_log.warn("deleteEventLog() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}catch(HinemosUnknownException e){
			m_log.warn("deleteEventLog() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}catch(NotifyNotFoundException e){
			m_log.warn("deleteEventLog() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (FinderException e) {
			m_log.warn("deleteEventLog() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}
		
		return ret;
	}
	
	/**
	 * 
	 * 仮想化操作ログを削除します。
	 * 
	 * @param keep_id 保存期間
	 * @param status 削除対象のステータス(true=全履歴、false=実行状態が「終了」の履歴)
	 * @return 削除件数
	 * 
	 * @throws CreateException 
	 * @throws NamingException 
	 * @throws FinderException 
	 * @throws SQLException
	 * @throws RemoveException
	 * @throws HinemosUnknownException 
	 * @throws NotifyNotFoundException 
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.transaction
	 *     type="NotSupported"
	 *     
	 * @ejb.permission
     *     unchecked="true"
     *     method-intf="Remote"
     *     
	 */
	public int deleteVmLog(String keep_id, boolean status) throws FinderException, NamingException, CreateException, SQLException, RemoveException, HinemosUnknownException, NotifyNotFoundException {
		m_log.debug("deleteVmLog() : keep_id = " + keep_id + ", status = " + status);
		MaintenanceVM vm = new MaintenanceVM();
		
		int ret = 0;
		
		try{
			ret = vm.delete(keep_id, status);
		}catch(NamingException e){
			m_log.warn("deleteEventLog() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}catch(CreateException e){
			m_log.warn("deleteEventLog() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}catch(SQLException e){
			m_log.warn("deleteEventLog() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}catch(RemoveException e){
			m_log.warn("deleteEventLog() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}catch(HinemosUnknownException e){
			m_log.warn("deleteEventLog() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}catch(NotifyNotFoundException e){
			m_log.warn("deleteEventLog() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (FinderException e) {
			m_log.warn("deleteEventLog() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}
		
		return ret;
	}
	
	/**
	 * 
	 * メンテナンスを実行するためのスケジュール情報を登録します。
	 * 
	 * @throws RemoteException 
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="MaintenanceWrite"
     *     method-intf="Remote"
	 * 
	 */
	public void addSchedule(ScheduleInfo info)  throws NamingException, SchedulerException, ParseException, RemoteException {
	    m_log.debug("addSchedule() : scheduleId=" + info.getId());
	    
	    Principal pri = m_context.getCallerPrincipal();
	    
	    ModifySchedule modify = new ModifySchedule();
	    try {
            modify.addSchedule(info, pri.getName());
        } catch (ParseException e) {
            AplLogger apllog = new AplLogger("MAINTENANCE", "maintenance");
            String[] args = {info.getId()};
            apllog.put("SYS", "004", args);
			m_log.warn("addSchedule() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
            throw e;
        } catch (NamingException e) {
            AplLogger apllog = new AplLogger("MAINTENANCE", "maintenance");
            String[] args = {info.getId()};
            apllog.put("SYS", "004", args);
			m_log.warn("addSchedule() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
            throw e;
        } catch (SchedulerException e) {
            AplLogger apllog = new AplLogger("MAINTENANCE", "maintenance");
            String[] args = {info.getId()};
            apllog.put("SYS", "004", args);
			m_log.warn("addSchedule() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
            throw e;
        } catch (RemoteException e) {
            AplLogger apllog = new AplLogger("MAINTENANCE", "maintenance");
            String[] args = {info.getId()};
            apllog.put("SYS", "004", args);
			m_log.warn("addSchedule() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
            throw e;
		}
	}
	
	/**
	 * 
	 * メンテナンスを実行するためのスケジュール情報を修正します。
	 * @throws RemoteException 
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="MaintenanceWrite"
     *     method-intf="Remote"
	 * 
	 */
	public void modifySchedule(ScheduleInfo info)  throws NamingException, SchedulerException, ParseException, RemoteException{
	    m_log.debug("modifySchedule() : scheduleId=" + info.getId());
	    
	    Principal pri = m_context.getCallerPrincipal();
	    
	    ModifySchedule modify = new ModifySchedule();
	    try {
            modify.modifySchedule(info, pri.getName());
        } catch (NamingException e) {
            AplLogger apllog = new AplLogger("MAINTENANCE", "maintenance");
            String[] args = {info.getId()};
            apllog.put("SYS", "006", args);
			m_log.warn("modifySchedule() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
            throw e;
        } catch (SchedulerException e) {
            AplLogger apllog = new AplLogger("MAINTENANCE", "maintenance");
            String[] args = {info.getId()};
            apllog.put("SYS", "006", args);
			m_log.warn("modifySchedule() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
            throw e;
        } catch (ParseException e) {
            AplLogger apllog = new AplLogger("MAINTENANCE", "maintenance");
            String[] args = {info.getId()};
            apllog.put("SYS", "006", args);
			m_log.warn("modifySchedule() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
            throw e;
        } catch (RemoteException e) {
            AplLogger apllog = new AplLogger("MAINTENANCE", "maintenance");
            String[] args = {info.getId()};
            apllog.put("SYS", "006", args);
			m_log.warn("modifySchedule() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
            throw e;
		}
		
	}
	
	/**
	 * 
	 * メンテナンスを実行するためのスケジュール情報を削除します。
	 * @throws RemoteException 
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="MaintenanceWrite"
     *     method-intf="Remote"
	 * 
	 */
	public void deleteSchedule(String scheduleId) throws NamingException, SchedulerException, RemoteException {
		m_log.debug("deleteSchedule() : scheduleId=" + scheduleId);
	    
	    ModifySchedule modify = new ModifySchedule();
	    try {
            modify.deleteSchedule(scheduleId);
        } catch (NamingException e) {
            AplLogger apllog = new AplLogger("MAINTENANCE", "maintenance");
            String[] args = {scheduleId};
            apllog.put("SYS", "005", args);
			m_log.warn("deleteSchedule() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
            throw e;
        } catch (SchedulerException e) {
            AplLogger apllog = new AplLogger("MAINTENANCE", "maintenance");
            String[] args = {scheduleId};
            apllog.put("SYS", "005", args);
			m_log.warn("deleteSchedule() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
             throw e;
        } catch (RemoteException e) {
            AplLogger apllog = new AplLogger("MAINTENANCE", "maintenance");
            String[] args = {scheduleId};
            apllog.put("SYS", "005", args);
			m_log.warn("deleteSchedule() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
             throw e;
		}
	}
	
	/**
	 * メンテナンス情報を取得します。
	 *
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *    role-name="MaintenanceRead"
     *    method-intf="Remote"
	 * 
	 * @return 
	 * 
	 * @throws CreateException 
	 * @throws NamingException 
	 * @throws FinderException 
	 * @throws HinemosUnknownException 
	 * @throws NotifyNotFoundException 
	 */
	public MaintenanceInfoDataImp getMaintenanceInfo(String maintenanceId) throws FinderException, NamingException, CreateException, NotifyNotFoundException, HinemosUnknownException {
		m_log.debug("getMaintenanceInfoData()");
		
		SelectMaintenanceInfo select = new SelectMaintenanceInfo();
		MaintenanceInfoDataImp info = select.getMaintenanceInfo(maintenanceId);
		
		return info;
		
	}
	
	/**
	 * メンテナンス情報の一覧を取得します。<BR>
	 * 下記のようにして生成されるArrayListを、要素として持つArrayListが一覧として返されます。
	 * 
	 * <p>
	 * MaintenanceInfoLocal maintenance = (MaintenanceInfoLocal)itr.next();
	 * 
	 * ArrayList info = new ArrayList();
	 * info.add(maintenance.getMaintenance_id());
	 * info.add(maintenance.getDescription());
	 * info.add(maintenance.getType_id());
	 * info.add(maintenance.getKeep_id());
	 * info.add(maintenance.getCalendar_id());
	 * info.add(new Date(maintenance.getSchedule_date().getTime()));
	 * info.add(maintenance.getValid_flg().intValue());
	 * info.add(maintenance.getRegUser());
	 * info.add(new Date(maintenance.getRegDate().getTime()));
	 * info.add(maintenance.getUpdateUser());
	 * info.add(new Date(maintenance.getUpdateDate().getTime()));
	 * list.add(info);
	 * </p>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="MaintenanceRead"
     *     method-intf="Remote"
	 * 
	 * @return メンテナンス情報の一覧を保持するArrayList
	 * 
	 * @throws CreateException 
	 * @throws NamingException 
	 * @throws FinderException 
	 * @throws SchedulerException 
	 */
	public ArrayList getMaintenanceList() throws FinderException, NamingException, CreateException, SchedulerException {
		m_log.debug("getMaintenanceList()");
		
		SelectMaintenanceInfo select = new SelectMaintenanceInfo();
		ArrayList list = select.getMaintenanceList();
		
		return list;
		
	}
	
	/**
	 * メンテナンス種別の一覧を取得します。<BR>
	 * 下記のようにして生成されるArrayListを、要素として持つArrayListが一覧として返されます。
	 * 
	 * <p>
	 * MaintenanceTypeMstLocal mst = (MaintenanceTypeMstLocal)itr.next();
	 * ArrayList info = new ArrayList();
	 * info.add(mst.getType_id());
	 * info.add(mst.getName_id());
	 * info.add(mst.getOrder_no());
	 * ist.add(info);
	 * </p>
	 * 
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="MaintenanceRead"
	 *     method-intf="Remote"
	 * 
	 * @return メンテナンス種別の一覧を保持するArrayList
	 * @throws CreateException 
	 * @throws NamingException 
	 * @throws FinderException 
	 */
	public ArrayList getMaintenanceTypeList() throws FinderException, NamingException, CreateException {
		m_log.debug("getMaintenanceTypeList()");
		
		SelectMaintenanceTypeMst select = new SelectMaintenanceTypeMst();
		ArrayList list = select.getMaintenanceTypeMst();
		
		return list;
		
	}
	
	/**
	 * 保存期間の表示用文字列の一覧を取得します。<BR>
	 * 下記のようにして生成されるArrayListを、要素として持つArrayListが一覧として返されます。
	 * 
	 * <p>
	 * MaintenanceKeepMstLocal mst = (MaintenanceKeepMstLocal)itr.next();
	 * list.add(mst.getName_id());
	 * </p>
	 * 
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="MaintenanceRead"
	 *     method-intf="Remote"
	 * 
	 * @return
	 * @throws CreateException 
	 * @throws NamingException 
	 * @throws FinderException 
	 */
	public ArrayList getMaintenanceKeepList() throws FinderException, NamingException, CreateException {
		m_log.debug("getMaintenanceKeepList()");
		
		SelectMaintenanceKeepMst select = new SelectMaintenanceKeepMst();
		ArrayList list = select.getMaintenanceKeepMst();
		
		return list;
		
	}
	
	
	/**
	 * メンテナンス情報に設定されている、スケジュールを取得します。<BR>
	 * スケジュール情報はArrayListに格納されて返されます。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     unchecked="MaintenanceRead"
     *     method-intf="Remote"
     * 
     * @param maintenanceId メンテナンスID
	 * 
	 * @return ArrayListに格納されたスケジュール情報
	 * 
	 * @see com.clustercontrol.maintenance.factory.SelectSchedule#getSchedule(String)
	 * 
	 * @throws NamingException
	 * @throws SchedulerException
	 */
	public ArrayList getSchedule(String maintenanceId) throws NamingException, SchedulerException {
	    m_log.debug("getSchedule()");
	    
	    SelectSchedule select = new SelectSchedule();
	    //FIXME FinderException以外はそのままthrow
//	    Schedule schedule = select.getSchedule(maintenanceId);
	    ArrayList info = select.getSchedule(maintenanceId);

	    return info;
	}
	
	/**
	 * スケジュール情報の一覧を取得します。<BR>
	 * スケジュール情報はArrayListで表現され、これを要素としてもつArrayListが返されます。
	 *  
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     unchecked="MaintenanceRead"
     *     method-intf="Remote"
	 * 
	 * @return スケジュール情報(それぞれがArrayList)が格納されたArrayList
	 * 
	 * @see com.clustercontrol.maintenance.factory.SelectSchedule#getScheduleList()
	 * 
	 * @throws NamingException
	 * @throws SchedulerException
	 */
	public ArrayList getScheduleList() throws NamingException, SchedulerException {
	    m_log.debug("getScheduleList()");
	    
	    SelectSchedule select = new SelectSchedule();
	    //FIXME FinderException以外はそのままthrow
	    ArrayList list = select.getScheduleList();

	    return list;
	}
	
	
	/**
	 * 
	 * メンテナンス機能をスケジュール実行します。<BR>
	 * Quartzからスケジュール実行時に呼び出されます。
	 * @throws HinemosUnknownException 
	 * @throws NotifyNotFoundException 
	 * @ejb.interface-method
	 * 
	 * @ejb.transaction
	 *     type="NotSupported"
	 *     
	 * @ejb.permission
     *     unchecked="true"
     *     method-intf="Remote"
	 */
	public void scheduleRunMaintenance(String maintenanceId, String calendarId) throws HinemosUnknownException, NotifyNotFoundException {
	    m_log.debug("scheduleRunMaintenance() : maintenanceId=" + maintenanceId + ", calendarId=" + calendarId);
	    
    	//カレンダをチェック
	    try {
			boolean check = false;
			if(calendarId != null && calendarId.length() > 0){
				CalendarControllerLocal calendar = 
					CalendarControllerUtil.getLocalHome().create();
					
				//カレンダによる実行可/不可のチェック
				if(calendar.isRun(calendarId, new Date()).booleanValue()){
					check = true;
				}
			}
			else{
				check = true;
			}
			
			if(!check)
				return;
			
			//メンテナンス実行
			runMaintenance(maintenanceId);
		} catch (CreateException e) {
			m_log.error("scheduleRunMaintainance() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw new HinemosUnknownException(e.getMessage(),e);
		} catch (NamingException e) {
			m_log.error("scheduleRunMaintainance() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw new HinemosUnknownException(e.getMessage(),e);
		} catch (CalendarNotFountException e) {
			m_log.error("scheduleRunMaintainance() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw new HinemosUnknownException(e.getMessage(),e);
		} catch (HinemosUnknownException e) {
			m_log.error("scheduleRunMaintainance() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw e;
		} catch (NotifyNotFoundException e) {
			m_log.error("scheduleRunMaintainance() : " + e.getClass().getSimpleName() +
					", " + e.getMessage(), e);
			m_context.setRollbackOnly();
			throw e;
		}
	}
	
	/**
	 * 
	 * メンテナンス機能を実行するメソッドです。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.transaction
	 *     type="NotSupported"
	 *     
	 * @ejb.permission
     *     unchecked="true"
     *     method-intf="Remote"
	 * 
	 * @param maintenanceId
	 * @throws HinemosUnknownException 
	 * @throws NotifyNotFoundException 
	 */
	public void runMaintenance(String maintenanceId) throws HinemosUnknownException, NotifyNotFoundException {
		m_log.debug("runMaintenance() : maintenanceId=" + maintenanceId);
		
		OperationMaintenance operation = new OperationMaintenance();
		try {
			operation.runMaintenance(maintenanceId);
		} catch (HinemosUnknownException e) {
			m_log.warn("scheduleRunMaintainance() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (NotifyNotFoundException e) {
			m_log.warn("scheduleRunMaintainance() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}
	}
	
	
	/**
	 * コンテキスト情報を設定します。<BR>
	 * Session Bean がインスタンスプールに格納される際に行う処理を実装します。
	 * 
	 * @see javax.ejb.SessionBean#setSessionContext(javax.ejb.SessionContext)
	 */
	public void setSessionContext(SessionContext ctx) throws EJBException, RemoteException {
		m_context = ctx;
	}
	
	
	public void ejbRemove() throws EJBException, RemoteException {}

	public void ejbActivate() throws EJBException, RemoteException {}

	public void ejbPassivate() throws EJBException, RemoteException {}

}
