/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.logtransfer.ejb.session;

import java.rmi.RemoteException;
import java.security.Principal;
import java.util.ArrayList;
import java.util.Collection;
import javax.ejb.CreateException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;
import javax.ejb.SessionContext;
import javax.jms.JMSException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.UsedFacilityException;
import com.clustercontrol.bean.PluginConstant;
import com.clustercontrol.commons.ejb.session.CheckFacility;
import com.clustercontrol.logtransfer.bean.LogTransferInfo;
import com.clustercontrol.logtransfer.ejb.entity.LogTransferInfoUtil;
import com.clustercontrol.logtransfer.factory.AddLogTransfer;
import com.clustercontrol.logtransfer.factory.DeleteLogTransfer;
import com.clustercontrol.logtransfer.factory.ModifyLogTransfer;
import com.clustercontrol.logtransfer.factory.SelectLogTransfer;

/**
 * ログ転送情報を制御するSeassion Bean <BR>
 *
 * <!-- begin-xdoclet-definition --> 
 * @ejb.bean name="LogTransferController"	
 *           jndi-name="LogTransferController"
 *           type="Stateless"
 *           transaction-type="Container"
 * 
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=LogTransferInfoLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=LogTransferFileInfoLocal"
 * 
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Home"
 *     
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="LocalHome"
 *     
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Local"
 * 
 *            
 * <!-- end-xdoclet-definition --> 
 * @generated
*/
public abstract class LogTransferControllerBean implements javax.ejb.SessionBean, CheckFacility {

	protected static Log m_log = LogFactory.getLog( LogTransferControllerBean.class );
	
	@SuppressWarnings("unused")
	private SessionContext m_context;
	
	
	/**
	 * セッションコンテキストを取得します。<BR>
	 */
	public void setSessionContext(SessionContext ctx) throws EJBException, RemoteException {
		m_context = ctx;
	}
	
	/**
	 * ログ転送情報をマネージャに登録します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="LogTransferWrite"
     *     method-intf="Remote"
	 * 
	 * @param info ログ転送情報
	 * @return 登録に成功した場合、true
	 * @throws CreateException
	 * @throws NamingException
	 * @throws JMSException
	 * @see com.clustercontrol.logtransfer.bean.LogTransferInfo
	 */
	public boolean addLogTransfer(LogTransferInfo info) throws CreateException, NamingException, JMSException {
		
		Principal pri = m_context.getCallerPrincipal();

		// ログ転送情報を登録
		AddLogTransfer logTransfer = new AddLogTransfer();
		
		boolean ret = false;
		
		try{
			ret = logTransfer.add(info, pri.getName());
		}catch(CreateException e){
			m_log.warn("addLogTransfer() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}catch(NamingException e){
			m_log.warn("addLogTransfer() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}
		
		return ret;
	}
	
	/**
	 * マネージャ上のログ転送情報を更新します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="LogTransferWrite"
     *     method-intf="Remote"
	 * 
	 * @param info ログ転送情報
	 * @return 変更に成功した場合、true
	 * @throws CreateException
	 * @throws FinderException
	 * @throws RemoveException
	 * @throws NamingException
	 * @throws JMSException
	 * 
	 * @see com.clustercontrol.logtransfer.bean.LogTransferInfo
	 */
	public boolean modifyLogTransfer(LogTransferInfo info) throws CreateException, FinderException, RemoveException, NamingException, JMSException {
		
		Principal pri = m_context.getCallerPrincipal();

		// ログ転送情報を更新
		ModifyLogTransfer logTransfer = new ModifyLogTransfer();
		
		boolean ret = false;
		
		try{
			ret = logTransfer.modify(info, pri.getName());
		}catch(CreateException e){
			m_log.warn("modifyLogTransfer() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}catch(NamingException e){
			m_log.warn("modifyLogTransfer() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (FinderException e) {
			m_log.warn("modifyLogTransfer() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		} catch (RemoveException e) {
			m_log.warn("modifyLogTransfer() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}
		
		return ret;
	}
	
	/**
	 * ログ転送情報をマネージャから削除します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="LogTransferWrite"
     *     method-intf="Remote"
	 * 
	 * @param transferId ログ転送ID
	 * @return 削除に成功した場合、true
	 * @throws FinderException
	 * @throws RemoveException
	 * @throws NamingException
	 * @throws JMSException
	 */
	public boolean deleteLogTransfer(String transferId) throws FinderException, RemoveException, NamingException, JMSException {

		// ログ転送情報を削除
		DeleteLogTransfer logTransfer = new DeleteLogTransfer();
		
		boolean ret = false;
		
		try{
			ret = logTransfer.delete(transferId);
		}catch(FinderException e){
			m_log.warn("deleteLogTransfer() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}catch(RemoveException e){
			m_log.warn("deleteLogTransfer() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}catch(NamingException e){
			m_log.warn("deleteLogTransfer() : " + e.getClass().getSimpleName() +
					", " + e.getMessage());
			m_context.setRollbackOnly();
			throw e;
		}
		
		return ret;
	}
	
	/**
	 * ログ転送情報を取得します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="LogTransferRead"
     *     method-intf="Remote"
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @param transferId ログ転送ID
	 * @return ログ転送情報
	 * @throws CreateException
	 * @throws FinderException
	 * @throws NamingException
	 * @see com.clustercontrol.logtransfer.bean.LogTransferInfo 
	 */
	public LogTransferInfo getLogTransfer(String transferId) throws CreateException, FinderException, NamingException {

		// ログ転送情報を取得
		SelectLogTransfer logTransfer = new SelectLogTransfer();
		LogTransferInfo info = logTransfer.getLogTransfer(transferId);

		return info;
	}
	
	/**
	 * ログ転送一覧リストを取得します。<BR>
	 * 
	 * 戻り値はArrayListのArrayListで内部のArrayListは下記のように
	 * セットされる。
	 * 
	 * <Pre>
	 * ArrayList info = new ArrayList();
	 *  info.add(logTransfer.getTransferId());
	 *  info.add(logTransfer.getDescription());
	 *  info.add(facilityPath);
	 *  info.add(logTransfer.getValidFlg().intValue());
	 *  info.add(logTransfer.getRegUser());
	 *  info.add(logTransfer.getRegDate() == null ? null:new Date(logTransfer.getRegDate().getTime()));
	 *  info.add(logTransfer.getUpdateUser());
	 *  info.add(logTransfer.getUpdateDate() == null ? null:new Date(logTransfer.getUpdateDate().getTime()));
	 *
	 *  list.add(info);
	 * </Pre>
	 * 
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="LogTransferRead"
     *     method-intf="Remote"
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @return Objectの2次元配列
	 * @throws CreateException
	 * @throws FinderException
	 * @throws NamingException
	 */
	public ArrayList getLogTransferList() throws CreateException, FinderException, NamingException {

		// ログ転送一覧を取得
		SelectLogTransfer logTransfer = new SelectLogTransfer();
		ArrayList list = logTransfer.getLogTransferList();

		return list;
	}
	
	
	/**
	 * ファシリティIDが使用されているかチェックします。<BR>
	 * <P>
	 * 使用されていたら、UsedFacilityException がスローされる。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="LogTransferRead"
	 *     method-intf="Remote"
	 *     
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @param facilityId ファシリティ
	 * @throws UsedFacilityException
	 * 
	 * @see com.clustercontrol.commons.ejb.session.CheckFacility#isUseFacilityId(java.lang.String)
	 * @see com.clustercontrol.bean.PluginConstant;
	 */
	public void isUseFacilityId(String facilityId) throws UsedFacilityException {
		try {
			// ファシリティIDが監視対象になっている転送情報を取得する。
			Collection ct = LogTransferInfoUtil.getLocalHome().findByFacilityId(facilityId);
			if(ct != null && ct.size() > 0) {
				throw new UsedFacilityException(PluginConstant.TYPE_LOG_TRANSFER);
			}
		} catch (FinderException e) {
			m_log.debug("isUseFacilityId() : " + e.getMessage());
		} catch (NamingException e) {
			m_log.debug("isUseFacilityId() : " + e.getMessage());
		}
	}
}
