/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.jobmanagement.factory;

import java.io.File;
import java.io.FileInputStream;
import java.sql.Time;
import java.text.DecimalFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Properties;

import javax.ejb.CreateException;
import javax.ejb.DuplicateKeyException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;

import javax.ejb.RemoveException;
import javax.jms.DeliveryMode;
import javax.jms.JMSException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.CalendarNotFountException;
import com.clustercontrol.FacilityNotFoundException;
import com.clustercontrol.JobInfoNotFoundException;
import com.clustercontrol.JobMasterNotFoundException;
import com.clustercontrol.bean.EndStatusConstant;
import com.clustercontrol.bean.JobConstant;
import com.clustercontrol.bean.JobParamTypeConstant;
import com.clustercontrol.bean.JudgmentObjectConstant;
import com.clustercontrol.bean.OutputBasicInfo;
import com.clustercontrol.bean.ProcessingMethodConstant;
import com.clustercontrol.bean.StatusConstant;
import com.clustercontrol.bean.YesNoConstant;
import com.clustercontrol.calendar.ejb.session.CalendarControllerLocal;
import com.clustercontrol.calendar.ejb.session.CalendarControllerUtil;
import com.clustercontrol.commons.util.SendQueue;
import com.clustercontrol.jobmanagement.bean.AgentCheckConstant;
import com.clustercontrol.jobmanagement.bean.CommandConstant;
import com.clustercontrol.jobmanagement.bean.CommandTypeConstant;
import com.clustercontrol.jobmanagement.bean.ConditionTypeConstant;
import com.clustercontrol.jobmanagement.bean.DelayNotifyConstant;
import com.clustercontrol.jobmanagement.bean.EndStatusCheckConstant;
import com.clustercontrol.jobmanagement.bean.JobTriggerInfo;
import com.clustercontrol.jobmanagement.bean.OperationConstant;
import com.clustercontrol.jobmanagement.bean.QueueConstant;
import com.clustercontrol.jobmanagement.bean.RunStatusConstant;
import com.clustercontrol.jobmanagement.bean.SystemParameterConstant;
import com.clustercontrol.jobmanagement.bean.TopicConstant;
import com.clustercontrol.jobmanagement.ejb.entity.JobCommandInfoLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobCommandInfoPK;
import com.clustercontrol.jobmanagement.ejb.entity.JobCommandInfoUtil;
import com.clustercontrol.jobmanagement.ejb.entity.JobCommandMasterLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobEndInfoLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobEndInfoPK;
import com.clustercontrol.jobmanagement.ejb.entity.JobEndInfoUtil;
import com.clustercontrol.jobmanagement.ejb.entity.JobEndMasterLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobFileInfoLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobFileInfoUtil;
import com.clustercontrol.jobmanagement.ejb.entity.JobFileMasterLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobInfoLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobInfoUtil;
import com.clustercontrol.jobmanagement.ejb.entity.JobMasterLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobMasterPK;
import com.clustercontrol.jobmanagement.ejb.entity.JobMasterUtil;
import com.clustercontrol.jobmanagement.ejb.entity.JobNoticeInfoLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobNoticeInfoUtil;
import com.clustercontrol.jobmanagement.ejb.entity.JobNoticeMasterLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobParamInfoUtil;
import com.clustercontrol.jobmanagement.ejb.entity.JobParamMasterLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobParamMasterUtil;
import com.clustercontrol.jobmanagement.ejb.entity.JobRelationInfoLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobRelationInfoUtil;
import com.clustercontrol.jobmanagement.ejb.entity.JobRelationMasterLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobRelationMasterUtil;
import com.clustercontrol.jobmanagement.ejb.entity.JobSessionJobLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobSessionJobPK;
import com.clustercontrol.jobmanagement.ejb.entity.JobSessionJobUtil;
import com.clustercontrol.jobmanagement.ejb.entity.JobSessionLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobSessionNodeLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobSessionNodePK;
import com.clustercontrol.jobmanagement.ejb.entity.JobSessionNodeUtil;
import com.clustercontrol.jobmanagement.ejb.entity.JobSessionPK;
import com.clustercontrol.jobmanagement.ejb.entity.JobSessionUtil;
import com.clustercontrol.jobmanagement.ejb.entity.JobStartInfoLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobStartInfoUtil;
import com.clustercontrol.jobmanagement.ejb.entity.JobStartJobInfoLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobStartJobInfoUtil;
import com.clustercontrol.jobmanagement.ejb.entity.JobStartJobMasterLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobStartMasterLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobStartTimeInfoLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobStartTimeInfoUtil;
import com.clustercontrol.jobmanagement.ejb.entity.JobStartTimeMasterLocal;
import com.clustercontrol.jobmanagement.message.RunInstructionInfo;
import com.clustercontrol.jobmanagement.message.RunResultInfo;
import com.clustercontrol.jobmanagement.util.ParameterUtil;
import com.clustercontrol.jobmanagement.util.SendTopic;
import com.clustercontrol.notify.bean.NotifyRelationInfo;
import com.clustercontrol.notify.bean.NotifyRequestMessage;
import com.clustercontrol.notify.ejb.entity.NotifyRelationInfoLocal;
import com.clustercontrol.notify.ejb.entity.NotifyRelationInfoUtil;
import com.clustercontrol.repository.bean.FacilityAttributeConstant;
import com.clustercontrol.repository.ejb.session.RepositoryControllerLocal;
import com.clustercontrol.repository.ejb.session.RepositoryControllerUtil;

/**
 * ジョブ操作を行うクラスです。
 *
 * @version 3.0.0
 * @since 1.0.0
 */
public class OperationJob {
	/** ログ出力のインスタンス */
	protected static Log m_log = LogFactory.getLog( OperationJob.class );
	/** メッセージ送信リトライ回数 */
	protected static Integer m_messageRetry = null;
	/** メッセージ送信タイムアウト間隔 */
	protected static Integer m_messageTimeout = null;
	
	/**
	 * コンストラクタ
	 * <P>
	 * ジョブ管理用プロパティファイルよりメッセージ送信リトライ回数とメッセージ送信タイムアウト間隔を取得します。
	 */
	public OperationJob(){
		if(m_messageRetry == null || m_messageTimeout == null){
		    Properties properties = new Properties();
		    String etcdir = System.getProperty("hinemos.manager.etc.dir");
			String propertyFile = etcdir + File.separator + "hinemos.properties";
			
			if(m_messageRetry == null){
				try {
					// プロパティファイルからキーと値のリストを読み込みます
					properties.load(new FileInputStream(propertyFile));
					
					String retry = properties.getProperty("job.message.retry", "false");
					m_messageRetry = Integer.valueOf(retry);
					
				} catch (Exception e) {
					m_log.error("OperationJob():" + e.getMessage());
					m_messageRetry = Integer.valueOf(5);
				}
			}
			
			if(m_messageTimeout == null){
				try {
					// プロパティファイルからキーと値のリストを読み込みます
					properties.load(new FileInputStream(propertyFile));
					
					String timeout = properties.getProperty("job.message.timeout", "false");
					m_messageTimeout = Integer.valueOf(timeout);
					
				} catch (Exception e) {
					m_log.error("SelectJob():" + e.getMessage());
					m_messageTimeout = Integer.valueOf(60);
				}
			}
		}
	}
	
	/**
	 * ジョブを実行します。
	 * 
	 * @throws NamingException
	 * @throws JobInfoNotFoundException 
	 */
	public void runJob(String sessionId) throws NamingException, JobInfoNotFoundException {
		m_log.debug("runJob() start : sessionid = " + sessionId);
		
		_runJob(sessionId);
		_runWaitJob(sessionId);
		
	}
	
	/**
	 * 未終了のセッションID一覧を取得します。
	 * 
	 * @return
	 * @throws NamingException
	 * @throws JobInfoNotFoundException 
	 */
	public ArrayList<String> getUnendSessionList() throws NamingException, JobInfoNotFoundException {
		m_log.debug("getUnendSessionList() start");
		
		ArrayList<String> list = new ArrayList<String>();
		
		Collection collection = null;
		try {
			collection = JobSessionUtil.getLocalHome().findUnendSessions();
		} catch (FinderException e) {
			m_log.warn("JobSessionUtil.getLocalHome().findUnendSessions() : " + e.getMessage());
			JobInfoNotFoundException je = new JobInfoNotFoundException(e.getMessage(), e);
			throw je;
		}
		
		if(collection != null && collection.size() > 0){
			Iterator itr = collection.iterator();
			while(itr.hasNext()){
				JobSessionLocal session = (JobSessionLocal)itr.next();
				m_log.debug("getUnendSessionList() target sessionid is " + session.getSession_id());				
				list.add(session.getSession_id());
			}
		}

		return list;
	}
	
	
	
	/**
	 * ジョブを実行します。
	 * 
	 * @param sessionId セッションID
	 * @param jobunitId 所属ジョブユニットのジョブID
	 * @param jobId ジョブID
	 * @throws NamingException
	 * @throws JobInfoNotFoundException 
	 * 
	 * @see com.clustercontrol.bean.EndStatusConstant
	 * @see com.clustercontrol.bean.JobConstant
	 * @see com.clustercontrol.bean.StatusConstant
	 * @see com.clustercontrol.jobmanagement.factory.OperationJob#startJob(String, String, String)
	 * @see com.clustercontrol.jobmanagement.factory.OperationJob#startJobMain1(String, String, String, String)
	 * @see com.clustercontrol.jobmanagement.factory.Notice#notify(String, String, Integer)
	 */
	public void runJob(String sessionId, String jobunitId, String jobId) throws NamingException, JobInfoNotFoundException {
		m_log.debug("runJob() : sessionId=" + sessionId + ", jobId=" + jobId);
		
		//セッションIDとジョブIDから、セッションジョブを取得
		JobSessionJobLocal sessionJob = getJobSessionJobLocal(sessionId, jobunitId, jobId);
		
		//実行状態を実行中にする
		sessionJob.setStatus(Integer.valueOf(StatusConstant.TYPE_RUNNING));
		
		//開始・再実行日時を設定
		sessionJob.setStart_date(new Date());
		
		//通知処理
		new Notice().notify(sessionId, jobunitId, jobId, Integer.valueOf(EndStatusConstant.TYPE_BEGINNING));
		
		//ジョブを取得
		JobInfoLocal job = sessionJob.getJobInfo();
		int type = job.getJob_type().intValue();
		sessionJob = null;
		job = null;
		if(type == JobConstant.TYPE_JOBUNIT){
			//ジョブユニットの場合、ジョブ開始処理を行う
			startJob(sessionId, jobunitId, jobId);
		}
		else{
			//ジョブネット、ジョブの場合、ジョブ開始処理メイン1を行う
			startJobMain1(sessionId, jobunitId, jobId, null);
		}
	}
	
	/**
	 * ジョブを実行します。
	 * 
	 * @throws NamingException
	 * @throws JobInfoNotFoundException 
	 * 
	 * @see com.clustercontrol.bean.JobConstant
	 * @see com.clustercontrol.bean.StatusConstant
	 * @see com.clustercontrol.jobmanagement.factory.OperationJob#checkEndStatus(String, String)
	 * @see com.clustercontrol.jobmanagement.factory.OperationJob#setEndStaus(String, String, Integer, Integer, String)
	 * @see com.clustercontrol.jobmanagement.factory.OperationJob#startJob(String, String)
	 * @see com.clustercontrol.jobmanagement.factory.OperationJob#startJobMain1(String, String, String)
	 * @see com.clustercontrol.jobmanagement.factory.Notice#notify(String, String, Integer)
	 */
	private void _runJob(String sessionId) throws NamingException, JobInfoNotFoundException {
		m_log.debug("_runJob() start : sessionId = " + sessionId);
		//実行状態が実行中のセッションジョブを取得
		Collection collection = null;
		try {
			collection = JobSessionJobUtil.getLocalHome().findByStatus(
					sessionId, Integer.valueOf(StatusConstant.TYPE_RUNNING));
		} catch (FinderException e) {
			m_log.warn("JobSessionJobUtil.getLocalHome().findByStatus() : " + e.getMessage());
			JobInfoNotFoundException je = new JobInfoNotFoundException(e.getMessage(), e);
			je.setSessionId(sessionId);
			je.setStatus(Integer.valueOf(StatusConstant.TYPE_RUNNING));
			throw je;
		}
		
		if(collection != null && collection.size() > 0){
			Iterator itr = collection.iterator();
			while(itr.hasNext()){
				//セッションジョブを取得
				JobSessionJobLocal sessionJob = (JobSessionJobLocal)itr.next();
				String jobunitId = sessionJob.getJobunit_id();
				String jobId = sessionJob.getJob_id();
				
				//ジョブを取得
				JobInfoLocal job = sessionJob.getJobInfo();
				int type = job.getJob_type().intValue();
				job = null;

				m_log.debug("_runJob() target : sessionId = " + sessionId + ", jobunitId = " + jobunitId + ", jobId = " + jobId + ", type = " + type);
				if(type == JobConstant.TYPE_JOBUNIT){
					//ジョブユニットが終了可能かチェック
					if(checkJobNetEnd(sessionId, jobId)){
						//終了の場合
						
						//実行状態を終了にする
						sessionJob.setStatus(Integer.valueOf(StatusConstant.TYPE_END));
						//終了日時を設定
						sessionJob.setEnd_date(new Date());
						
						sessionJob = null;
						
						//終了状態を判定し、終了状態と終了値を設定
						Integer endStatus = checkEndStatus(sessionId, jobunitId, jobId);
						
						//実行状態、終了状態、終了値、終了日時を設定
						setEndStaus(sessionId, jobunitId, jobId, 
								Integer.valueOf(StatusConstant.TYPE_END), endStatus, null);
						
						//通知処理
						new Notice().notify(sessionId, jobunitId, jobId, endStatus);
					}
					else{
						sessionJob = null;
						
						//ジョブユニットの場合、ジョブ開始処理を行う
						startJob(sessionId, jobunitId, jobId);
					}
				}
				else{
					sessionJob = null;
					
					//ジョブネット、ジョブの場合、ジョブ開始処理メイン1を行う
					startJobMain1(sessionId, jobunitId, jobId, null);
				}
			}
		}
	}
	
	/**
	 * ジョブ（実行状態が待機）を実行します。
	 * 
	 * @throws NamingException
	 * @throws JobInfoNotFoundException 
	 * 
	 * @see com.clustercontrol.bean.JobConstant
	 * @see com.clustercontrol.bean.StatusConstant
	 * @see com.clustercontrol.jobmanagement.factory.OperationJob#runJob(String, String, String)
	 */
	private void _runWaitJob(String sessionId) throws NamingException, JobInfoNotFoundException {
		m_log.debug("_runWaitJob() start : sessionId = " + sessionId);
		//実行状態が待機のセッションジョブを取得
		Collection collection = null;
		try {
			collection = JobSessionJobUtil.getLocalHome().findByStatus(
					sessionId, Integer.valueOf(StatusConstant.TYPE_WAIT));
		} catch (FinderException e1) {
			m_log.warn("JobSessionJobUtil.getLocalHome().findByStatus() : " + e1.getMessage());
			JobInfoNotFoundException je = new JobInfoNotFoundException(e1.getMessage(), e1);
			je.setSessionId(sessionId);
			je.setStatus(Integer.valueOf(StatusConstant.TYPE_WAIT));
			throw je;
		}
		
		if(collection != null && collection.size() > 0){
			Iterator itr = collection.iterator();
			while(itr.hasNext()){
				//セッションジョブを取得
				JobSessionJobLocal sessionJob = (JobSessionJobLocal)itr.next();
				String jobunitId = sessionJob.getJobunit_id();
				String jobId = sessionJob.getJob_id();
				
				//ジョブを取得
				JobInfoLocal job = sessionJob.getJobInfo();
				int type = job.getJob_type().intValue();
				job = null;
				
				m_log.debug("_runWaitJob() target : sessionId = " + sessionId + ", jobunitId = " + jobunitId + ", jobId = " + jobId + ", type = " + type);
				if(type == JobConstant.TYPE_JOBUNIT){
					//実行状態が待機のジョブユニットを実行する
					try {
						m_log.debug("_runWaitJob() : RUN WAIT JOB : sessionId=" + sessionId + ", jobId=" + jobId);
						
						runJob(sessionId, jobunitId, jobId);
					} catch (Exception e) {
						m_log.debug("_runWaitJob() : RUN WAIT JOB ERROR : sessionId=" + sessionId + ", jobId=" + jobId);
						
						//実行に失敗した場合、実行状態を実行失敗にする
						sessionJob.setStatus(Integer.valueOf(StatusConstant.TYPE_ERROR));
					}
				}
				sessionJob = null;
			}
		}
	}
	
	/**
	 * ジョブ（実行状態が停止処理中）をチェックします。
	 * 
	 * @throws NamingException
	 * @throws JobInfoNotFoundException 
	 * 
	 * @see com.clustercontrol.bean.JobConstant
	 * @see com.clustercontrol.bean.StatusConstant
	 * @see com.clustercontrol.jobmanagement.factory.OperationJob#checkTimeout(String, String)
	 */
	public void checkStoppingJob() throws NamingException, JobInfoNotFoundException {
		m_log.debug("runStoppingJob()");
		//実行状態が停止処理中のセッションノードを取得
		Collection collection = null;
		try {
			collection = JobSessionNodeUtil.getLocalHome().findByStatus(
					Integer.valueOf(StatusConstant.TYPE_STOPPING));
		} catch (FinderException e) {
			m_log.warn("JobSessionNodeUtil.getLocalHome().findByStatus() : " + e.getMessage());
			JobInfoNotFoundException je = new JobInfoNotFoundException(e.getMessage(), e);
			je.setStatus(Integer.valueOf(StatusConstant.TYPE_WAIT));
			throw je;
		}
		
		if(collection != null && collection.size() > 0){
			Iterator itr = collection.iterator();
			while(itr.hasNext()){
				//セッションノードを取得
				JobSessionNodeLocal sessionNode = (JobSessionNodeLocal)itr.next();
				String sessionId = sessionNode.getSession_id();
				String jobunitId = sessionNode.getJobunit_id();
				String jobId = sessionNode.getJob_id();
				String facilityId = sessionNode.getFacility_id();
				sessionNode = null;
				
				//停止処理のエージェントタイムアウトをチェック
				checkStopTimeout(sessionId, jobunitId, jobId, facilityId);
			}
		}
	}
	
	/**
	 * ジョブ開始処理を行います。
	 * 
	 * @param sessionId セッションID
	 * @param jobId ジョブID
	 * @throws NamingException
	 * @throws JobInfoNotFoundException 
	 * 
	 * @see com.clustercontrol.jobmanagement.factory.OperationJob#startJobMain1(String, String, String)
	 */
	protected void startJob(String sessionId, String jobunitId, String jobId) throws NamingException, JobInfoNotFoundException {
		m_log.debug("startJob() : sessionId=" + sessionId + ", jobunitId=" + jobunitId + ", jobId=" + jobId);
		
		//セッションIDとジョブIDから、直下のジョブを取得（実行状態が待機または実行中）
		Collection collection = null;
		try {
			collection = JobRelationInfoUtil.getLocalHome().findByStartStatus(sessionId, jobId);
		} catch (FinderException e) {
			m_log.warn("JobRelationInfoUtil.getLocalHome().findByStartStatus() : " + e.getMessage());
			JobInfoNotFoundException je = new JobInfoNotFoundException(e.getMessage(), e);
			je.setSessionId(sessionId);
			je.setJobId(jobId);
			throw je;
		}
		if(collection != null && collection.size() > 0){
			Iterator itr = collection.iterator();
			while(itr.hasNext()){
				JobRelationInfoLocal relation = (JobRelationInfoLocal)itr.next();
				JobSessionJobLocal sessionJob = relation.getJobSessionJob();
				String startSessionId = sessionJob.getSession_id();
				String startJobId = sessionJob.getJob_id();
				sessionJob = null;
				
				//ジョブ開始処理メイン1を行う
				startJobMain1(startSessionId, jobunitId, startJobId, null);
			}
		}
	}
	
	/**
	 * ジョブ開始処理メイン1を行います。
	 * 
	 * @param sessionId
	 * @param jobId
	 * @throws NamingException
	 * @throws JobInfoNotFoundException 
	 * 
	 * @see com.clustercontrol.bean.JobConstant
	 * @see com.clustercontrol.bean.StatusConstant
	 * @see com.clustercontrol.jobmanagement.factory.OperationJob#checkStartCondition(String, String)
	 * @see com.clustercontrol.jobmanagement.factory.OperationJob#checkCalendar(String, String)
	 * @see com.clustercontrol.jobmanagement.factory.OperationJob#startJobMain2(String, String, String)
	 * @see com.clustercontrol.jobmanagement.factory.OperationJob#checkStartDelayMain(String, String)
	 * @see com.clustercontrol.jobmanagement.factory.OperationJob#checkJobEnd(JobSessionJobLocal)
	 * @see com.clustercontrol.jobmanagement.factory.OperationJob#checkEndStatus(String, String)
	 * @see com.clustercontrol.jobmanagement.factory.OperationJob#setEndStaus(String, String, Integer, Integer, String)
	 * @see com.clustercontrol.jobmanagement.factory.OperationJob#endJob(String, String)
	 * @see com.clustercontrol.jobmanagement.factory.OperationJob#startNode(String, String, String)
	 * @see com.clustercontrol.jobmanagement.factory.OperationJob#setStatus(String, String, Integer)
	 * @see com.clustercontrol.jobmanagement.factory.OperationJob#startJob(String, String)
	 * @see com.clustercontrol.jobmanagement.factory.Notice#notify(String, String, Integer)
	 */
	protected void startJobMain1(String sessionId, String jobunitId, String jobId, String facilityId) throws NamingException, JobInfoNotFoundException {
		m_log.debug("startJobMain1() : sessionId=" + sessionId + ", jobunitId=" + jobunitId + ", jobId=" + jobId + ", facilityId=" + facilityId);
		
		//セッションIDとジョブIDから、セッションジョブを取得
		JobSessionJobLocal sessionJob = getJobSessionJobLocal(sessionId, jobunitId, jobId);
		
		//実行状態チェック
		if(sessionJob.getStatus().intValue() == StatusConstant.TYPE_WAIT){
			//待機中の場合
			sessionJob = null;
			
			//開始条件をチェックする
			if(checkStartCondition(sessionId, jobunitId, jobId)){
				//開始OKの場合
				
				//カレンダをチェック
				if(checkCalendar(sessionId, jobunitId, jobId)){
					//実行可能の場合
					
					//ジョブ開始処理メイン2を行う
					startJobMain2(sessionId, jobunitId, jobId, facilityId);
				}
			}
			else {
				//開始遅延チェック
				checkStartDelayMain(sessionId, jobunitId, jobId);
			}
		}
		else if(sessionJob.getStatus().intValue() == StatusConstant.TYPE_RUNNING){
			//実行中の場合
	
			JobInfoLocal job = sessionJob.getJobInfo();
			int type = job.getJob_type().intValue();
			job = null;
			
			if(type == JobConstant.TYPE_JOB){
				//ジョブが終了可能かチェック
				if(checkJobEnd(sessionJob)){
					//終了の場合

					//終了状態を取得
					Integer endStatus = checkEndStatus(sessionId, jobunitId, jobId);
					//終了日時を設定
					sessionJob.setEnd_date(new Date());
					
					sessionJob = null;
					
					//実行状態、終了状態、終了値、終了日時を設定
					setEndStaus(sessionId, jobunitId, jobId, 
							Integer.valueOf(StatusConstant.TYPE_END), endStatus, "");
					
					//通知処理
					new Notice().notify(sessionId, jobunitId, jobId, endStatus);
					
					//ジョブ終了時関連処理（再帰呼び出し）
					endJob(sessionId, jobunitId, jobId);
				}
				else{
					//終了条件をチェックする
					if(!checkEndDelay(sessionId, jobunitId, jobId)){
						//遅延チェックで操作が行われない場合
					
						sessionJob = null;

						startNode(sessionId, jobunitId, jobId, null);
					}
				}
			}
			else{
				//ジョブネットが終了可能かチェック
				if(checkJobNetEnd(sessionId, jobId)){
					//終了の場合
					
					//実行状態を終了にする
					sessionJob.setStatus(Integer.valueOf(StatusConstant.TYPE_END));
					//終了日時を設定
					sessionJob.setEnd_date(new Date());
					
					sessionJob = null;
					
					//終了状態を判定し、終了状態と終了値を設定
					Integer endStatus = checkEndStatus(sessionId, jobunitId, jobId);
					
					//実行状態、終了状態、終了値、終了日時を設定
					setEndStaus(sessionId, jobunitId, jobId, 
							Integer.valueOf(StatusConstant.TYPE_END), endStatus, null);
					
					//通知処理
					new Notice().notify(sessionId, jobunitId, jobId, endStatus);
					
					//ジョブ終了時関連処理
					endJob(sessionId, jobunitId, jobId);
				}
				else{
					//終了条件をチェックする
					if(!checkEndDelay(sessionId, jobunitId, jobId)){
						//遅延チェックで操作が行われない場合
					
						sessionJob = null;

						startJob(sessionId, jobunitId, jobId);
					}
				}
			}
		}
		else if(sessionJob.getStatus().intValue() == StatusConstant.TYPE_SKIP){
			//スキップの場合
			
			//開始条件をチェックする
			if(checkStartCondition(sessionId, jobunitId, jobId)){
				//実行状態を終了にする
				sessionJob.setStatus(Integer.valueOf(StatusConstant.TYPE_END));
				//終了日時を設定
				sessionJob.setEnd_date(new Date());
				sessionJob = null;
				
				//ジョブ終了時関連処理
				endJob(sessionId, jobunitId, jobId);
			}
		}
	}
	
	/**
	 * ジョブ開始処理メイン2を行います。
	 * 
	 * @param sessionId
	 * @param jobId
	 * @throws NamingException
	 * @throws JobInfoNotFoundException 
	 * 
	 * @see com.clustercontrol.bean.EndStatusConstant
	 * @see com.clustercontrol.bean.JobConstant
	 * @see com.clustercontrol.bean.StatusConstant
	 * @see com.clustercontrol.jobmanagement.factory.OperationJob#startNode(String, String, String)
	 * @see com.clustercontrol.jobmanagement.factory.OperationJob#startJob(String, String)
	 */
	protected void startJobMain2(String sessionId, String jobunitId, String jobId, String facilityId) throws NamingException, JobInfoNotFoundException {
		m_log.debug("startJobMain2() : sessionId=" + sessionId + ", jobId=" + jobId + ", facilityId=" + facilityId);
		
		//セッションIDとジョブIDから、セッションジョブを取得
		JobSessionJobLocal sessionJob = getJobSessionJobLocal(sessionId, jobunitId, jobId);
		
		//実行状態を実行中にする
		sessionJob.setStatus(Integer.valueOf(StatusConstant.TYPE_RUNNING));
		//開始・再実行日時を設定
		sessionJob.setStart_date(new Date());
		
		//ジョブ
		JobInfoLocal job = sessionJob.getJobInfo();
		int type = job.getJob_type().intValue();
		sessionJob = null;
		job = null;
		
		//通知処理
		new Notice().notify(sessionId, jobunitId, jobId, Integer.valueOf(EndStatusConstant.TYPE_BEGINNING));

		if(type == JobConstant.TYPE_JOB){
			//ノードへの実行指示
			startNode(sessionId, jobunitId, jobId, facilityId);
		}
		else{
			//ジョブ開始処理（再帰呼び出し）
			startJob(sessionId, jobunitId, jobId);
		}
	}
	
	/**
	 * ジョブの待ち条件のチェックを行います。
	 * 
	 * @param sessionId セッションID
	 * @param jobId ジョブID
	 * @return true：実行可、false：実行不可
	 * @throws NamingException
	 * @throws JobInfoNotFoundException 
	 * 
	 * @see com.clustercontrol.bean.JudgmentObjectConstant
	 * @see com.clustercontrol.bean.StatusConstant
	 * @see com.clustercontrol.bean.YesNoConstant
	 * @see com.clustercontrol.jobmanagement.bean.ConditionTypeConstant
	 */
	protected boolean checkStartCondition(String sessionId, String jobunitId, String jobId) throws NamingException, JobInfoNotFoundException {
		m_log.debug("checkStartCondition() : sessionId=" + sessionId + ", jobunitId=" + jobunitId + ", jobId=" + jobId);
		
		//セッションIDとジョブIDから、セッションジョブを取得
		JobSessionJobLocal sessionJob = getJobSessionJobLocal(sessionId, jobunitId, jobId);
		
		//待機中の場合
		JobInfoLocal job = sessionJob.getJobInfo();
		JobStartInfoLocal start = job.getJobStartInfo();
		
		//待ち条件ジョブを取得
		Collection startJobs = start.getJobStartJobInfo();
		//待ち条件時刻を取得
		JobStartTimeInfoLocal startTime = start.getJobStartTimeInfo();
		
		ArrayList<Boolean> jobResult = new ArrayList<Boolean>();
		
		//statusCheck 待ち条件判定を行うかを表すフラグ
		//AND条件の場合　実行中の先行ジョブがある場合 :false　全ての先行ジョブが終了している場合 :true
		//OR条件の場合 :true
		//先行ジョブが存在しない場合（時刻のみの制御）：true
		boolean statusCheck = true;

		//startCheck 待ち条件チェックの結果、ジョブを実行するかを表すフラグ
		//AND条件の場合　全ての待ち条件が満たされている場合：ture
		//OR条件の場合　待ち条件のうちどれか一つでも満たされている場合：true
		boolean startCheck = true;
		
		//allEndCheck 先行ジョブが全て終了しているかを表すフラグ
		boolean allEndCheck = true;
		
		//possibilityCheck 実行可能性を表すフラグ
		//以下の場合、実行可能性なし
		//ANDの場合：待ち条件ジョブの実行結果に1つでもNGがある場合
		//ORの場合：待ち条件がジョブのみで、全てのジョブが終了し、実行結果がNGである場合
		boolean possibilityCheck = true;
		
		//待ち条件のチェック
		if(startJobs != null && 
				startJobs.size() > 0 ||
				startTime != null){
			
			//待ち条件が設定されている場合
			
			//待ち条件ジョブ判定
			Iterator itr = startJobs.iterator();
			while(itr.hasNext()){
				//待ち条件ジョブを取得
				JobStartJobInfoLocal startJob = (JobStartJobInfoLocal)itr.next();
				
				//セッションIDとジョブIDから、対象セッションジョブを取得
				JobSessionJobLocal targetSessionJob = getJobSessionJobLocal(
						sessionJob.getSession_id(), startJob
								.getTarget_jobunit_id(), startJob
								.getTarget_job_id());
				
				//対象セッションジョブ(先行ジョブ)の実行状態をチェック
				if(targetSessionJob.getStatus().intValue() == StatusConstant.TYPE_END || 
						targetSessionJob.getStatus().intValue() == StatusConstant.TYPE_MODIFIED){
					//終了または、変更済の場合
					
					if(startJob.getTarget_job_type().intValue() == JudgmentObjectConstant.TYPE_JOB_END_STAUS){
						//終了状態での比較
						
						Integer endStatus = getEndStatus(targetSessionJob);
						if(endStatus instanceof Integer){
							//対象セッションジョブの実行状態と待ち条件の終了状態を比較
							if(endStatus.compareTo(startJob.getTarget_job_end_value()) == 0){
								jobResult.add(new Boolean(true));
								
								//OR条件の場合、これ以上ループを回す必要はない
								if(start.getCondition_type().intValue() == ConditionTypeConstant.TYPE_OR)
									break;
							}
							else{
								jobResult.add(new Boolean(false));
							}
						}
						else{
							jobResult.add(new Boolean(false));
						}
					}
					else if(startJob.getTarget_job_type().intValue() == JudgmentObjectConstant.TYPE_JOB_END_VALUE){
						//終了値での比較
						
						Integer endValue = targetSessionJob.getEnd_value();
						if(endValue instanceof Integer){
							//対象セッションジョブの実行状態と待ち条件の終了値を比較
							if(endValue.compareTo(startJob.getTarget_job_end_value()) == 0){
								jobResult.add(new Boolean(true));
								
								//OR条件の場合、これ以上ループを回す必要はない
								if(start.getCondition_type().intValue() == ConditionTypeConstant.TYPE_OR)
									break;
							}
							else{
								jobResult.add(new Boolean(false));
							}
						}
						else{
							jobResult.add(new Boolean(false));
						}
					}
					else{
						jobResult.add(new Boolean(false));
					}
				}
				else{
					
					// 終了していないジョブが存在するため、allEndCheckフラグをfalseに変更
					allEndCheck = false;
					
					// 待ち条件が「AND」の場合、待ち条件ジョブの判定を終了
					if(start.getCondition_type().intValue() == ConditionTypeConstant.TYPE_AND) {
						statusCheck = false;
						startCheck = false;
						break;
					}
					// 待ち条件が「OR」の場合
					else{
						jobResult.add(new Boolean(false));
					}
				}
			}
			
			// 待ち条件判定開始
			if(statusCheck){
				//ANDまたはOR条件に一致するかチェック
				if(start.getCondition_type().intValue() == ConditionTypeConstant.TYPE_AND){
					//AND条件の場合
					startCheck = true;
					for(int i = 0; i < jobResult.size(); i++){
						if(!((Boolean)jobResult.get(i)).booleanValue()){
							startCheck = false;
							break;
						}
					}
				}
				else{
					//OR条件の場合
					startCheck = false;
					for(int i = 0; i < jobResult.size(); i++){
						if(((Boolean)jobResult.get(i)).booleanValue()){
							startCheck = true;
							break;
						}
					}
				}
				
				
				//実行可能性のチェック
				//ANDまたはOR条件により、開始できる可能性を取得する
				if(!startCheck){
					//ANDの場合：待ち条件ジョブの実行結果に1つでもNGがある場合(startCheckがfalseの時点で条件を満たす)
					if(start.getCondition_type().intValue() == ConditionTypeConstant.TYPE_AND){
						possibilityCheck = false;
					}
					//ORの場合：待ち条件がジョブのみで、全てのジョブが終了し、実行結果がNGである場合
					else{
						if(startTime == null && allEndCheck){
							possibilityCheck = false;
						}
					}
				}
				
				
				// 待ち条件時間のチェック
				if(start.getCondition_type().intValue() == ConditionTypeConstant.TYPE_AND){
					// 待ち条件が「AND」の場合
					//待ち条件ジョブ判定を満たしていたら、待ち条件時間をチェック
					if(startCheck){
						if(startTime != null){
							//セッションIDから、セッションを取得
							JobSessionLocal session = null;
							try {
								session = JobSessionUtil.getLocalHome().findByPrimaryKey(new JobSessionPK(sessionId));
							} catch (FinderException e) {
								m_log.warn("JobSessionUtil.getLocalHome().findByPrimaryKey() : " + e.getMessage());
								JobInfoNotFoundException je = new JobInfoNotFoundException(e.getMessage(), e);
								je.setSessionId(sessionId);
								throw je;
							}
							Date sessionDate = session.getSchedule_date();
							session = null;
							
							Date now = new Date();
							Calendar work = Calendar.getInstance();
							Calendar time = Calendar.getInstance();
							work.setTime(startTime.getStart_time());
							work.getTime();
							
							time.set(Calendar.HOUR_OF_DAY, work.get(Calendar.HOUR_OF_DAY));
							time.set(Calendar.MINUTE, work.get(Calendar.MINUTE));
							time.set(Calendar.SECOND, work.get(Calendar.SECOND));
							time.set(Calendar.MILLISECOND, 0);
							Date startDate = time.getTime();
							
							if(startDate.after(sessionDate) && 
									startDate.before(now)){
								startCheck = true;
							}
							else{
								startCheck = false;
							}
						}
					}
				}
				else {
					// 待ち条件が「OR」の場合
					// 待ち条件ジョブ判定を満たしていない場合、待ち条件時間をチェック
					if(!startCheck) {
						if(startTime != null){
							//セッションIDから、セッションを取得
							JobSessionLocal session = null;
							try {
								session = JobSessionUtil.getLocalHome().findByPrimaryKey(new JobSessionPK(sessionId));
							} catch (FinderException e) {
								m_log.warn("JobSessionUtil.getLocalHome().findByPrimaryKey() : " + e.getMessage());
								JobInfoNotFoundException je = new JobInfoNotFoundException(e.getMessage(), e);
								je.setSessionId(sessionId);
								throw je;
							}
							Date sessionDate = session.getSchedule_date();
							session = null;
							
							Date now = new Date();
							Calendar work = Calendar.getInstance();
							Calendar time = Calendar.getInstance();
							work.setTime(startTime.getStart_time());
							work.getTime();
							
							time.set(Calendar.HOUR_OF_DAY, work.get(Calendar.HOUR_OF_DAY));
							time.set(Calendar.MINUTE, work.get(Calendar.MINUTE));
							time.set(Calendar.SECOND, work.get(Calendar.SECOND));
							time.set(Calendar.MILLISECOND, 0);
							Date startDate = time.getTime();
							
							if(startDate.after(sessionDate) && 
									startDate.before(now)){
								startCheck = true;
							}
							else{
								startCheck = false;
							}
						}
					}
				}
			}
		}
		
		//開始チェック結果が開始NGの場合
		if(!startCheck){
//			boolean possibility = true;
//			//ANDまたはOR条件により、開始できる可能性を取得する
//			//ANDの場合
//			if(start.getCondition_type().intValue() == ConditionTypeConstant.TYPE_AND){
//				if(statusCheck){
//					possibility = false;
//				}
//			}
//			//ORの場合
//			else{
//				if(statusCheck && allEndCheck && startTime == null){
//					possibility = false;
//				}
//			}
			
			//開始できる可能性なしの場合
			if(!possibilityCheck){
				//条件を満たさなければ終了する場合
				if(start.getUnmatch_end_flg().intValue() == YesNoConstant.TYPE_YES){
					m_log.debug("checkStartCondition() : unmatch end flg is true. end job : " + " jobid : " + jobId +" : status :" + sessionJob.getStatus().intValue());
					//実行状態を終了にする
					sessionJob.setStatus(Integer.valueOf(StatusConstant.TYPE_END));
					//終了値を設定
					sessionJob.setEnd_value(start.getUnmatch_end_value());
					//終了日時を設定
					sessionJob.setEnd_date(new Date());
					
					//ジョブ終了時関連処理
					endJob(sessionJob.getSession_id(), sessionJob.getJobunit_id(), sessionJob.getJob_id());
				}
			}
		}
		
		return startCheck;
	}
	
	/**
	 * 開始遅延処理を行います。
	 * 
	 * @param sessionId セッションID
	 * @param jobId ジョブID
	 * @return true=操作あり, false=操作なし
	 * @throws NamingException
	 * @throws JobInfoNotFoundException 
	 * 
	 * @see com.clustercontrol.bean.StatusConstant
	 * @see com.clustercontrol.jobmanagement.factory.OperationJob#checkStartDelay(String, String)
	 * @see com.clustercontrol.jobmanagement.factory.Notice#notify(String, String, Integer)
	 */
	protected void checkStartDelayMain(String sessionId, String jobunitId, String jobId) throws NamingException, JobInfoNotFoundException {
		m_log.debug("checkStartDelayMain() : sessionId=" + sessionId + ", jobunitId=" + jobunitId + ", jobId=" + jobId);
		
		//セッションIDとジョブIDから、セッションジョブを取得
		JobSessionJobLocal sessionJob = getJobSessionJobLocal(sessionId, jobunitId, jobId);
		
		//実行状態チェック
		if(sessionJob.getStatus().intValue() == StatusConstant.TYPE_WAIT){
			sessionJob = null;
			
			//開始遅延チェック
			checkStartDelay(sessionId, jobunitId, jobId);
		}
		
		//セッションIDとジョブIDから、直下のジョブを取得
		Collection collection = null;
		try {
			collection = JobRelationInfoUtil.getLocalHome().findByParentJobId(sessionId, jobId);
		} catch (FinderException e) {
			m_log.warn("JobRelationInfoUtil.getLocalHome().findByParentJobId() : " + e.getMessage());
			JobInfoNotFoundException je = new JobInfoNotFoundException(e.getMessage(), e);
			je.setSessionId(sessionId);
			je.setParentJobId(jobId);
			throw je;
		}
		if(collection != null && collection.size() > 0){
			Iterator itr = collection.iterator();
			while(itr.hasNext()){
				JobRelationInfoLocal relation = (JobRelationInfoLocal)itr.next();
				JobSessionJobLocal childSessionJob = relation.getJobSessionJob();
				String childSessionId = childSessionJob.getSession_id();
				String childJobUnitId = childSessionJob.getJobunit_id();
				String childJobId = childSessionJob.getJob_id();
				childSessionJob = null;
				
				//開始遅延チェックメイン処理を行う（再帰呼び出し）
				checkStartDelayMain(childSessionId, childJobUnitId, childJobId);
			}
		}
	}

	/**
	 * 開始遅延をチェックします。
	 * 
	 * @param sessionId セッションID
	 * @param jobId ジョブID
	 * @return true=操作あり, false=操作なし
	 * @throws NamingException
	 * @throws JobInfoNotFoundException 
	 * 
	 * @see com.clustercontrol.bean.YesNoConstant
	 * @see com.clustercontrol.jobmanagement.bean.ConditionTypeConstant
	 * @see com.clustercontrol.jobmanagement.bean.DelayNotifyConstant
	 * @see com.clustercontrol.jobmanagement.bean.OperationConstant
	 * @see com.clustercontrol.jobmanagement.factory.OperateSkipOfJob#skipJob(String, String, Integer)
	 * @see com.clustercontrol.jobmanagement.factory.OperateWaitOfJob#waitJob(String, String)
	 * @see com.clustercontrol.jobmanagement.factory.Notice#delayNotify(String, String, boolean)
	 */
	protected void checkStartDelay(String sessionId, String jobunitId, String jobId) throws NamingException, JobInfoNotFoundException {
		m_log.debug("checkStartDelay() : sessionId=" + sessionId + ", jobunitId=" + jobunitId + ", jobId=" + jobId);
		
		//セッションIDとジョブIDから、セッションジョブを取得
		JobSessionJobLocal sessionJob = getJobSessionJobLocal(sessionId, jobunitId, jobId);
		
		JobInfoLocal job = sessionJob.getJobInfo();
		JobStartInfoLocal start = job.getJobStartInfo();
		
		ArrayList<Boolean> result = new ArrayList<Boolean>();
		boolean delayCheck = true;
		
		if(start != null && 
				start.getStart_delay().intValue() == YesNoConstant.TYPE_YES){
			//開始遅延が設定されている場合
			Date sessionDate = null;
			
			if(start.getStart_delay_session().intValue() == YesNoConstant.TYPE_YES){
				//セッション開始後の時間が設定されている場合
			
				//セッション開始日時を取得
				JobSessionLocal session;
				try {
					session = JobSessionUtil.getLocalHome().findByPrimaryKey(new JobSessionPK(sessionId));
				} catch (FinderException e) {
					m_log.warn("JobSessionUtil.getLocalHome().findByPrimaryKey() : " + e.getMessage());
					JobInfoNotFoundException je = new JobInfoNotFoundException(e.getMessage(), e);
					je.setSessionId(sessionId);
					je.setJobunitId(jobunitId);
					je.setJobId(jobId);
					throw je;
				}
				sessionDate = session.getSchedule_date();
				
				Date now = new Date();
				Calendar work = Calendar.getInstance();
				work.setTime(sessionDate);
				work.getTime();
				work.add(Calendar.MINUTE, start.getStart_delay_session_value().intValue());
				Date check = work.getTime();
				
				if(now.after(check)){
					result.add(new Boolean(true));
				}
				else{
					result.add(new Boolean(false));
				}	
			}
			
			if(start.getStart_delay_time().intValue() == YesNoConstant.TYPE_YES){
				//時刻が設定されている場合

				if(start.getStart_delay_time_value() instanceof Time){
					//セッション開始日時を取得
					if(sessionDate == null){
						JobSessionLocal session;
						try {
							session = JobSessionUtil.getLocalHome().findByPrimaryKey(new JobSessionPK(sessionId));
						} catch (FinderException e) {
							m_log.warn("JobSessionUtil.getLocalHome().findByPrimaryKey() : " + e.getMessage());
							JobInfoNotFoundException je = new JobInfoNotFoundException(e.getMessage(), e);
							je.setSessionId(sessionId);
							throw je;
						}
						sessionDate = session.getSchedule_date();
					}
					
					Date now = new Date();
					Calendar work = Calendar.getInstance();
					Calendar time = Calendar.getInstance();
					work.setTime(start.getStart_delay_time_value());
					work.getTime();
					
					time.set(Calendar.HOUR_OF_DAY, work.get(Calendar.HOUR_OF_DAY));
					time.set(Calendar.MINUTE, work.get(Calendar.MINUTE));
					time.set(Calendar.SECOND, work.get(Calendar.SECOND));
					time.set(Calendar.MILLISECOND, 0);
					Date startDate = time.getTime();
					
					if(startDate.after(sessionDate) && 
							startDate.before(now)){
						result.add(new Boolean(true));
					}
					else{
						result.add(new Boolean(false));
					}
				}
				else{
					result.add(new Boolean(false));
				}
			}
			
			//ANDまたはOR条件に一致するかチェック
			if(result.size() > 0){
				if(start.getStart_delay_condition_type().intValue() == ConditionTypeConstant.TYPE_AND){
					//AND条件の場合
					delayCheck = true;
					for(int i = 0; i < result.size(); i++){
						if(!((Boolean)result.get(i)).booleanValue()){
							delayCheck = false;
							break;
						}
					}
				}
				else{
					//OR条件の場合
					delayCheck = false;
					for(int i = 0; i < result.size(); i++){
						if(((Boolean)result.get(i)).booleanValue()){
							delayCheck = true;
							break;
						}
					}
				}
			}
			else{
				delayCheck = false;
			}
			
			//開始遅延チェック結果が遅延の場合
			if(delayCheck){
				
				//通知
				if(start.getStart_delay_notify().intValue() == YesNoConstant.TYPE_YES){
					//遅延通知状態を取得
					int flg = sessionJob.getDelay_notify_flg().intValue();
					//遅延通知状態から通知済みフラグを取得
					int notifyFlg = DelayNotifyConstant.getNotify(flg);

					if(notifyFlg == DelayNotifyConstant.NONE || notifyFlg == DelayNotifyConstant.END){
						//通知済みフラグが「通知・操作なし」又は「終了遅延通知済み」の場合
						
						//通知処理
						new Notice().delayNotify(sessionId, jobunitId, jobId, true);
						
						if(notifyFlg == DelayNotifyConstant.NONE)
							sessionJob.setDelay_notify_flg(Integer.valueOf(DelayNotifyConstant.START));
						else if(notifyFlg == DelayNotifyConstant.END)
							sessionJob.setDelay_notify_flg(Integer.valueOf(DelayNotifyConstant.START_AND_END));
					}
				}
				
				//操作
				if(start.getStart_delay_operation().intValue() == YesNoConstant.TYPE_YES){
					int type = start.getStart_delay_operation_type().intValue();
					
					if(type == OperationConstant.TYPE_STOP_SKIP){
						//スキップ
						new OperateSkipOfJob().skipJob(
								sessionId, 
								jobunitId, 
								jobId, 
								start.getStart_delay_operation_end_value());
					}
					else if(type == OperationConstant.TYPE_STOP_WAIT){
						//保留
						new OperateWaitOfJob().waitJob(sessionId, jobunitId, jobId);
					}
				}
			}
			
		}
		sessionJob = null;
	}
	
	/**
	 * 終了遅延処理を行います。
	 * 
	 * @param sessionId セッションID
	 * @param jobId ジョブID
	 * @return true=操作あり, false=操作なし
	 * @throws NamingException
	 * @throws JobInfoNotFoundException 
	 * 
	 * @see com.clustercontrol.bean.YesNoConstant
	 * @see com.clustercontrol.jobmanagement.bean.DelayNotifyConstant
	 * @see com.clustercontrol.jobmanagement.bean.ConditionTypeConstant
	 * @see com.clustercontrol.jobmanagement.bean.OperationConstant
	 * @see com.clustercontrol.jobmanagement.factory.OperateStopOfJob#stopJob(String, String, String)
	 * @see com.clustercontrol.jobmanagement.factory.OperateSuspendOfJob#suspendJob(String, String)
	 * @see com.clustercontrol.jobmanagement.factory.Notice#delayNotify(String, String, boolean)
	 */
	protected boolean checkEndDelay(String sessionId, String jobunitId, String jobId) throws NamingException, JobInfoNotFoundException {
		m_log.debug("checkEndDelay() : sessionId=" + sessionId + ", jobId=" + jobId);
		
		//セッションIDとジョブIDから、セッションジョブを取得
		JobSessionJobLocal sessionJob = getJobSessionJobLocal(sessionId, jobunitId, jobId);
		
		JobInfoLocal job = sessionJob.getJobInfo();
		JobStartInfoLocal start = job.getJobStartInfo();
		
		ArrayList<Boolean> result = new ArrayList<Boolean>();
		boolean delayCheck = true;
		boolean operation = false;
		
		if(start != null && 
				start.getEnd_delay().intValue() == YesNoConstant.TYPE_YES){
			//終了遅延が設定されている場合
			Date sessionDate = null;
			
			if(start.getEnd_delay_session().intValue() == YesNoConstant.TYPE_YES){
				//セッション開始後の時間が設定されている場合
			
				//セッション開始日時を取得
				JobSessionLocal session;
				try {
					session = JobSessionUtil.getLocalHome().findByPrimaryKey(new JobSessionPK(sessionId));
				} catch (FinderException e) {
					m_log.warn("JobSessionUtil.getLocalHome().findByPrimaryKey() : " + e.getMessage());
					JobInfoNotFoundException je = new JobInfoNotFoundException(e.getMessage(), e);
					je.setSessionId(sessionId);
					throw je;
				}
				sessionDate = session.getSchedule_date();
				
				Date now = new Date();
				Calendar work = Calendar.getInstance();
				work.setTime(sessionDate);
				work.getTime();
				work.add(Calendar.MINUTE, start.getEnd_delay_session_value().intValue());
				Date check = work.getTime();
				
				if(now.after(check)){
					result.add(Boolean.valueOf(true));
				}
				else{
					result.add(Boolean.valueOf(false));
				}	
			}
			
			if(start.getEnd_delay_job().intValue() == YesNoConstant.TYPE_YES){
				//ジョブ開始後の時間が設定されている場合
			
				//ジョブ開始日時を取得
				Date startDate = sessionJob.getStart_date();
				
				Date now = new Date();
				Calendar work = Calendar.getInstance();
				work.setTime(startDate);
				work.getTime();
				work.add(Calendar.MINUTE, start.getEnd_delay_job_value().intValue());
				Date check = work.getTime();
				
				if(now.after(check)){
					result.add(Boolean.valueOf(true));
				}
				else{
					result.add(Boolean.valueOf(false));
				}	
			}
			
			if(start.getEnd_delay_time().intValue() == YesNoConstant.TYPE_YES){
				//時刻が設定されている場合

				if(start.getEnd_delay_time_value() instanceof Time){
					//セッション開始日時を取得
					if(sessionDate == null){
						JobSessionLocal session;
						try {
							session = JobSessionUtil.getLocalHome().findByPrimaryKey(new JobSessionPK(sessionId));
						} catch (FinderException e) {
							m_log.warn("JobSessionUtil.getLocalHome().findByPrimaryKey() : " + e.getMessage());
							JobInfoNotFoundException je = new JobInfoNotFoundException(e.getMessage(), e);
							je.setSessionId(sessionId);
							throw je;
						}
						sessionDate = session.getSchedule_date();
					}
					
					Date now = new Date();
					Calendar work = Calendar.getInstance();
					Calendar time = Calendar.getInstance();
					work.setTime(start.getEnd_delay_time_value());
					work.getTime();
					
					time.set(Calendar.HOUR_OF_DAY, work.get(Calendar.HOUR_OF_DAY));
					time.set(Calendar.MINUTE, work.get(Calendar.MINUTE));
					time.set(Calendar.SECOND, work.get(Calendar.SECOND));
					time.set(Calendar.MILLISECOND, 0);
					Date startDate = time.getTime();
					
					if(startDate.after(sessionDate) && 
							startDate.before(now)){
						result.add(Boolean.valueOf(true));
					}
					else{
						result.add(Boolean.valueOf(false));
					}
				}		
				else{
					result.add(Boolean.valueOf(false));
				}
			}
			
			//ANDまたはOR条件に一致するかチェック
			if(result.size() > 0){
				if(start.getEnd_delay_condition_type().intValue() == ConditionTypeConstant.TYPE_AND){
					//AND条件の場合
					delayCheck = true;
					for(int i = 0; i < result.size(); i++){
						if(!((Boolean)result.get(i)).booleanValue()){
							delayCheck = false;
							break;
						}
					}
				}
				else{
					//OR条件の場合
					delayCheck = false;
					for(int i = 0; i < result.size(); i++){
						if(((Boolean)result.get(i)).booleanValue()){
							delayCheck = true;
							break;
						}
					}
				}
			}
			else{
				delayCheck = false;
			}
			
			//終了遅延チェック結果が遅延の場合
			if(delayCheck){
				
				//通知
				if(start.getEnd_delay_notify().intValue() == YesNoConstant.TYPE_YES){
					//遅延通知状態を取得
					int flg = sessionJob.getDelay_notify_flg().intValue();
					//遅延通知状態から通知済みフラグを取得
					int notifyFlg = DelayNotifyConstant.getNotify(flg);
					
					if(notifyFlg == DelayNotifyConstant.NONE || notifyFlg == DelayNotifyConstant.START){
						//通知済みフラグが「通知・操作なし」又は「開始遅延通知済み」の場合
						
						//通知処理
						new Notice().delayNotify(sessionId, jobunitId, jobId, false);
						
						if(notifyFlg == DelayNotifyConstant.NONE)
							sessionJob.setDelay_notify_flg(Integer.valueOf(DelayNotifyConstant.END));
						else if(notifyFlg == DelayNotifyConstant.START)
							sessionJob.setDelay_notify_flg(Integer.valueOf(DelayNotifyConstant.START_AND_END));
					}
				}
				
				//操作
				if(start.getEnd_delay_operation().intValue() == YesNoConstant.TYPE_YES){
					int type = start.getEnd_delay_operation_type().intValue();
					//遅延通知状態を取得
					int flg = sessionJob.getDelay_notify_flg().intValue();
					
					if(type == OperationConstant.TYPE_STOP_AT_ONCE){
						//停止
						
						//遅延通知状態に操作済みフラグを設定
						int notifyFlg = DelayNotifyConstant.addOperation(
								flg, DelayNotifyConstant.STOP_AT_ONCE);
						sessionJob.setDelay_notify_flg(Integer.valueOf(notifyFlg));
						
						new OperateStopOfJob().stopJob(sessionId, jobunitId, jobId, null);
					}
					else if(type == OperationConstant.TYPE_STOP_SUSPEND){
						//中断
						
						//遅延通知状態から操作済みフラグを取得
						int operationFlg = DelayNotifyConstant.getOperation(flg);
						
						if(operationFlg != DelayNotifyConstant.STOP_SUSPEND ){
							//操作済みフラグが停止[中断]以外の場合
							
							//遅延通知状態に操作済みフラグを設定
							int notifyFlg = DelayNotifyConstant.addOperation(
									flg, DelayNotifyConstant.STOP_SUSPEND);
							sessionJob.setDelay_notify_flg(Integer.valueOf(notifyFlg));

							new OperateSuspendOfJob().suspendJob(sessionId, jobunitId, jobId);
						}
					}
					else if(type == OperationConstant.TYPE_STOP_SET_END_VALUE){
						//終了値の指定
						
						//遅延通知状態に操作済みフラグを設定
						int notifyFlg = DelayNotifyConstant.addOperation(
								flg, DelayNotifyConstant.STOP_SET_END_VALUE);
						sessionJob.setDelay_notify_flg(Integer.valueOf(notifyFlg));
						
						new OperateStopOfJob().stopJob(sessionId, jobunitId, jobId, null);
					}
					operation = true;
				}
			}
		}
		
		return operation;
	}
	
	/**
	 * 終了状態を取得します。
	 * 
	 * @param sessionJob セッションジョブ
	 * @return 終了状態
	 * @throws NamingException
	 * 
	 * @see com.clustercontrol.bean.EndStatusConstant
	 */
	protected Integer getEndStatus(JobSessionJobLocal sessionJob) throws NamingException{
		m_log.debug("getEndStatus() : sessionId=" + sessionJob.getSession_id() + ", jobId=" + sessionJob.getJob_id());
		
		/*
		Integer status = null;
		
		if(sessionJob.getEnd_value() instanceof Integer){
			
			JobInfoLocal job = sessionJob.getJobInfo();
			Collection collection = job.getJobEndInfo();
			if(collection != null && collection.size() > 0){
				Iterator itr = collection.iterator();
				while(itr.hasNext()){
					JobEndInfoLocal end = (JobEndInfoLocal)itr.next();
					
					if(end.getEnd_value().compareTo(sessionJob.getEnd_value()) == 0){
						status = end.getEnd_status();
						break;
					}
				}
			}
		}
		
		return status;
		*/
		
		return sessionJob.getEnd_status();
	}
	
	/**
	 * ノードへの実行指示を行います。
	 * 
	 * @param sessionId セッションID
	 * @param jobId ジョブID
	 * @param facilityId ファシリティID
	 * @return true：成功、false：失敗
	 * @throws NamingException
	 * @throws JobInfoNotFoundException 
	 * 
	 * @see com.clustercontrol.bean.ProcessingMethodConstant
	 * @see com.clustercontrol.bean.StatusConstant
	 * @see com.clustercontrol.jobmanagement.bean.AgentCheckConstant
	 * @see com.clustercontrol.jobmanagement.bean.CommandTypeConstant
	 * @see com.clustercontrol.jobmanagement.bean.TopicConstant
	 * @see com.clustercontrol.jobmanagement.message.RunInstructionInfo
	 * @see com.clustercontrol.jobmanagement.util.SendTopic
	 */
	protected boolean startNode(String sessionId, String jobunitId, String jobId, String facilityId) throws NamingException, JobInfoNotFoundException {
		m_log.debug("startNode() : sessionId=" + sessionId + ", jobId=" + jobId + ", facilityId=" + facilityId);
		
		//エージェントタイムアウトチェック
		checkTimeout(sessionId, jobunitId, jobId);
		
		boolean startCommand = false;
		
		//セッションIDとジョブIDから、セッションジョブを取得
		JobSessionJobLocal sessionJob = getJobSessionJobLocal(sessionId, jobunitId, jobId);
		
		JobInfoLocal job = sessionJob.getJobInfo();
		JobCommandInfoLocal command = job.getJobCommandInfo();
		
		Collection collection = sessionJob.getJobSessionNode();
		if(collection != null && collection.size() > 0){
			//実行フラグを実行OKに設定
			boolean run = true;
			
			//コマンドの実行が正常終了するまで順次リトライの場合
			if(command.getProcess_mode().intValue() == ProcessingMethodConstant.TYPE_RETRY){
				//実行中のノードが存在するかチェック
				Iterator itr = collection.iterator();
				while(itr.hasNext()){
					JobSessionNodeLocal sessionNode = (JobSessionNodeLocal)itr.next();
					
					//実行状態が実行中の場合
					if(sessionNode.getStatus().intValue() == StatusConstant.TYPE_RUNNING){
						//実行フラグを実行NGに設定
						run = false;
						startCommand = true;
					}
				}
			}
			
			//実行フラグが実行OKの場合
			if(run){
				Iterator itr = collection.iterator();
				while(itr.hasNext()){
					JobSessionNodeLocal sessionNode = (JobSessionNodeLocal)itr.next();
					
					//ファシリティIDが指定されてない場合、指定されたファシリティIDが一致した場合
					if(facilityId == null || 
							(facilityId != null && facilityId.compareTo(sessionNode.getFacility_id()) == 0)){
						//実行状態が待機の場合
						if(sessionNode.getStatus().intValue() == StatusConstant.TYPE_WAIT){
							//実行状態を実行中にする
							sessionNode.setStatus(Integer.valueOf(StatusConstant.TYPE_RUNNING));
							
							//実行情報を作成
							RunInstructionInfo info = new RunInstructionInfo();
							info.setSessionId(sessionJob.getSession_id());
							info.setJobunitId(sessionJob.getJobunit_id());
							info.setJobId(sessionJob.getJob_id());
							info.setFacilityId(sessionNode.getFacility_id());
							info.setCommandType(CommandTypeConstant.CHECK);
							
							//エージェントチェックをチェック中にする
							sessionNode.setAgent_check(Integer.valueOf(AgentCheckConstant.CHECK));
							//チェック日時を設定
							sessionNode.setCheck_date(new Date());
							
							try {
								//Topicに送信
								m_log.debug("startNode() : send RunInstructionInfo(command type : Agent Check) : sessionId=" + sessionId + ", jobId=" + jobId + ", facilityId=" + sessionNode.getFacility_id());
								
								SendTopic send = new SendTopic(TopicConstant.TOPIC_NAME_EXECUTE);
								send.put(info);
								startCommand = true;
							} catch (Exception e) {
								m_log.debug("startNode() : RunInstructionInfo(command type : Agent Check) send error : sessionId=" + sessionId + ", jobId=" + jobId + ", facilityId=" + sessionNode.getFacility_id() + ", " + e.getMessage());
							}
							
							sessionNode = null;
							
							//コマンドの実行が正常終了するまで順次リトライの場合
							if(command.getProcess_mode().intValue() == ProcessingMethodConstant.TYPE_RETRY){
								break;
							}
						}
					}
					sessionNode = null;
				}
			}
		}
		
		return startCommand;
	}
	
	/**
	 * エージェントタイムアウトチェックを行います。
	 * 
	 * @param sessionId セッションID
	 * @param jobId ジョブID
	 * @throws NamingException
	 * @throws JobInfoNotFoundException 
	 * 
	 * @see com.clustercontrol.bean.StatusConstant
	 * @see com.clustercontrol.jobmanagement.bean.AgentCheckConstant
	 * @see com.clustercontrol.jobmanagement.bean.CommandTypeConstant
	 * @see com.clustercontrol.jobmanagement.bean.QueueConstant
	 * @see com.clustercontrol.jobmanagement.bean.RunStatusConstant
	 * @see com.clustercontrol.jobmanagement.bean.TopicConstant
	 * @see com.clustercontrol.jobmanagement.message.RunInstructionInfo
	 * @see com.clustercontrol.jobmanagement.message.RunResultInfo
	 * @see com.clustercontrol.jobmanagement.util.SendQueue
	 * @see com.clustercontrol.jobmanagement.util.SendTopic
	 */
	protected void checkTimeout(String sessionId, String jobunitId, String jobId) throws NamingException, JobInfoNotFoundException {
		m_log.debug("checkTimeout() : sessionId=" + sessionId + ", jobId=" + jobId);
		
		//セッションIDとジョブIDから、セッションジョブを取得
		JobSessionJobLocal sessionJob = getJobSessionJobLocal(sessionId, jobunitId, jobId);
		
		Collection collection = sessionJob.getJobSessionNode();
		if(collection != null && collection.size() > 0){
			
			//待ち条件ジョブ判定
			Iterator itr = collection.iterator();
			while(itr.hasNext()){
				JobSessionNodeLocal sessionNode = (JobSessionNodeLocal)itr.next();
				
				//実行状態が実行中の場合
				if(sessionNode.getStatus().intValue() == StatusConstant.TYPE_RUNNING){
					String facilityId = sessionNode.getFacility_id();
					
					Boolean retry = null;
					Calendar checkTime = Calendar.getInstance();
					checkTime.setTime(sessionNode.getCheck_date());
					checkTime.add(Calendar.SECOND, m_messageTimeout.intValue());
					Date now = new Date();
					
					//チェック中の場合
					if(sessionNode.getAgent_check().intValue() == AgentCheckConstant.CHECK){
						if(now.after(checkTime.getTime())){
							if(sessionNode.getRetry_count().compareTo(m_messageRetry) >= 0){
								//エージェントチェックをリトライ
								retry = Boolean.valueOf(false);
							}
							else{
								//エージェントチェックをリトライ
								retry = Boolean.valueOf(true);
							}
						}
					}
					else if(sessionNode.getAgent_check().intValue() == AgentCheckConstant.OK){
						if(sessionNode.getStart_date() == null && 
								now.after(checkTime.getTime())){
							if(sessionNode.getRetry_count().compareTo(m_messageRetry) >= 0){
								//エージェントチェックをリトライ
								retry = Boolean.valueOf(false);
							}
							else{
								//エージェントチェックをリトライ
								retry = Boolean.valueOf(true);
							}
						}
					}
					
					if(retry != null){
						if(retry.booleanValue()){
							//実行情報を作成
							RunInstructionInfo info = new RunInstructionInfo();
							info.setSessionId(sessionJob.getSession_id());
							info.setJobunitId(sessionJob.getJobunit_id());
							info.setJobId(sessionJob.getJob_id());
							info.setFacilityId(sessionNode.getFacility_id());
							info.setCommandType(CommandTypeConstant.CHECK);
							
							//エージェントチェックをチェック中にする
							sessionNode.setAgent_check(Integer.valueOf(AgentCheckConstant.CHECK));
							//チェック日時を設定
							sessionNode.setCheck_date(checkTime.getTime());
							int retryCount = sessionNode.getRetry_count();
							retryCount++;
							sessionNode.setRetry_count(Integer.valueOf(retryCount));
							
							sessionNode = null;
							
							try {
								//Topicに送信
								m_log.debug("checkTimeout() : send RunInstructionInfo(command type : Agent Check) : sessionId=" + sessionId + ", jobId=" + jobId + ", facilityId=" + facilityId + ", retry=" + retryCount);
								
								SendTopic send = new SendTopic(TopicConstant.TOPIC_NAME_EXECUTE);
								send.put(info);
							} catch (Exception e) {
								m_log.debug("checkTimeout() : RunInstructionInfo(command type : Agent Check) send error : sessionId=" + sessionId + ", jobId=" + jobId + ", facilityId=" + facilityId + ", " + e.getMessage());
							}
						}
						else{
							//エージェントチェックをチェックNGにする
							sessionNode.setAgent_check(Integer.valueOf(AgentCheckConstant.NG));
							
							m_log.debug("checkTimeout() : Agent Check NG : sessionId=" + sessionId + ", jobId=" + jobId + ", facilityId=" + facilityId);
							
							//実行結果情報を作成
							RunResultInfo info = new RunResultInfo();
							info.setSessionId(sessionJob.getSession_id());
							info.setJobunitId(sessionJob.getJobunit_id());
							info.setJobId(sessionJob.getJob_id());
							info.setFacilityId(sessionNode.getFacility_id());
							info.setCommand("");
							info.setCommandType(CommandTypeConstant.NORMAL);
							info.setStatus(RunStatusConstant.ERROR);
							info.setMessage("Agent Timeout Error");
							info.setErrorMessage("");
							
							sessionNode = null;
							SendQueue queue = null;
							
							try {
								//Queueに送信
								m_log.debug("checkTimeout() : send RunresultInfo : sessionId=" + sessionId + ", jobId=" + jobId + ", facilityId=" + facilityId);
								
								queue = new SendQueue(QueueConstant.QUEUE_NAME_STATUS);
								
								queue.put(info, DeliveryMode.PERSISTENT);
							} catch (Exception e) {
								m_log.debug("checkTimeout() : RunresultInfo send error : sessionId=" + sessionId + ", jobId=" + jobId + ", facilityId=" + facilityId + ", " + e.getMessage());
							} finally {
								if (queue != null) {
									try {
										queue.terminate();
									} catch (JMSException e) {
										m_log.error("Terminate SendQueue Failed : " + e);
									}
								}
							}
						}
					}
				}
				sessionNode = null;
			}
		}
	}

	/**
	 * 停止処理のエージェントタイムアウトチェックを行います。
	 * 
	 * @param sessionId セッションID
	 * @param jobId ジョブID
	 * @param  facilityId ファシリティID
	 * @throws NamingException
	 * @throws JobInfoNotFoundException 
	 * 
	 * @see com.clustercontrol.bean.StatusConstant
	 * @see com.clustercontrol.jobmanagement.bean.AgentCheckConstant
	 * @see com.clustercontrol.jobmanagement.bean.CommandTypeConstant
	 * @see com.clustercontrol.jobmanagement.bean.QueueConstant
	 * @see com.clustercontrol.jobmanagement.bean.RunStatusConstant
	 * @see com.clustercontrol.jobmanagement.bean.TopicConstant
	 * @see com.clustercontrol.jobmanagement.message.RunInstructionInfo
	 * @see com.clustercontrol.jobmanagement.message.RunResultInfo
	 * @see com.clustercontrol.jobmanagement.util.SendQueue
	 * @see com.clustercontrol.jobmanagement.util.SendTopic
	 */
	protected void checkStopTimeout(String sessionId, String jobunitId, String jobId, String facilityId) throws NamingException, JobInfoNotFoundException {
		m_log.debug("checkStopTimeout() : sessionId=" + sessionId + ", jobId=" + jobId + ", facilityId=" + facilityId);
		
		//セッションIDとジョブIDから、セッションノードを取得
		JobSessionNodeLocal sessionNode;
		try {
			sessionNode = JobSessionNodeUtil.getLocalHome().findByPrimaryKey(
					new JobSessionNodePK(sessionId, jobunitId, jobId, facilityId));
		} catch (FinderException e1) {
			m_log.warn("JobSessionNodeUtil.getLocalHome().findByPrimaryKey() : " + e1.getMessage());
			JobInfoNotFoundException je = new JobInfoNotFoundException(e1.getMessage(), e1);
			je.setSessionId(sessionId);
			je.setJobunitId(jobunitId);
			je.setJobId(jobId);
			je.setFacilityId(facilityId);
			throw je;
		}
		
		//実行状態が停止処理中の場合
		if(sessionNode.getStatus().intValue() == StatusConstant.TYPE_STOPPING){
			Boolean retry = null;
			Calendar checkTime = Calendar.getInstance();
			checkTime.setTime(sessionNode.getCheck_date());
			checkTime.add(Calendar.SECOND, m_messageTimeout.intValue());
			Date now = new Date();
			
			if(sessionNode.getStart_date() == null && 
					now.after(checkTime.getTime())){
				if(sessionNode.getRetry_count().compareTo(m_messageRetry) >= 0){
					//エージェントチェックをリトライ
					retry = Boolean.valueOf(false);
				}
				else{
					//エージェントチェックをリトライ
					retry = Boolean.valueOf(true);
				}
			}
			
			if(retry != null){
				if(retry.booleanValue()){
					JobCommandInfoLocal command;
					try {
						command = JobCommandInfoUtil.getLocalHome().findByPrimaryKey(
								new JobCommandInfoPK(sessionId, jobunitId, jobId));
					} catch (FinderException e1) {
						m_log.warn("JobCommandInfoUtil.getLocalHome().findByPrimaryKey() : " + e1.getMessage());
						JobInfoNotFoundException je = new JobInfoNotFoundException(e1.getMessage(), e1);
						je.setSessionId(sessionId);
						je.setJobunitId(jobunitId);
						je.setJobId(jobId);
						throw je;
					}
					
					//実行指示情報を作成
					RunInstructionInfo info = new RunInstructionInfo();
					info.setSessionId(sessionId);
					info.setJobunitId(jobunitId);
					info.setJobId(jobId);
					info.setFacilityId(facilityId);
					//コマンド内のパラメータを置き換える
					String stopCommand = ParameterUtil.replaceSessionParameterValue(
							sessionId, 
							command.getStop_command());
					info.setCommand(stopCommand);
					info.setUser(command.getEffective_user());
					info.setCommandType(CommandTypeConstant.STOP);
					command = null;
					
					//チェック日時を設定
					sessionNode.setCheck_date(checkTime.getTime());
					int retryCount = sessionNode.getRetry_count();
					retryCount++;
					sessionNode.setRetry_count(Integer.valueOf(retryCount));
					
					sessionNode = null;
					
					try {
						//Topicに送信
						m_log.debug("checkStopTimeout() : send RunInstructionInfo(command type:STOP) : sessionId=" + sessionId + ", jobId=" + jobId + ", facilityId=" + facilityId + ", retry=" + retryCount);
						
						SendTopic send = new SendTopic(TopicConstant.TOPIC_NAME_EXECUTE);
						send.put(info);
					} catch (Exception e) {
						m_log.debug("checkStopTimeout() : RunInstructionInfo(command type:STOP) send error : sessionId=" + sessionId + ", jobId=" + jobId + ", facilityId=" + facilityId + ", " + e.getMessage());
					}
				}
				else{
					//エージェントチェックをチェックNGにする
					sessionNode.setAgent_check(Integer.valueOf(AgentCheckConstant.NG));
					
					m_log.debug("checkStopTimeout() : AgentCheck NG : sessionId=" + sessionId + ", jobId=" + jobId + ", facilityId=" + facilityId);
					
					//実行結果情報を作成
					RunResultInfo info = new RunResultInfo();
					info.setSessionId(sessionId);
					info.setJobunitId(jobunitId);
					info.setJobId(jobId);
					info.setFacilityId(facilityId);
					info.setCommand("");
					info.setCommandType(CommandTypeConstant.STOP);
					info.setStatus(RunStatusConstant.ERROR);
					info.setMessage("Agent Timeout Error");
					info.setErrorMessage("");
					
					sessionNode = null;
					SendQueue queue = null;
					
					try {
						//Queueに送信
						m_log.debug("checkStopTimeout() : send RunResultInfo : sessionId=" + sessionId + ", jobId=" + jobId + ", facilityId=" + facilityId);
						
						queue = new SendQueue(QueueConstant.QUEUE_NAME_STATUS);
						
						queue.put(info, DeliveryMode.PERSISTENT);
					} catch (Exception e) {
						m_log.debug("checkStopTimeout() : RunResultInfo send error : sessionId=" + sessionId + ", jobId=" + jobId + ", facilityId=" + facilityId + ", " + e.getMessage());
					} finally {
						if (queue != null) {
							try {
								queue.terminate();
							} catch (JMSException e) {
								m_log.error("Terminate SendQueue Failed : " + e);
							}						
						}
					}
				}
			}
		}
		sessionNode = null;
	}
	
	/**
	 * ジョブ終了時関連処理を行います。
	 * 
	 * @param sessionId セッションID
	 * @param jobId ジョブID
	 * @throws NamingException
	 * @throws JobInfoNotFoundException 
	 * 
	 * @see com.clustercontrol.bean.StatusConstant
	 * @see com.clustercontrol.jobmanagement.factory.OperationJob#endJob(String, String)
	 * @see com.clustercontrol.jobmanagement.factory.OperationJob#startJobMain1(String, String, String)
	 * @see com.clustercontrol.jobmanagement.factory.OperationJob#checkEndStatus(String, String)
	 * @see com.clustercontrol.jobmanagement.factory.OperationJob#setEndStaus(String, String, Integer, Integer, String)
	 * @see com.clustercontrol.jobmanagement.factory.Notice#notify(String, String, Integer)
	 */
	protected void endJob(String sessionId, String jobunitId, String jobId) throws NamingException, JobInfoNotFoundException {
		m_log.debug("endJob() : sessionId=" + sessionId + ", jobId=" + jobId);
		
		//終了ジョブが待ち条件に指定されているか
		Collection collection;
		try {
			collection = JobStartJobInfoUtil.getLocalHome().findByTargetJobId(sessionId, jobId);
		} catch (FinderException e) {
			m_log.warn("JobStartJobInfoUtil.getLocalHome().findByTargetJobId() : " + e.getMessage());
			JobInfoNotFoundException je = new JobInfoNotFoundException(e.getMessage(), e);
			je.setSessionId(sessionId);
			je.setTargetJobId(jobId);
			throw je;
		}
		if(collection != null && collection.size() > 0){
			//待ち条件に指定されている場合
			
			Iterator itr = collection.iterator();
			while(itr.hasNext()){
				JobStartJobInfoLocal startJob = (JobStartJobInfoLocal)itr.next();
				
				//セッションIDとジョブIDから、セッションジョブを取得
				JobSessionJobLocal sessionJob = getJobSessionJobLocal(startJob
						.getSession_id(), startJob.getJobunit_id(), startJob
						.getJob_id());
				String startSessionId = sessionJob.getSession_id();
				String startJobUnitId = sessionJob.getJobunit_id();
				String startJobId = sessionJob.getJob_id();
				int status = sessionJob.getStatus().intValue();
				sessionJob = null;
				startJob = null;
				
				if(status == StatusConstant.TYPE_END || 
						status == StatusConstant.TYPE_MODIFIED){
					//実行状態が終了または変更済の場合
					
					//ジョブ終了時関連処理
					endJob(startSessionId, startJobUnitId, startJobId);
				}
				else{
					//実行状態が終了・変更済以外の場合
					
					//ジョブ開始処理メイン1を行う
					startJobMain1(startSessionId, startJobUnitId, startJobId, null);
				}
			}
		}
		else{
			//待ち条件に指定されていない場合
			
			//セッションIDとジョブIDから、セッションジョブを取得
			JobSessionJobLocal sessionJob = getJobSessionJobLocal(sessionId, jobunitId, jobId);
			
			//リレーションを取得し、親ジョブのジョブIDを取得
			JobRelationInfoLocal relation = sessionJob.getJobRelationInfo();
			String parentJobUnitId = relation.getParent_jobunit_id();
			String parentJobId = relation.getParent_job_id();
			relation = null;
			
			//同一階層のジョブが全て完了したかチェック
			boolean endAll = true;
			Collection sameHierarchy = null;
			try {
				sameHierarchy = JobRelationInfoUtil.getLocalHome().findByParentJobId(sessionId, parentJobId);
			} catch (FinderException e) {
				m_log.warn("JobRelationInfoUtil.getLocalHome().findByParentJobId() : " + e.getMessage());
				JobInfoNotFoundException je = new JobInfoNotFoundException(e.getMessage(), e);
				je.setSessionId(sessionId);
				je.setParentJobId(parentJobId);
				throw je;
			}
			if (sameHierarchy != null) {
				Iterator itr = sameHierarchy.iterator();
				while(itr.hasNext()){
					JobRelationInfoLocal relation1 = (JobRelationInfoLocal)itr.next();
					JobSessionJobLocal sessionJob1 = relation1.getJobSessionJob();
					
					//実行状態が終了または変更済以外の場合、同一階層のジョブは未完了
					if(sessionJob1.getStatus().intValue() != StatusConstant.TYPE_END &&
							sessionJob1.getStatus().intValue() != StatusConstant.TYPE_MODIFIED){
						endAll = false;
						break;
					}
				}
			}
				
				//同一階層のジョブが全て完了の場合
			if(endAll){
				if(parentJobId.compareTo("TOP") != 0){
					//セッションIDとジョブIDから、セッションジョブを取得
					JobSessionJobLocal parentSessionJob = getJobSessionJobLocal(sessionId, parentJobUnitId, parentJobId);
					
					if(parentSessionJob.getStatus().intValue() == StatusConstant.TYPE_RUNNING){
						parentSessionJob = null;
						
						//終了状態を判定し、終了状態と終了値を設定
						Integer endStatus = checkEndStatus(sessionId, parentJobUnitId, parentJobId);
						
						//実行状態、終了状態、終了値、終了日時を設定
						setEndStaus(sessionId, parentJobUnitId, parentJobId, 
								Integer.valueOf(StatusConstant.TYPE_END), endStatus, null);
						
						//通知処理
						new Notice().notify(sessionId, parentJobUnitId, parentJobId, endStatus);
						
						//ジョブ終了時関連処理（再帰呼び出し）
						endJob(sessionId, parentJobUnitId, parentJobId);
					}
				}
			}
			
		}
	}
	
	/**
	 * 終了状態をチェックします
	 * 
	 * @param sessionId セッションID
	 * @param jobId ジョブID
	 * @return 終了状態
	 * @throws NamingException
	 * @throws JobInfoNotFoundException 
	 * 
	 * @see com.clustercontrol.bean.EndStatusConstant
	 * @see com.clustercontrol.bean.JobConstant
	 * @see com.clustercontrol.bean.ProcessingMethodConstant
	 * @see com.clustercontrol.jobmanagement.bean.EndStatusCheckConstant
	 * @see com.clustercontrol.jobmanagement.factory.EndJudgment#judgment(ArrayList)
	 */
	protected Integer checkEndStatus(String sessionId, String jobunitId, String jobId) throws NamingException, JobInfoNotFoundException{
		m_log.debug("checkEndStatus() : sessionId=" + sessionId + ", jobId=" + jobId);
		
		//セッションIDとジョブIDから、セッションジョブを取得
		JobSessionJobLocal sessionJob = getJobSessionJobLocal(sessionId, jobunitId, jobId);
		
		//終了状態（正常）を取得
		JobEndInfoLocal endInfoNormal = null;
		try {
			endInfoNormal = JobEndInfoUtil.getLocalHome().findByPrimaryKey(
					new JobEndInfoPK(sessionId, jobunitId, jobId, Integer.valueOf(EndStatusConstant.TYPE_NORMAL)));
		} catch (FinderException e) {
			m_log.warn("JobEndInfoUtil.getLocalHome().findByPrimaryKey() : " + e.getMessage());
			JobInfoNotFoundException je = new JobInfoNotFoundException(e.getMessage(), e);
			je.setSessionId(sessionId);
			je.setJobId(jobId);
			
			throw je;
		}
		
		
		//終了状態（警告）を取得
		JobEndInfoLocal endInfoWarning = null;
		try {
			endInfoWarning = JobEndInfoUtil.getLocalHome().findByPrimaryKey(
					new JobEndInfoPK(sessionId, jobunitId, jobId, Integer.valueOf(EndStatusConstant.TYPE_WARNING)));
		} catch (FinderException e) {
			m_log.warn("JobRelationInfoUtil.getLocalHome().findByParentJobId() : " + e.getMessage());
			JobInfoNotFoundException je = new JobInfoNotFoundException(e.getMessage(), e);
			je.setSessionId(sessionId);
			je.setJobId(jobId);
			je.setEndStatus(Integer.valueOf(EndStatusConstant.TYPE_WARNING));
			throw je;
		}
		
		ArrayList<Integer> statusList = new ArrayList<Integer>();
		
		if(sessionJob.getJobInfo().getJob_type().intValue() == JobConstant.TYPE_JOB){
			//ジョブの場合
			
			JobInfoLocal job = sessionJob.getJobInfo();
			JobCommandInfoLocal command = job.getJobCommandInfo();
			
			//セッションジョブからセッションノードを取得
			Collection collection = sessionJob.getJobSessionNode();
			Iterator itr = collection.iterator();
			while(itr.hasNext()){
				JobSessionNodeLocal sessionNode = (JobSessionNodeLocal)itr.next();
				
				Integer endValue = sessionNode.getEnd_value();
				if(endValue instanceof Integer){
					Integer status = null;
					if(endValue.compareTo(endInfoNormal.getEnd_value_from()) >= 0
							&& endValue.compareTo(endInfoNormal.getEnd_value_to()) <= 0){
						//終了状態（正常）の範囲内ならば、正常とする
						status = Integer.valueOf(EndStatusConstant.TYPE_NORMAL);
						statusList.add(status);
					}
					else if(endValue.compareTo(endInfoWarning.getEnd_value_from()) >= 0
							&& endValue.compareTo(endInfoWarning.getEnd_value_to()) <= 0){
						//終了状態（警告）の範囲内ならば、警告とする
						status = Integer.valueOf(EndStatusConstant.TYPE_WARNING);
						statusList.add(status);
					}
					else{
						//終了状態（異常）の範囲内ならば、異常とする
						status = Integer.valueOf(EndStatusConstant.TYPE_ABNORMAL);
						statusList.add(status);
					}
					
					//コマンドの実行が正常終了するまで順次リトライの場合
					if(command.getProcess_mode().intValue() == ProcessingMethodConstant.TYPE_RETRY){
						if(status.intValue() == EndStatusConstant.TYPE_NORMAL){
							statusList.clear();
							statusList.add(Integer.valueOf(EndStatusConstant.TYPE_NORMAL));
							break;
						}
					}
				}
			}
			//配下にセッションノードが存在しない場合
			if(collection.size() == 0){
				statusList.clear();
				statusList.add(Integer.valueOf(EndStatusConstant.TYPE_ABNORMAL));
			}
		}
		else{
			//ジョブ以外の場合
			
			Integer endStatusCheck = sessionJob.getEnd_staus_check_flg();
			if(endStatusCheck == null || (endStatusCheck != null && 
					endStatusCheck.intValue() == EndStatusCheckConstant.NO_WAIT_JOB)){
				//待ち条件にしていされていないジョブのみで判定
				
				//セッションIDとジョブIDの直下のジョブを取得
				Collection collection = null;
				try {
					collection = JobRelationInfoUtil.getLocalHome().findByParentJobId(sessionId, jobId);
				} catch (FinderException e1) {
					m_log.warn("JobRelationInfoUtil.getLocalHome().findByParentJobId() : " + e1.getMessage());
					JobInfoNotFoundException je = new JobInfoNotFoundException(e1.getMessage(), e1);
					je.setSessionId(sessionId);
					je.setParentJobId(jobId);
					throw je;
				}
				Iterator itr = collection.iterator();
				while(itr.hasNext()){
					JobRelationInfoLocal relation = (JobRelationInfoLocal)itr.next();
					JobSessionJobLocal childSessionJob = relation.getJobSessionJob();
					
					//待ち条件に指定されているかチェック
					Collection targetJobList = null;
					try {
						targetJobList = JobStartJobInfoUtil.getLocalHome().findByTargetJobId(sessionId, childSessionJob.getJob_id());
					} catch (FinderException e) {
						m_log.warn("JobStartJobInfoUtil.getLocalHome().findByTargetJobId() : " + e.getMessage());
						JobInfoNotFoundException je = new JobInfoNotFoundException(e.getMessage(), e);
						je.setSessionId(sessionId);
						je.setTargetJobId(childSessionJob.getJob_id());
						throw je;
					}
					if(targetJobList == null || 
							(targetJobList != null && targetJobList.size() == 0)){
						//待ち条件に指定されていないジョブ及びジョブネットを対象にする
						
						Integer endValue = childSessionJob.getEnd_value();
						if(endValue.compareTo(endInfoNormal.getEnd_value_from()) >= 0
								&& endValue.compareTo(endInfoNormal.getEnd_value_to()) <= 0){
							//終了状態（正常）の範囲内ならば、正常とする
							statusList.add(Integer.valueOf(EndStatusConstant.TYPE_NORMAL));
						}
						else if(endValue.compareTo(endInfoWarning.getEnd_value_from()) >= 0
								&& endValue.compareTo(endInfoWarning.getEnd_value_to()) <= 0){
							//終了状態（警告）の範囲内ならば、警告とする
							statusList.add(Integer.valueOf(EndStatusConstant.TYPE_WARNING));
						}
						else{
							//終了状態（異常）の範囲内ならば、異常とする
							statusList.add(Integer.valueOf(EndStatusConstant.TYPE_ABNORMAL));
						}
					}
				}
				//配下にセッションジョブが存在しない場合
				if(collection.size() == 0){
					statusList.clear();
					statusList.add(Integer.valueOf(EndStatusConstant.TYPE_ABNORMAL));
				}
			}
			else{
				//全ジョブで判定
				
				//セッションIDとジョブIDの直下のジョブを取得
				Collection collection = null;
				try {
					collection = JobRelationInfoUtil.getLocalHome().findByParentJobId(sessionId, jobId);
				} catch (FinderException e) {
					m_log.warn("JobRelationInfoUtil.getLocalHome().findByParentJobId() : " + e.getMessage());
					JobInfoNotFoundException je = new JobInfoNotFoundException(e.getMessage(), e);
					je.setSessionId(sessionId);
					je.setParentJobId(jobId);
					throw je;
				}
				Iterator itr = collection.iterator();
				while(itr.hasNext()){
					JobRelationInfoLocal relation = (JobRelationInfoLocal)itr.next();
					JobSessionJobLocal childSessionJob = relation.getJobSessionJob();
					
					Integer endValue = childSessionJob.getEnd_value();
					if(endValue.compareTo(endInfoNormal.getEnd_value_from()) >= 0
							&& endValue.compareTo(endInfoNormal.getEnd_value_to()) <= 0){
						//終了状態（正常）の範囲内ならば、正常とする
						statusList.add(Integer.valueOf(EndStatusConstant.TYPE_NORMAL));
					}
					else if(endValue.compareTo(endInfoWarning.getEnd_value_from()) >= 0
							&& endValue.compareTo(endInfoWarning.getEnd_value_to()) <= 0){
						//終了状態（警告）の範囲内ならば、警告とする
						statusList.add(Integer.valueOf(EndStatusConstant.TYPE_WARNING));
					}
					else{
						//終了状態（異常）の範囲内ならば、異常とする
						statusList.add(Integer.valueOf(EndStatusConstant.TYPE_ABNORMAL));
					}
				}
				//配下にセッションジョブが存在しない場合
				if(collection.size() == 0){
					statusList.clear();
					statusList.add(Integer.valueOf(EndStatusConstant.TYPE_ABNORMAL));
				}
			}
		}
		
		//終了判定を行う。
		Integer endStatus = EndJudgment.judgment(statusList);
		
		return endStatus;
	}
	
	/**
	 * 終了状態を設定します。
	 * 
	 * @param sessionId セッションID
	 * @param jobId ジョブID
	 * @param status 実行状態
	 * @param endStatus 終了状態
	 * @param result 結果
	 * @throws NamingException
	 * @throws JobInfoNotFoundException 
	 * @throws JobInfoNotFoundException 
	 * 
	 * @see com.clustercontrol.bean.EndStatusConstant
	 */
	protected void setEndStaus(
			String sessionId, 
			String jobunitId, 
			String jobId, 
			Integer status, 
			Integer endStatus, 
			String result) throws NamingException, JobInfoNotFoundException {
		
		m_log.debug("setEndStaus() : sessionId=" + sessionId + ", jobId=" + jobId);
		
		//セッションIDとジョブIDから、セッションジョブを取得
		JobSessionJobLocal sessionJob = getJobSessionJobLocal(sessionId, jobunitId, jobId);
		
		//終了状態（正常）を取得
		JobEndInfoLocal endInfoNormal = null;
		try {
			endInfoNormal = JobEndInfoUtil.getLocalHome().findByPrimaryKey(
					new JobEndInfoPK(sessionId, jobunitId, jobId, Integer.valueOf(EndStatusConstant.TYPE_NORMAL)));
		} catch (FinderException e) {
			m_log.warn("JobEndInfoUtil.getLocalHome().findByPrimaryKey() : " + e.getMessage());
			JobInfoNotFoundException je = new JobInfoNotFoundException(e.getMessage(), e);
			je.setSessionId(sessionId);
			je.setJobunitId(jobunitId);
			je.setJobId(jobId);
			je.setEndStatus(Integer.valueOf(EndStatusConstant.TYPE_NORMAL));
			throw je;
		}
		
		//終了状態（警告）を取得
		JobEndInfoLocal endInfoWarning = null;
		try {
			endInfoWarning = JobEndInfoUtil.getLocalHome().findByPrimaryKey(
					new JobEndInfoPK(sessionId, jobunitId, jobId, Integer.valueOf(EndStatusConstant.TYPE_WARNING)));
		} catch (FinderException e) {
			m_log.warn("JobEndInfoUtil.getLocalHome().findByPrimaryKey() : " + e.getMessage());
			JobInfoNotFoundException je = new JobInfoNotFoundException(e.getMessage(), e);
			je.setSessionId(sessionId);
			je.setJobunitId(jobunitId);
			je.setJobId(jobId);
			je.setEndStatus(Integer.valueOf(EndStatusConstant.TYPE_WARNING));
			throw je;
		}
		
		//終了状態（異常）を取得
		JobEndInfoLocal endInfoAbnormal = null;
		try {
			endInfoAbnormal = JobEndInfoUtil.getLocalHome().findByPrimaryKey(
					new JobEndInfoPK(sessionId, jobunitId, jobId, Integer.valueOf(EndStatusConstant.TYPE_ABNORMAL)));
		} catch (FinderException e) {
			m_log.warn("JobEndInfoUtil.getLocalHome().findByPrimaryKey() : " + e.getMessage());
			JobInfoNotFoundException je = new JobInfoNotFoundException(e.getMessage(), e);
			je.setSessionId(sessionId);
			je.setJobunitId(jobunitId);
			je.setJobId(jobId);
			je.setEndStatus(Integer.valueOf(EndStatusConstant.TYPE_ABNORMAL));
			throw je;
		}
		
		//実行状態を設定
		sessionJob.setStatus(status);
		
		//終了状態を設定
		sessionJob.setEnd_status(endStatus);
		//終了値を設定
		if(endInfoNormal.getEnd_status().compareTo(endStatus) == 0){
			sessionJob.setEnd_value(endInfoNormal.getEnd_value());
		}
		else if(endInfoWarning.getEnd_status().compareTo(endStatus) == 0){
			sessionJob.setEnd_value(endInfoWarning.getEnd_value());
		}
		else if(endInfoAbnormal.getEnd_status().compareTo(endStatus) == 0){
			sessionJob.setEnd_value(endInfoAbnormal.getEnd_value());
		}
		//終了日時を設定
		sessionJob.setEnd_date(new Date());
		//結果を設定
		sessionJob.setResult(result);
	}
	/**
	 * ノード終了処理を行います。
	 * 
	 * @param info 実行結果情報
	 * @throws NamingException
	 * @throws JobInfoNotFoundException 
	 * 
	 * @see com.clustercontrol.bean.StatusConstant
	 * @see com.clustercontrol.bean.EndStatusConstant
	 * @see com.clustercontrol.bean.YesNoConstant
	 * @see com.clustercontrol.jobmanagement.bean.CommandTypeConstant
	 * @see com.clustercontrol.jobmanagement.bean.AgentCheckConstant
	 * @see com.clustercontrol.jobmanagement.bean.QueueConstant
	 * @see com.clustercontrol.jobmanagement.bean.RunStatusConstant
	 * @see com.clustercontrol.jobmanagement.bean.TopicConstant
	 * @see com.clustercontrol.jobmanagement.message.RunInstructionInfo
	 * @see com.clustercontrol.jobmanagement.message.RunResultInfo
	 * @see com.clustercontrol.jobmanagement.factory.OperationJob#checkAllNodeStop(JobSessionJobLocal)
	 * @see com.clustercontrol.jobmanagement.factory.OperationJob#checkJobEnd(JobSessionJobLocal)
	 * @see com.clustercontrol.jobmanagement.factory.OperationJob#checkEndStatus(String, String)
	 * @see com.clustercontrol.jobmanagement.factory.OperationJob#setEndStaus(String, String, Integer, Integer, String)
	 * @see com.clustercontrol.jobmanagement.factory.OperationJob#endJob(String, String)
	 * @see com.clustercontrol.jobmanagement.factory.OperateStopOfJob#stopJob(String, String)
	 * @see com.clustercontrol.jobmanagement.util.SendTopic
	 * @see com.clustercontrol.jobmanagement.util.SendQueue
	 * @see com.clustercontrol.jobmanagement.util.ParameterUtil
	 * @see com.clustercontrol.jobmanagement.factory.Notice#notify(String, String, Integer)
	 */
	public void endNode(RunResultInfo info) throws NamingException, JobInfoNotFoundException {
		m_log.debug("endNode() : sessionId=" + info.getSessionId() + ", jobId=" + info.getJobId() + ", facilityId=" + info.getFacilityId());
		
		//セッションIDとジョブIDから、セッションジョブを取得
		JobSessionJobLocal sessionJob = getJobSessionJobLocal(info.getSessionId(), info.getJobunitId(), info.getJobId());

		JobInfoLocal job = sessionJob.getJobInfo();
		JobCommandInfoLocal command = job.getJobCommandInfo();
		
		//セッションノード取得
		JobSessionNodeLocal sessionNode;
		try {
			sessionNode = JobSessionNodeUtil.getLocalHome().findByPrimaryKey(
					new JobSessionNodePK(info.getSessionId(), info.getJobunitId(), info.getJobId(), info.getFacilityId()));
		} catch (FinderException e1) {
			m_log.warn("JobSessionNodeUtil.getLocalHome().findByPrimaryKey() : " + e1.getMessage());
			JobInfoNotFoundException je = new JobInfoNotFoundException(e1.getMessage(), e1);
			je.setSessionId(info.getSessionId());
			je.setJobunitId(info.getJobunitId());
			je.setJobId(info.getJobId());
			je.setFacilityId(info.getFacilityId());
			throw je;
		}
		
		//コマンドタイプチェック
		if(info.getCommandType() == CommandTypeConstant.CHECK){
			//コマンドタイプがチェックの場合

			//実行状態が実行中で、エージェントチェック中の場合
			if(sessionNode.getStatus().intValue() == StatusConstant.TYPE_RUNNING){
				//エージェントチェック結果をチェック
				if(sessionNode.getAgent_check().intValue() == AgentCheckConstant.CHECK){
					//チェック中の場合
					
					//エージェントチェックをチェックOKにする
					sessionNode.setAgent_check(Integer.valueOf(AgentCheckConstant.OK));
					//チェック日時を設定
					sessionNode.setCheck_date(new Date());
					
					//実行指示情報を作成
					RunInstructionInfo instructionInfo = new RunInstructionInfo();
					instructionInfo.setSessionId(sessionJob.getSession_id());
					instructionInfo.setJobunitId(sessionJob.getJobunit_id());
					instructionInfo.setJobId(sessionJob.getJob_id());
					instructionInfo.setFacilityId(sessionNode.getFacility_id());
					//コマンド内のパラメータを置き換える
					String startCommand = ParameterUtil.replaceSessionParameterValue(
							info.getSessionId(), 
							command.getStart_command());
					instructionInfo.setCommand(startCommand);
					instructionInfo.setUser(command.getEffective_user());
					instructionInfo.setCommandType(CommandTypeConstant.NORMAL);
					
					//特殊コマンド
					if(instructionInfo.getCommand().equals(CommandConstant.ADD_PUBLIC_KEY) || 
							instructionInfo.getCommand().equals(CommandConstant.DELETE_PUBLIC_KEY)){
						//公開鍵設定
						
						//セッションIDとジョブIDから、セッションジョブを取得
						//FIXME 引数のジョブIDって何ですか？
						JobSessionJobLocal argumentSessionJob = getJobSessionJobLocal(
								info.getSessionId(), info.getJobunitId(),
								command.getArgument_job_id());
						String result = argumentSessionJob.getResult();
						argumentSessionJob = null;
						
						//設定したい公開鍵のファシリティIDを設定
						instructionInfo.setPublicKey(result);
					}
					else if(instructionInfo.getCommand().equals(CommandConstant.GET_FILE_LIST)){
						//ファイルリスト取得
						
						//取得したいファイルリストのパスを設定
						instructionInfo.setFilePath(command.getArgument());
					}
					else if(instructionInfo.getCommand().equals(CommandConstant.GET_CHECKSUM)){
						//チェックサム取得
						
						//チェックサムを取得するファイルパスを設定
						instructionInfo.setFilePath(command.getArgument());
					}
					else if(instructionInfo.getCommand().equals(CommandConstant.CHECK_CHECKSUM)){
						//整合性チェック
						
						//セッションIDとジョブIDから、セッションジョブを取得
						JobSessionJobLocal argumentSessionJob = getJobSessionJobLocal(info.getSessionId(), info.getJobunitId(), command.getArgument_job_id());
						String result = argumentSessionJob.getResult();
						argumentSessionJob = null;
						
						//チェックサムを設定
						instructionInfo.setCheckSum(result);
						//整合性チェックするファイルパスを設定
						instructionInfo.setFilePath(command.getArgument());
					}

					try {
						//Topicに送信
						SendTopic send = new SendTopic(TopicConstant.TOPIC_NAME_EXECUTE);
						send.put(instructionInfo);
					} catch (Exception e) {
						m_log.debug("endNode() : RunInstructionInfo send error : " + e.getMessage());
					}
				}
			}
			else{
				//実行状態が終了、変更済、起動失敗、停止、中断以外の場合、実行結果を再受信
				if(sessionNode.getStatus().intValue() != StatusConstant.TYPE_END &&
						sessionNode.getStatus().intValue() != StatusConstant.TYPE_MODIFIED && 
						sessionNode.getStatus().intValue() != StatusConstant.TYPE_ERROR && 
						sessionNode.getStatus().intValue() != StatusConstant.TYPE_STOP && 
						sessionNode.getStatus().intValue() != StatusConstant.TYPE_SUSPEND){

					SendQueue queue = null;
					
					try {
						m_log.debug("endNode() : put RunResultInfo");
						queue = new SendQueue(QueueConstant.QUEUE_NAME_STATUS);
						
						queue.put(info, DeliveryMode.PERSISTENT);
					} catch (Exception e) {
						m_log.debug("endNode() : wait RunResultInfo : " + e.getMessage());
					} finally {
						if (queue != null) {
							try {
								queue.terminate();
							} catch (JMSException e) {
								m_log.error("Terminate SendQueue Failed : " + e);
							}					
						}
					}
				}
			}
		}
		else if(info.getCommandType() == CommandTypeConstant.NORMAL || 
				info.getCommandType() == CommandTypeConstant.STOP){
			//コマンドタイプが通常の場合
			
			//実行状態で分岐
			if(info.getStatus() == RunStatusConstant.START){
				//開始の場合
				
				if(sessionNode.getStart_date() == null){
					//開始・再実行日時を設定
					sessionNode.setStart_date(info.getTime());
					
					if(sessionNode.getAgent_check().intValue() != AgentCheckConstant.OK){
						//チェック中の場合
						m_log.debug("agent check OK : status=" + info.getStatus() + ", sessionId=" + info.getSessionId() + ", jobId=" + info.getJobId() + ", facilityId=" + info.getFacilityId());
						
						//エージェントチェックをチェックOKにする
						sessionNode.setAgent_check(Integer.valueOf(AgentCheckConstant.OK));
						//チェック日時を設定
						sessionNode.setCheck_date(new Date());
					}
				}
				else{
					m_log.debug("endNode() : this messsage is already received. drop message." + 
							" sessionId=" + info.getSessionId() + 
							", jobId=" + info.getJobId() + 
							", facilityId=" + info.getFacilityId());
				}
			}
			else{
				if(info.getStatus() == RunStatusConstant.END){
					//終了の場合
					
					//開始・再実行日時が設定済みならば、終了処理を行う。
					if(sessionJob.getStart_date() instanceof Date || sessionNode.getStart_date() instanceof Date){
						if(sessionJob.getEnd_date() == null || 
								sessionJob.getStatus().intValue() == StatusConstant.TYPE_RUNNING || 
								sessionJob.getStatus().intValue() == StatusConstant.TYPE_SUSPEND || 
								sessionJob.getStatus().intValue() == StatusConstant.TYPE_STOPPING){
					
							//実行状態バッファを設定
							if(sessionNode.getStatus().intValue() == StatusConstant.TYPE_RUNNING){
								//実行状態が実行中の場合、実行状態バッファに終了を設定
								sessionNode.setStatus_buffer(Integer.valueOf(StatusConstant.TYPE_END));
								//終了・中断日時を設定
								sessionNode.setEnd_date(info.getTime());
							}
							else if(sessionNode.getStatus().intValue() == StatusConstant.TYPE_STOPPING){
								if(info.getCommandType() == CommandTypeConstant.NORMAL && 
										sessionNode.getStatus_buffer().intValue() != StatusConstant.TYPE_STOP){
									//実行状態が停止処理中の場合、実行状態バッファに停止処理中を設定
									sessionNode.setStatus_buffer(Integer.valueOf(StatusConstant.TYPE_STOPPING));
								}
								else if(info.getCommandType() == CommandTypeConstant.STOP){
									//実行状態が停止処理中の場合、実行状態バッファにコマンド停止を設定
									sessionNode.setStatus_buffer(Integer.valueOf(StatusConstant.TYPE_STOP));
								}
							}
							//終了値を設定
							sessionNode.setEnd_value(Integer.valueOf(info.getEndValue()));
							//メッセージを設定
							sessionNode.setMessage(info.getMessage() + info.getErrorMessage());
							
							//特殊コマンド
							if(info.getCommand().equals(CommandConstant.GET_PUBLIC_KEY)){
								//公開鍵取得
								sessionNode.setResult(info.getPublicKey());
							}
							else if(info.getCommand().equals(CommandConstant.GET_CHECKSUM)){
								//チェックサム取得
								sessionNode.setResult(info.getCheckSum());
							}
						}
						else{
							m_log.debug("endNode() : this messsage is already received. drop message." + 
									" sessionId=" + info.getSessionId() + 
									", jobId=" + info.getJobId() + 
									", facilityId=" + info.getFacilityId());
						}
					}
					else{
						// 起動時刻が無い状態で停止通知を受信した場合
						m_log.warn("endNode() : this messsage does not have start time. drop message." + 
								" sessionId=" + info.getSessionId() + 
								", jobId=" + info.getJobId() + 
								", facilityId=" + info.getFacilityId());
					}
				}
				else if(info.getStatus() == RunStatusConstant.ERROR){
					//失敗の場合
					
					//実行状態、終了値、終了・中断日時を設定
					if(sessionNode.getStatus().intValue() == StatusConstant.TYPE_RUNNING){
						//エラー時に終了にする
						if(command.getError_end_flg().intValue() == YesNoConstant.TYPE_YES){
							//実行状態が実行中の場合、実行状態バッファに終了を設定
							sessionNode.setStatus_buffer(Integer.valueOf(StatusConstant.TYPE_END));
							
							//終了・中断日時を設定
							sessionNode.setEnd_date(new Date());
							//終了値を設定
							sessionNode.setEnd_value(command.getError_end_value());
						}
						else{
							//実行状態が実行中の場合、実行状態バッファに実行失敗を設定
							sessionNode.setStatus_buffer(Integer.valueOf(StatusConstant.TYPE_ERROR));
						}
					}
					else if(sessionNode.getStatus().intValue() == StatusConstant.TYPE_STOPPING){
						//実行状態が停止処理中の場合、実行状態バッファにコマンド停止を設定
						sessionNode.setStatus_buffer(Integer.valueOf(StatusConstant.TYPE_STOP));
					}
					
					//メッセージを設定
					sessionNode.setMessage(info.getMessage() + info.getErrorMessage());
				}
				
				//履歴削除情報を作成
				if(sessionNode.getAgent_check().intValue() == AgentCheckConstant.OK){
					m_log.debug("send delete history : status=" + info.getStatus() + ", sessionId=" + info.getSessionId() + ", jobId=" + info.getJobId() + ", facilityId=" + info.getFacilityId());
					RunInstructionInfo instructionInfo = new RunInstructionInfo();
					instructionInfo.setSessionId(sessionJob.getSession_id());
					instructionInfo.setJobunitId(sessionJob.getJobunit_id());
					instructionInfo.setJobId(sessionJob.getJob_id());
					instructionInfo.setFacilityId(sessionNode.getFacility_id());
					instructionInfo.setCommand("");
					if(info.getCommandType() == CommandTypeConstant.NORMAL)
						instructionInfo.setCommandType(CommandTypeConstant.DELETE_NORMAL_HISTORY);
					else if(info.getCommandType() == CommandTypeConstant.STOP)
						instructionInfo.setCommandType(CommandTypeConstant.DELETE_STOP_HISTORY);
					
					try {
						//Topicに送信
						SendTopic send = new SendTopic(TopicConstant.TOPIC_NAME_EXECUTE);
						send.put(instructionInfo);
					} catch (Exception e) {
						m_log.debug("endNode() : RunInstructionInfo send error : " + e.getMessage());
					}
				}
				else {
					m_log.debug("set status buffer : status=" + info.getStatus() + ", sessionId=" + info.getSessionId() + ", jobId=" + info.getJobId() + ", facilityId=" + info.getFacilityId());
					//実行状態バッファを実行状態に設定
					sessionNode.setStatus(sessionNode.getStatus_buffer());
				}
			}
		}
		else if(info.getCommandType() == CommandTypeConstant.DELETE_NORMAL_HISTORY || 
				info.getCommandType() == CommandTypeConstant.DELETE_STOP_HISTORY) {
			//実行状態バッファを実行状態に設定
			sessionNode.setStatus(sessionNode.getStatus_buffer());
		}
		
		//実行状態チェック
		if(sessionNode.getStatus().intValue() == StatusConstant.TYPE_STOP){
			//実行状態がコマンド停止の場合
			sessionNode = null;
			
			if(sessionJob.getStatus().intValue() == StatusConstant.TYPE_STOPPING){
				//全ノード停止チェック
				if(checkAllNodeStop(sessionJob)){
					//全ノード停止の場合
					
					//実行状態にコマンド停止を設定
					sessionJob.setStatus(Integer.valueOf(StatusConstant.TYPE_STOP));
					
					//ジョブ停止関連処理
					new OperateStopOfJob().stopJob(
							info.getSessionId(), 
							info.getJobunitId(), 
							info.getJobId());
					
					JobStartInfoLocal start = job.getJobStartInfo();
					
					//遅延通知状態を取得
					int flg = sessionJob.getDelay_notify_flg().intValue();
					//遅延通知状態から操作済みフラグを取得
					int operationFlg = DelayNotifyConstant.getOperation(flg);
					if(operationFlg == DelayNotifyConstant.STOP_SET_END_VALUE){
						//操作済みフラグが停止[終了値の指定]の場合、停止[終了値の変更]を行う
						new OperateMaintenanceOfJob().maintenanceJob(
								info.getSessionId(), 
								info.getJobunitId(), 
								info.getJobId(), 
								null, 
								start.getEnd_delay_operation_end_value());
					}
				}
			}
		}
		else{
			//実行状態がコマンド停止以外の場合
			String result = sessionNode.getResult();
			sessionNode = null;
			
			if(sessionJob.getStatus().intValue() != StatusConstant.TYPE_SUSPEND){
				
				//ジョブ終了チェック
				if(checkJobEnd(sessionJob)){
					//ジョブ終了の場合
					sessionJob = null;
					
					//終了状態を取得
					Integer endStatus = checkEndStatus(info.getSessionId(), info.getJobunitId(), info.getJobId());
					
					if(endStatus.intValue() == EndStatusConstant.TYPE_NORMAL && 
							info.getCommand().equals(CommandConstant.GET_FILE_LIST)){
						new CreateFileJob().createFileJobNet(
								info.getSessionId(), 
								info.getJobunitId(), 
								info.getJobId(), 
								info.getFileList());
					}

					//実行状態、終了状態、終了値、終了日時を設定
					setEndStaus(info.getSessionId(), info.getJobunitId(), info.getJobId(), 
							Integer.valueOf(StatusConstant.TYPE_END), endStatus, result);
					
					//通知処理
					new Notice().notify(info.getSessionId(), info.getJobunitId(), info.getJobId(), endStatus);
					
					//ジョブ終了時関連処理（再帰呼び出し）
					endJob(info.getSessionId(), info.getJobunitId(), info.getJobId());
				}
			}
		}
	}
	
	/**
	 * ジョブネットの終了チェックを行います。
	 * 
	 * @param sessionId セッションID
	 * @param jobId ジョブID
	 * @return true：終了、false：未終了
	 * @throws NamingException
	 * @throws JobInfoNotFoundException 
	 * 
	 * @see com.clustercontrol.bean.StatusConstant
	 */
	protected boolean checkJobNetEnd(String sessionId, String jobId) throws NamingException, JobInfoNotFoundException{
		m_log.debug("checkJobNetEnd() : sessionId=" + sessionId + ", jobId=" + jobId);
		
		//直下のジョブが全て完了したかチェック
		boolean endAll = true;
		try {
			Collection sameHierarchy = JobRelationInfoUtil.getLocalHome().findByParentJobId(sessionId, jobId);
			Iterator itr = sameHierarchy.iterator();
			while(itr.hasNext()){
				JobRelationInfoLocal relation1 = (JobRelationInfoLocal)itr.next();
				JobSessionJobLocal sessionJob1 = relation1.getJobSessionJob();
				
				//実行状態が終了または変更済以外の場合、同一階層のジョブは未完了
				if(sessionJob1.getStatus().intValue() != StatusConstant.TYPE_END &&
						sessionJob1.getStatus().intValue() != StatusConstant.TYPE_MODIFIED){
					endAll = false;
					break;
				}
			}
		} catch (FinderException e) {
			m_log.warn("JobRelationInfoUtil.getLocalHome().findByParentJobId() : " + e.getMessage());
			JobInfoNotFoundException je = new JobInfoNotFoundException(e.getMessage(), e);
			je.setSessionId(sessionId);
			je.setParentJobId(jobId);
			throw je;
		}
		
		return endAll;
	}
	
	/**
	 * ジョブの終了チェックを行います。
	 * 
	 * @param sessionJob セッションジョブ
	 * @return true：終了、false：未終了
	 * @throws NamingException
	 * @throws JobInfoNotFoundException 
	 * 
	 * @see com.clustercontrol.bean.EndStatusConstant
	 * @see com.clustercontrol.bean.ProcessingMethodConstant
	 * @see com.clustercontrol.jobmanagement.factory.OperationJob#checkEndStatus(String, String)
	 * @see com.clustercontrol.jobmanagement.factory.OperationJob#checkAllNodeEnd(JobSessionJobLocal)
	 */
	protected boolean checkJobEnd(JobSessionJobLocal sessionJob) throws NamingException, JobInfoNotFoundException{
		m_log.debug("checkJobEnd() : sessionId=" + sessionJob.getSession_id() + ", jobId=" + sessionJob.getJob_id());
		
		JobInfoLocal job = sessionJob.getJobInfo();
		JobCommandInfoLocal command = job.getJobCommandInfo();
		
		//ジョブ終了フラグをfalseにする
		boolean end = false;
		
		//実行タイプをチェック
		if(command.getProcess_mode().intValue() == ProcessingMethodConstant.TYPE_RETRY){
			//正常するまで順次リトライ
			
			//終了状態を取得
			Integer endStatus = checkEndStatus(sessionJob.getSession_id(), sessionJob.getJobunit_id(), sessionJob.getJob_id());
			
			if(endStatus instanceof Integer &&
					endStatus.intValue() == EndStatusConstant.TYPE_NORMAL){
				//正常終了の場合
				
				//ジョブ終了フラグをtrueにする
				end = true;
			}
			else{
				//次のノードで実行
				if(!startNode(sessionJob.getSession_id(), sessionJob.getJobunit_id(), sessionJob.getJob_id(), null)){
					//実行ノードなしの場合
					
					//ジョブ終了フラグをtrueにする
					end = true;
					
					//全ノード終了をチェック
					end = checkAllNodeEnd(sessionJob);;
				}
			}
		}
		else{
			//すべてのノードで実行
			
			//全ノード終了をチェック
			end = checkAllNodeEnd(sessionJob);
		}
		
		return end;
	}
	
	/**
	 * 全ノードの終了チェックを行います。
	 * 
	 * @param sessionJob セッションジョブ
	 * @return true：終了、false：未終了
	 * @throws NamingException
	 * 
	 * @see com.clustercontrol.bean.StatusConstant
	 */
	protected boolean checkAllNodeEnd(JobSessionJobLocal sessionJob) throws  NamingException{
		m_log.debug("checkAllNodeEnd() : sessionId=" + sessionJob.getSession_id() + ", jobId=" + sessionJob.getJob_id());
		
		//終了フラグをfalseにする
		boolean end = false;
		
		//実行状態が終了または変更済以外が存在するかチェック
		boolean runNode = false;
		Collection collection = sessionJob.getJobSessionNode();
		if(collection != null && collection.size() > 0){
			Iterator itr = collection.iterator();
			while(itr.hasNext()){
				JobSessionNodeLocal sessionNode = (JobSessionNodeLocal)itr.next();
				//実行状態が終了または変更済以外かチェック
				if(sessionNode.getStatus().intValue() != StatusConstant.TYPE_END && 
						sessionNode.getStatus().intValue() != StatusConstant.TYPE_MODIFIED){
					runNode = true;
				}
			}
		}
		
		//全ノード終了の場合
		if(!runNode){
			//ジョブ終了フラグをtrueにする
			end = true;
		}
		
		return end;
	}
	
	/**
	 * 全ノードの停止チェックを行います。
	 * 
	 * @param sessionJob セッションジョブ
	 * @return true：停止、false：未停止あり
	 * @throws NamingException
	 * 
	 * @see com.clustercontrol.bean.StatusConstant
	 */
	protected boolean checkAllNodeStop(JobSessionJobLocal sessionJob) throws NamingException{
		m_log.debug("checkAllNodeStop() : sessionId=" + sessionJob.getSession_id() + ", jobId=" + sessionJob.getJob_id());
		
		//停止フラグをfalseにする
		boolean stop = false;
		
		//実行状態が停止処理中または実行中が存在するかチェック
		boolean runNode = false;
		Collection collection = sessionJob.getJobSessionNode();
		if(collection != null && collection.size() > 0){
			Iterator itr = collection.iterator();
			while(itr.hasNext()){
				JobSessionNodeLocal sessionNode = (JobSessionNodeLocal)itr.next();
				//実行状態が停止処理中かチェック
				if(sessionNode.getStatus().intValue() == StatusConstant.TYPE_STOPPING || 
						sessionNode.getStatus().intValue() == StatusConstant.TYPE_RUNNING){
					runNode = true;
				}
			}
		}
		
		//全ノード停止の場合
		if(!runNode){
			//停止フラグをtrueにする
			stop = true;
		}
		
		return stop;
	}

	/**
	 * ジョブの実行結果を削除します。
	 * 
	 * @param sessionId セッションID
	 * @param jobId ジョブID
	 * @throws NamingException
	 * @throws EJBException
	 * @throws RemoveException
	 * @throws JobInfoNotFoundException 
	 * 
	 * @see com.clustercontrol.bean.JobConstant
	 */
	protected void clearJob(String sessionId, String jobunitId, String jobId) throws NamingException, EJBException, RemoveException, JobInfoNotFoundException {
		m_log.debug("clearJob() : sessionId=" + sessionId + ", jobId=" + jobId);
		
		//セッションIDとジョブIDから、セッションジョブを取得
		JobSessionJobLocal sessionJob = getJobSessionJobLocal(sessionId, jobunitId, jobId);
		
		if(sessionJob != null){
			JobInfoLocal job = sessionJob.getJobInfo();
			
			if(job.getJob_type().intValue() == JobConstant.TYPE_JOB){
				//セッションノード削除
				Collection collection = sessionJob.getJobSessionNode();
                if(collection != null && collection.size() > 0){
                    Object[] sessionNodeArray = collection.toArray();
                    for(int j = 0; j < sessionNodeArray.length; j++){
                    	JobSessionNodeLocal sessionNode = (JobSessionNodeLocal)sessionNodeArray[j];
                        if(sessionNode != null){
                        	sessionNode.remove();
                        }
                    }
                }
                
                //リレーションを削除
				JobRelationInfoLocal relation = sessionJob.getJobRelationInfo();
                if(relation != null){
                    relation.remove();
                }
                //コマンドを削除
                JobCommandInfoLocal command = job.getJobCommandInfo();
                if(command != null){
                    command.remove();
                }
                //ファイル転送を削除
                JobFileInfoLocal file = job.getJobFileInfo();
                if(file != null){
                	file.remove();
                }
                //開始条件(待ち条件)を削除
                JobStartInfoLocal start = job.getJobStartInfo();
                if(start != null){
                    //開始条件(ジョブ)を削除
                    collection = start.getJobStartJobInfo();
                    if(collection != null && collection.size() > 0){
                        Object[] startJobArray = collection.toArray();
                        for(int j = 0; j < startJobArray.length; j++){
                        	JobStartJobInfoLocal startJob = (JobStartJobInfoLocal)startJobArray[j];
                            if(startJob != null){
                            	startJob.remove();
                            }
                        }
                    }
                    //開始条件(時刻)を削除
                    JobStartTimeInfoLocal startTime = start.getJobStartTimeInfo();
                    if(startTime != null){
                    	startTime.remove();
                    }
                    
                    start.remove();
                }
                //通知メッセージを削除
                collection = job.getJobNoticeInfo();
                if(collection != null && collection.size() > 0){
                    Object[] noticeArray = collection.toArray();
                    for(int j = 0; j < noticeArray.length; j++){
                        JobNoticeInfoLocal notice = (JobNoticeInfoLocal)noticeArray[j];
                        if(notice != null){
                            notice.remove();
                        }
                    }
                }
                //終了状態を削除
                collection = job.getJobEndInfo();
                if(collection != null && collection.size() > 0){
                    Object[] endArray = collection.toArray();
                    for(int j = 0; j < endArray.length; j++){
                        JobEndInfoLocal end = (JobEndInfoLocal)endArray[j];
                        if(end != null){
                            end.remove();
                        }
                    }
                }
                
                //ジョブを削除
                job.remove();
                
                //セッションジョブ削除
                sessionJob.remove();
			}
			else{
				//ジョブ以外の場合

				//セッションIDとジョブIDから、直下のジョブを取得
				Collection collection = null;
				try {
					collection = JobRelationInfoUtil.getLocalHome().findByParentJobId(sessionId, jobId);
				} catch (FinderException e) {
					m_log.warn("JobRelationInfoUtil.getLocalHome().findByParentJobId() : " + e.getMessage());
					JobInfoNotFoundException je = new JobInfoNotFoundException(e.getMessage(), e);
					je.setSessionId(sessionId);
					je.setParentJobId(jobId);
					throw je;
				}
				if(collection != null && collection.size() > 0){
					Iterator itr = collection.iterator();
					while(itr.hasNext()){
						JobRelationInfoLocal relation = (JobRelationInfoLocal)itr.next();
						
						//ジョブを削除
						clearJob(relation.getSession_id(), relation.getJobunit_id(), relation.getJob_id());
					}
				}
				
                //リレーションを削除
				JobRelationInfoLocal relation = sessionJob.getJobRelationInfo();
                if(relation != null){
                    relation.remove();
                }
                //コマンドを削除
                JobCommandInfoLocal command = job.getJobCommandInfo();
                if(command != null){
                    command.remove();
                }
                //ファイル転送を削除
                JobFileInfoLocal file = job.getJobFileInfo();
                if(file != null){
                	file.remove();
                }
                //開始条件(待ち条件)を削除
                JobStartInfoLocal start = job.getJobStartInfo();
                if(start != null){
                    //開始条件(ジョブ)を削除
                    collection = start.getJobStartJobInfo();
                    if(collection != null && collection.size() > 0){
                        Object[] startJobArray = collection.toArray();
                        for(int j = 0; j < startJobArray.length; j++){
                        	JobStartJobInfoLocal startJob = (JobStartJobInfoLocal)startJobArray[j];
                            if(startJob != null){
                            	startJob.remove();
                            }
                        }
                    }
                    //開始条件(時刻)を削除
                    JobStartTimeInfoLocal startTime = start.getJobStartTimeInfo();
                    if(startTime != null){
                    	startTime.remove();
                    }
                    
                    start.remove();
                }
                //通知メッセージを削除
                collection = job.getJobNoticeInfo();
                if(collection != null && collection.size() > 0){
                    Object[] noticeArray = collection.toArray();
                    for(int j = 0; j < noticeArray.length; j++){
                        JobNoticeInfoLocal notice = (JobNoticeInfoLocal)noticeArray[j];
                        if(notice != null){
                            notice.remove();
                        }
                    }
                }
                //終了状態を削除
                collection = job.getJobEndInfo();
                if(collection != null && collection.size() > 0){
                    Object[] endArray = collection.toArray();
                    for(int j = 0; j < endArray.length; j++){
                        JobEndInfoLocal end = (JobEndInfoLocal)endArray[j];
                        if(end != null){
                            end.remove();
                        }
                    }
                }
                
                //ジョブを削除
                job.remove();
                
                //セッションジョブ削除
                sessionJob.remove();
			}
		}
	}
	
	/**
	 * 実行状態を設定します。
	 * 
	 * @param sessionId セッションID
	 * @param jobId ジョブID
	 * @param status 実行状態
	 * @throws NamingException
	 * @throws JobInfoNotFoundException 
	 */
	protected void setStatus(String sessionId, String jobunitId, String jobId, Integer status) throws NamingException, JobInfoNotFoundException {
		m_log.debug("setStatus() : sessionId=" + sessionId + ", jobunitId=" + jobunitId + ", jobId=" + jobId + ", status=" + status);
		
		//セッションIDとジョブIDから、セッションジョブを取得
		JobSessionJobLocal sessionJob = getJobSessionJobLocal(sessionId, jobunitId, jobId);
		
		if(sessionJob != null){
			//実行状態を設定
			sessionJob.setStatus(status);
			
			//リレーションを取得し、親ジョブのジョブIDを取得
			JobRelationInfoLocal relation = sessionJob.getJobRelationInfo();
			String parentJobUnitId = relation.getParent_jobunit_id();
			String parentJobId = relation.getParent_job_id();
			relation = null;
			
			if(parentJobId.compareTo("TOP") != 0){
				setStatus(sessionId, parentJobUnitId, parentJobId, status);
			}
		}
	}
	
	/**
	 * ジョブ情報を作成します。
	 * 
	 * @param jobId ジョブID
	 * @param info ログ出力情報
	 * @param triggerInfo 実行契機情報
	 * @return セッションID
	 * @throws CreateException
	 * @throws NamingException
	 * @throws FacilityNotFoundException 
	 * @throws JobMasterNotFoundException 
	 * @throws JobInfoNotFoundException 
	 * @throws FinderException 
	 * 
	 * @see com.clustercontrol.notify.message.NotifyRequestMessage
	 * @see com.clustercontrol.jobmanagement.factory.OperationJob#createJobInfo(JobMasterLocal, String, String, NotifyRequestMessage)
	 */
	public String createJobInfo(String jobunitId, String jobId, OutputBasicInfo info, JobTriggerInfo triggerInfo) throws CreateException, NamingException, FacilityNotFoundException, JobMasterNotFoundException, JobInfoNotFoundException, FinderException {
		m_log.debug("createJobInfo() : jobunitId=" + jobunitId + ", jobId=" + jobId);
		
		String sessionId = null;
		Date createTime = new Date();
		
		//ジョブIDからジョブを取得する
		JobMasterLocal job = null;
		try {
			job = JobMasterUtil.getLocalHome().findByPrimaryKey(new JobMasterPK(jobunitId, jobId));
		} catch (FinderException e1) {
			m_log.warn("JobMasterUtil.getLocalHome().findByPrimaryKey() : " + e1.getMessage());
			JobMasterNotFoundException je = new JobMasterNotFoundException(e1.getMessage(), e1);
			je.setJobunitId(jobunitId);
			je.setJobId(jobId);
			throw je;
		}
		if(job != null){
			//セッションを作成
			boolean createFlg = false;
			for(int i = 0; !createFlg; i++){
				try {
					SimpleDateFormat dateFormat = new SimpleDateFormat("yyyyMMddHHmmss");
					sessionId = dateFormat.format(createTime);
					
					DecimalFormat format = new DecimalFormat("-000");
					sessionId = sessionId + format.format(i);
					
					//JobSessionを作成
					JobSessionUtil.getLocalHome().create(
							sessionId, 
							job.getJobunit_id(),
							job.getJob_id(), 
							new Date(), 
							Integer.valueOf(0),
							triggerInfo.getTrigger_type(),
							triggerInfo.getTrigger_info());
					
					createFlg = true;
				} catch (DuplicateKeyException e) {
					//プライマリーキー重複
					//再度JobSessionの作成を行う
				} catch (CreateException e) {
					if(e.getMessage().indexOf("duplicate") != -1){
						//プライマリーキー重複
						//DuplicateKeyExceptionをスロー
						throw new DuplicateKeyException();
					}
					throw e;
				}
			}
			
			//ジョブ情報作成
			//FIXME 動作確認
			createJobInfo(job, null, "TOP", sessionId, info);
		}
		
		return sessionId;
	}
	
	/**
	 * ジョブ情報を作成します。<BR>
	 * 再帰呼び出しを行います。
	 * 
	 * @param job ジョブマスタ
	 * @param parentJobUnitId 親ジョブが所属するジョブユニットのジョブID
	 * @param parentJobId 親ジョブID
	 * @param sessionId セッションID
	 * @param info ログ出力情報
	 * @throws CreateException
	 * @throws NamingException
	 * @throws FacilityNotFoundException 
	 * @throws JobInfoNotFoundException 
	 * @throws FinderException 
	 */
	protected void createJobInfo(
			JobMasterLocal job, 
			String parentJobUnitId, 
			String parentJobId, 
			String sessionId, 
			OutputBasicInfo info) throws CreateException, NamingException, FacilityNotFoundException, JobInfoNotFoundException, FinderException {
		
		//ジョブ情報データ作成
		createJobInfoData(job, parentJobUnitId, parentJobId, sessionId, info);
		
		//ジョブIDから子ジョブを取得
		Collection collection = null;
		try {
			collection = JobRelationMasterUtil.getLocalHome().findByParentJobunitIdAndJobId(job.getJobunit_id(), job.getJob_id());
		} catch (FinderException e) {
			m_log.warn("JobRelationMasterUtil.getLocalHome().findByParentJobId() : " + e.getMessage());
			JobInfoNotFoundException je = new JobInfoNotFoundException(e.getMessage(), e);
			je.setParentJobId(job.getJob_id());
			throw je;
		}
		
		//子ジョブのジョブ情報を登録
		if(collection != null && collection.size() > 0){
			Iterator itr = collection.iterator();
			while(itr.hasNext()){
				//リレーションを取得
				JobRelationMasterLocal children = (JobRelationMasterLocal)itr.next();
				//ジョブを取得
				JobMasterLocal childJob = children.getJobMaster();
				//ジョブ情報作成
				createJobInfo(childJob, job.getJobunit_id(), job.getJob_id(), sessionId, null);
			}
		}
	}
	
	/**
	 * ジョブ情報を作成します。
	 * 
	 * @param job ジョブマスタ
	 * @param parentJobId 親ジョブID
	 * @param sessionId セッションID
	 * @param info ログ出力情報
	 * @throws CreateException
	 * @throws NamingException
	 * @throws FacilityNotFoundException 
	 * @throws JobInfoNotFoundException 
	 * @throws FinderException 
	 * 
	 * @see com.clustercontrol.bean.JobConstant
	 * @see com.clustercontrol.bean.JobParamTypeConstant
	 * @see com.clustercontrol.bean.StatusConstant
	 * @see com.clustercontrol.bean.YesNoConstant
	 * @see com.clustercontrol.jobmanagement.bean.EndStatusCheckConstant
	 * @see com.clustercontrol.jobmanagement.bean.SystemParameterConstant
	 * @see com.clustercontrol.jobmanagement.util.ParameterUtil
	 * @see com.clustercontrol.repository.bean.FacilityAttributeConstant
	 * @see com.clustercontrol.repository.ejb.session.RepositoryControllerLocal
	 */
	@SuppressWarnings("unchecked")
	protected void createJobInfoData(
			JobMasterLocal job, 
			String parentJobUnitId, 
			String parentJobId, 
			String sessionId, 
			OutputBasicInfo info) throws CreateException, NamingException, FacilityNotFoundException, JobInfoNotFoundException, FinderException {

		RepositoryControllerLocal repository = null;
		
		//JobSessionJobを作成
		JobSessionJobLocal sessionJob = JobSessionJobUtil.getLocalHome().create(
				sessionId, 
				job.getJobunit_id(), 
				job.getJob_id(), 
				null, 
				Integer.valueOf(StatusConstant.TYPE_WAIT), 
				null, 
				null, 
				null, 
				null, 
				null,
				Integer.valueOf(EndStatusCheckConstant.NO_WAIT_JOB),
				Integer.valueOf(DelayNotifyConstant.NONE));
		
		//JobRelationInfoLocalを作成
		JobRelationInfoUtil.getLocalHome().create(
				sessionId, 
				job.getJobunit_id(),
				job.getJob_id(),
				parentJobUnitId, 
				parentJobId);
		
		//JobInfoLocalを作成
		JobInfoUtil.getLocalHome().create(
				sessionId, 
				job.getJobunit_id(),
				job.getJob_id(), 
				job.getJob_name(), 
				job.getDescription(), 
				job.getJob_type(), 
				job.getReg_date(), 
				job.getUpdate_date(), 
				job.getReg_user(), 
				job.getUpdate_user());
		
		//ジョブネットとジョブの場合、待ち条件を作成する
		JobStartMasterLocal start = null;
		if(parentJobId.compareTo("TOP") != 0 &&
				(job.getJob_type().intValue() == JobConstant.TYPE_JOBNET || 
						job.getJob_type().intValue() == JobConstant.TYPE_JOB || 
						job.getJob_type().intValue() == JobConstant.TYPE_FILEJOB)){
			start = job.getJobStartMaster();
			
			if(start != null){
				//JobStartInfoLocal(待ち条件)を作成
				JobStartInfoUtil.getLocalHome().create(
						sessionId, 
						job.getJobunit_id(), 
						job.getJob_id(), 
						start.getCondition_type(), 
						start.getSuspend(), 
						start.getSkip(), 
						start.getSkip_end_status(), 
						start.getSkip_end_value(), 
						start.getUnmatch_end_flg(), 
						start.getUnmatch_end_status(), 
						start.getUnmatch_end_value(), 
						start.getCalendar(), 
						start.getCalendar_id(), 
						start.getCalendar_end_value(), 
						start.getStart_delay(), 
						start.getStart_delay_session(), 
						start.getStart_delay_session_value(), 
						start.getStart_delay_time(), 
						start.getStart_delay_time_value(), 
						start.getStart_delay_condition_type(), 
						start.getStart_delay_notify(), 
						start.getStart_delay_notify_priority(), 
						start.getStart_delay_operation(), 
						start.getStart_delay_operation_type(), 
						start.getStart_delay_operation_end_value(), 
						start.getEnd_delay(), 
						start.getEnd_delay_session(), 
						start.getEnd_delay_session_value(), 
						start.getEnd_delay_job(), 
						start.getEnd_delay_job_value(), 
						start.getEnd_delay_time(), 
						start.getEnd_delay_time_value(), 
						start.getEnd_delay_condition_type(), 
						start.getEnd_delay_notify(), 
						start.getEnd_delay_notify_priority(), 
						start.getEnd_delay_operation(), 
						start.getEnd_delay_operation_type(), 
						start.getEnd_delay_operation_end_value());
				
				//JobStartJobInfoLocal(待ち条件(ジョブ))を作成
				Collection startJobList = start.getJobStartJobMaster();
				if(startJobList != null && startJobList.size() > 0){
					Iterator itr = startJobList.iterator();
					while(itr.hasNext()){
						JobStartJobMasterLocal startJob = (JobStartJobMasterLocal)itr.next();
						if(startJob != null){
							//JobStartJobInfoLocalを作成
							JobStartJobInfoUtil.getLocalHome().create(
									sessionId, 
									job.getJobunit_id(), 
									job.getJob_id(), 
									startJob.getTarget_jobunit_id(), 
									startJob.getTarget_job_id(), 
									startJob.getTarget_job_type(), 
									startJob.getTarget_job_end_value());
						}
					}
				}
				
				//JobStartTimeInfoLocal(待ち条件(時刻))を作成
				JobStartTimeMasterLocal startTime = start.getJobStartTimeMaster();
				if(startTime != null){
					//JobStartTimeInfoLocalを作成
					JobStartTimeInfoUtil.getLocalHome().create(
							sessionId, 
							job.getJobunit_id(), 
							job.getJob_id(), 
							startTime.getStart_time());
				}
				
				//開始時保留、開始時スキップをチェック
				if(start.getSuspend().intValue() == YesNoConstant.TYPE_YES){
					//JobSessionJobの実行状態に保留中を設定
					sessionJob.setStatus(Integer.valueOf(StatusConstant.TYPE_RESERVING));
				}
				else if(start.getSkip().intValue() == YesNoConstant.TYPE_YES){
					//JobSessionJobの実行状態にスキップを設定
					sessionJob.setStatus(Integer.valueOf(StatusConstant.TYPE_SKIP));
					//JobSessionJobの終了値を設定
					sessionJob.setEnd_value(start.getSkip_end_value());
				}
			}
		}
		
		//パラメータを作成
		if(parentJobId.equals("TOP")){
			//最上位のジョブIDを取得
//			String topJobId = getTopJob(job);
			String topJobId = job.getJobunit_id();
			Collection paramList = null;
			try {
				paramList = JobParamMasterUtil.getLocalHome().findByJobId(topJobId);
			} catch (FinderException e) {
				m_log.warn("JobSessionJobUtil.getLocalHome().findByPrimaryKey() : " + e.getMessage());
				JobInfoNotFoundException je = new JobInfoNotFoundException(e.getMessage(), e);
				je.setJobId(topJobId);
				throw je;
			}
			if(paramList != null){
				Iterator itr = paramList.iterator();
				while(itr.hasNext()){
					JobParamMasterLocal param = (JobParamMasterLocal)itr.next();
					if(param != null){
						
						String value = null;
						if(param.getParam_type().intValue() == JobParamTypeConstant.TYPE_SYSTEM){
//							// システム変数
							
							// システム変数に格納する情報が、どの機能のものかチェック
							int functionType = ParameterUtil.checkFunctionType(param.getParam_id());
							
							if (functionType == ParameterUtil.TYPE_JOB){
								// ジョブ管理の情報を取得する場合
								value = ParameterUtil.getJobParameterValue(sessionId, param.getParam_id());
								
							} else if (functionType == ParameterUtil.TYPE_MONITOR){
								// 監視管理の情報を取得する場合
								value = ParameterUtil.getParameterValue(param.getParam_id(), info);
							}
							
//							if(param.getParam_id().equals(SystemParameterConstant.START_DATE)){
//								//セッション開始日時
//								JobSessionLocal session = JobSessionUtil.getLocalHome().findByPrimaryKey(new JobSessionPK(sessionId));
//								value = DateFormat.getDateTimeInstance().format(session.getSchedule_date());
//							}
//							else{
//								//その他
//								value = ParameterUtil.getParameterValue(param.getParam_id(), info);
//							}
						}
						else{
							//ユーザ変数
							value = param.getValue();
						}
						
						//JobParamInfoLocal(パラメータ)を作成
						JobParamInfoUtil.getLocalHome().create(
								sessionId, 
								job.getJobunit_id(), 
								job.getJob_id(),
								param.getParam_id(), 
								param.getParam_type(), 
								param.getDescription(), 
								value);
					}
				}
			}
		}
		
		//ジョブの場合、実行コマンドを作成する
		JobCommandMasterLocal command = null;
		if(job.getJob_type().intValue() == JobConstant.TYPE_JOB){
			command = job.getJobCommandMaster();
			if(command != null){
				//JobCommandInfoLocal(実行コマンド)を作成
				JobCommandInfoUtil.getLocalHome().create(
						sessionId, 
						job.getJobunit_id(), 
						job.getJob_id(), 
						command.getFacility_id(), 
						command.getProcess_mode(), 
						command.getStart_command(), 
						command.getStop_command(), 
						command.getEffective_user(), 
						command.getError_end_flg(), 
						command.getError_end_value(), 
						command.getArgument_job_id(), 
						command.getArgument());
			}
			
			//リポジトリ(RepositoryControllerLocal)を取得
			if(repository == null)
				repository = RepositoryControllerUtil.getLocalHome().create();
			
			String facilityId = null;
			if(command.getFacility_id().equals(SystemParameterConstant.HEADER + 
            		SystemParameterConstant.FACILITY_ID + 
            		SystemParameterConstant.FOOTER)){
				facilityId = ParameterUtil.getSessionParameterValue(sessionId, 
						SystemParameterConstant.FACILITY_ID);
			}
			else{
				facilityId = command.getFacility_id();
			}
			
			//JobSessionJobにファシリティパスを設定
			sessionJob.setScope_text(repository.getFacilityPath(facilityId, null));
			
			//実行コマンドの対象ファシリティ
			ArrayList<String> nodeIdList = repository.getExecTargetFacilityIdList(facilityId);
//			ArrayList<String> nodeIdList = null;
//			if(repository.isNode(facilityId)){
//				nodeIdList = new ArrayList<String>();
//				nodeIdList.add(facilityId);
//			}
//			else{
//				nodeIdList = repository.getNodeFacilityIdList(facilityId, RepositoryControllerBean.ALL);
//			}
			
			
			if(nodeIdList instanceof ArrayList){
				for(int i = 0; i < nodeIdList.size(); i++){
					ArrayList Attribute = new ArrayList();
					Attribute.add(FacilityAttributeConstant.CN);
					HashMap map = repository.getNodeDetail((String)nodeIdList.get(i), Attribute);
					
					//JobSessionNodeを作成
					JobSessionNodeUtil.getLocalHome().create(
							sessionId, 
							job.getJobunit_id(), 
							job.getJob_id(), 
							(String)nodeIdList.get(i), 
							(String)map.get(FacilityAttributeConstant.CN), 
							Integer.valueOf(StatusConstant.TYPE_WAIT), 
							null, 
							null, 
							null, 
							null, 
							null, 
							null, 
							Integer.valueOf(0), 
							null, 
							null);
				}
			}
		}
		
		//ファイル転送ジョブの場合、ファイル転送ジョブネットを作成する
		JobFileMasterLocal file = null;
		if(job.getJob_type().intValue() == JobConstant.TYPE_FILEJOB){
			file = job.getJobFileMaster();
			if(file != null){
				//JobFileInfoLocal(実行コマンド)を作成
				JobFileInfoUtil.getLocalHome().create(
						sessionId, 
						job.getJobunit_id(), 
						job.getJob_id(), 
						file.getProcess_mode(), 
						file.getSrc_facility_id(), 
						file.getDest_facility_id(), 
						file.getSrc_file(), 
						file.getSrc_work_dir(), 
						file.getDest_directory(), 
						file.getDest_work_dir(), 
						file.getCompression_flg(), 
						file.getCheck_flg(), 
						file.getEffective_user());
			}
			
			//ファイルリスト取得ジョブを作成
			new CreateFileJob().createGetFileListJob(job, sessionId);
		}
		
		//通知メッセージを作成
		Collection noticeList = job.getJobNoticeMaster();
		String notifyGroupIdMst = null;
		//JobNoticeInfo用に通知グループIDを新規作成
		String notifyGroupIdSession = NotifyRelationInfo.createNotifyGroupIdJobInfo(sessionId, job.getJobunit_id(), job.getJob_id(), 0);

		if(noticeList != null){
			Iterator itr = noticeList.iterator();
			while(itr.hasNext()){
				JobNoticeMasterLocal notice = (JobNoticeMasterLocal)itr.next();
				if(notice != null){
					//JobNoticeInfoLocal(通知メッセージ)を作成
					JobNoticeInfoUtil.getLocalHome().create(
							sessionId, 
							job.getJobunit_id(), 
							job.getJob_id(),
							notice.getNotice_type(), 
							notice.getPriority(), 
							notifyGroupIdSession);
					if(notifyGroupIdMst == null){
						notifyGroupIdMst = notice.getNotifyGroupId();
					}
				}
			}
			
			//JobNoticeInfo用に通知関連テーブルにグループを新規作成
			Collection ct = NotifyRelationInfoUtil.getLocalHome().findByNotifyGroupId(notifyGroupIdMst);
			itr = ct.iterator();
			while(itr.hasNext()) {
				NotifyRelationInfoLocal relation = (NotifyRelationInfoLocal)itr.next();
				NotifyRelationInfoUtil.getLocalHome().create(
						notifyGroupIdSession,
						relation.getNotifyId(),
						relation.getNotifyType(),
						relation.getNotifyFlg()
				);	
			}
		}
		
		//終了状態を作成
		Collection endList = job.getJobEndMaster();
		if(endList != null){
			Iterator itr = endList.iterator();
			while(itr.hasNext()){
				JobEndMasterLocal end = (JobEndMasterLocal)itr.next();
				if(end != null){
					//JobEndInfoLocal(終了状態)を作成
					JobEndInfoUtil.getLocalHome().create(
							sessionId, 
							job.getJobunit_id(), 
							job.getJob_id(),
							end.getEnd_status(), 
							end.getEnd_value(), 
							end.getEnd_value_from(), 
							end.getEnd_value_to());
				}
			}
		}
	}
	
	/**
	 * 最上位ジョブのジョブIDを取得します。<BR>
	 * 再帰呼び出しを行います。
	 * 
	 * @param job ジョブマスタ
	 * @return ジョブID
	 * @throws NamingException
	 * @throws JobInfoNotFoundException 
	 */
	protected String getTopJob(JobMasterLocal job) throws NamingException, JobInfoNotFoundException {

		//ジョブIDから子ジョブを取得
		JobRelationMasterLocal relation = job.getJobRelationMaster();
		if(relation.getParent_job_id().equals("TOP")){
			return job.getJob_id();
		}
		
		JobMasterLocal parentJob = null;
		try {
			parentJob = JobMasterUtil.getLocalHome().findByPrimaryKey(
					new JobMasterPK(relation.getParent_jobunit_id(), relation.getParent_job_id()));
		} catch (FinderException e) {
			m_log.warn("JobMasterUtil.getLocalHome().findByPrimaryKey() : " + e.getMessage());
			JobInfoNotFoundException je = new JobInfoNotFoundException(e.getMessage(), e);
			je.setJobunitId(relation.getParent_jobunit_id());
			je.setJobId(relation.getParent_job_id());
			throw je;
		}
		return getTopJob(parentJob);
	}
	
	/**
	 * カレンダチェックを行います。
	 * 
	 * @param sessionId セッションID
	 * @param jobId ジョブID
	 * @return true:実行可 false:実行不可
	 * @throws NamingException
	 * @throws JobInfoNotFoundException 
	 * 
	 * @see com.clustercontrol.bean.StatusConstant
	 * @see com.clustercontrol.bean.YesNoConstant
	 * @see com.clustercontrol.jobmanagement.factory.OperationJob#endJob(String, String)
	 * @see com.clustercontrol.calendar.ejb.session.CalendarControllerLocal#isRun(java.lang.String, java.util.Date)
	 */
	protected boolean checkCalendar(String sessionId, String jobunitId, String jobId) throws NamingException, JobInfoNotFoundException {
		m_log.debug("checkCalendar() : sessionId=" + sessionId + ", jobunitId=" + jobunitId + ", jobId=" + jobId);
		
		boolean check = false;
		
		//セッションIDとジョブIDから、セッションジョブを取得
		JobSessionJobLocal sessionJob = getJobSessionJobLocal(sessionId, jobunitId, jobId);
		
		JobSessionLocal session = sessionJob.getJobSession();
		JobInfoLocal job = sessionJob.getJobInfo();
		JobStartInfoLocal start = job.getJobStartInfo();
		
		//カレンダをチェック
		if(start.getCalendar().intValue() == YesNoConstant.TYPE_YES){
			try {
				CalendarControllerLocal calendar = 
					CalendarControllerUtil.getLocalHome().create();
				
				//カレンダによる実行可/不可のチェック
				if(calendar.isRun(start.getCalendar_id(), session.getSchedule_date()).booleanValue()){
					check = true;
				}
			} catch (CalendarNotFountException e) {
			} catch (CreateException e) {
			} catch (NamingException e) {
			}
		}
		else{
			check = true;
		}
		
		//実行不可の場合
		if(!check){
			//実行状態を終了にする
			sessionJob.setStatus(Integer.valueOf(StatusConstant.TYPE_END));
			//終了日時を設定
			sessionJob.setEnd_date(new Date());
			//終了値を設定
			sessionJob.setEnd_value(start.getCalendar_end_value());
			sessionJob = null;
			
			//ジョブ終了時関連処理
			endJob(sessionId, jobunitId, jobId);
		}
		
		return check;
	}
	
	
	private JobSessionJobLocal getJobSessionJobLocal(String sessionId, String jobunitId, String jobId) throws JobInfoNotFoundException, NamingException {
		//セッションIDとジョブIDから、セッションジョブを取得
		JobSessionJobLocal sessionJob = null;
		try {
			sessionJob = JobSessionJobUtil.getLocalHome().findByPrimaryKey(
					new JobSessionJobPK(sessionId, jobunitId, jobId));
		} catch (FinderException e) {
			m_log.warn("JobSessionJobUtil.getLocalHome().findByPrimaryKey() : " + e.getMessage());
			JobInfoNotFoundException je = new JobInfoNotFoundException(e.getMessage(), e);
			je.setSessionId(sessionId);
			je.setJobunitId(jobunitId);
			je.setJobId(jobId);
			throw je;
		}
		
		return sessionJob;
	}
}
