/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.jobmanagement.factory;

import java.util.Collection;
import java.util.Date;
import java.util.Iterator;

import javax.ejb.FinderException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.JobInfoNotFoundException;
import com.clustercontrol.bean.JobConstant;
import com.clustercontrol.bean.StatusConstant;
import com.clustercontrol.jobmanagement.ejb.entity.JobRelationInfoLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobRelationInfoUtil;
import com.clustercontrol.jobmanagement.ejb.entity.JobSessionJobLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobSessionJobPK;
import com.clustercontrol.jobmanagement.ejb.entity.JobSessionJobUtil;

/**
 * ジョブ操作の中断に関する処理を行うクラスです。
 *
 * @version 2.1.0
 * @since 2.1.0
 */
public class OperateSuspendOfJob extends OperationJob {
	/** ログ出力のインスタンス */
	protected static Log m_log = LogFactory.getLog( OperateSuspendOfJob.class );

	/**
	 * コンストラクタ
	 */
	public OperateSuspendOfJob(){
		super();
	}
	
	/**
	 * ジョブを停止[中断]します。
	 * 
	 * @param sessionId セッションID
	 * @param jobunitId 所属ジョブユニットのジョブID
	 * @param jobId ジョブID
	 * @throws NamingException
	 * @throws JobInfoNotFoundException 
	 */
	public void suspendJob(String sessionId, String jobunitId, String jobId) throws NamingException, JobInfoNotFoundException {
		m_log.debug("suspendJob() : sessionId=" + sessionId + ", jobunitId=" + jobunitId + ", jobId=" + jobId);
		
		//セッションIDとジョブIDから、セッションジョブを取得
		JobSessionJobLocal sessionJob;
		try {
			sessionJob = JobSessionJobUtil.getLocalHome().findByPrimaryKey(
					new JobSessionJobPK(sessionId, jobunitId, jobId));
		} catch (FinderException e) {
			m_log.warn("JobSessionJobLocal.findByPrimaryKey() : " + e.getMessage());
			JobInfoNotFoundException je = new JobInfoNotFoundException(e.getMessage(), e);
			je.setSessionId(sessionId);
			je.setJobunitId(jobunitId);
			je.setJobId(jobId);
			throw je;
		}
		
		if(sessionJob != null){
			//実行状態が実行中の場合、実行状態を中断にする
			if(sessionJob.getStatus().intValue() == StatusConstant.TYPE_RUNNING){
				sessionJob.setStatus(Integer.valueOf(StatusConstant.TYPE_SUSPEND));
				
				//終了・中断日時を設定
				sessionJob.setEnd_date(new Date());
				
				//セッションIDとジョブIDから、直下のジョブを取得（実行状態が実行中）
				Collection collection;
				try {
					collection = JobRelationInfoUtil.getLocalHome().findByStatus(sessionId, jobId, Integer.valueOf(StatusConstant.TYPE_RUNNING));
				} catch (FinderException e) {
					m_log.warn("JobRelationInfoLocal.findByStatus : " + e.getMessage());
					JobInfoNotFoundException je = new JobInfoNotFoundException(e.getMessage(), e);
					je.setSessionId(sessionId);
					je.setJobId(jobId);
					throw je;
				}
				if(collection != null && collection.size() > 0){
					Iterator itr = collection.iterator();
					while(itr.hasNext()){
						JobRelationInfoLocal relation = (JobRelationInfoLocal)itr.next();
						
						//ジョブ中断処理を行う
						suspendJob(relation.getSession_id(), relation.getJobunit_id(), relation.getJob_id());
					}
				}
			}
		}
	}
	
	/**
	 * ジョブを開始[中断解除]します。
	 * 
	 * @param sessionId セッションID
	 * @param jobunitId 所属ジョブユニットのジョブID
	 * @param jobId ジョブID
	 * @throws NamingException
	 * @throws JobInfoNotFoundException 
	 * 
	 * @see com.clustercontrol.jobmanagement.factory.OperationJob#checkJobEnd(JobSessionJobLocal)
	 * @see com.clustercontrol.jobmanagement.factory.OperationJob#checkEndStatus(String, String, String)
	 * @see com.clustercontrol.jobmanagement.factory.OperationJob#setEndStaus(String, String, String, Integer, Integer, String)
	 * @see com.clustercontrol.jobmanagement.factory.OperationJob#endJob(String, String, String)
	 * @see com.clustercontrol.jobmanagement.factory.OperateSuspendOfJob#releaseSuspend(String, String)
	 * @see com.clustercontrol.jobmanagement.factory.Notice#notify(String, String, Integer)
	 */
	public void releaseSuspendJob(String sessionId, String jobunitId, String jobId) throws NamingException, JobInfoNotFoundException {
		m_log.debug("releaseSuspendJob() : sessionId=" + sessionId + ", jobunitId=" + jobunitId + ", jobId=" + jobId);
		
		//ジョブ中断解除処理
		releaseSuspend(sessionId, jobunitId, jobId);
		
		//セッションIDとジョブIDから、セッションジョブを取得
		JobSessionJobLocal sessionJob;
		try {
			sessionJob = JobSessionJobUtil.getLocalHome().findByPrimaryKey(
					new JobSessionJobPK(sessionId, jobunitId, jobId));
		} catch (FinderException e) {
			m_log.warn("JobSessionJobUtil.getLocalHome().findByPrimaryKey() : " + e.getMessage());
			JobInfoNotFoundException je = new JobInfoNotFoundException(e.getMessage(), e);
			je.setSessionId(sessionId);
			je.setJobunitId(jobunitId);
			je.setJobId(jobId);
			throw je;
		}
		
		//セッションIDとジョブIDから、直下のジョブを取得（実行状態が中断）
		Collection collection = null;
		try {
			collection = JobRelationInfoUtil.getLocalHome().findByStatus(sessionId, jobId, Integer.valueOf(StatusConstant.TYPE_SUSPEND));
		} catch (FinderException e) {
			m_log.warn("JobRelationInfoUtil.getLocalHome().findByStatus() : " + e.getMessage());
			JobInfoNotFoundException je = new JobInfoNotFoundException(e.getMessage(), e);
			je.setSessionId(sessionId);
			je.setParentJobId(jobId);
			
			throw je;
		}
		if(collection != null && collection.size() > 0){
			Iterator itr = collection.iterator();
			while(itr.hasNext()){
				JobRelationInfoLocal relation = (JobRelationInfoLocal)itr.next();
				String relationJobUnitId = relation.getJobunit_id();
				String relationJobId = relation.getJob_id();
				relation = null;
				
				//ジョブ中断解除処理を行う
				releaseSuspendJob(sessionId, relationJobUnitId, relationJobId);
			}
		}
		
		if(sessionJob != null){
			boolean endAll = true;
			
			//ジョブ
			if(sessionJob.getJobInfo().getJob_type().intValue() == JobConstant.TYPE_JOB){
				//ジョブ終了チェック
				endAll = checkJobEnd(sessionJob);
			}
			else{
				try {
					//直下のジョブが全て完了したかチェック
					Collection sameHierarchy = JobRelationInfoUtil.getLocalHome().findByParentJobId(sessionId, jobId);
					
					Iterator itr = sameHierarchy.iterator();
					while(itr.hasNext()){
						JobRelationInfoLocal relation1 = (JobRelationInfoLocal)itr.next();
						JobSessionJobLocal sessionJob1 = relation1.getJobSessionJob();
						
						//実行状態が終了または変更済以外の場合、直下のジョブは未完了
						if(sessionJob1.getStatus().intValue() != StatusConstant.TYPE_END &&
								sessionJob1.getStatus().intValue() != StatusConstant.TYPE_MODIFIED){
							endAll = false;
							break;
						}
					}
					itr = null;
					sameHierarchy = null;
				} catch (FinderException e) {
					m_log.warn("JobRelationInfoLocal.findByParentJobId() : " + e.getMessage());
					JobInfoNotFoundException je = new JobInfoNotFoundException(e.getMessage(), e);
					je.setSessionId(sessionId);
					je.setParentJobId(jobId);
					throw je;
				}
			}
			
			//直下のジョブが全て完了の場合
			if(endAll){
				//終了状態を判定し、終了状態と終了値を設定
				Integer endStatus = checkEndStatus(sessionId, jobunitId, jobId);
				
				//実行状態、終了状態、終了値、終了日時を設定
				setEndStaus(sessionId, jobunitId, jobId, 
						Integer.valueOf(StatusConstant.TYPE_END), endStatus, null);
				
				//通知処理
				new Notice().notify(sessionId, jobunitId, jobId, endStatus);
				
				//リレーションを取得し、親ジョブのジョブIDを取得
				JobRelationInfoLocal relation = sessionJob.getJobRelationInfo();
				String parentJobId = relation.getParent_job_id();
				relation = null;
				
				//ジョブ終了時関連処理
				if(parentJobId.compareTo("TOP") != 0){
					endJob(sessionId, jobunitId, jobId);
				}
			}
		}
	}
	
	/**
	 * ジョブ中断解除処理を行います。
	 * 
	 * @param sessionId セッションID
	 * @param jobunitId 所属ジョブユニットのジョブID
	 * @param jobId ジョブID
	 * @throws NamingException
	 * @throws JobInfoNotFoundException 
	 */
	protected void releaseSuspend(String sessionId, String jobunitId, String jobId) throws NamingException, JobInfoNotFoundException {
		m_log.debug("releaseSuspend() : sessionId=" + sessionId + ", jobunitId=" + jobunitId + ", jobId=" + jobId);
		
		//セッションIDとジョブIDから、セッションジョブを取得
		JobSessionJobLocal sessionJob = null;
		try {
			sessionJob = JobSessionJobUtil.getLocalHome().findByPrimaryKey(
					new JobSessionJobPK(sessionId, jobunitId, jobId));
		} catch (FinderException e) {
			m_log.warn("JobSessionJobUtil.getLocalHome().findByPrimaryKey() : " + e.getMessage());
			JobInfoNotFoundException je = new JobInfoNotFoundException(e.getMessage(), e);
			je.setSessionId(sessionId);
			je.setParentJobId(jobId);
			throw je;
		}
		
		if(sessionJob != null){
			//実行状態が中断の場合
			if(sessionJob.getStatus().intValue() == StatusConstant.TYPE_SUSPEND){
				//実行状態を実行中にする
				sessionJob.setStatus(Integer.valueOf(StatusConstant.TYPE_RUNNING));
				
				//リレーションを取得し、親ジョブのジョブIDを取得
				JobRelationInfoLocal relation = sessionJob.getJobRelationInfo();
				String parentJobUnitId = relation.getParent_jobunit_id();
				String parentJobId = relation.getParent_job_id();
				relation = null;
				sessionJob = null;
				
				if(parentJobId.compareTo("TOP") != 0){
					releaseSuspend(sessionId, parentJobUnitId, parentJobId);
				}
			}
		}
	}
}
