/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.repository.dialog;

import java.util.ArrayList;

import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.TableTree;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;

import com.clustercontrol.bean.Property;
import com.clustercontrol.bean.PropertyConstant;
import com.clustercontrol.composite.PropertySheet;
import com.clustercontrol.dialog.CommonDialog;
import com.clustercontrol.dialog.ValidateResult;
import com.clustercontrol.repository.action.AddScopeProperty;
import com.clustercontrol.repository.action.GetScopeProperty;
import com.clustercontrol.repository.action.ModifyScopeProperty;
import com.clustercontrol.repository.bean.ScopeConstant;
import com.clustercontrol.util.Messages;
import com.clustercontrol.util.PropertyUtil;

/**
 * スコープの作成・変更ダイアログクラス<BR>
 * 
 * @version 2.2.0
 * @since 1.0.0
 */
public class ScopeCreateDialog extends CommonDialog {

    // ----- instance フィールド ----- //

    /** 初期表示ノードのファシリティID */
    private String facilityId = "";
    
    /** 親ファシリティID */
    private String parentFacilityId = "";

    /** 変更用ダイアログ判別フラグ */
    private boolean isModifyDialog = false;

    /** ノード属性プロパティシート */
    private PropertySheet propertySheet = null;

    // ----- コンストラクタ ----- //

    /**
     * 指定した形式のダイアログのインスタンスを返します。
     * 
     * @param parent
     *            親のシェルオブジェクト
     * @param facilityId
     *            初期表示するノードのファシリティID
     * @param isModifyDialog
     *            変更用ダイアログとして利用する場合は、true
     */
    public ScopeCreateDialog(Shell parent, String facilityId,
            boolean isModifyDialog) {
        super(parent);

        this.facilityId = facilityId;
        this.isModifyDialog = isModifyDialog;
    }

    // ----- instance メソッド ----- //

    /**
     * ダイアログの初期サイズを返します。
     * 
     * @return 初期サイズ
     */
    @Override
	protected Point getInitialSize() {
        return new Point(450, 400);
    }

    /**
     * ダイアログエリアを生成します。
     * 
     * @param parent
     *            親のインスタンス
     */
    @Override
	protected void customizeDialog(Composite parent) {
        Shell shell = this.getShell();

        // タイトル
        shell.setText(Messages
                .getString("dialog.repository.scope.create.modify"));

        // レイアウト
        GridLayout layout = new GridLayout(1, true);
        layout.marginWidth = 10;
        layout.marginHeight = 10;
        parent.setLayout(layout);

        /*
         * 属性プロパティシート
         */

        // ラベル
        Label label = new Label(parent, SWT.LEFT);
        GridData gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        gridData.horizontalSpan = 1;
        label.setLayoutData(gridData);
        label.setText(Messages.getString("attribute") + " : ");

        // プロパティシート
        TableTree table = new TableTree(parent, SWT.H_SCROLL | SWT.V_SCROLL
                | SWT.FULL_SELECTION | SWT.MULTI | SWT.BORDER);
        gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.verticalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        gridData.grabExcessVerticalSpace = true;
        gridData.horizontalSpan = 1;
        table.setLayoutData(gridData);

        this.propertySheet = new PropertySheet(table);

        // プロパティ取得及び設定
        Property property = null;
        if (this.isModifyDialog) {
            property = new GetScopeProperty().getProperty(this.facilityId,
                    PropertyConstant.MODE_MODIFY);
        } else {
            property = new GetScopeProperty().getProperty(null,
                    PropertyConstant.MODE_ADD);
        }
        this.propertySheet.setInput(property);

        // ラインを引く
        Label line = new Label(parent, SWT.SEPARATOR | SWT.HORIZONTAL);
        gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        gridData.horizontalSpan = 1;
        line.setLayoutData(gridData);

        // 画面中央に
        Display display = shell.getDisplay();
        shell.setLocation((display.getBounds().width - shell.getSize().x) / 2,
                (display.getBounds().height - shell.getSize().y) / 2);
    }

    /**
     * 入力値チェックをします。
     * 
     * @return 検証結果
     */
    @Override
	protected ValidateResult validate() {
    	ValidateResult result = null;

		//ファシリティIDの入力チェック
        ArrayList values = PropertyUtil.getPropertyValue(getInputData(), ScopeConstant.FACILITY_ID);
		String value = (String)values.get(0);
		if(value.compareTo("") == 0){
            result = new ValidateResult();
            result.setID(Messages.getString("message.hinemos.1"));
            result.setMessage(Messages.getString("message.repository.21"));
            result.setValid(false);
            return result;
		}
		
		//ファシリティ名の入力チェック
        values = PropertyUtil.getPropertyValue(getInputData(), ScopeConstant.FACILITY_NAME);
		value = (String)values.get(0);
		if(value.compareTo("") == 0){
            result = new ValidateResult();
            result.setID(Messages.getString("message.hinemos.1"));
            result.setMessage(Messages.getString("message.repository.22"));
            result.setValid(false);
            return result;
		}
		
        return result;
    }
    
    /**
     * 入力値をマネージャに登録します。
     * 
     * @return true：正常、false：異常
     * 
     * @see com.clustercontrol.dialog.CommonDialog#action()
     */
    @Override
	protected boolean action() {
    	boolean result = false;
    	
    	Property property = this.getInputData();
    	if(property != null){
            if(!this.isModifyDialog()){
            	// 作成の場合
            	result = new AddScopeProperty().add(parentFacilityId, property);
            }
            else{
            	// 変更の場合
                result = new ModifyScopeProperty().edit(property);
            }
    	}
    	
    	return result;
    }

    /**
     * 変更用ダイアログなのかを返します。
     * 
     * @return 変更用ダイアログの場合、true
     */
    public boolean isModifyDialog() {
        return this.isModifyDialog;
    }

    /**
     * 入力値を保持したデータモデルを生成します。
     * 
     * @return データモデル
     */
    public Property getInputData() {
        return (Property) this.propertySheet.getInput();
    }
    
    /**
     * 親ファシリティIDを返します。
     * 
     * @return 親ファシリティID
     */
    public String getParentFacilityId() {
		return parentFacilityId;
	}

	/**
	 * 親ファシリティIDを設定します。
	 * 
	 * @param parentFacilityId 親ファシリティID
	 */
	public void setParentFacilityId(String parentFacilityId) {
		this.parentFacilityId = parentFacilityId;
	}

	/**
     * ＯＫボタンのテキストを返します。
     * 
     * @return ＯＫボタンのテキスト
     */
    @Override
	protected String getOkButtonText() {
        return Messages.getString("register");
    }

    /**
     * キャンセルボタンのテキストを返します。
     * 
     * @return キャンセルボタンのテキスト
     */
    @Override
	protected String getCancelButtonText() {
        return Messages.getString("cancel");
    }
}