/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License 
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied 
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 PURPOSE.  See the GNU General Public License for more details.
 
 */

package com.clustercontrol.monitor.run.util;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import com.clustercontrol.monitor.run.bean.MonitorInfo;
import com.clustercontrol.monitor.run.bean.MonitorStringValueInfo;

/**
 * 文字列監視の判定情報を管理するクラス<BR>
 * シングルトン。
 * 
 * @version 2.1.0
 * @since 2.1.0
 */
public class StringValueInfoManager {
	
	/** シングルトン用インスタンス。 */
	private static StringValueInfoManager INSTANCE = null;

	
    /**
	 * 本クラスのインスタンスを返します。<BR>
	 * シングルトン用インスタンスが<code> null </code>ならば、インスタンスを生成します。<BR>
	 * シングルトン用インスタンスが存在すれば、シングルトン用インスタンスを返します。
	 * 
	 * @return 唯一のインスタンス
	 */
	public static StringValueInfoManager getInstance() {
		if (INSTANCE == null) {
			synchronized (StringValueInfoManager.class) {
				if (INSTANCE == null) {
					INSTANCE = new StringValueInfoManager();
				}
			}
		}
		
		return INSTANCE;
	}
	
	
	/** 判定情報のキャッシュ。 */
	private Map<String, MonitorStringValueInfo> m_cashe = null;
	
	/** 順序を管理する判定情報のリスト。 */
	private ArrayList<MonitorStringValueInfo> m_orderList = null;
	
	/** 監視情報。 */
	private MonitorInfo m_monitorInfo = null;
	
	/** ID採番用カウンタ */
	private int _nextId = 0;
	
	/**
	 * コンストラクタ。<BR>
	 * アクセスを禁止します。
	 */
	private StringValueInfoManager() {
	}
	
	/**
	 * 初期処理を行います。
	 * 
	 * @param monitorInfo 監視情報
	 * 
	 * @see #loadLog()
	 */
	public void initialize(MonitorInfo monitorInfo) {
		
		m_monitorInfo = monitorInfo;
		_nextId = 0;
		this.loadLog();
	}
	
	
	/**
	 * 全ての判定情報の配列を返します。
	 * <p>
	 * 順序の番号順に整列した配列を返します。
	 * 
	 * @return 判定情報一覧
	 */
	public Object[] get() {
		
		Object[] records = this.m_orderList.toArray();
		return records;
	}
	
	/**
	 * 全ての判定情報のリストを返します。
	 * <p>
	 * 順序の番号順に整列した配列を返します。
	 * 
	 * @return 判定情報一覧
	 */
	public ArrayList getMonitorStringValueInfo() {
		return m_orderList;
	}
	
	/**
	 * 引数で指定した識別キーの判定情報を返します。
	 * <p>
	 * 判定情報キャッシュより判定情報を取得します。
	 * 
	 * @param identifier 識別キー
	 * @return 判定情報
	 */
	public MonitorStringValueInfo get(String identifier) {
		return this.m_cashe.get(identifier);
	}
	
	/**
	 * 引数で指定した判定情報を追加します。
	 * <p>
	 * 順序と識別キーを割り当て、判定情報をキャッシュとリストに保持します。
	 * 
	 * @param info 判定情報
	 * @return 成功した場合、<code> true </code>
	 */
	public boolean add(MonitorStringValueInfo info) {
		
		int order = this.m_cashe.size() + 1;
		
		// 順序は一番後ろとする。
		info.setOrderNo(order);
		// 暫定の識別子を振り分ける。
		info.setIdentifier(getNextId());
		
		this.m_cashe.put(info.getIdentifier(), info);
		this.m_orderList.add(info);
		return true;
	}
	
	/**
	 * 引数で指定した判定情報を変更します。
	 * <p>
	 * 判定情報のキャッシュとリストを置き換えます。
	 * 
	 * @param info 判定情報
	 * @return 成功した場合、<code> true </code>
	 */
	public boolean modify(MonitorStringValueInfo info) {
		if (!this.m_cashe.containsKey(info.getIdentifier())) {
			return false;
		}
		
		this.m_cashe.put(info.getIdentifier(), info);
		this.m_orderList.set(info.getOrderNo() - 1, info);
		return true;
	}
	
	/**
	 * 引数で指定した識別キーの判定情報を削除します。
	 * <p>
	 * <ol>
	 * <li>判定情報のキャッシュより、引数で指定された判定情報を削除します。</li>
	 * <li>判定情報のリストより、引数で指定された判定情報を削除します。</li>
	 * <li>リストに保持されている判定情報の順序を振りなおします。</li>
	 * </ol>
	 * 
	 * @param identifier 識別キー
	 * @return 成功した場合、<code> true </code>
	 */
	public boolean delete(String identifier) {
		if (!this.m_cashe.containsKey(identifier)) {
			return false;
		}
		
		MonitorStringValueInfo info = this.m_cashe.remove(identifier);
		
		this.m_orderList.remove(info.getOrderNo() - 1);
		
		// 順序を割り当てなおします。
		int order = 0;
		Iterator ite = this.m_orderList.iterator();
		while (ite.hasNext()) {
			info = (MonitorStringValueInfo) ite.next();
			info.setOrderNo(++order);
		}
		
		return true;
	}
	
	/**
	 * 引数で指定した判定情報の順序をひとつ上げます。
	 * 
	 * @param identifier 識別キー
	 * @return 成功した場合、<code> true </code>
	 * 
	 * @see #change(int, int)
	 */
	public boolean upOrder(String identifier) {
		if (!this.m_cashe.containsKey(identifier)) {
			return false;
		}
		
		MonitorStringValueInfo info = this.m_cashe.get(identifier);
		int oldOrder = info.getOrderNo();
		int newOrder = oldOrder - 1;
		if (newOrder < 1) {
			return false;
		}
		
		return this.change(oldOrder, newOrder);
	}
	
	/**
	 * 引数で指定した判定情報の順序をひとつ下げます。
	 * 
	 * @param identifier 識別キー
	 * @return 成功した場合、<code> true </code>
	 * 
	 * @see #change(int, int)
	 */
	public boolean downOrder(String identifier) {
		if (!this.m_cashe.containsKey(identifier)) {
			return false;
		}
		
		MonitorStringValueInfo info = this.m_cashe.get(identifier);
		int oldOrder = info.getOrderNo();
		int newOrder = oldOrder + 1;
		if (newOrder > this.m_cashe.size()) {
			return false;
		}
		
		return this.change(oldOrder, newOrder);
	}
	
    /**
     * 判定情報をロードします。
     * <p>
     * 監視情報より判定情報を取得し、判定情報のキャッシュとリストに保持します。
     */
    private void loadLog() {
        this.m_cashe = new HashMap<String, MonitorStringValueInfo>();
        this.m_orderList = new ArrayList<MonitorStringValueInfo>();
        
        if(m_monitorInfo != null){
        	ArrayList<MonitorStringValueInfo> records = m_monitorInfo.getStringValueInfo();
        	
        	if(records != null){
        		int index = 0;
            	Iterator<MonitorStringValueInfo> ite = records.iterator();
            	while (ite.hasNext()) {
                	Object o = ite.next();
                	MonitorStringValueInfo info = (MonitorStringValueInfo)o;

                	info.setIdentifier(getNextId());
                    this.m_cashe.put(info.getIdentifier(), info);
                    this.m_orderList.add(info);
                    index++;
                }
        	}
        }
    }
	
	/**
	 * 引数で指定した順序の判定情報同士の順序を入れ替えます。
	 * <p>
	 * 引数で指定する値は、リストのインデックス値ではなく、判定情報の順序を指定します。<BR>
	 * 判定情報の順序を変更し、リストの位置を入れ替えてにセットします。
	 * 
	 * @param index1 判定情報１の順序の値
	 * @param index2 判定情報２の順序の値
	 * @return 正常に終了した場合、<code> true </code>
	 */
	private boolean change(int index1, int index2) {
		MonitorStringValueInfo info1 = this.m_orderList.get(--index1);
		MonitorStringValueInfo info2 = this.m_orderList.get(--index2);
		
		int order1 = info1.getOrderNo();
		int order2 = info2.getOrderNo();
		
		// 順序の値を入れ替えます。
		info1.setOrderNo(order2);
		info2.setOrderNo(order1);
		
		// リストの位置を入れ替えます。
		this.m_orderList.set(index1, info2);
		this.m_orderList.set(index2, info1);
		
		return true;
	}
	
	private String getNextId() {
		return "ID" + String.format("%06d", _nextId++);
	}
	
}