/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.jobmanagement.composite;

import java.util.ArrayList;

import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Table;

import com.clustercontrol.composite.CommonTableViewer;
import com.clustercontrol.jobmanagement.action.GetNodeDetail;
import com.clustercontrol.jobmanagement.action.GetNodeDetailTableDefine;
import com.clustercontrol.jobmanagement.bean.JobNodeDetail;
import com.clustercontrol.jobmanagement.bean.NodeDetailTableDefine;
import com.clustercontrol.jobmanagement.composite.action.NodeDetailSelectionChangedListener;
import com.clustercontrol.util.Messages;

/**
 * ジョブ[ノード詳細]ビュー用のコンポジットクラスです。
 * 
 * @version 1.0.0
 * @since 1.0.0
 */
public class NodeDetailComposite extends Composite {
	/** テーブルビューアー */
    protected CommonTableViewer m_viewer = null;
    /** セッションID */
    protected String m_sessionId = null;
    /** 所属ジョブユニットのジョブID */
    protected String m_jobunitId = null;
    /** ジョブID */
    protected String m_jobId = null;
    /** ファシリティID */
    protected String m_facilityId = null;
    /** ID用ラベル */
    protected Label m_idLabel = null;

    /**
     * コンストラクタ
     * 
     * @param parent 親のコンポジット
     * @param style スタイル
     * 
     * @see org.eclipse.swt.SWT
     * @see org.eclipse.swt.widgets.Composite#Composite(Composite parent, int style)
     * @see #initialize()
     */
    public NodeDetailComposite(Composite parent, int style) {
        super(parent, style);
        initialize();
    }

    /**
     * コンポジットを構築します。
     */
    private void initialize() {
        GridLayout layout = new GridLayout(1, true);
        this.setLayout(layout);
        layout.marginHeight = 0;
        layout.marginWidth = 0;

        //セッションID・ジョブIDラベル作成
        m_idLabel = new Label(this, SWT.LEFT);
        GridData gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        m_idLabel.setLayoutData(gridData);

        //ノード詳細テーブル作成
        Table table = new Table(this, SWT.H_SCROLL | SWT.V_SCROLL
                | SWT.FULL_SELECTION | SWT.SINGLE);
        table.setHeaderVisible(true);
        table.setLinesVisible(true);

        gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.verticalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        gridData.grabExcessVerticalSpace = true;
        gridData.horizontalSpan = 1;
        table.setLayoutData(gridData);

        m_viewer = new CommonTableViewer(table);
        m_viewer.createTableColumn(GetNodeDetailTableDefine.get(),
                NodeDetailTableDefine.SORT_COLUMN_INDEX,
                NodeDetailTableDefine.SORT_ORDER);

        m_viewer
                .addSelectionChangedListener(new NodeDetailSelectionChangedListener(
                        this));

    }

    /**
     * テーブルビューアーを更新します。<BR>
     * 引数で指定されたセッションIDとジョブIDのノード詳細一覧情報を取得し、
     * 共通テーブルビューアーにセットします。
     * <p>
	 * <ol>
	 * <li>引数で指定されたセッションIDとジョブIDのノード詳細一覧情報を取得します。</li>
	 * <li>共通テーブルビューアーにノード詳細一覧情報をセットします。</li>
	 * </ol>
     * 
     * @param sessionId セッションID
     * @param jobunitId 所属ジョブユニットのジョブID
     * @param jobId ジョブID
     * 
     * @see com.clustercontrol.jobmanagement.action.GetNodeDetail#getNodeDetail(String, String)
     */
    public void update(String sessionId, String jobunitId, String jobId) {
        ArrayList<JobNodeDetail> nodeDetailInfo = null;

        //ノード詳細情報取得
        GetNodeDetail getNodeDetail = new GetNodeDetail();
        if (sessionId != null && jobId != null) {
            nodeDetailInfo = getNodeDetail.getNodeDetail(sessionId, jobunitId, jobId);
        } else {
            nodeDetailInfo = new ArrayList<JobNodeDetail>();
        }
        ArrayList<Object> listInput = new ArrayList<Object>();
    	for (JobNodeDetail info : nodeDetailInfo) {
    		ArrayList<Object> a = new ArrayList<Object>();
    		a.add(info.getStatus());
    		a.add(info.getEndValue());
    		a.add(info.getFacilityId());
    		a.add(info.getNodeName());
    		a.add(info.getStartDate());
    		a.add(info.getEndDate());
    		a.add(info.getMessage());
    		listInput.add(a);
    	}
    	m_viewer.setInput(listInput);
        m_sessionId = sessionId;
        m_jobunitId = jobunitId;
        m_jobId = jobId;

        //セッションID・ジョブIDを表示
        if (m_sessionId != null && m_jobId != null) {
            m_idLabel.setText(Messages.getString("session.id") + " : "
                    + m_sessionId + ",   " + Messages.getString("job.id")
                    + " : " + m_jobId);
        } else {
            m_idLabel.setText(Messages.getString("session.id") + " : " + ",   "
                    + Messages.getString("job.id") + " : ");
        }
    }

    /**
     * このコンポジットが利用するテーブルビューアを返します。
     * 
     * @return テーブルビューア
     */
    public TableViewer getTableViewer() {
        return m_viewer;
    }

    /**
     * このコンポジットが利用するテーブルを返します。
     * 
     * @return テーブル
     */
    public Table getTable() {
        return m_viewer.getTable();
    }

    /**
     * セッションIDを返します。
     * 
     * @return セッションID
     */
    public String getSessionId() {
        return m_sessionId;
    }

    /**
     * セッションIDを設定します。
     * 
     * @param sessionId セッションID
     */
    public void setSessionId(String sessionId) {
        m_sessionId = sessionId;
    }

    /**
     * ジョブIDを返します。
     * 
     * @return ジョブID
     */
    public String getJobId() {
        return m_jobId;
    }

    /**
     * ジョブIDを設定します。
     * 
     * @param jobId ジョブID
     */
    public void setJobId(String jobId) {
        m_jobId = jobId;
    }

    /**
     * ファシリティIDを返します。
     * 
     * @return ファシリティID
     */
    public String getFacilityId() {
        return m_facilityId;
    }

    /**
     * ファシリティIDを設定します。
     * 
     * @param facilityId ファシリティID
     */
    public void setFacilityId(String facilityId) {
        m_facilityId = facilityId;
    }

	/**
	 * 所属ジョブユニットのジョブIDを取得します。
	 * 
	 * @return 所属ジョブユニットのジョブID
	 */
	public String getJobunitId() {
		return m_jobunitId;
	}

	/**
	 * 所属ジョブユニットのジョブIDを設定します。
	 * 
	 * @param jobunitId 所属ジョブユニットのジョブID
	 */
	public void setJobunitId(String jobunitId) {
		m_jobunitId = jobunitId;
	}




}