/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.jobmanagement.composite;

import java.util.ArrayList;

import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Table;

import com.clustercontrol.composite.CommonTableViewer;
import com.clustercontrol.jobmanagement.action.GetForwardFile;
import com.clustercontrol.jobmanagement.action.GetForwardFileTableDefine;
import com.clustercontrol.jobmanagement.bean.ForwardFileTableDefine;
import com.clustercontrol.jobmanagement.bean.JobForwardFile;
import com.clustercontrol.util.Messages;

/**
 * ジョブ[ファイル転送]ビュー用のコンポジットクラスです。
 * 
 * @version 2.0.0
 * @since 2.0.0
 */
public class ForwardFileComposite extends Composite {
	/** テーブルビューアー */
    protected CommonTableViewer m_viewer = null;
    /** セッションID */
    protected String m_sessionId = null;
    /** 所属ジョブユニットのジョブID */
    protected String m_jobunitId = null;
    /** ジョブID */
    protected String m_jobId = null;
    /** ID用ラベル */
    protected Label m_idLabel = null;

    /**
     * コンストラクタ
     * 
     * @param parent 親のコンポジット
     * @param style スタイル
     * 
     * @see org.eclipse.swt.SWT
     * @see org.eclipse.swt.widgets.Composite#Composite(Composite parent, int style)
     * @see #initialize()
     */
    public ForwardFileComposite(Composite parent, int style) {
        super(parent, style);
        initialize();
    }

    /**
     * コンポジットを配置します。
     */
    private void initialize() {
        GridLayout layout = new GridLayout(1, true);
        this.setLayout(layout);
        layout.marginHeight = 0;
        layout.marginWidth = 0;

        //セッションID・ジョブIDラベル作成
        m_idLabel = new Label(this, SWT.LEFT);
        GridData gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        m_idLabel.setLayoutData(gridData);

        //ファイル転送テーブル作成
        Table table = new Table(this, SWT.H_SCROLL | SWT.V_SCROLL
                | SWT.FULL_SELECTION | SWT.SINGLE);
        table.setHeaderVisible(true);
        table.setLinesVisible(true);

        gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.verticalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        gridData.grabExcessVerticalSpace = true;
        gridData.horizontalSpan = 1;
        table.setLayoutData(gridData);

        m_viewer = new CommonTableViewer(table);
        m_viewer.createTableColumn(GetForwardFileTableDefine.get(),
        		ForwardFileTableDefine.SORT_COLUMN_INDEX,
        		ForwardFileTableDefine.SORT_ORDER);
    }

    /**
     * テーブルビューアーを更新します。<BR>
     * 引数で指定されたセッションIDとジョブIDのファイル転送一覧情報を取得し、
     * 共通テーブルビューアーにセットします。
     * <p>
	 * <ol>
	 * <li>引数で指定されたセッションIDとジョブIDのファイル転送一覧情報を取得します。</li>
	 * <li>共通テーブルビューアーにファイル転送一覧情報をセットします。</li>
	 * </ol>
     * 
     * @param sessionId セッションID
     * @param jobId ジョブID
     * 
     * @see com.clustercontrol.jobmanagement.action.GetForwardFile#get(String, String)
     */
    public void update(String sessionId, String jobunitId, String jobId) {
        ArrayList<JobForwardFile> list = null;

        //ファイル転送情報取得
        GetForwardFile getForwardFile = new GetForwardFile();
        if (sessionId != null && jobId != null) {
            list = getForwardFile.get(sessionId, jobunitId, jobId);
        } else {
            list = new ArrayList<JobForwardFile>();
        }

        ArrayList<Object> listInput = new ArrayList<Object>();
    	for (JobForwardFile info : list) {
    		ArrayList<Object> a = new ArrayList<Object>();
    		a.add(info.getStatus());
    		a.add(info.getEndStatus());
    		a.add(info.getFile());
    		a.add(info.getSrcFacility());
    		a.add(info.getSrcFacilityName());
    		a.add(info.getDstFacilityId());
    		a.add(info.getDstFacilityName());
    		a.add(info.getStartDate());
    		a.add(info.getEndDate());
    		listInput.add(a);
    	}
        m_viewer.setInput(listInput);

        m_sessionId = sessionId;
        m_jobunitId = jobunitId;
        m_jobId = jobId;

        //セッションID・ジョブIDを表示
        if (m_sessionId != null && m_jobId != null) {
            m_idLabel.setText(Messages.getString("session.id") + " : "
                    + m_sessionId + ",   " + Messages.getString("job.id")
                    + " : " + m_jobId);
        } else {
            m_idLabel.setText(Messages.getString("session.id") + " : " + ",   "
                    + Messages.getString("job.id") + " : ");
        }
    }

    /**
     * このコンポジットが利用するテーブルビューアを返します。
     * 
     * @return テーブルビューア
     */
    public TableViewer getTableViewer() {
        return m_viewer;
    }

    /**
     * このコンポジットが利用するテーブルを返します。
     * 
     * @return テーブル
     */
    public Table getTable() {
        return m_viewer.getTable();
    }

    /**
     * セッションIDを返します。
     * 
     * @return セッションID
     */
    public String getSessionId() {
        return m_sessionId;
    }

    /**
     * セッションIDを設定します。
     * 
     * @param sessionId セッションID
     */
    public void setSessionId(String sessionId) {
        m_sessionId = sessionId;
    }

    /**
     * ジョブIDを返します。
     * 
     * @return ジョブID
     */
    public String getJobId() {
        return m_jobId;
    }

    /**
     * ジョブIDを設定します。
     * 
     * @param jobId ジョブID
     */
    public void setJobId(String jobId) {
        m_jobId = jobId;
    }

	/**
	 * 
	 * 所属ジョブユニットのジョブIDを返します。
	 * 
	 * @return 所属ジョブユニットのジョブID
	 */
	public String getJobunitId() {
		return m_jobunitId;
	}

	/**
	 * 
	 * 所属ジョブユニットのジョブIDを設定します。
	 * 
	 * @param jobunitId 所属ジョブユニットのジョブID
	 */
	public void setJobunitId(String jobunitId) {
		m_jobunitId = jobunitId;
	}




}