/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.jobmanagement.action;

import java.rmi.AccessException;
import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Locale;

import org.eclipse.jface.dialogs.MessageDialog;

import com.clustercontrol.bean.DataRangeConstant;
import com.clustercontrol.bean.Property;
import com.clustercontrol.bean.PropertyConstant;
import com.clustercontrol.jobmanagement.bean.JobOperationConstant;
import com.clustercontrol.jobmanagement.bean.OperationConstant;
import com.clustercontrol.jobmanagement.ejb.session.JobController;
import com.clustercontrol.jobmanagement.util.EjbConnectionManager;
import com.clustercontrol.util.Messages;

/**
 * ジョブ操作用プロパティを取得するクライアント側アクションクラス<BR>
 * 
 * マネージャにSessionBean経由でアクセスし、ジョブ操作用プロパティを取得する
 * 
 * @version 1.0.0
 * @since 1.0.0
 */
public class GetJobOperationProperty {
	
	/**
	 * 開始用プロパティ取得
	 * 
	 * @param sessionId セッションID
	 * @param jobunitId 所属ジョブユニットのジョブID
	 * @param jobId ジョブID
	 * @param facilityId ファシリティID
	 * @return ジョブ開始操作用プロパティ
	 * 
	 * @see com.clustercontrol.jobmanagement.ejb.session.JobController
	 * @see com.clustercontrol.jobmanagement.util.EjbConnectionManager
	 * @see com.clustercontrol.jobmanagement.ejb.session.JobControllerBean#getStartProperty(java.lang.String, java.lang.String, java.lang.String, java.util.Locale)
	 */
	public Property getStartProperty(String sessionId, String jobunitId, String jobId, String facilityId) {
		Locale locale = Locale.getDefault();
	    //セッションID
		Property session =
			new Property(JobOperationConstant.SESSION, Messages.getString("session.id", locale), PropertyConstant.EDITOR_TEXT);
	    //ジョブユニットID
		Property jobUnit =
			new Property(JobOperationConstant.JOB_UNIT, Messages.getString("jobunit.id", locale), PropertyConstant.EDITOR_TEXT);
	    //ジョブID
		Property job =
			new Property(JobOperationConstant.JOB, Messages.getString("job.id", locale), PropertyConstant.EDITOR_TEXT);
	    //ファシリティID
		Property facility =
			new Property(JobOperationConstant.FACILITY, Messages.getString("facility.id", locale), PropertyConstant.EDITOR_TEXT);
		//制御
		Property control =
			new Property(JobOperationConstant.CONTROL, Messages.getString("control", locale), PropertyConstant.EDITOR_SELECT);
		
		JobController jobController = 
			EjbConnectionManager.getConnectionManager().getJobController();
		ArrayList<String> values = null;
		try {
			values = jobController.getAvailableStartOperation(sessionId, jobunitId, jobId, facilityId, locale);
		} catch (RemoteException e) {
			if(e instanceof AccessException){
				// アクセス権なしの場合、エラーダイアログを表示する
				MessageDialog.openInformation(null, Messages.getString("message"),
						Messages.getString("message.accesscontrol.16"));
			} else {
				e.printStackTrace();
			}
		}
		
		//値を初期化
		Object controlValues[][] = {values.toArray(), values.toArray()};
		
		control.setSelectValues(controlValues);
		if(values.size() >= 1){
		    control.setValue(values.get(0));
		}
		else{
		    control.setValue("");
		}
		
		session.setValue(sessionId);
		jobUnit.setValue(jobunitId);
		job.setValue(jobId);
		if(facilityId != null && facilityId.length() > 0){
		    facility.setValue(facilityId);
		}
		else{
		    facility.setValue("");
		}
		
		//変更の可/不可を設定
		session.setModify(PropertyConstant.MODIFY_NG);
		jobUnit.setModify(PropertyConstant.MODIFY_NG);
		job.setModify(PropertyConstant.MODIFY_NG);
		facility.setModify(PropertyConstant.MODIFY_NG);
		control.setModify(PropertyConstant.MODIFY_OK);

		Property property = new Property(null, null, "");
		
		// 初期表示ツリーを構成。
		property.removeChildren();
		property.addChildren(session);
		property.addChildren(jobUnit);
		property.addChildren(job);
		if(facilityId != null && facilityId.length() > 0){
		    property.addChildren(facility);
		}
		property.addChildren(control);
		
		return property;
	}
	
	
	/**
	 * 停止用プロパティ取得
	 * 
	 * @param sessionId セッションID
	 * @param jobunitId 所属ジョブユニットのジョブID
	 * @param jobId ジョブID
	 * @param facilityId ファシリティID
	 * @return ジョブ停止操作用プロパティ
	 * 
	 * @see com.clustercontrol.jobmanagement.ejb.session.JobController
	 * @see com.clustercontrol.jobmanagement.util.EjbConnectionManager
	 * @see com.clustercontrol.jobmanagement.ejb.session.JobControllerBean#getStopProperty(java.lang.String, java.lang.String, java.lang.String, java.util.Locale)
	 */
	public Property getStopProperty(String sessionId, String jobunitId, String jobId, String facilityId) {
		Locale locale = Locale.getDefault();
		JobController jobController = 
			EjbConnectionManager.getConnectionManager().getJobController();
		
	    //セッションID
		Property session =
			new Property(JobOperationConstant.SESSION, Messages.getString("session.id", locale), PropertyConstant.EDITOR_TEXT);
	    //ジョブユニットID
		Property jobUnit =
			new Property(JobOperationConstant.JOB_UNIT, Messages.getString("jobunit.id", locale), PropertyConstant.EDITOR_TEXT);
	    //ジョブID
		Property job =
			new Property(JobOperationConstant.JOB, Messages.getString("job.id", locale), PropertyConstant.EDITOR_TEXT);
	    //ファシリティID
		Property facility =
			new Property(JobOperationConstant.FACILITY, Messages.getString("facility.id", locale), PropertyConstant.EDITOR_TEXT);
		//制御
		Property control =
			new Property(JobOperationConstant.CONTROL, Messages.getString("control", locale), PropertyConstant.EDITOR_SELECT);
	    //終了値
		Property endValue =
			new Property(JobOperationConstant.END_VALUE, Messages.getString("end.value", locale), PropertyConstant.EDITOR_NUM, 
					DataRangeConstant.SMALLINT_HIGH, DataRangeConstant.SMALLINT_LOW);
		
	    ArrayList<Property> endList = new ArrayList<Property>();
	    endList.add(endValue);
	    
	    HashMap<String, Object> skipEndMap = new HashMap<String, Object>();
	    skipEndMap.put("value", OperationConstant.STRING_STOP_SKIP);
	    skipEndMap.put("property", endList);
	    
	    HashMap<String, Object> mainteEndMap = new HashMap<String, Object>();
	    mainteEndMap.put("value", OperationConstant.STRING_STOP_MAINTENANCE);
	    mainteEndMap.put("property", endList);

	    HashMap<String, Object> forceEndMap = new HashMap<String, Object>();
	    forceEndMap.put("value", OperationConstant.STRING_STOP_FORCE);
	    forceEndMap.put("property", endList);

	    ArrayList<String> values1 = null;
		try {
			values1 = jobController.getAvailableStopOperation(sessionId, jobunitId, jobId, facilityId, locale);
		} catch (RemoteException e) {
			if(e instanceof AccessException){
				// アクセス権なしの場合、エラーダイアログを表示する
				MessageDialog.openInformation(null, Messages.getString("message"),
						Messages.getString("message.accesscontrol.16"));
			} else {
				e.printStackTrace();
			}
		}
		
		ArrayList<Object> values2 = new ArrayList<Object>();
		if(values1.contains(OperationConstant.STRING_STOP_AT_ONCE)) {
			values2.add(OperationConstant.STRING_STOP_AT_ONCE);
		}
		if(values1.contains(OperationConstant.STRING_STOP_SUSPEND)){
		    values2.add(OperationConstant.STRING_STOP_SUSPEND);
		}
		if(values1.contains(OperationConstant.STRING_STOP_WAIT)){
			values2.add(OperationConstant.STRING_STOP_WAIT);
		}
		if(values1.contains(OperationConstant.STRING_STOP_SKIP)) {
			values2.add(skipEndMap);
		}
		if(values1.contains(OperationConstant.STRING_STOP_MAINTENANCE)) {
			values2.add(mainteEndMap);
		}
		if(values1.contains(OperationConstant.STRING_STOP_FORCE)) {
		    values2.add(forceEndMap);
		}
		
		
		//値を初期化
		Object controlValues[][] = {values1.toArray(), values2.toArray()};
		
		control.setSelectValues(controlValues);
		if(values1.size() >= 1){
		    control.setValue(values1.get(0));
		}
		else{
		    control.setValue("");
		}
		
		endValue.setValue("");
		
		session.setValue(sessionId);
		jobUnit.setValue(jobunitId);
		job.setValue(jobId);
		if(facilityId != null && facilityId.length() > 0){
		    facility.setValue(facilityId);
		}
		else{
		    facility.setValue("");
		}
		
		//変更の可/不可を設定
		session.setModify(PropertyConstant.MODIFY_NG);
		jobUnit.setModify(PropertyConstant.MODIFY_NG);
		job.setModify(PropertyConstant.MODIFY_NG);
		facility.setModify(PropertyConstant.MODIFY_NG);
		control.setModify(PropertyConstant.MODIFY_OK);
		endValue.setModify(PropertyConstant.MODIFY_OK);

		Property property = new Property(null, null, "");
		
		// 初期表示ツリーを構成。
		property.removeChildren();
		property.addChildren(session);
		property.addChildren(jobUnit);
		property.addChildren(job);
		if(facilityId != null && facilityId.length() > 0){
		    property.addChildren(facility);
		}
		property.addChildren(control);
		
		if(OperationConstant.STRING_STOP_SKIP.compareTo((String)control.getValue()) == 0 || 
				OperationConstant.STRING_STOP_MAINTENANCE.compareTo((String)control.getValue()) == 0 || 
				OperationConstant.STRING_STOP_FORCE.compareTo((String)control.getValue()) == 0 ){
		    control.addChildren(endValue);
		}
		
		
		
		
		return property;
	}
}
