/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.collectiverun.composite;

import java.rmi.AccessException;
import java.util.ArrayList;

import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.dialogs.MessageDialogWithToggle;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Table;

import com.clustercontrol.ClusterControlPlugin;
import com.clustercontrol.bean.Property;
import com.clustercontrol.bean.ViewListInfo;
import com.clustercontrol.collectiverun.action.GetHistoryInfo;
import com.clustercontrol.collectiverun.action.GetHistoryTabelDefine;
import com.clustercontrol.collectiverun.bean.HistoryTableDefine;
import com.clustercontrol.collectiverun.composite.action.HistorySelectionChangedListener;
import com.clustercontrol.collectiverun.preference.CollectiveRunPreferencePage;
import com.clustercontrol.composite.CommonTableViewer;
import com.clustercontrol.util.Messages;

/**
 * 一括制御[履歴]コンポジット
 * 
 * 
 * @version 2.1.1
 * @since 1.0.0
 *  
 */
public class HistoryComposite extends Composite {
    private CommonTableViewer m_viewer = null;
    
    protected Label m_labelType = null;

    protected Label m_labelCount = null;

    protected String m_sessionId = null;
    
    private boolean openAccessInformation = false;
    private boolean openInformation = false;

    /**
     * コンストラクタ
     * 
     * @param parent
     * @param style
     */
    public HistoryComposite(Composite parent, int style) {
        super(parent, style);
        initialize();
    }

    /**
     * 初期化処理
     *  
     */
    private void initialize() {
        GridLayout layout = new GridLayout(1, true);
        this.setLayout(layout);
        layout.marginHeight = 0;
        layout.marginWidth = 0;
        
        m_labelType = new Label(this, SWT.LEFT);
        GridData gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.verticalAlignment = GridData.FILL;
        m_labelType.setLayoutData(gridData);

        Table table = new Table(this, SWT.H_SCROLL | SWT.V_SCROLL
                | SWT.FULL_SELECTION | SWT.SINGLE);
        table.setHeaderVisible(true);
        table.setLinesVisible(true);

        gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.verticalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        gridData.grabExcessVerticalSpace = true;
        gridData.horizontalSpan = 1;
        table.setLayoutData(gridData);

        m_labelCount = new Label(this, SWT.RIGHT);
        gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.verticalAlignment = GridData.FILL;
        m_labelCount.setLayoutData(gridData);
        
        m_viewer = new CommonTableViewer(table);
        m_viewer.createTableColumn(GetHistoryTabelDefine.get(),
                HistoryTableDefine.SORT_COLUMN_INDEX,
                HistoryTableDefine.SORT_ORDER);

        this.update();

        m_viewer.addSelectionChangedListener(
        		new HistorySelectionChangedListener(this));
    }

    /**
     * 更新処理
     * 
     * @since 1.0.0
     */
    @Override
	public void update() {
    	ViewListInfo historyInfo = null;

        //履歴情報取得
        GetHistoryInfo history = new GetHistoryInfo();
        try {
        	int histories = ClusterControlPlugin.getDefault().getPreferenceStore().getInt(
        			CollectiveRunPreferencePage.P_HISTORY_MAX_HISTORIES);
            historyInfo = history.getHistory(histories);
        } catch (AccessException e) {
        	if(!openAccessInformation){
        		openAccessInformation = true;
        		// アクセス権なしの場合、エラーダイアログを表示する
        		MessageDialog.openInformation(null, Messages.getString("message"),
	                    Messages.getString("message.accesscontrol.16"));
        	}
        }
        
        if (historyInfo == null) {
        	historyInfo = new ViewListInfo();
        	historyInfo.setList(new ArrayList());
        }
        else if(historyInfo.getList() == null){
        	historyInfo.setList(new ArrayList());
        }
        
        if(ClusterControlPlugin.getDefault().getPreferenceStore().getBoolean(
        		CollectiveRunPreferencePage.P_HISTORY_MESSAGE_FLG)){
        	if(historyInfo.getTotal() > historyInfo.getList().size()){ 
        		if(!openInformation){
        			openInformation = true;
        			// 最大表示件数を超える場合、エラーダイアログを表示する
        			MessageDialogWithToggle.openInformation(
        					null, 
        					Messages.getString("message"),
        					Messages.getString("message.job.33"), 
        					Messages.getString("message.will.not.be.displayed"), 
        					false, 
        					ClusterControlPlugin.getDefault().getPreferenceStore(), 
        					CollectiveRunPreferencePage.P_HISTORY_MESSAGE_FLG);
        			openInformation = false;
        		}
        	} 
        }

        m_viewer.setInput(historyInfo.getList());

        selectHistory(historyInfo.getList());
        
        m_labelType.setText("");
        Object[] args = { new Integer(historyInfo.getTotal()) };
        m_labelCount.setText(Messages.getString("records", args));
    }

    /**
     * 更新処理
     * 
     * @param condition
     *            フィルタ条件
     * @since 1.0.0
     */
    public void update(Property condition) {
    	ViewListInfo historyInfo = null;

        //履歴情報取得
        GetHistoryInfo history = new GetHistoryInfo();
        try {
        	int histories = ClusterControlPlugin.getDefault().getPreferenceStore().getInt(
        			CollectiveRunPreferencePage.P_HISTORY_MAX_HISTORIES);
            historyInfo = history.getHistory(condition, histories);
        } catch (AccessException e) {
        	if(!openAccessInformation){
        		openAccessInformation = true;
        		// アクセス権なしの場合、エラーダイアログを表示する
        		MessageDialog.openInformation(null, Messages.getString("message"),
	                    Messages.getString("message.accesscontrol.16"));
        	}
        }
        
        if (historyInfo == null) {
        	historyInfo = new ViewListInfo();
        	historyInfo.setList(new ArrayList());
        }
        else if(historyInfo.getList() == null){
        	historyInfo.setList(new ArrayList());
        }
        
        if(ClusterControlPlugin.getDefault().getPreferenceStore().getBoolean(
        		CollectiveRunPreferencePage.P_HISTORY_MESSAGE_FLG)){
        	if(historyInfo.getTotal() > historyInfo.getList().size()){ 
        		if(!openInformation){
        			openInformation = true;
        			// 最大表示件数を超える場合、エラーダイアログを表示する
        			MessageDialogWithToggle.openInformation(
        					null, 
        					Messages.getString("message"),
        					Messages.getString("message.job.33"), 
        					Messages.getString("message.will.not.be.displayed"), 
        					false, 
        					ClusterControlPlugin.getDefault().getPreferenceStore(), 
        					CollectiveRunPreferencePage.P_HISTORY_MESSAGE_FLG);
        			openInformation = false;
        		}
        	} 
        }

        m_viewer.setInput(historyInfo.getList());

        selectHistory(historyInfo.getList());
        
        m_labelType.setText(Messages.getString("filtered.list"));
        Object[] args = { new Integer(historyInfo.getTotal()) };
        m_labelCount.setText(Messages.getString("filtered.records", args));
    }

    /**
     * 選択処理
     * 
     * @param historyInfo
     *            履歴情報
     * @since 1.0.0
     */
    public void selectHistory(ArrayList historyInfo) {
        if (m_sessionId != null && m_sessionId.length() > 0) {
            int index = -1;
            for (int i = 0; i < historyInfo.size(); i++) {
                ArrayList line = (ArrayList) historyInfo.get(i);
                String sessionId = (String) line
                        .get(HistoryTableDefine.SESSION_ID);
                if (m_sessionId.compareTo(sessionId) == 0) {
                    index = i;
                    break;
                }
            }
            if (index == -1) {
                m_sessionId = null;
            } else {
                m_viewer.setSelection(new StructuredSelection(historyInfo
                        .get(index)), true);
            }
        }
    }

    /**
     * テーブルビューワ取得
     * 
     * @return m_viewer
     */
    public CommonTableViewer getTableViewer() {
        return this.m_viewer;
    }

    /**
     * @return Returns the sessionId.
     */
    public String getSessionId() {
        return m_sessionId;
    }

    /**
     * @param sessionId
     *            The sessionId to set.
     */
    public void setSessionId(String sessionId) {
        m_sessionId = sessionId;
    }
}