/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
 */

package com.clustercontrol.action;

import java.io.IOException;
import java.rmi.RemoteException;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Locale;
import java.util.Set;

import org.eclipse.swt.widgets.Composite;

import com.clustercontrol.bean.FacilityTreeItem;
import com.clustercontrol.composite.FacilityTreeComposite;
import com.clustercontrol.repository.action.GetFacilityTree;
import com.clustercontrol.repository.ejb.session.RepositoryController;
import com.clustercontrol.util.EjbConnectionManager;

/**
 * スコープツリーアイテムを取得するクライアント側アクションクラス<BR>
 * 
 * @version 1.0.0
 * @since 1.0.0
 */
public class FacilityTree {
	// ファシリティツリーのクライアント側ローカルキャッシュを保持
	private static FacilityTreeItem m_facilityTreeCache = null;

	// ファシリティツリーを表示しているビューComposite
	private static final Set<Composite> m_DisplayViewSet = new HashSet<Composite>();

	/**
	 * スコープツリー構造を返します。
	 * 
	 * @return スコープツリー構造
	 */
	public FacilityTreeItem getTreeItem() {
		//System.out.println("FacilityTree.getTreeItem() start : scopeOnly =  " + scopeOnly);
		FacilityTreeItem item = null;

		if (m_facilityTreeCache == null) {
			// 存在しない場合はキャッシュ再交信再取得
			refleshCache();
		}
		// 要素を取得
		try {
			item = m_facilityTreeCache.deepCopy();
		} catch (IOException e) {
			e.printStackTrace();
		} catch (ClassNotFoundException e) {
			e.printStackTrace();
		}
		
		//System.out.println("FacilityTree.getTreeItem() stop : scopeOnly =  " + scopeOnly);
		return item;
	}

	/**
	 * セパレータを取得します。<BR>
	 * 
	 * スコープの区切り文字（セパレータ）を取得します。
	 * 
	 * @return セパレータ文字列
	 */
	public String getSeparator() {
		RepositoryController repository = EjbConnectionManager
				.getConnectionManager().getRepositoryController();
		String separator = ">";
		try {
			separator = repository.getSeparator();
		} catch (RemoteException e) {
		}
		return separator;
	}

	/**
	 * 保持しているローカルキャッシュをクリアして画面を再描画する
	 */
	public static void clearCache() {
		//System.out.println("FacilityTree.clearCache() start");

		// キャッシュ情報を削除する
		//System.out.println("FacilityTree.clearCache() : clear cache");

		m_facilityTreeCache = null;

		//System.out.println("FacilityTree.clearCache() stop");
	}

	/**
	 * キャッシュおよび画面のリフレッシュ
	 */
	public static void reflesh() {
		//System.out.println("FacilityTree.reflesh() start");

		refleshCache();
		refleshComposite();

		//System.out.println("FacilityTree.reflesh() stop");
	}

	/**
	 * キャッシュのリフレッシュ
	 */
	public static void refleshCache() {
		//System.out.println("FacilityTree.refleshCache() : start");

		// 先にキャッシュクリア
		// clearCache() ;

		// 再登録
		FacilityTreeItem item = null;
		try {
			GetFacilityTree action = new GetFacilityTree();
			
			item = action.getFacilityTree(Locale.getDefault());
			if (item != null) {
				m_facilityTreeCache = item;
			} else {
				System.out.println("error 136");
			}
			
		} catch (Exception e) {
			e.printStackTrace();
			return;
		}

		//System.out.println("FacilityTree.refleshCache() : stop");
	}

	/**
	 * ファシリティツリー画面を再描画する
	 */
	public static void refleshComposite() {
		//System.out.println("FacilityTree.refleshComposite() : start");

		synchronized (m_DisplayViewSet) {

			Iterator<Composite> iterator = m_DisplayViewSet.iterator();
			while (iterator.hasNext()) {
				FacilityTreeComposite composite = (FacilityTreeComposite) iterator
						.next();
				if (!composite.isDisposed()) {
					//System.out.println("FacilityTree.refleshComposite() : target is "
							//+ composite.getClass().getCanonicalName());
					composite.update();
				}
			}
		}

		//System.out.println("refleshComposite() : stop");
	}

	/**
	 * キャッシュ更新時にリフレッシュするCompositeを一覧に追加
	 * 
	 * @param composite
	 */
	public void addComposite(FacilityTreeComposite composite) {
		//System.out.println("FacilityTree.addComposite() : start");

		synchronized (m_DisplayViewSet) {
			//System.out.println("FacilityTree.addComposite()  composite is "
					//+ composite.getClass().getCanonicalName());
			m_DisplayViewSet.add(composite);
		}

		//System.out.println("FacilityTree.addComposite() : stop");
	}

	/**
	 * キャッシュ更新時にリフレッシュするCompositeを一覧から削除
	 * 
	 * @param composite
	 */
	public void delComposite(FacilityTreeComposite composite) {
		//System.out.println("FacilityTree.delComposite() : start");

		synchronized (m_DisplayViewSet) {
			m_DisplayViewSet.remove(composite);
		}
		//System.out.println("FacilityTree.delComposite() : stop");
	}
}