/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.agent;

import java.util.Collection;
import java.util.Date;
import java.util.Iterator;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;

import javax.jms.ExceptionListener;
import javax.jms.JMSException;
import javax.jms.Message;
import javax.jms.MessageListener;
import javax.jms.ObjectMessage;
import javax.jms.Session;
import javax.jms.Topic;
import javax.jms.TopicConnection;
import javax.jms.TopicConnectionFactory;
import javax.jms.TopicSession;
import javax.jms.TopicSubscriber;
import javax.naming.InitialContext;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.agent.util.AgentProperties;
import com.clustercontrol.repository.bean.TopicConstant;
import com.clustercontrol.repository.message.UpdateRepositoryInfo;
import com.clustercontrol.util.Dummy;

/**
 * リポジトリ情報の更新通知を受信するクラス<BR>
 * 
 * リポジトリの情報（スコープ、ノード）が更新されると
 * 更新が発生したことをTopicを通じてエージェントに通知されます。<BR>
 * その更新を受信するクラスです。
 * 
 * Topicへの接続と、メッセージの受信を行う。
 * @version 2.0.0
 * @since 2.0.0
 * 
 */
public class UpdateRepositoryInfoReceiveTopic implements MessageListener, ExceptionListener {
	private static final String TOPIC_CON_FACTORY = "ConnectionFactory";
	private static long m_repositoryTopicReconnectionInterval = 10;
	
	private TopicConnectionFactory m_factory;
	protected TopicConnection m_con;
	protected Topic m_topic;
	protected TopicSession m_session;
	protected TopicSubscriber m_subscriber;
	
	protected Agent m_agent;
	
	private volatile boolean m_errFlg = false;
	
	// heartbeat送受完了日時(jms妥当性判定用)
	private volatile static Date heartbeatReceiveDate = new Date();
	public static final Object _instanceHeartbeatDateLock = new Object();
	
	/** UpdateRepositoryInfoReceiveTopicクラス用スケジューラ * */
	private ScheduledExecutorService m_scheduler = Executors.newScheduledThreadPool(10);
    private ScheduledFuture<?> m_future;
    
	//ロガー
	private static Log log = LogFactory.getLog(UpdateRepositoryInfoReceiveTopic.class);
	
	/**
	 * コンストラクタ
	 * @param facilityIdList ファシリティIDのリスト　
	 * @param sendQueue　メッセージ送信用クラス
	 * @param props　プロパティファイル情報
	 */
	@SuppressWarnings("unchecked")
	public UpdateRepositoryInfoReceiveTopic(Agent agent) {
		super();
		 
		m_agent = agent;
		
		// 再接続処理実行間隔取得
		String interval = AgentProperties.getProperty("repository.topic.reconnection.interval");
		if (interval != null) {
			try {
				// プロパティファイルには秒で記述
				m_repositoryTopicReconnectionInterval = Integer.parseInt(interval);
				log.info("repository.topic.reconnection.interval = " + m_repositoryTopicReconnectionInterval + " sec");
			} catch (NumberFormatException e) {
				log.error("repository.topic.reconnection.interval",e);
			}
		}
		
		//接続処理
		initial();
	}
	
	/**
	 * メッセージを受信した際に動作するメソッド
	 * 
	 */
	/* 
	 * トピック受信処理
	 * (non-Javadoc)
	 * @see javax.jms.MessageListener#onMessage(javax.jms.Message)
	 */
	public void onMessage(Message message) {
		log.debug("onMessage() start.");
		
		if(message instanceof ObjectMessage){
			ObjectMessage objectMessage = (ObjectMessage)message;
			Object obj;
			try {
				obj = objectMessage.getObject();
			} catch (JMSException e) {
				log.error("onMessage()", e);
				return;
			}
			
			if(obj instanceof UpdateRepositoryInfo){
				log.debug("onMessage() get UpdateRepositoryInfo");
				
				m_agent.setFacility();
			
			}else if(obj instanceof Dummy){
				// TOPIC接続疎通のためのオブジェクトのため何もしない
				Dummy dummy = (Dummy)obj;
				if(dummy.getFacilityIdList() != null || dummy.getFacilityIdList().size() > 0)
					log.debug("onMessage() get Dummy : facilityId = " + dummy.getFacilityIdList().get(0));
				else
					log.debug("onMessage() get Dummy");
				
				// 最終受信日時を格納する
				synchronized (_instanceHeartbeatDateLock) {
					heartbeatReceiveDate = new Date();
				}
			}
			else{
				log.error("onMessage():Invalid message type received. " + obj.getClass().getCanonicalName());
				return;
			}
		}else{
			log.error("onMessage():Invalid message type received. " + message.getClass());
			return;
		}
	}
	
	/* 通信エラーハンドラ
	 * (non-Javadoc)
	 * @see javax.jms.ExceptionListener#onException(javax.jms.JMSException)
	 */
	public void onException(JMSException arg0) {
		log.error(arg0.getMessage(), arg0);
		
		m_agent.reConnection();
	}
	
	/**
	 * errFlg を設定します。
	 * 
	 * @param errFlg
	 *            
	 */
	synchronized private void setErrFlg(boolean errFlg) {
		log.debug("setErrFlg() start : errFlg = " + errFlg);
		if (m_errFlg == false && errFlg == true) {
			
			log.debug("setErrFlg() set ReSetupTask to timer");
			m_future = m_scheduler.scheduleWithFixedDelay(
					new ReSetupTask(), 
					m_repositoryTopicReconnectionInterval, 
					m_repositoryTopicReconnectionInterval, 
					TimeUnit.SECONDS);
		}
		m_errFlg = errFlg;
		log.debug("setErrFlg() finish");
	}

	/**
	 * 再初期化を行います。<BR>
	 *  
	 */
	synchronized private boolean reInitial() {
		boolean ret = false;
		
		log.info("reInitial() start");

		terminate();
		
		if (initial()) {
			
			log.info("reInitial() success!");
			ret = true;
			
			//エラーフラグ解除
			setErrFlg(false);
			
		} else {
			log.warn("reInitial() fail!");
		}
		
		return ret;
	}
	/**
	 * サーバ接続の終了処理
	 *  
	 */
	public void terminate() {
		log.debug("terminate() start");

		terminateSumscriber();
		
		
		try {
			if (m_session != null)
				m_session.close();
		} catch (JMSException e) {
			log.debug("terminate() session close error", e);
		}
		
		try {
			if (m_con != null)
				m_con.close();
		} catch (JMSException e1) {
			log.debug("terminate() connection close error", e1);
		}
		
		log.debug("terminate() finish");
	}
	/**
	 * トピック受信処理の終了 
	 */
	private void terminateSumscriber() {
		try {
			if (m_subscriber != null)
				m_subscriber.close();
		} catch (JMSException e) {
			log.debug("terminateSumscriber() subscriber close error", e);
		}
	}
	
	
	
	/**
	 * 初期化処理を行います。<BR>
	 * 
	 * JMSへの接続、トピック受信設定を行います。<BR>
	 * @return
	 */
	private boolean initial() {
		
		log.info("initial() start");

		InitialContext con = null;
		
		try {
			//InitialContextの生成
			con = new InitialContext(AgentProperties.getProperties());

			//コネクションファクトリ生成
			m_factory = (TopicConnectionFactory)con.lookup(TOPIC_CON_FACTORY);
			
			//コネクション生成
			m_con = m_factory.createTopicConnection();
			
			//セッション生成
			m_session = m_con.createTopicSession(false,
					Session.AUTO_ACKNOWLEDGE);
			
			//メッセージTopic取得
			m_topic = (Topic)con.lookup(TopicConstant.TOPIC_NAME_EXECUTE);
			
			
			//エラーハンドらセット
			m_con.setExceptionListener(this);
			
			m_con.start();
			
			//トピック接続開始
			initialTopic();
			
			
		} catch (Exception e) {
			log.error("initial() fail!", e);
			setErrFlg(true);
			return false;
		} finally {
			try {
				if (con != null)
					con.close();
			} catch (Exception e1) {
				log.warn("initial() close fail!", e1);
			}
		}
		
		log.info("initial() success!");
		return true;
		
	}
	
	/**
	 * トピック受信設定を行います。
	 * 
	 * @return　true:成功　false:失敗
	 * @since
	 */
	private boolean initialTopic() {
		log.info("initialTopic() start");
		
		//現在のTopic受信を終了
		terminateSumscriber();
		
		//新しいファシリティIDでトピック受信開始
		try {
	        //ファシリティIDでメッセージセレクターを作成
	        StringBuffer msgSelector = new StringBuffer();
			msgSelector.append("All='true'");

			Collection<String> m_facilityIdList = m_agent.getFacilityIdList();
	        for (Iterator iter = m_facilityIdList.iterator(); iter.hasNext();) {
				String facilityId = (String) iter.next();
				msgSelector.append(" OR ( FacilityId='");
				msgSelector.append(facilityId);
				msgSelector.append("' AND Agent='job' )");
			}	        
	        log.debug("initialTopic() selector = " + msgSelector.toString());
	        if(msgSelector.length() != 0){
		        log.debug("initialTopic() create ubscriber with selector = " + msgSelector.toString());
				m_subscriber = m_session.createSubscriber(m_topic, msgSelector.toString(), false);
				
				//コールバックを登録する
				m_subscriber.setMessageListener(this);
	        	
	        }else{
	        	log.debug("initialTopic() facilityId information is not exists");
	        }
	        
		} catch (Exception e) {
			log.error("initialTopic() fail!", e);
			setErrFlg(true);
			return false;
		} finally {
		}
		
		log.info("initialTopic() success!");
		return true;
	}
	
	/**
	 * 再接続処理を行う
	 */
	synchronized public void reconnect() {
		
		setErrFlg(true);
		
		terminate();
        
	}
	
	/**
	 * 接続状況を返す
	 * @return 接続状況(再接続中 : true, 接続中　: false)
	 */
	synchronized public boolean isReconnecting() {
		return m_errFlg;
	}
	
	/**
	 * Heartbeanメッセージの最終受信日時を取得するメソッド
	 * @return Heartbeatメッセージの最終受信日時
	 */
	public static Date getHeartbeatReceiveDate() {
		synchronized (_instanceHeartbeatDateLock) {
			return heartbeatReceiveDate;
		}
	}
	
	/**
	 * EJB再接続タイマータスク<BR>
	 * 
	 * 通信エラーとなった場合に定周期で呼ばれ再接続を行います。
	 */
	protected class ReSetupTask implements Runnable {

		/**
		 * デフォルトコンストラクタ
		 */
		public ReSetupTask() {
		}
		
		/**
		 * コネクションクローズ
		 */
		public void run() {
			log.debug("UpdateRepositoryInfoReceiveTopic.ReSetupTask.run() start");
			try {
				if (reInitial()) {
					//このタスクをタイマーから解除
					log.debug("UpdateRepositoryInfoReceiveTopic.ReSetupTask.run() task cancel");
					if(m_future != null){
						m_future.cancel(true);
					}
				}
			} catch (Exception e) {
				log.error("UpdateRepositoryInfoReceiveTopic.ReSetupTask.run()", e);
			}
		}
		
	}
	
}
