/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.agent;

import java.util.Calendar;
import java.util.Date;
import java.util.Hashtable;
import java.util.Properties;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.agent.util.AgentProperties;
import com.clustercontrol.agent.util.RunHistoryUtil;
import com.clustercontrol.jobmanagement.bean.CommandTypeConstant;
import com.clustercontrol.jobmanagement.message.RunInstructionInfo;
import com.clustercontrol.jobmanagement.message.RunResultInfo;

/**
 * エージェントスレッドクラス<BR>
 * 
 * エージェントでは複数の処理に対応するスレッドクラスがあります。
 * その共通の親クラスです。
 *
 * @version 2.1.0
 * @since 2.1.0
 */
public class AgentThread extends Thread {
	private static Log log = LogFactory.getLog(AgentThread.class);
	
	protected RunInstructionInfo m_info = null;
	protected SendQueue m_sendQueue = null;
	protected Hashtable<String, Date> m_runHistory = null;
	protected int m_endMessageResendInterval = 30;
	
	//オリジナルメッセージのサイズ上限（Byte）
	protected int m_limit_orgmsg = 8192;
	
	/**
	 * コンストラクタ
	 */
	public AgentThread() {
		
	}
	
	/**
	 * コンストラクタ
	 * 
	 * @param info
	 * @param sendQueue
	 * @param runHistory
	 * @param props
	 */
	public AgentThread(RunInstructionInfo info, SendQueue sendQueue, Hashtable<String, Date> runHistory) {
		m_info = info;
		m_sendQueue = sendQueue;
		m_runHistory = runHistory;
		
		//終了メッセージ再送間隔取得
		String interval = AgentProperties.getProperty("end.message.resend.interval");
		if (interval != null) {
			try {
				m_endMessageResendInterval = Integer.parseInt(interval);
				log.info("end.message.resend.interval = " + m_endMessageResendInterval + " sec");
			} catch (NumberFormatException e) {
				log.error("end.message.resend.interval",e);
			}
		}
		
		//オリジナルメッセージサイズ上限を取得
		String limit_orgmsg = AgentProperties.getProperty("limit.size.orgmsg");
		if (limit_orgmsg != null) {
			try {
				m_limit_orgmsg = Integer.parseInt(limit_orgmsg);
				log.info("limit.size.orgmsg = " + m_limit_orgmsg + " byte");
			} catch (NumberFormatException e) {
				log.error("limit.orgmsg",e);
			}
		}
		
	}
	
	/**
	 * 実行履歴削除メッセージ送信します。<BR>
	 * 
	 * エージェント側でも実行情報の履歴を持っています。<BR>
	 * マネージャに実行終了を送信するとその後に履歴削除の
	 * 指示が送信されます。<BR>
	 * その実行履歴削除の完了メッセージを送信します。
	 * 
	 */
	public void sendDeleteHistory(RunResultInfo info) {
		Date checkDate = new Date();
		
		int commandType = info.getCommandType();
		
		while(true){
			if(RunHistoryUtil.findRunHistory(m_info, m_runHistory) == null){
		        //実行履歴削除完了メッセージ作成
				if(commandType == CommandTypeConstant.NORMAL)
					info.setCommandType(CommandTypeConstant.DELETE_NORMAL_HISTORY);
				else if(commandType == CommandTypeConstant.STOP)
					info.setCommandType(CommandTypeConstant.DELETE_STOP_HISTORY);
		        
		        log.info("delete history SessionID=" + m_info.getSessionId() + ", JobID=" + m_info.getJobId());
		        
		        //送信
		        m_sendQueue.put(info);

		        break;
			}
			else{
				Calendar checkTime = Calendar.getInstance();
				checkTime.setTime(checkDate);
				checkTime.add(Calendar.SECOND, m_endMessageResendInterval);
				Date now = new Date();
				
				if(now.after(checkTime.getTime())){
			        //終了メッセージメッセージ作成
					info.setCommandType(commandType);
	
			        //送信
			        m_sendQueue.put(info);
			        
			        checkDate = new Date();
				}
			}
			
			try {
				sleep(1000);
			} catch (InterruptedException e) {
			}
		}
	}
}
