/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License 
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied 
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 PURPOSE.  See the GNU General Public License for more details.
 
 */

package com.clustercontrol.syslogng.forward;

import java.io.File;
import java.io.Serializable;
import java.rmi.RemoteException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Properties;
import java.util.Timer;
import java.util.TimerTask;

import javax.ejb.CreateException;
import javax.ejb.FinderException;
import javax.jms.ExceptionListener;
import javax.jms.JMSException;
import javax.jms.ObjectMessage;
import javax.jms.Queue;
import javax.jms.QueueConnection;
import javax.jms.QueueConnectionFactory;
import javax.jms.QueueSender;
import javax.jms.QueueSession;
import javax.jms.Session;
import javax.naming.InitialContext;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jnp.interfaces.NamingContext;

import com.clustercontrol.bean.OutputNotifyGroupInfo;
import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.calendar.ejb.session.CalendarController;
import com.clustercontrol.calendar.ejb.session.CalendarControllerHome;
import com.clustercontrol.notify.bean.QueueConstant;
import com.clustercontrol.repository.bean.FacilityAttributeConstant;
import com.clustercontrol.repository.bean.FacilityTreeAttributeConstant;
import com.clustercontrol.repository.ejb.session.RepositoryController;
import com.clustercontrol.repository.ejb.session.RepositoryControllerHome;
import com.clustercontrol.syslogng.bean.LogFilterInfo;
import com.clustercontrol.syslogng.ejb.session.SyslogNGController;
import com.clustercontrol.syslogng.ejb.session.SyslogNGControllerHome;

/**
 * EJBアクセスを行うクラス<BR>
 * syslog処理のためのEJBアクセスを行います。
 * 
 * @version 3.0.0
 * @since 1.0.0
 */
public class EJBContoroller {
	
	/** プロパティのキー：コネクションファクトリー。 */
	private static final String QUEUE_CON_FACTORY = "ConnectionFactory";
	
	/** プロパティのキー：キューへの接続ユーザ。 */
	private static final String QUEUE_USER_NAME = "queue.user.name";
	
	/** プロパティのキー：キューへの接続パスワード。 */
	private static final String QUEUE_USER_PASSWORD = "queue.user.password";
	
	/** プロパティのキー：リトライ間隔（ミリ秒）。 */
	private final static String MSG_RETRY_INTERVAL = "msg.retry.interval";
	
	/** プロパティのキー：キャッシュ更新間隔（ミリ秒）。 */
	private final static String FILTER_CHACHE_INTERVAL = "filter.cache.interval";
	
	/** プロパティのキー：メッセージファイルのパス。 */
	private final static String MSG_FILE_PATH = "msg.file.msg.path";
	
	/** プロパティのキー：メッセージファイルの拡張子。 */
	private final static String MSG_FILE_SUFFIX = "msg.file.msg.syffix";
	
	/** ディレクトリ下に作成するファイルの上限 */
	private final static String MAX_FILE_COUNT = "msg.max.file.count";
	
	/** プロパティのキー：キューの名前。 */
	private final static String QUEUE_NAME = "queue.name";
	
	/** コネクション。 */
	private QueueConnection m_con;
	
	/** キュー。 */
	private Queue m_queue;
	
	/** セッション。 */
	private QueueSession m_session;
	
	/** コネクションファクトリー。 */
	private QueueConnectionFactory m_factory;
	
	/** エラーフラグ。 */
	private boolean m_errFlg = false;
	
	/** リトライ間隔（ミリ秒）。 */
	private long m_retryInterval = 10000;
	
	/** キャッシュ更新間隔（ミリ秒）。 */
	private long m_cacheInterval = 600000;
	
	/** プロパティファイル情報。 */
	private Properties m_props;
	
	/** メッセージ管理のインスタンス。 */
	private LogManager m_logManager;
	
	/** ファイル入出力のインスタンス。 */
	private FileUtil m_fileUtil;
	
	/** メッセージサービス再接続用タイマー。 */
	protected static Timer m_timer = new Timer(true);
	
	/** ログ出力のインスタンス。 */
	static Log log = LogFactory.getLog(EJBContoroller.class);
	
	
	/** Lockオブジェクト */
	private Object reconnectLock = new Object();
	
	/**
	 * コンストラクタ。
	 * 
	 * @param logManager メッセージ管理
	 * @param props プロパティファイル情報
	 */
	public EJBContoroller(LogManager logManager, Properties props) {
		
		//インスタンス変数の初期化
		
		//ログマネージャ
		m_logManager = logManager;
		
		//プロパティファイル
		m_props = props;
		
		//エラーリトライインターバル
		String retryInterval = props.getProperty(MSG_RETRY_INTERVAL);
		if (retryInterval != null) {
			try {
				m_retryInterval = Long.parseLong(retryInterval);
			} catch (NumberFormatException e) {
			}
		}
		
		//キャッシュ更新インターバル
		String casheInterval = props.getProperty(FILTER_CHACHE_INTERVAL);
		if (casheInterval != null) {
			try {
				m_cacheInterval = Long.parseLong(casheInterval);
			} catch (NumberFormatException e) {
			}
		}
		
		//ファイル入出力ユーティリティ
		String filePath = m_props.getProperty(MSG_FILE_PATH, ".");
		String msgSuffix = m_props.getProperty(MSG_FILE_SUFFIX, ".msg");
		int maxFileCount;
		try {
			maxFileCount = Integer.parseInt(props.getProperty(MAX_FILE_COUNT, ".txt"));
		} catch (Exception e){
			maxFileCount = -1;
		}
		
		m_fileUtil = new FileUtil(filePath, msgSuffix, maxFileCount);
		
		
		//サーバー接続処理
		if (initial()) {
			sendMsgRetry();
		}
		
		//フィルタ条件の読み込み
		loadFilterInfo();
		
		//キャッシュ更新タイマー開始
		m_timer.schedule(new ReflashFilterTask(), m_cacheInterval,
				m_cacheInterval);
		
	}
	
	/**
	 * メッセージ送信の初期処理を行います。
	 * <p>
	 * <ol>
	 *  <li>コネクションファクトリと宛先を取得します。</li>
	 *  <li>コネクションファクトリより、JMSプロバイダとのコネクションを生成します。</li>
	 *  <li>コネクションよりセッションを生成します。</li>
	 * </ol>
	 * 
	 * @return 成功した場合、<code> true </code>
	 */
	private boolean initial() {
		
		log.info("EJB接続初期化");
		
		InitialContext con = null;
		
		try {
			//InitialContextの生成
			con = new InitialContext(m_props);
			
			//コネクションファクトリ生成
			m_factory = (QueueConnectionFactory) con.lookup(QUEUE_CON_FACTORY);
			
			//コネクション生成
			if (m_props.getProperty(QUEUE_USER_NAME) != null) {
				//ユーザ認証
				m_con = m_factory.createQueueConnection(m_props
						.getProperty(QUEUE_USER_NAME), m_props
						.getProperty(QUEUE_USER_PASSWORD));
			} else {
				//ユーザ認証なし
				m_con = m_factory.createQueueConnection();
			}
			
			//コネクションが切れた際のハンドリング
			m_con.setExceptionListener(new JMSExceptionListener());
			
			//セッション生成
			m_session = m_con.createQueueSession(false,
					Session.AUTO_ACKNOWLEDGE);
			
			String queueName = QueueConstant.QUEUE_NAME_NOTIFYCONTROL;
			//キューの名前の取得
			if (m_props.getProperty(QUEUE_NAME) != null && !m_props.getProperty(QUEUE_NAME).equals("")) {
				log.info("queue name : " + m_props.getProperty(QUEUE_NAME));
				queueName = m_props.getProperty(QUEUE_NAME);
			}
			//メッセージQueue取得
			m_queue = (Queue) con.lookup(queueName);
			
		} catch (Exception e) {
			log.error("Init", e);
			setErrFlg(true);
			return false;
		} finally {
			try {
				if (con != null)
					con.close();
			} catch (Exception e1) {
			}
		}
		return true;
		
	}
	
	/**
	 * フィルタとリポジトリ情報を読み込みます。
	 * <p>
	 * <ol>
	 * <li>フィルタ情報を読み込みます。</li>
	 * <li>フィルタに設定されているスコープのノード情報を読み込みます。</li>
	 * <li>結果をフィルタとファシリティ情報に格納します。</li>
	 * </ol>
	 * 
	 * @see com.clustercontrol.syslogng.forward.LoginManager#getNamingContext()
	 * @see com.clustercontrol.repository.ejb.session.RepositoryControllerBean#getExecTargetFacilityIdList(String)
	 * @see com.clustercontrol.repository.ejb.session.RepositoryControllerBean#getNodeFacilityIdList(java.lang.String, int, boolean)
	 * @see com.clustercontrol.repository.ejb.session.RepositoryControllerBean#getNodeDetail(java.util.ArrayList, java.util.ArrayList)
	 */
	public void loadFilterInfo() {
		
		log.debug("loadFilterInfo() : start");
		
		// 最後に失敗した例外格納用
		Exception lastException = null;
		
		// EJB認証失敗のケースを想定して指定回数リトライする
		boolean flg = false;
		for (int i = 0; i < 10; i++) {
			log.debug("loadFilterInfo() : number of trials = " + i);
			
			NamingContext con;
			RepositoryControllerHome m_repositoryHome;
			SyslogNGControllerHome m_logFilterHome;
			
			ArrayList<LogFilterRepositoryInfo> filterRepList = null;
			RepositoryController repositoryController = null;
			SyslogNGController logFilter = null;
			
			try{
				//
				//ログイン処理
				//
				log.debug("Login Hinemos Manager");
				con = LoginManager.getContextManager(m_props).getNamingContext();
				
				//リポジトリ情報ホームインターフェース
				m_repositoryHome = (RepositoryControllerHome) con
				.lookup(RepositoryControllerHome.JNDI_NAME);
				
				//syslog-ng監視情報ホームインターフェース
				m_logFilterHome =  (SyslogNGControllerHome) con
				.lookup(SyslogNGControllerHome.JNDI_NAME);
				
				//
				// syslog-ng監視条件取得
				//
				log.debug("Load syslog-ng Filter Info");
				logFilter = m_logFilterHome.create();
				repositoryController = m_repositoryHome.create();
				
				
				//フィルタ情報取得（ダイアログで見えるものリストと同じ？）
				ArrayList filterList = logFilter.getFilterInfoList();
				
				//実際に実行するときのフィルタ情報
				filterRepList = new ArrayList<LogFilterRepositoryInfo>();
				
				// syslog-ng監視条件には対象のfacilityIDしか含まないので、
				//　リポジトリにアクセスを行い、フィルタ条件情報（ホスト名）を作成
				for (Iterator iter = filterList.iterator(); iter.hasNext();) {
					
					//１つ１つのフィルタ条件を有効フラグを確認して、
					//実行用のフィルタ条件としていく
					LogFilterInfo filterInfo = (LogFilterInfo) iter.next();
					if (ValidConstant.typeToBoolean(filterInfo.getValid())) {
						
						LogFilterRepositoryInfo filterRepInfo = 
							new LogFilterRepositoryInfo(filterInfo);
						
						String facilityID = filterInfo.getFacilityId();
						if(log.isDebugEnabled()){
							log.debug("getList For:" + facilityID);
						}
						
						//未登録ノード対応
						//登録ノードのホスト名のリストをここでは入れておき、
						//containsのメソッドで除外する
						
						if(facilityID.equals(FacilityTreeAttributeConstant.UNREGISTEREFD_SCOPE)){
							
							facilityID=FacilityTreeAttributeConstant.REGISTEREFD_SCOPE;
							
						}
							
						//配下の一覧
						ArrayList facilityList = repositoryController.getExecTargetFacilityIdList(facilityID);
						
						//ノード名取得
						ArrayList<String> attributeList = new ArrayList<String>();
						attributeList.add(FacilityAttributeConstant.NODENAME);
						attributeList.add(FacilityAttributeConstant.CN);
						
						HashMap facilityAttrMap = repositoryController
						.getNodeDetail(facilityList, attributeList);
						
						filterRepInfo.setMap(facilityAttrMap);
						
						filterRepList.add(filterRepInfo);
						
					}	
				}
				
				//
				// フィルタ条件を設定
				//
				log.debug("Set New syslog-ng Filter Info");
				m_logManager.add(filterRepList);
				flg = true;
			
			} catch (NamingException e) {
				log.debug("Login Hinemos Manager : fail",e);
				lastException = e;
			} catch (CreateException e) {
				log.debug("Load syslog-ng Filter Info : fail",e);
				lastException = e;
			} catch (FinderException e) {
				log.debug("Load syslog-ng Filter Info : fail",e);
				lastException = e;
			} catch (RemoteException e) {
				log.debug("Load syslog-ng Filter Info : fail",e);
				lastException = e;
			} finally {
				
				log.debug("Logout Hinemos Manager");

				//ログアウト
				try {
					LoginManager.getContextManager(m_props).logout();
				} catch (NamingException e) {
					log.error("Logout Hinemos Manager : fail",e);
				}
			}
			
			//
			// 処理成功時に抜ける
			//
			if(flg) {
				log.debug("loadFilterInfo() : success");
				return;
			}
			
			// 処理失敗時はタイミングをずらす為にsleep
			try {
				Thread.sleep(i * 100);
			} catch (InterruptedException e) {
				log.debug("Catch InterruptedException for isRun()",e);
				Thread.currentThread().interrupt();
			}
		}	

		if(lastException != null){
			log.error("loadFilterInfo() : fail",lastException);
		}
		return;		
		
	}
	
	/**
	 * 引数で指定されたログ出力メッセージが、指定されたカレンダIDの稼働日に
	 * 出力されたメッセージかチェックします。
	 * 
	 * @param id カレンダID
	 * @param msg ログ出力メッセージ
	 * @return 稼働日の場合、<code> true </code>
	 * 
	 * @see com.clustercontrol.syslogng.forward.LoginManager#getNamingContext()
	 * @see com.clustercontrol.calendar.ejb.session.CalendarControllerBean#isRun(java.lang.String, java.util.Date)
	 */
	public boolean isRun(String id, OutputNotifyGroupInfo msg) {
		
		if(log.isDebugEnabled()){
			log.debug("isRun() : start");
		}
		
		// 最後に失敗した例外格納用
		Exception lastException = null;

		// EJB認証失敗のケースを想定して指定回数リトライする
		for (int i = 0; i < 10; i++) {
			CalendarController calendarController = null;

			if(log.isDebugEnabled()){
				log.debug("isRun() : start id = " + id + ", msg = " + msg.getMessage() + ", number of trials = " + i);
			}
			
			try{
				//
				//ログイン処理
				//
				if(log.isDebugEnabled()){
					log.debug("Login Hinemos Manager");
				}
				NamingContext con = LoginManager.getContextManager(m_props).getNamingContext();
				//カレンダ情報ホームインターフェース
				CalendarControllerHome calendarHome = (CalendarControllerHome) con.lookup(CalendarControllerHome.JNDI_NAME);
				calendarController = calendarHome.create();

				//
				//カレンダ稼働日チェック処理
				//
				try {
					if(calendarController.isRun(id, msg.getGenerationDate()).booleanValue()){
						// 稼動日
						if(log.isDebugEnabled()){
							log.debug("isRun() : success");
						}
						return true;
					}
					else{
						// 非稼働日
						if(log.isDebugEnabled()){
							log.debug("isRun() : calendarID = " + id + ", It is non-operating day");
						}
						return false;
					}
				} catch (FinderException e) {
					// 指定されたカレンダIDがすでに存在しない場合は、カレンダ指定なしと判断する
					log.info("isRun() : calendarID = " + id + " is not exist! msg = " + msg.getMessage());
					return true;
				}
			} catch (Exception e) {
				if(log.isDebugEnabled()){
					log.debug("iisRun() : fail",e);
				}
				lastException = e;
			} finally {
				//
				// ログアウト
				//
				if(log.isDebugEnabled()){
					log.debug("Logout Hinemos Manager");
				}
				try {
					LoginManager.getContextManager(m_props).logout();
				} catch (NamingException e) {
					log.error("Logout Hinemos Manager : fail",e);
				}
			}
			
			// 処理失敗時はタイミングをずらす為にsleep
			try {
				Thread.sleep(i * 100);
			} catch (InterruptedException e) {
				if(log.isDebugEnabled()){
					log.debug("Catch InterruptedException for isRun()",e);
				}
				Thread.currentThread().interrupt();
			}
		}
		
		//
		// 失敗はファイル出力(指定回数失敗したら諦める)
		//
		log.info("Write Message To Files");
		m_logManager.addMsg((String) msg.getMessageOrg());
		setErrFlg(true);

		//
		// 指定回数失敗したらカレンダ指定無しと判断する
		//
		if(lastException != null){
			log.error("isRun() : fail id = " + id + ", msg = " + msg.getMessage(),lastException);
		}
		return true;	

	}
	
	/**
	 * ログ出力メッセージを送信します。<BR>
	 * キューへの送信を試みます。
	 * 失敗した場合、メッセージファイルを出力します。
	 * 
	 * @param msg ログ出力メッセージ
	 * 
	 * @see #_sendMsg(Object)
	 * @see com.clustercontrol.syslogng.forward.FileUtil#write(Object)
	 */
	synchronized public void sendMsg(OutputNotifyGroupInfo msg) {
		
		
		if (!_sendMsg(msg)) {
			//送信失敗はファイル出力
			log.debug("オブジェクトメッセージファイル書き込み!");
			m_fileUtil.write(msg);
		}
	}
	
	/**
	 * 引数で指定されたメッセージをキューへ送信します。
	 * 
	 * @param msg メッセージ
	 * @return 成功した場合、<code> true </code>
	 */
	private boolean _sendMsg(Object msg) {
		
		if (isErrFlg()) {
			return false;
		}
		
		QueueSender sender = null;
		try {
			//Sender作成
			sender = m_session.createSender(m_queue);
			//メッセージ作成
			ObjectMessage mess = m_session.createObjectMessage((Serializable)msg);
			//送信
			log.debug("オブジェクトメッセージ送信!");
			sender.send(mess);
			
		} catch (JMSException e) {
			log.error(e);
			setErrFlg(true);
			return false;
			
		} finally {
			try {
				if (sender != null)
					sender.close();
			} catch (Exception e1) {
			}
		}
		return true;
		
	}
	
	/**
	 * メッセージをキューへ再送信します。
	 * <p>
	 * <ol>
	 *  <li>メッセージファイル一覧を取得します。</li>
	 *  <li>メッセージファイルを読み込みます。</li>
	 *  <li>読み込んだメッセージをキューへ送信します。</li>
	 * </ol>
	 * 
	 * @see com.clustercontrol.syslogng.forward.FileUtil#getFileList()
	 * @see com.clustercontrol.syslogng.forward.FileUtil#readMsgFile(File)
	 * @see #_sendMsg(Object)
	 */
	synchronized private void sendMsgRetry() {
		
		log.debug("オブジェクトメッセージ再送信処理!");
		
		//ファイル一覧取得
		Collection col = m_fileUtil.getFileList();
		
		for (Iterator iter = col.iterator(); iter.hasNext();) {
			File file = (File) iter.next();
			//ファイルを読み込み
			Object msg = m_fileUtil.readMsgFile(file);
			
			//送信
			if (!_sendMsg(msg)) {
				//失敗したら処理中止
				break;
			}
			//送信したファイルを削除
			file.delete();
		}
		
	}
	
	/**
	 * サーバ接続の終了処理を行います。
	 * <p>
	 * <ol>
	 * <li>メッセージサービス再接続用タイマーを解除します。</li>
	 * <li>メッセージ送信の終了処理を行います。</li>
	 * </ol>
	 * 
	 * @see #_terminate()
	 */
	public void terminate() {
		
		m_timer.cancel();
		
		_terminate();
	}
	
	/**
	 * メッセージ送信の終了処理を行います。
	 * <p>
	 * <ol>
	 * <li>セッションをクローズします。</li>
	 * <li>コネクションをクローズします。</li>
	 * </ol>
	 */
	public void _terminate() {
		
		try {
			if (m_session != null)
				m_session.close();
			
			log.debug("Session close was successful.");
		} catch (JMSException e) {
			log.error(e.getMessage(),e);
		}
		
		try {
			if (m_con != null)
				m_con.close();
			
			log.debug("Connection close was successful.");
		} catch (JMSException e) {
			log.error(e.getMessage(),e);
		}
	}
	
	
	
	/**
	 * エラーフラグを設定します。
	 * 
	 * @param errFlg エラーの場合、<code> true </code>
	 */
	synchronized private void setErrFlg(boolean errFlg) {
		if (m_errFlg == false && errFlg == true) {
			
			m_timer.schedule(new ReSetupTask(), m_retryInterval,
					m_retryInterval);
			
		}
		m_errFlg = errFlg;
	}
	
	/**
	 * エラーフラグを返します。
	 * 
	 * @return エラーフラグ
	 */
	synchronized private boolean isErrFlg() {
		return m_errFlg;
	}
	
	/**
	 * 再接続処理を行います。
	 * <p>
	 * <ol>
	 *  <li>終了処理を行います。</li>
	 *  <li>初期処理を行います。</li>
	 *  <li>エラーフラグを解除します。</li>
	 *  <li>フィルタとリポジトリ情報を読み込みます。</li>
	 *  <li>メッセージをキューへ再送信します。</li>
	 * </ol>
	 * 
	 * @see #_terminate()
	 * @see #initial()
	 * @see #setErrFlg(boolean)
	 * @see #loadFilterInfo()
	 * @see #sendMsgRetry()
	 */
	synchronized private boolean reInitial() {
		
		boolean ret = false;
		
		log.info("再接続処理!");
		
		_terminate();
		
		if (initial()) {
			
			ret = true;
			
			log.info("再接続処理:成功!");
			
			//エラーフラグ解除
			setErrFlg(false);
			
			loadFilterInfo();
			
			sendMsgRetry();
			
		} else {
			log.info("再接続処理:失敗!");
		}
		
		return ret;
	}
	
	/**
	 * 再接続処理を行う
	 */
	public void reconnect() {
		
		synchronized (reconnectLock) {
			
			setErrFlg(true);
        
			_terminate();
		}
        
	}
	
	/**
	 * フィルタ情報取得タイマータスクです。<BR>
	 * 定周期でフィルタ情報を取得し、キャッシュを更新します。
	 */
	protected class ReflashFilterTask extends TimerTask {
		
		/**
		 * キャッシュを更新します。
		 * 
		 * @see com.clustercontrol.syslogng.forward.EJBContoroller#loadFilterInfo()
		 */
		public void run() {
			
			loadFilterInfo();
			
		}
	}
	
	/**
	 * EJB再接続タイマータスクです。
	 * <BR>
	 * 通信エラーとなった場合に定周期で呼ばれ、再接続を行います。
	 */
	protected class ReSetupTask extends TimerTask {
		
		/**
		 * 再接続を行い、成功した場合、このタスクをタイマーから削除します。
		 * 
		 * @see com.clustercontrol.syslogng.forward.EJBContoroller#reInitial()
		 */
		public void run() {
			log.info("timer task");
			if (reInitial()) {
				//このタスクをタイマーから解除
				cancel();
			}
		}
		
	}
	
	
	/** 
	 * 通信エラーハンドラです。
	 * <BR>
	 * JMSException発生時に再接続用のタイマースレッド（ReSetupTask）を起動します。
	 * 
	 */
	protected class JMSExceptionListener implements ExceptionListener {
		
		/**
		 * 接続をクローズし、再接続のタイマータスクを起動します。
		 *
		 * @see javax.jms.ExceptionListener#onException(javax.jms.JMSException)
		 */
		public void onException(JMSException e) {
			log.info("onException : try reconnect : " + e.getMessage(), e);
			
			m_logManager.reConnection();
		}
	}
	
}
