/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.
 
 */

package com.clustercontrol.performance.util;

import java.util.ArrayList;
import java.util.Date;

//FIXME クラス全体に配列が多用されており解読が困難
/**
 * ４つグラフの共通部分（設定情報）を保持するクラス
 * 
 * @version 1.0
 * @since 1.0
 */
public class RealtimeCollectorInfo {
	
	private int interval;
	private int requestInterval;
	private String facilityId;
	private int graphNum;
	
	private ArrayList subScopeFacilityId; // 制御用。
	private ArrayList subScopeFacilityName; // 見出用。制御用と使い方が異なるので別途配列を用意する。
	private int subScopeNum;
	
	private Date timestamp;
	private int timestampId;
	
	private Date collectStartTime;
	private Date collectEndTime;
	private Date graphStartTime;
	private Date graphEndTime;
	
	private int graphPlotNum;
	
	/**
	 * コンストラクター
	 *
	 */
	public RealtimeCollectorInfo() {
		super();
		initialize();
	}
	
	/**
	 * 初期化を行います。
	 *
	 */
	private void initialize() {
		// グラフの更新間隔
		interval = 0;
		// マネージャ側でのデータ収集間隔
		requestInterval = 0;
		// 選択されたfacilityId。則ち、グラフ表示の基準となるスコープの値
		// サブスコープ表示の場合は上位facilityに相当。
		facilityId = null;
		// Viewに表示するグラフ数
		graphNum = 4;  //現行仕様では4で固定。将来は増やすかも。
		
		subScopeNum = 0;
		timestamp = new Date(System.currentTimeMillis());
		timestampId = 0;
		
		collectStartTime = null; // 実績で利用.リアルタイムではnullのままとなる.timestampと共通化すべき.
		collectEndTime = null; // 実績で利用.リアルタイムではnullのままとなる.
		graphStartTime = null; // 実績で利用.リアルタイムではnullのままとなる.
		graphEndTime = null; // 実績で利用.リアルタイムではnullのままとなる.
		
		graphPlotNum = 0;	// 0はauto.
	}
	
	/**
	 * 更新間隔を設定します。
	 * @param interval 更新間隔
	 */
	public void setInterval(int interval) {
		this.interval = interval;
	}
	
	/** 
	 * 更新間隔を取得します。
	 * @return 更新間隔
	 */
	public int getInterval() {
		return this.interval;
	}
	
	/**
	 * ファシリティIDを設定します。
	 * @param facilityId ファシリティID
	 */
	public void setFacilityId(String facilityId) {
		this.facilityId = facilityId;
	}
	
	/**
	 * ファシリティIDを取得します。
	 * @return ファシリティID
	 */
	public String getFacilityId() {
		return this.facilityId;
	}
	
	/**
	 * グラフ番号を設定します。
	 * @param graphNum グラフ番号
	 */
	public void setGraphNum(int graphNum) {
		this.graphNum = graphNum;
	}
	
	/**
	 * グラフ番号を取得します。
	 * @return グラフ番号
	 */
	public int getGraphNum() {
		return this.graphNum;
	}
	
	/**
	 * サブスコープのファシリティIDを取得します。
	 * @param i グラフ番号
	 * @return
	 */
	public String getSubScopeFacilityId(int i) {
		return subScopeFacilityId.get(i).toString();
	}
	
	/**
	 * サブスコープのファシリティIDのリストを設定します。
	 * @param subScopeFacilityId サブスコープのファシリティIDのリスト
	 */
	public void setSubScopeFacilityId(ArrayList subScopeFacilityId) {
		this.subScopeFacilityId = subScopeFacilityId;
	}
	
	/**
	 * サブスコープのファシリティIDのリストを取得します。
	 * @param subScopeFacilityId サブスコープのファシリティIDのリスト
	 */
	public ArrayList getSubScopeFacilityName() {
		return subScopeFacilityName;
	}
	
	/**
	 * サブスコープのファシリティ名のリストを設定します。
	 * @param subScopeFacilityName サブスコープのファシリティ名のリスト
	 */
	public void setSubScopeFacilityName(ArrayList subScopeFacilityName) {
		this.subScopeFacilityName = subScopeFacilityName;
	}
	
	/**
	 * マネージャ側でのデータ収集間隔を取得します。
	 * @return マネージャ側でのデータ収集間隔
	 */
	public int getRequestInterval() {
		return requestInterval;
	}
	
	/**
	 * マネージャ側でのデータ収集間隔を設定します。
	 * @param requestInterval マネージャ側でのデータ収集間隔
	 */
	public void setRequestInterval(int requestInterval) {
		this.requestInterval = requestInterval;
	}
	
	/**
	 * サブスコープの数を取得します。
	 * @return サブスコープの数
	 */
	public int getSubScopeNum() {
		return subScopeNum;
	}
	
	/**
	 * サブスコープの数を設定します。
	 * @param subScopeNum サブスコープの数
	 */
	public void setSubScopeNum(int subScopeNum) {
		this.subScopeNum = subScopeNum;
	}
	
	public boolean isCurrentCollect(Date timestamp, int timestampId) {
		boolean ret = false;
		
		if (this.timestamp == timestamp && this.timestampId == timestampId) {
			ret = true;
		}
		return ret;
	}
	
	public int setCurrentCollect(Date timestamp) { //getよりsetがいい?
		if (timestamp.after(this.timestamp)) {
			this.timestamp = timestamp;
			this.timestampId = 0;
		} else if (this.timestamp == timestamp) {
			this.timestampId = this.timestampId + 1;
		} // これ以外のケースは想定外。後でハンドリングを記述する.
		return this.timestampId;
	}
	
	/**
	 * グラフの描画開始時刻を取得する。
	 * @return グラフの描画開始時刻
	 */
	public Date getGraphStartTime() {
		return graphStartTime;
	}
	
	/**
	 * グラフの描画開始時刻を設定する。
	 * @param graphStartTime グラフの描画開始時刻を
	 */
	public void setGraphStartTime(Date graphStartTime) {
		this.graphStartTime = graphStartTime;
	}
	
	/**
	 * グラフの描画終了時刻を取得する。
	 * @return グラフの描画終了時刻
	 */
	public Date getGraphEndTime() {
		return graphEndTime;
	}
	
	/**
	 * グラフの描画終了時刻を設定する。
	 * @param graphEndTime グラフの描画終了時刻
	 */
	public void setGraphEndTime(Date graphEndTime) {
		this.graphEndTime = graphEndTime;
	}
	
	/**
	 * 収集開始時刻を取得する。
	 * @return 収集開始時刻
	 */
	public Date getCollectStartTime() {
		return collectStartTime;
	}
	
	/**
	 * 収集開始時刻を設定する。
	 * @param collectStartTime 収集開始時刻
	 */
	public void setCollectStartTime(Date collectStartTime) {
		this.collectStartTime = collectStartTime;
	}
	
	/**
	 * 収集終了時刻を取得する。
	 * @return 収集終了時刻
	 */
	public Date getCollectEndTime() {
		return collectEndTime;
	}
	
	/**
	 * 収集終了時刻を設定する。
	 * @param collectEndTime 収集終了時刻
	 */
	public void setCollectEndTime(Date collectEndTime) {
		this.collectEndTime = collectEndTime;
	}
	
	/**
	 * グラフのプロット数を取得する。
	 * @return グラフのプロット数を取得する。
	 */
	public int getGraphPlotNum() {
		return graphPlotNum;
	}
	
	/**
	 * グラフのプロット数を設定する。
	 * @param graphPlotNum グラフのプロット数
	 */
	public void setGraphPlotNum(int graphPlotNum) {
		this.graphPlotNum = graphPlotNum;
	}
}