/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.jobmanagement.composite;

import java.util.ArrayList;

import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TableTreeViewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.TableTree;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Table;

import com.clustercontrol.bean.CommonTableTreeItem;
import com.clustercontrol.composite.CommonTableTreeViewer;
import com.clustercontrol.jobmanagement.action.GetJobDetail;
import com.clustercontrol.jobmanagement.action.GetJobDetailTableDefine;
import com.clustercontrol.jobmanagement.bean.JobDetailTableDefine;
import com.clustercontrol.jobmanagement.composite.action.JobDetailSelectionChangedListener;
import com.clustercontrol.jobmanagement.composite.action.SessionJobDoubleClickListener;
import com.clustercontrol.util.Messages;

/**
 * ジョブ[ジョブ詳細]ビュー用のコンポジットクラスです。
 * 
 * @version 1.0.0
 * @since 1.0.0
 */
public class DetailComposite extends Composite {
	/** テーブルビューアー */
    protected CommonTableTreeViewer m_viewer = null;
    /** セッションID */
    protected String m_sessionId = null;
    /** ジョブID */
    protected String m_jobId = null;
    /** セッションID用ラベル */
    protected Label m_sessionIdLabel = null;

    /**
     * コンストラクタ
     * 
     * @param parent 親のコンポジット
     * @param style スタイル
     * 
     * @see org.eclipse.swt.SWT
     * @see org.eclipse.swt.widgets.Composite#Composite(Composite parent, int style)
     * @see #initialize()
     */
    public DetailComposite(Composite parent, int style) {
        super(parent, style);
        initialize();
    }

    /**
     * コンポジットを配置します。
     */
	private void initialize() {
        GridLayout layout = new GridLayout(1, true);
        this.setLayout(layout);
        layout.marginHeight = 0;
        layout.marginWidth = 0;

        //セッションIDラベル作成
        m_sessionIdLabel = new Label(this, SWT.LEFT);
        GridData gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        m_sessionIdLabel.setLayoutData(gridData);

        //ジョブ詳細テーブル作成
        TableTree tree = new TableTree(this, SWT.H_SCROLL | SWT.V_SCROLL
                | SWT.FULL_SELECTION | SWT.SINGLE);

        Table table = tree.getTable();
        table.setHeaderVisible(true);
        table.setLinesVisible(true);

        gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.verticalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        gridData.grabExcessVerticalSpace = true;
        gridData.horizontalSpan = 1;
        tree.setLayoutData(gridData);

        m_viewer = new CommonTableTreeViewer(tree);
        m_viewer.createTableColumn(GetJobDetailTableDefine.get(),
                JobDetailTableDefine.SORT_COLUMN_INDEX,
                JobDetailTableDefine.SORT_ORDER);

        m_viewer.addSelectionChangedListener(
        		new JobDetailSelectionChangedListener(this));
        
        m_viewer.addDoubleClickListener(
        		new SessionJobDoubleClickListener(this));

        update(null);
    }

    /**
     * テーブルビューアーを更新します。<BR>
     * 引数で指定されたセッションIDのジョブ詳細一覧情報を取得し、
     * 共通テーブルビューアーにセットします。
     * <p>
	 * <ol>
	 * <li>引数で指定されたセッションIDのジョブ詳細一覧情報を取得します。</li>
	 * <li>共通テーブルビューアーにジョブ詳細一覧情報をセットします。</li>
	 * </ol>
     * 
     * @param sessionId セッションID
     *  
     * @see com.clustercontrol.jobmanagement.action.GetJobDetail#getJobDetail(String)
     * @see #selectDetail(CommonTableTreeItem)
     * @see #setJobId(String)
     */
    public void update(String sessionId) {
        //ジョブ詳細情報取得
        GetJobDetail getJobDetail = new GetJobDetail();
        CommonTableTreeItem item = null;
        if (sessionId != null && sessionId.length() > 0) {
            item = getJobDetail.getJobDetail(sessionId);
        }
        m_viewer.setInput(item);
        m_viewer.expandAll();

        if (m_sessionId != null && m_sessionId.length() > 0
                && sessionId != null && sessionId.length() > 0
                && m_sessionId.compareTo(sessionId) == 0) {
            selectDetail(item);
        } else {
            setJobId(null);
        }

        m_sessionId = sessionId;

        //セッションIDを表示
        if (m_sessionId != null) {
            m_sessionIdLabel.setText(Messages.getString("session.id") + " : "
                    + m_sessionId);
        } else {
            m_sessionIdLabel.setText(Messages.getString("session.id") + " : ");
        }
    }

    /**
     * ジョブ詳細の行を選択します。<BR>
     * 前回選択したジョブIDと同じジョブIDの行を選択します。
     * 
     * @param item テーブルツリーアイテム
     */
    public void selectDetail(CommonTableTreeItem item) {
        if (getJobId() != null && getJobId().length() > 0) {
            if (m_viewer.getSelection().isEmpty()) {
                boolean select = false;
                ArrayList list = item.getData();
                if (list instanceof ArrayList) {
                    String jobId = (String) list
                            .get(JobDetailTableDefine.JOB_ID);
                    if (getJobId().compareTo(jobId) == 0) {
                        select = true;
                    }
                }

                if (select) {
                    m_viewer.setSelection(new StructuredSelection(item), true);
                } else {
                    for (int i = 0; i < item.getChildren().length; i++) {
                        CommonTableTreeItem children = item.getChildren()[i];
                        selectDetail(children);
                    }
                }
            }
        }
    }

    /**
     * このコンポジットが利用するテーブルツリービューアを返します。
     * 
     * @return テーブルツリービューア
     */
    public TableTreeViewer getTableTreeViewer() {
        return m_viewer;
    }

    /**
     * このコンポジットが利用するテーブルを返します。
     * 
     * @return テーブル
     */
	public Table getTable() {
        return m_viewer.getTableTree().getTable();
    }

    /**
     * このコンポジットが利用するテーブルツリーを返します。
     * 
     * @return テーブルツリー
     */
	public TableTree getTableTree() {
        return m_viewer.getTableTree();
    }

    /**
     * セッションIDを返します。
     * 
     * @return セッションID
     */
    public String getSessionId() {
        return m_sessionId;
    }

    /**
     * セッションIDを設定します。
     * 
     * @param sessionId セッションID
     */
    public void setSessionId(String sessionId) {
        m_sessionId = sessionId;
    }

    /**
     * ジョブIDを返します。
     * 
     * @return ジョブID
     */
    public String getJobId() {
        return m_jobId;
    }

    /**
     * ジョブIDを設定します。
     * 
     * @param jobId ジョブID
     */
    public void setJobId(String jobId) {
        m_jobId = jobId;
    }
}