/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.syslogng.dialog;

import java.util.ArrayList;

import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableItem;

import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.bean.YesNoConstant;
import com.clustercontrol.dialog.CommonDialog;
import com.clustercontrol.dialog.ValidateResult;
import com.clustercontrol.syslogng.action.LogManager;
import com.clustercontrol.syslogng.bean.LogFilterInfo;
import com.clustercontrol.syslogng.composite.LogListComposite;
import com.clustercontrol.util.Messages;

/**
 * syslog-ng[一覧]ダイアログクラス<BR>
 * 
 * @version 2.2.0
 * @since 1.0.0
 */
public class LogListDialog extends CommonDialog {
	
	/** フィルタ情報一覧 コンポジット。 */
	private LogListComposite logList = null;
	
	/** 追加 ボタン。 */
	private Button buttonAdd = null;
	
	/** 変更 ボタン。 */
	private Button buttonModify = null;
	
	/** 削除 ボタン。 */
	private Button buttonDelete = null;
	
	/** 上へ ボタン。 */
	private Button buttonUp = null;
	
	/** 下へ ボタン。 */
	private Button buttonDown = null;
	
	/** 有効 ボタン。 */
	private Button buttonValid = null;
	
	/** 無効 ボタン。 */
	private Button buttonInvalid = null;
	
	
	/**
	 * ダイアログのインスタンスを返します。
	 * 
	 * @param parent 親のシェルオブジェクト
	 */
	public LogListDialog(Shell parent) {
		super(parent);
		setShellStyle(getShellStyle() | SWT.RESIZE | SWT.MAX);
	}
	
	/**
	 * ダイアログの初期サイズを返します。
	 * 
	 * @return 初期サイズ
	 */
	protected Point getInitialSize() {
		return new Point(800, 420);
	}
	
	/**
	 * ダイアログエリアを生成します。
	 * 
	 * @param parent 親のコンポジット
	 * 
	 * @see com.clustercontrol.syslogng.composite.LogListComposite
	 * @see com.clustercontrol.syslogng.action.LogManager#add(LogFilterInfo)
	 * @see com.clustercontrol.syslogng.action.LogManager#modify(LogFilterInfo)
	 * @see com.clustercontrol.syslogng.action.LogManager#delete(String)
	 * @see com.clustercontrol.syslogng.action.LogManager#upOrder(String)
	 * @see com.clustercontrol.syslogng.action.LogManager#downOrder(String)
	 * @see #setValid(int)
	 */
	protected void customizeDialog(Composite parent) {
		Shell shell = this.getShell();
		
		// タイトル
		shell.setText(Messages.getString("dialog.syslog-ng.list"));
		
		// レイアウト
		GridLayout layout = new GridLayout(8, true);
		layout.marginWidth = 10;
		layout.marginHeight = 10;
		layout.numColumns = 8;
		parent.setLayout(layout);
		
		/*
		 * ログ一覧
		 */
		
		this.logList = new LogListComposite(parent, SWT.BORDER);
		GridData gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.verticalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.grabExcessVerticalSpace = true;
		gridData.horizontalSpan = 7;
		this.logList.setLayoutData(gridData);
		
		/*
		 * 操作ボタン
		 */
		
		Composite composite = new Composite(parent, SWT.NONE);
		layout = new GridLayout(1, true);
		layout.numColumns = 1;
		composite.setLayout(layout);
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.verticalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.grabExcessVerticalSpace = true;
		gridData.horizontalSpan = 1;
		composite.setLayoutData(gridData);
		
		// 追加ボタン
		this.buttonAdd = this
		.createButton(composite, Messages.getString("add"));
		this.buttonAdd.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				LogCreateDialog dialog = new LogCreateDialog(getParentShell());
				if (dialog.open() == IDialogConstants.OK_ID) {
					logList.update();
				}
			}
		});
		
		// 変更ボタン
		@SuppressWarnings("unused") Label dummy = new Label(composite, SWT.NONE);
		this.buttonModify = this.createButton(composite, Messages
				.getString("modify"));
		this.buttonModify.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				String logId = getSelectLogId();
				if (logId != null) {
					LogCreateDialog dialog = new LogCreateDialog(
							getParentShell(), logId);
					if (dialog.open() == IDialogConstants.OK_ID) {
						Table table = logList.getTableViewer().getTable();
						int selectIndex = table.getSelectionIndex();
						logList.update();
						table.setSelection(selectIndex);
					}
				}
				else{
                	MessageDialog.openWarning(
                			null, 
                			Messages.getString("warning"), 
                			Messages.getString("message.monitor.1"));
				}
			}
		});
		
		// 削除ボタン
		dummy = new Label(composite, SWT.NONE);
		this.buttonDelete = this.createButton(composite, Messages
				.getString("delete"));
		this.buttonDelete.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				LogFilterInfo filterInfo = logList.getFilterItem();
				
				if (filterInfo != null) {
                    String monitorId = filterInfo.getMonitorId();

                    String[] args = { monitorId };
                    if (MessageDialog.openConfirm(
                			null, 
                			Messages.getString("confirmed"), 
                			Messages.getString("message.monitor.39", args))) {
                    	
                        LogManager.getInstance().delete(getSelectLogId());
                        logList.update();
                    }
				}
				else{
                	MessageDialog.openWarning(
                			null, 
                			Messages.getString("warning"), 
                			Messages.getString("message.monitor.1"));
				}
			}
		});
		
		// 上へボタン
		dummy = new Label(composite, SWT.NONE);
		dummy = new Label(composite, SWT.NONE);
		this.buttonUp = this.createButton(composite, Messages.getString("up"));
		this.buttonUp.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				String logId = getSelectLogId();
				if (logId != null) {
					if (LogManager.getInstance().upOrder(logId)) {
						logList.update();
						selectItem(logId);
					}
				}
				else{
                	MessageDialog.openWarning(
                			null, 
                			Messages.getString("warning"), 
                			Messages.getString("message.monitor.1"));
				}
			}
		});
		
		// 下へボタン
		this.buttonDown = this.createButton(composite, Messages
				.getString("down"));
		this.buttonDown.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				String logId = getSelectLogId();
				if (logId != null) {
					if (LogManager.getInstance().downOrder(getSelectLogId())) {
						logList.update();
						selectItem(logId);
					}
				}
				else{
                	MessageDialog.openWarning(
                			null, 
                			Messages.getString("warning"), 
                			Messages.getString("message.monitor.1"));
				}
			}
		});
		
		// 有効ボタン
		dummy = new Label(composite, SWT.NONE);
		this.buttonValid = this.createButton(composite, Messages
				.getString("valid"));
		this.buttonValid.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				//一括で有効に変更
				setValid(ValidConstant.TYPE_VALID);
			}
		});
		
		// 無効ボタン
		this.buttonInvalid = this.createButton(composite, Messages
				.getString("invalid"));
		this.buttonInvalid.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				//一括で無効に変更
				setValid(ValidConstant.TYPE_INVALID);
			}
		});
		
		// ラインを引く
		Label line = new Label(parent, SWT.SEPARATOR | SWT.HORIZONTAL);
		gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.horizontalSpan = 8;
		line.setLayoutData(gridData);
		
		// 画面中央に
		Display display = shell.getDisplay();
		shell.setLocation((display.getBounds().width - shell.getSize().x) / 2,
				(display.getBounds().height - shell.getSize().y) / 2);
	}
	
	/**
	 * 入力値チェックをします。
	 * 
	 * @return 検証結果
	 */
	protected ValidateResult validate() {
		return super.validate();
	}
	
	/**
	 * ＯＫボタンのテキストを返します。
	 * 
	 * @return ＯＫボタンのテキスト
	 */
	protected String getOkButtonText() {
		return Messages.getString("register");
	}
	
	/**
	 * キャンセルボタンのテキストを返します。
	 * 
	 * @return キャンセルボタンのテキスト
	 */
	protected String getCancelButtonText() {
		return Messages.getString("cancel");
	}
	
	/**
	 * ボタンを返します。
	 * 
	 * @param parent 親のコンポジット
	 * @param label ラベル文字列
	 * @return 生成されたボタン
	 */
	private Button createButton(Composite parent, String label) {
		Button button = new Button(parent, SWT.NONE);
		
		GridData gridData = new GridData();
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		button.setLayoutData(gridData);
		
		button.setText(label);
		
		return button;
	}
	
	/**
	 * 選択されているフィルタ情報のフィルタIDを返します。
	 * 
	 * @return フィルタID。選択されていない場合は、<code> null </code>。
	 */
	private String getSelectLogId() {
		StructuredSelection selection = (StructuredSelection) this.logList
		.getTableViewer().getSelection();
		
		if (selection.getFirstElement() instanceof LogFilterInfo) {
			LogFilterInfo filterInfo = (LogFilterInfo) selection
			.getFirstElement();
			return filterInfo.getLogId();
		} else {
			return null;
		}
	}
	
	/**
	 * 引数で指定されたフィルタ情報の行を選択状態にします。
	 * 
	 * @param logId フィルタID
	 */
	private void selectItem(String logId) {
		Table table = this.logList.getTableViewer().getTable();
		TableItem[] items = table.getItems();
		
		if (items == null || logId == null) {
			return;
		}
		
		for (int i = 0; i < items.length; i++) {
			
			if (items[i].getData() instanceof LogFilterInfo) {
				LogFilterInfo filterInfo = (LogFilterInfo) items[i].getData();
				if (logId.equals(filterInfo.getLogId())) {
					table.select(i);
					return;
				}
			}
		}
	}
	
	/**
	 * 選択されているフィルタ情報の有効／無効を変更します。
	 * <p>
	 * <ol>
	 *  <li>現在選択されているフィルタ情報一覧を取得します。</li>
	 *  <li>取得したフィルタ情報の有効／無効を変更します。</li>
	 *  <li>フィルタ情報一覧を更新します。</li>
	 * </ol>
	 * 
	 * @param valid 有効／無効
	 * 
	 * @see com.clustercontrol.bean.ValidConstant
	 * @see com.clustercontrol.syslogng.composite.LogListComposite#getSelectionData()
	 * @see com.clustercontrol.syslogng.action.LogManager#modify(LogFilterInfo)
	 * @see com.clustercontrol.syslogng.composite.LogListComposite#update()
	 */
	public void setValid(int valid) {
		
		//選択された監視項目IDを取得
		ArrayList<LogFilterInfo> list = logList.getSelectionData();
		if (list != null && list.size() > 0) {
			
			StringBuilder logIds = new StringBuilder();
			for(int i = 0; i < list.size(); i++){
				if(logIds.length() > 0){
					logIds.append(", ");
				}
				LogFilterInfo info = list.get(i);
				if(info instanceof LogFilterInfo){
					logIds.append(info.getMonitorId());
				}
			}

            String[] confirmArgs = { logIds.toString() };
            String message;
            if(valid == YesNoConstant.TYPE_YES)
            	message = Messages.getString("message.monitor.47",confirmArgs);
            else
            	message = Messages.getString("message.monitor.48",confirmArgs);
            if (!MessageDialog.openConfirm(
        			null, 
        			Messages.getString("confirmed"), 
        			message)) {
            	return;
            }
			
			for(int i = 0; i < list.size(); i++){
				String logId = null;
				if(list.get(i) instanceof LogFilterInfo){
					logId = ((LogFilterInfo)list.get(i)).getLogId();
				}
				
				if(logId != null && !logId.equals("")){
					// ログIDの情報を取得
					LogFilterInfo info = LogManager.getInstance().get(logId);
					
					//有効・無効を設定
					info.setValid(valid);
					
					//監視情報を更新
					boolean result = LogManager.getInstance().modify(info);
					if(!result){
                        String[] args = { logId };
                    	MessageDialog.openError(
                    			null, 
                    			Messages.getString("failed"), 
                    			Messages.getString("message.monitor.36", args));
					}
				}
			}
			
			int selectIndex = logList.getTableViewer().getTable().getSelectionIndex();
			logList.update();
			logList.getTableViewer().getTable().setSelection(selectIndex);
		}
		else{
        	MessageDialog.openWarning(
        			null, 
        			Messages.getString("warning"), 
        			Messages.getString("message.monitor.1"));
        }
    }
}