/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.
 
 */

package com.clustercontrol.performance.util.code;

import java.io.Serializable;
import java.util.ArrayList;

import com.clustercontrol.bean.YesNoConstant;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorCategoryMstData;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorDeviceInfoData;
import com.clustercontrol.performance.monitor.ejb.entity.CollectorItemCodeMstData;

/**
 * 収集項目の親子関係を保持するユーティリティクラス
 * 
 * @version 1.0
 * @since 1.0
 */
public class CollectorItemTreeItem  implements Serializable {
	private static final long serialVersionUID = -8592461412968942952L;

	/** カテゴリ */
    private CollectorCategoryMstData category = null;

    /** 親 */
    private CollectorItemTreeItem parent = null;

    /** 情報オブジェクト */
    private Object data = null;

    /** 子の格納リスト */
    private ArrayList<CollectorItemTreeItem> children = null;
    
    /** 収集方法 */
    private String collectMethod;
    
    /** デバイス情報 */
    private CollectorDeviceInfoData deviceData = null;

    /**
     * コンストラクター
     * @param parent 親の収集項目要素
     * @param data 収集項目要素
     * @param deviceData デバイス情報
     */
    public CollectorItemTreeItem(
    		CollectorItemTreeItem parent, 
    		Object data,
    		String collectMethod,
    		CollectorDeviceInfoData deviceData) {
    	// 収集方法を設定
    	this.collectMethod = collectMethod;
    	
    	// デバイス情報を生成
    	if(deviceData != null){
    		this.deviceData = deviceData;
    	} else{
    		this.deviceData = new CollectorDeviceInfoData(
    				null, //collectorId,
    				null, //facilityId,
    				"", //deviceNameは空文字とする
    				"", //deviceDisplayNameはから文字とする
    				new Long(-1), // index,
    				null, //deviceId
    				null //deviceType
    				);
    	}
       		
    	// 要素が収集項目カテゴリの場合は、自分自身のカテゴリを設定する。
    	if (data instanceof CollectorCategoryMstData){
            this.setCategory((CollectorCategoryMstData)data);
        } else
        // 要素が収集項目コードの場合は、親のカテゴリと同じカテゴリとする。
        if (data instanceof CollectorItemCodeMstData) {
            this.setCategory(parent.getCategory());
//            this.category.addIte((CollectorItemCodeMstData)data);
        }

        this.setParent(parent);  // 親を設定
        this.data = data;  // 情報オブジェクトを設定

        if (parent != null) {
            parent.addChildren(this);
        }

        this.children = new ArrayList<CollectorItemTreeItem>();
    }
    
    /**
     * 親を設定します。
     * <p>
     * 
     * @return 親
     */
    private void setParent(CollectorItemTreeItem parent) {
        this.parent = parent;
    }

    /**
     * 子を追加します。
     * <p>
     * 
     * この際、childeの親はこのオブジェクトとして設定されます。
     * 
     * @param child
     *            子
     */
    private void addChildren(CollectorItemTreeItem child) {
        child.setParent(this);
        children.add(child);
    }

    /**
     * 収集項目コードを取得します。
     * 
     * @return itemCode 収集項目コード
     */
    public String getItemCode() {
    	if(data instanceof CollectorItemCodeMstData){
            return ((CollectorItemCodeMstData)data).getItemCode();
    	} else {
    		return null;
    	}
    }

    /**
     * 収集項目名を取得します。
     * 
     * @return 収集項目名
     */
    public String getItemName() {
    	if(data instanceof CollectorItemCodeMstData){
            return ((CollectorItemCodeMstData)data).getItemName();
    	} else {
    		return null;
    	}
    }

    /**
     * 詳細情報表示可能か否か（子項目があるか否か）を取得します。
     */
    public boolean hasBreakdown() {
        return (children.size() != 0);
    }

    /**
     * deviceFlag を取得します。
     * 
     * @return deviceFlag
     */
    public boolean isDeviceFlag() {
    	if(data instanceof CollectorItemCodeMstData){
            return ((CollectorItemCodeMstData)data).getDeviceSupport() == YesNoConstant.TYPE_YES;
    	} else {
    		return false;
    	}
    }

    /**
     * デバイス種別を取得します。
     * 
     * @return deviceType
     */
    public String getDeviceType() {
    	if(data instanceof CollectorItemCodeMstData){
            return ((CollectorItemCodeMstData)data).getDeviceType();
    	} else {
    		return null;
    	}
    }
    
    /**
     * isRangeFixed を取得します。
     * 
     * @return isRangeFixed
     */
    public boolean isRangeFixed() {
    	if(data instanceof CollectorItemCodeMstData){
            return ((CollectorItemCodeMstData)data).getGraphRange() == YesNoConstant.TYPE_YES;
    	} else {
    		return false;
    	}
    }

    /**
     * measure を取得します。
     * 
     * @return measure
     */
    public String getMeasure() {
    	if(data instanceof CollectorItemCodeMstData){
            return ((CollectorItemCodeMstData)data).getMeasure();
    	} else{
    		return null;
    	}
    }

    /**
     * category を取得します。
     * 
     * @return category
     */
    public CollectorCategoryMstData getCategory() {
        return category;
    }

    /**
     * category を設定します。
     * 
     * @param category
     */
    private void setCategory(CollectorCategoryMstData category) {
        this.category = category;
    }

    /**
     * children を取得します。
     * 
     * @return children
     */
    public ArrayList<CollectorItemTreeItem> getChildren() {
        return children;
    }

    /**
     * children を設定します。
     * 
     * @param children
     */
//    private void setChildren(ArrayList children) {
//        this.children = children;
//    }

    /**
     * コード を取得します。
     * 
     * @return data
     */
    public String getCode() {
    	if (data instanceof CollectorCategoryMstData){
            return ((CollectorCategoryMstData)data).getCategoryCode();
        } else if (data instanceof CollectorItemCodeMstData) {
            return ((CollectorItemCodeMstData)data).getItemCode();
        }
    	return null;
    }

    /**
     * 項目名 を取得します。
     * 
     * @return data
     */
    public String getName() {
    	if (data instanceof CollectorCategoryMstData){
            return ((CollectorCategoryMstData)data).getCategoryName();
        } else if (data instanceof CollectorItemCodeMstData) {
            return ((CollectorItemCodeMstData)data).getItemName();
        }
    	return null;
    }
    
    /**
     * 項目名（デバイス名込）を取得します。
     * 
     * @return data
     */
    public String getFullName() {
    	if (data instanceof CollectorCategoryMstData){
    		return ((CollectorCategoryMstData)data).getCategoryName();
    	} else if (data instanceof CollectorItemCodeMstData) {
    		String itemName = ((CollectorItemCodeMstData)data).getItemName();
    		if(deviceData != null && 
    				deviceData.getDeviceType() != null &&
    				!"".equals(deviceData.getDeviceType())){
    			itemName = itemName + "[" + deviceData.getDisplayName() + "]";
    		}
    		return itemName;
        }
    	return null;
    }

    /**
     * parent を取得します。
     * 
     * @return parent
     */
    public CollectorItemTreeItem getParent() {
        return parent;
    }
    
    /**
     * 子の要素の数を取得します。
     * @return 子の要素の数
     */
	public int size() {
		return children.size();
	}
	
	/**
	 * デバイス情報を取得します。
	 * @return デバイス情報
	 */
	public CollectorDeviceInfoData getDeviceData(){
		return deviceData;
	}

	/**
	 * 収集方法を取得します。
	 * @return 収集方法
	 */
	public String getCollectMethod() {
		return collectMethod;
	}
	
    /**
     * 項目名（デバイス名込）を取得します。
     * 
     * 
     * @return data
     */
    public String getCollectorItemInfoData() {
    	if (data instanceof CollectorCategoryMstData){
            return ((CollectorCategoryMstData)data).getCategoryName();
        } else if (data instanceof CollectorItemCodeMstData) {
        	String itemName = ((CollectorItemCodeMstData)data).getItemName();
			if(deviceData != null){
				itemName = itemName + "[" + deviceData.getDisplayName() + "]";
			}
            return itemName;
        }
    	return null;
    }
    
    public boolean isCategoryItem(){
    	if (data instanceof CollectorCategoryMstData){
    		return true;
    	} else {
    		return false;
    	}
    }
}