/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License 
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied 
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 PURPOSE.  See the GNU General Public License for more details.
 
 */

package com.clustercontrol.commons.util;

import java.util.Properties;

import javax.jms.JMSException;
import javax.jms.Session;
import javax.jms.Topic;
import javax.jms.TopicConnection;
import javax.jms.TopicConnectionFactory;
import javax.jms.TopicSession;
import javax.naming.InitialContext;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.commons.bean.ConnectionFactoryConstant;

/**
 * トピックへメッセージを送信するクラスです。
 *
 * @version 2.3.0
 * @since 2.3.0
 */
public abstract class CommonSendTopic {
	/** ログ出力のインスタンス */
    protected static Log m_log = LogFactory.getLog( CommonSendTopic.class );
    
    /** トピック名 */
    protected String m_topicName;
    /** コネクションファクトリー */
    protected TopicConnectionFactory m_connectionFactory;
    /** コネクション */
    protected TopicConnection m_connection;
    /** トピック */
    protected Topic m_topic;
    /** セッション */
    protected TopicSession m_session;
    
    /**
     * コンストラクタ<BR>
     * トピック名を設定します。
     * 初期化処理を行います。
     * 
     * @param topicName
     * @throws NamingException
     * @throws JMSException
     */
    public CommonSendTopic(String topicName) throws NamingException, JMSException{
        m_topicName = topicName;
        
        //初期化
        connect();
    }
    
    /**
     * メッセージ送信の初期処理を行います。
	 * <p>
	 * <ol>
	 * <li>コネクションファクトリと宛先を取得します。</li>
	 * <li>コネクションファクトリより、JMSプロバイダとのコネクションを生成します。</li>
	 * <li>コネクションよりセッションを生成します。</li>
	 * </ol>
     * 
     * @throws NamingException
     * @throws JMSException
     */
    protected void connect() throws NamingException, JMSException {
		InitialContext context = null;
		try{
			String serverName = System.getProperty("jboss.server.name");
			if(serverName.equals("all")){
				Properties properties = new Properties(); 
				properties.put(InitialContext.INITIAL_CONTEXT_FACTORY,"org.jnp.interfaces.NamingContextFactory");
				properties.put(InitialContext.URL_PKG_PREFIXES,"org.jboss.naming:org.jnp.interfaces");
				properties.put(InitialContext.PROVIDER_URL,"jnp://localhost:1100");
				
				context = new InitialContext(properties);
			}
			else{
				context = new InitialContext();
			}
//			m_connectionFactory = (TopicConnectionFactory)context.lookup("ConnectionFactory");
			m_connectionFactory = (TopicConnectionFactory)context.lookup(ConnectionFactoryConstant.JNDI_JVMIL);
			m_connection = m_connectionFactory.createTopicConnection();
			m_session = m_connection.createTopicSession( false, Session.AUTO_ACKNOWLEDGE );
			m_connection.start();
			m_topic = (Topic)context.lookup(m_topicName);
	    }
		catch(NamingException e){
			terminate();
			throw e;
		}
		catch(JMSException e){
			terminate();
			throw e;
		}
		finally{
			try {
				if (context != null)
					context.close();
			}
			catch (NamingException e) {
				m_log.error("connect(): can not closed  TopicName:" + m_topicName + e.getMessage(), e);
			}
		}
		m_log.debug("connect() : end  TopicName:" + m_topicName);
	}
	
    /**
     * メッセージ送信の終了処理を行います。
	 * <p>
	 * <ol>
	 * <li>セッションをクローズします。</li>
	 * <li>コネクションをクローズします。</li>
	 * </ol>
	 * 
     * @throws JMSException
     */
	public void terminate() {
		m_log.debug("terminate() : start  TopicName:" + m_topicName);
		try {
			if (m_connection != null){
				m_connection.stop();
			}
		}
		catch (JMSException e){
			m_log.error("terminate():  TopicName:" + m_topicName + e.getMessage(), e);
		}
		
		try {
			if (m_session != null){
				m_session.close();
				m_session = null;
			}
		}
		catch (JMSException e) {
			m_log.error("terminate():  TopicName:" + m_topicName + e.getMessage(), e);
		}
		
		try {
			if (m_connection != null){
				m_connection.close();
				m_connection = null;
			}
		}
		catch (JMSException e){
			m_log.error("terminate():  TopicName:" + m_topicName + e.getMessage(), e);
		}
		
		m_log.debug("terminate(): end  TopicName:" + m_topicName);
	}
}