/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.agent;

import java.util.Properties;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;

import javax.jms.ExceptionListener;
import javax.jms.JMSException;
import javax.jms.ObjectMessage;
import javax.jms.Queue;
import javax.jms.QueueConnection;
import javax.jms.QueueConnectionFactory;
import javax.jms.QueueSender;
import javax.jms.QueueSession;
import javax.jms.Session;
import javax.naming.InitialContext;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.agent.util.InternalLogger;
import com.clustercontrol.jobmanagement.bean.QueueConstant;
import com.clustercontrol.jobmanagement.message.RunResultInfo;

/**
 * ジョブ実行結果（チェック、開始含む）をQueue送信するクラス<BR>
 *
 * エージェントからの戻りメッセージはこのメソッドを用いて
 * マネージャに返送されます。
 *
 * @version $Revision: 3506 $
 * @since 1.0.0
 */
public class SendQueue implements ExceptionListener {

	private static final String QUEUE_CON_FACTORY = "ConnectionFactory";

	private static final String QUEUE_USER_NAME = "queue.user.name";

	private static final String QUEUE_USER_PASSWORD = "queue.user.password";
	
	private static final long SEND_TIMEOUT = 60 * 1000l;
	private static final long RECONNECT_TIMEOUT = 60 * 1000l;

	private Agent m_agent;
	
	private Properties m_props;

	private QueueConnection m_con;

	private Queue m_queue;

	private QueueSession m_session = null;

	private QueueConnectionFactory m_factory;

	private volatile boolean m_isErr = false;
	
	private long m_interval = 10000;

	//ロガー
	static private Log log = LogFactory.getLog(SendQueue.class);
	
	// reInitialメソッドが複数スレッドから呼ばれないよにするロック用オブジェクト
	private final Object reInitialLock = new Object();

	// Queue送信処理が複数スレッドから呼ばれないよにするロック用オブジェクト
	private final Object sendQueueLock = new Object();  
	
	// Queueコネクションに対する処理が複数スレッドから呼ばれないようにするためのロック用オブジェクト  
	// 再接続処理でコネクションを生成し直すと別オブジェクトとなる
	private volatile Object connectionLock = new Object();
	
	// 内部エラー出力抑制フラグ
	// エージェント → マネージャ間通信に失敗した場合にエラー出力を行うが、
	// 継続して通信できていない状態の場合は出力を制御する
	volatile private boolean internalErroInhibitflag = false;

	/**
	 * コンストラクタ
	 * @param props プロパティファイル情報
	 */
	public SendQueue(Agent agent, Properties props) {
		super();
		m_agent = agent;
		m_props = props;
		
		// SendQueue再接続処理実行間隔取得
		String interval = m_props.getProperty("sendqueue.reconnection.interval");
		if (interval != null) {
			try {
				// プロパティファイルには秒で記述されているため、1000をかける
				m_interval = Integer.parseInt(interval) * 1000;
			} catch (NumberFormatException e) {
				log.error("sendqueue.reconnection.interval",e);
			}
		}
		
		initial();
	}

	/**
	 *  再接続の処理を行います。<BR>
	 */
	private boolean reInitial() {
		m_isErr = true;
		
		// 単一スレッドのみ以下の処理が実行できるように、クラスインスタンスのロックオブジェクトで、
		// 同期化しシリアルに動作するようにする。
		synchronized(reInitialLock){
			if(m_isErr == false){
				// 別スレッドで接続が復旧しているため、何もせずに処理を抜ける
				return true;
			}
			
			log.debug("reInitial() start");

			ExecutorService es = null;
			Future<Boolean> task = null;
			try {
				// Executorオブジェクトの生成
				ReconnectorThreadFactory threadFactory = 
					new ReconnectorThreadFactory(Thread.currentThread().getName());
				es = Executors.newSingleThreadExecutor(threadFactory);
				task = es.submit(new Reconnector());

				return task.get(RECONNECT_TIMEOUT, TimeUnit.MILLISECONDS);
			} catch (Exception e) {
				log.error(e.getMessage(), e);
				m_isErr = true;
				return false;
			} finally {
				// タスクをキャンセル
				task.cancel(true);
				
				if (es != null) {
					es.shutdown();
				}
				log.debug("reInitial() end");
			}
		}
	}
	
	/**
	 * マネージャへの接続の終了処理を行います。<BR>
	 */
	public void terminate() {
		
		try {
			if (m_session != null)
				m_session.close();
		} catch (JMSException e) {
			log.error(e.getMessage(), e);
		}

		try {
			if (m_con != null)
				m_con.close();
		} catch (JMSException e) {
			log.error(e.getMessage(), e);
		}
	}

	/**
	 * メッセージを送信します。<BR>
	 * 
	 * マネージャからの実行に対する応答メッセージを送信します。<BR>
	 * 処理失敗は1回再試行します。
	 * @param msg
	 */
	public boolean put(RunResultInfo info) {
		// 単一スレッドのみ以下の処理が実行できるように、クラスインスタンスのロックオブジェクト（sendQueueLock）で、
		// 同期化しシリアルに動作するようにする。
		synchronized(sendQueueLock){
			log.debug("put() start  : "  + info.getCommandType() + ", " + info.getSessionId()+ ", " + info.getJobId());

			while (!ReceiveTopic.isHistoryClear()) {
				ExecutorService es = null;
				Future<Boolean> task = null;
				try {
					// Executorオブジェクトの生成
					SenderThreadFactory threadFactory = new SenderThreadFactory(info.getSessionId());
					es = Executors.newSingleThreadExecutor(threadFactory);

					// Queue送信タスクを実行する。
					// 別スレッドでQueue送信処理を実行することで、送信処理に無限の時間がかかっても、
					// Future.get()のタイムアウトにより、本スレッドに制御が戻るようにする。
					log.debug("put() submit : " + info.getCommandType() + ", "
							+ info.getSessionId() + ", " + info.getJobId());
					task = es.submit(new Sender(info));
					boolean sendQueueStatus = task.get(SEND_TIMEOUT, TimeUnit.MILLISECONDS);

					// Queue送信に成功した場合はループを抜ける
					if(sendQueueStatus == true){
						return true;
					}
				} catch (Exception e) {
					// Queue送信処理で例外が発生した場合、もしくは、Future.get()でタイムアウトが発生した場合に、
					// ここに制御が移る
					if(internalErroInhibitflag == true){
						// 抑制されている場合は簡略なメッセージとする
						log.error("Failed to connect to MGR");
					} else {
						log.error("Failed to connect to MGR", e);
					}
					
					// 抑制されていない場合に出力する
					if(internalErroInhibitflag == false){
						// syslogにログ出力
						InternalLogger.error("hinemos_jobagent: Failed to connect to MGR");
						// エラーログ出力抑制フラグを設定する
						internalErroInhibitflag = true;
					}
					
					// 一定時間sleepした後、QueueConnection、QueueSession を再接続する。
					try {
						Thread.sleep(m_interval);
					} catch (InterruptedException e1) {
						// 何もしない
						log.error(e1.getMessage(), e1);
					}

					m_isErr = true;
					reInitial();

					log.info("オブジェクトメッセージ送信リトライ");
				} finally {
					// タスクをキャンセル
					task.cancel(true);
					
					if (es != null) {
						es.shutdown();
					}
					
					log.debug("put() end    : " + info.getCommandType() + ", "
							+ info.getSessionId() + ", " + info.getJobId());
				}
			}

			return false;
		}
	}

	/**
	 * マネージャとの接続を初期化します。<BR>
	 * @return　true:成功　false:失敗
	 */
	private boolean initial() {

		log.info("EJB接続初期化");

		InitialContext con = null;

		try {
			//InitialContextの生成
			con = new InitialContext(m_props);

			//コネクションファクトリ生成
			m_factory = (QueueConnectionFactory) con.lookup(QUEUE_CON_FACTORY);

			//コネクション生成
			if (m_props.getProperty(QUEUE_USER_NAME) != null) {
				//ユーザ認証
				m_con = m_factory.createQueueConnection(m_props
						.getProperty(QUEUE_USER_NAME), m_props
						.getProperty(QUEUE_USER_PASSWORD));
			} else {
				//ユーザ認証なし
				m_con = m_factory.createQueueConnection();
			}


			//セッション生成
			m_session = m_con.createQueueSession(false, Session.AUTO_ACKNOWLEDGE);

			//エラーハンドラセット
			m_con.setExceptionListener(this);

			//コネクション開始
			m_con.start();

			//メッセージQueue取得
			m_queue = (Queue) con.lookup(QueueConstant.QUEUE_NAME_STATUS);

			// 新ロックオブジェクトを生成
			connectionLock = new Object();
			m_isErr = false;
			
			// エラー出力抑制フラグをクリア
			internalErroInhibitflag = false;
		} catch (Exception e) {
			log.error("Init", e);
			m_isErr = true;
			return false;
		} finally {
			try {
				if (con != null)
					con.close();
			} catch (Exception e) {
				log.error(e.getMessage(), e);
			}
		}
		return true;

	}


	/* 通信エラーハンドラ
	 * (non-Javadoc)
	 * @see javax.jms.ExceptionListener#onException(javax.jms.JMSException)
	 */
	public void onException(JMSException arg0) {

		log.error(arg0.getMessage(), arg0);

		m_agent.reConnection();

	}

	/**
	 * 再接続処理を行うようにフラグを設定する
	 */
	public void reconnect() {
		m_isErr = true;
	}
	
	/**
	 * Queueメッセージ送信処理を実行するタスク用クラス
	 */
	private class Sender implements Callable<Boolean> {
		private RunResultInfo m_info;

		public Sender(RunResultInfo info){
			m_info = info;
		}

		public Boolean call() throws Exception {
			if (m_isErr) {
				if(reInitial() == false){
					// 再接続に失敗した場合
					return false;
				}
			}

			if(m_session == null){
				return false;
			}

			QueueSender sender = null;
			try {
				// 複数スレッドがQueueConnection, QueueSessionのインスタンスにアクセスしないように、
				// 送信処理を同期化する。
				synchronized(connectionLock){
					//Sender作成
					sender = m_session.createSender(m_queue);
					//メッセージ作成
					ObjectMessage mess = m_session.createObjectMessage(m_info);
					//送信
					log.info("オブジェクトメッセージ送信 : SessionID=" + m_info.getSessionId() + ", JobID=" + m_info.getJobId() + ", CommandType=" + m_info.getCommandType());
					sender.send(mess);
				}
				
				return true;
			} catch (Exception e) {
				throw e;
			} finally {
				try {
					if (sender != null)
						sender.close();
				} catch (Exception e) {
					log.error(e.getMessage(), e);
				}
			}
		}
	}

	/**
	 * Queueメッセージ送信処理を実行するタスク用のThreadFactory
	 */
	private class SenderThreadFactory implements ThreadFactory {
		private final String m_threadName;
		
		public SenderThreadFactory(String threadName){
			m_threadName = threadName;
		}
		
		public Thread newThread(Runnable r) {
			return new Thread(r, "Sender-" + m_threadName);
		}
	}
	
	/**
	 * 再接続処理を実行するタスク用クラス
	 */
	private class Reconnector implements Callable<Boolean> {
		public Boolean call() throws Exception {
			boolean ret = false;

			log.info("再接続処理!");

			terminate();

			if (initial()) {

				ret = true;

				log.info("再接続処理:成功!");

				//エラーフラグ解除
				m_isErr = false;
				
			} else {
				log.info("再接続処理:失敗!");
			}

			return ret;
		}
	}

	/**
	 * 再接続処理を実行するタスク用のThreadFactory
	 */
	private class ReconnectorThreadFactory implements ThreadFactory {
		private final String m_threadName;
		
		public ReconnectorThreadFactory(String threadName){
			m_threadName = threadName;
		}
		
		public Thread newThread(Runnable r) {
			return new Thread(r, "Reconnector-" + m_threadName);
		}
	}
}
