/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.syslogng.forward;

import java.io.BufferedReader;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.InetAddress;
import java.net.NetworkInterface;
import java.net.URL;
import java.net.URLConnection;
import java.util.Date;
import java.util.Enumeration;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.log4j.PropertyConfigurator;
import org.apache.log4j.helpers.LogLog;
import org.jboss.util.stream.Streams;

import com.clustercontrol.syslogng.forward.util.LogForwardProperties;

/**
 * syslog転送メインクラス<BR>
 * 標準入力でsyslogメッセージを受け取り、メッセージ管理クラスへ渡します。
 * 
 * @version 1.0.0
 * @since 1.0.0
 */
public class LogForward {
	
	/** LogForwardクラス用スケジューラ * */
	private ScheduledExecutorService m_scheduler = Executors.newScheduledThreadPool(10);
	
	/** メッセージ管理のインスタンス。 */
	private LogManager m_manageLog;

	/** ログ出力のインスタンス。 */
	static Log log = LogFactory.getLog(LogForward.class);

	/** log4j設定 */
	public String m_log4jFileName = null;
	/** log4j設定ファイル再読み込み間隔 */
	private long m_reconfigLog4jInterval = 600;

	/**
	 * メイン処理です。<BR>
	 * syslog転送を開始します。
	 * 
	 * @param args プロパティファイル名（配列の１個目）
	 * @throws Exception
	 * 
	 * @see #forward()
	 */
	public static void main(String[] args) throws Exception {

		// 引数チェック
		if(args.length != 2){
			System.out.println("Usage : java LogForward [LogForward.properties File Path] [log4j.properties File Path]");
			System.exit(1);
		}
		
		// Systemプロパティ
		log.info("starting Hinemos syslogforward...");
		log.info("java.vm.version = " + System.getProperty("java.vm.version"));
		log.info("java.vm.vendor = " + System.getProperty("java.vm.vendor"));
		log.info("java.home = " + System.getProperty("java.home"));
		log.info("os.name = " + System.getProperty("os.name"));
		log.info("os.arch = " + System.getProperty("os.arch"));
		log.info("os.version = " + System.getProperty("os.version"));
		log.info("user.name = " + System.getProperty("user.name"));
		log.info("user.dir = " + System.getProperty("user.dir"));
		log.info("user.country = " + System.getProperty("user.country"));
		log.info("user.language = " + System.getProperty("user.language"));
		log.info("file.encoding = " + System.getProperty("file.encoding"));
		
		// OS情報(識別したホスト名、IPアドレス)
		log.info("nodename = " + InetAddress.getLocalHost().getHostName());
		Enumeration networkInterfaces = NetworkInterface.getNetworkInterfaces();
		if (null != networkInterfaces) {
			while (networkInterfaces.hasMoreElements()) {
				NetworkInterface ni = (NetworkInterface) networkInterfaces.nextElement();
				Enumeration inetAddresses = ni.getInetAddresses();
				while (inetAddresses.hasMoreElements()) {
					InetAddress in = (InetAddress) inetAddresses.nextElement();
					String hostAddress = in.getHostAddress();
					log.info("ipaddress = " + hostAddress);
				}
			}
		}
		
		// Agent設定の初期化
		log.info("LogForward.properties = " + args[0]);
		// ログ再読み込みの機構初期化
		log.info("log4j.properties = " + args[1]);
		
		LogForward logForward = new LogForward(args[0], args[1]);

		try {
			log.info("Hinemos syslogforward started");
			logForward.forward();
		} catch (Exception e) {
			log.error("Hinemos syslogforward unknown Error!!", e);
			System.exit(1);
		}
		
		//------------
		//-- 終了処理
		//------------

		log.info("Hinemos syslogforward stopped");
		
		System.exit(0);
	}

	/**
	 * コンストラクタ。
	 * 
	 * @param propFileName プロパティファイル名
	 * @throws Exception
	 * 
	 * @see com.clustercontrol.syslogng.forward.LogManager
	 */
	LogForward(String propFileName, String log4jFileName) throws Exception {
	
		////
		//プロパティファイル読み込み初期化
		////
		LogForwardProperties.init(propFileName);

		////
		// log4j設定ファイル再読み込み設定
		////
		if(log4jFileName == null){
			log.error("log4j.properties does not specify!");
		}
		m_log4jFileName = log4jFileName;
		
		// log4j再読み込みタスク
		m_scheduler.scheduleWithFixedDelay(new ReloadLog4jTask(), 
				m_reconfigLog4jInterval, m_reconfigLog4jInterval, TimeUnit.SECONDS);
		
		////
		//ログ処理スレッドを開始
		////
		m_manageLog = new LogManager();
		m_manageLog.start();

		//スレッド待機待ち
		while (true) {
			if (m_manageLog.isWaiting())
				break;
		}
		
		////
		//終了呼び出し設定
		////
    	Runtime.getRuntime().addShutdownHook(new Thread() {
    	    @Override
			public void run() { m_manageLog.interrupt(); }
    	});
	}
	
	/**
	 * メッセージを送信します。<BR>
	 * 引数で渡されたメッセージを、メッセージ管理へ渡します。
	 * 
	 * @param msg メッセージ
	 */
	private void put(String msg) {

		if (msg == null || msg.equals("")){
			return;
		}

		if(log.isDebugEnabled()){
			log.debug("put() : Message = " + msg);
		}

		m_manageLog.add(msg);
	}

	/**
	 * メッセージを転送します。<BR>
	 * 標準入力でメッセージを受け取り、メッセージ管理へ渡します。
	 * 
	 * @throws Exception
	 * 
	 * @see #put(String)
	 */
	public void forward() throws Exception {

		log.debug("forward() start");

		BufferedReader br = new BufferedReader(new InputStreamReader(System.in));
		String line;
		// 半永久的に標準入力から読み出し続けて転送処理を行う。入力の終端に達した場合
		// （readLine==null）か、何らかの例外が発生した場合のみSyslogForwardを終了する
		try {
			while ( (line = br.readLine()) != null) {
				put(line);
			}
		} catch (IOException e) {
			log.error("forward()", e);
		} finally {
			// 例外の有無に関わらずLogManagerをinterruptする （ログマネージャは中で
			// スレッドを持ち、ログマネージャスレッドの終了をSyslogForwardが待つため）
			m_manageLog.interrupt();
		}
		
		log.debug("forward() finish");
	}

	/**
	 * ローカルディスクに存在するファイルのURLオブジェクトを作成する
	 * 
	 * @param localFilePath
	 * @return
	 */
	private URL toUrl(String localFilePath){
		URL url = null;
		
		// ファイルの存在チェック
		FileInputStream in = null;
		try{
			in = new FileInputStream(localFilePath);
		}catch (Exception e) {
			log.error(localFilePath + " does not exists!", e);
		}finally{
			if(in != null){
				try {
					in.close();
				} catch (Exception e) {
				}
			}
		}
		
		// URLクラスの作成
		try{
			url = new URL("file", "localhost", localFilePath);
		}catch (Exception e) {
			log.error(localFilePath + " : unknown exception", e);
		}
		
		return url;
	}
	
	/**
	 * ログ設定を再読み込みするタスク
	 * @author Hinemos
	 *
	 */
	protected class ReloadLog4jTask implements Runnable {

		/**
		 * デフォルトコンストラクタ
		 */
		public ReloadLog4jTask() {
		}
		
		// 最終更新時刻
		private long lastConfigured = -1;		
		// log4j設定ファイルURL
		URL configURL = null;
		
		/**
		 * ログ設定ファイルリロードタスクの開始
		 */
		public void run() {
			log.debug("ReloadLog4jTask.run() Checking if configuration changed");
			
			// File Load
			configURL = toUrl(m_log4jFileName);
			
			// 対象ファイルが存在する場合に限り実行
			if(configURL != null){
				
				// File Check
				try{
					URLConnection conn = configURL.openConnection();
					long lastModified = conn.getLastModified();
					
					if (lastConfigured < lastModified)
					{
						log.debug("ReloadLog4jTask.run() do re-configuration");
						reconfigure(conn);
					}
					
				} catch (IOException e) {
					log.warn("ReloadLog4jTask.run() Failed to check URL: " + configURL, e);
				}
			}
			log.debug("ReloadLog4jTask.run() finish");
		}
		
		/**
		 * ログ設定のリロード
		 * @param conn
		 */
		private void reconfigure(final URLConnection conn) {
			log.info("Configuring from URL: " + configURL);
			
			if(log.isDebugEnabled()){
				try
				{
					System.out.println((new Date()).toString() + " : Configuring URL Data Dump");
					java.io.InputStream is = conn.getInputStream();
					Streams.copy(is, System.out);
				}
				catch (Exception e)
				{
					log.error("Failed to dump config", e);
				}
			}
			
			// reconfigure
			try{
				PropertyConfigurator.configure(configURL);
				
				/* Set the LogLog.QuietMode. As of log4j1.2.8 this needs to be set to
					avoid deadlock on exception at the appender level. See bug#696819.
				 */
				LogLog.setQuietMode(true);
				
				lastConfigured = System.currentTimeMillis();
				
				log.debug("ReloadLog4jTask.reconfigure() lastConfigured = " + lastConfigured);
				
			} catch (Exception e) {
				log.error("Failed to configure from URL: " + configURL, e);
			}
		}
	}
}
