/*
 
Copyright (C) 2006 - 2008 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.snmptrap.ejb.session;

import java.rmi.RemoteException;
import java.security.Principal;
import java.util.ArrayList;
import java.util.Collection;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.SessionBean;
import javax.ejb.SessionContext;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.HinemosUnknownException;
import com.clustercontrol.MonitorDuplicateException;
import com.clustercontrol.MonitorNotFoundException;
import com.clustercontrol.UsedFacilityException;
import com.clustercontrol.bean.PluginConstant;
import com.clustercontrol.commons.ejb.session.CheckFacility;
import com.clustercontrol.snmptrap.bean.MonitorSnmpTrapInfo;
import com.clustercontrol.snmptrap.bean.SnmpTrapMasterInfo;
import com.clustercontrol.snmptrap.ejb.entity.SnmpTrapInfoUtil;
import com.clustercontrol.snmptrap.ejb.entity.SnmpTrapMibMasterData;
import com.clustercontrol.snmptrap.factory.AddMibMaster;
import com.clustercontrol.snmptrap.factory.AddMonitorInfo;
import com.clustercontrol.snmptrap.factory.AddOidMaster;
import com.clustercontrol.snmptrap.factory.DeleteMibMaster;
import com.clustercontrol.snmptrap.factory.DeleteMonitorInfo;
import com.clustercontrol.snmptrap.factory.DeleteOidMaster;
import com.clustercontrol.snmptrap.factory.ModifyMibMaster;
import com.clustercontrol.snmptrap.factory.ModifyMonitorInfo;
import com.clustercontrol.snmptrap.factory.ModifyOidMaster;
import com.clustercontrol.snmptrap.factory.SelectMibMaster;
import com.clustercontrol.snmptrap.factory.SelectMonitorInfo;

/**
 * SNMPトラップ監視を制御するSession Bean <BR>
 * 
 * @ejb.bean name="MonitorSnmpTrapController"
 *           jndi-name="MonitorSnmpTrapController"
 *           type="Stateless"
 *           transaction-type="Container"
 * 
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=SnmpTrapInfoLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=SnmpTrapMasterLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=SnmpTrapMibMasterLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=SnmpTrapOidInfoLocal"
 * 
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Home"
 *     
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="LocalHome"
 *     
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Local"
 *     
 * @version 2.4.0
 * @since 2.1.0
 * 
 */
public abstract class MonitorSnmpTrapControllerBean implements SessionBean, CheckFacility {
	
	protected static Log m_log = LogFactory.getLog( MonitorSnmpTrapControllerBean.class );
	
	@SuppressWarnings("unused")
	private SessionContext m_context;
	
	public void setSessionContext(SessionContext ctx) throws EJBException, RemoteException {
		m_context = ctx;
	}
	
	/**
	 * SNMPTRAP監視情報をマネージャに登録します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="SnmptrapMonitorWrite"
     *     method-intf="Remote"
	 * 
	 * @param info 監視情報
	 * @return 登録に成功した場合、true
	 * @throws MonitorNotFoundException
	 * @throws MonitorDuplicateException
	 * @throws HinemosUnknownException 
	 */
	public boolean addMonitor(MonitorSnmpTrapInfo info) throws MonitorNotFoundException, MonitorDuplicateException, HinemosUnknownException {
		
		Principal pri = m_context.getCallerPrincipal();
		
		// SNMPTRAP監視情報を登録
		AddMonitorInfo add = new AddMonitorInfo();
		boolean flag;
		try {
			flag = add.add(info, pri.getName());
		} catch (MonitorNotFoundException e) {
			throw e;
		} catch (MonitorDuplicateException e) {
			throw e;
		} catch (Exception e) {
			m_log.error("addMonitor " + e.getMessage(), e);
			throw new HinemosUnknownException(e.getMessage(), e);
		}
		return flag;
	}
	
	/**
	 * マネージャのSNMP TRAP監視情報を更新します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="SnmptrapMonitorWrite"
     *     method-intf="Remote"
	 * 
	 * @param info 監視情報
	 * @return 変更に成功した場合、true
	 * @throws MonitorNotFoundException 
	 * @throws HinemosUnknownException
	 */
	public boolean modifyMonitor(MonitorSnmpTrapInfo info) throws MonitorNotFoundException, HinemosUnknownException{
		
		Principal pri = m_context.getCallerPrincipal();

		// SNMPTRAP監視情報を更新
		ModifyMonitorInfo modify = new ModifyMonitorInfo();
		boolean flag;
		try {
			flag = modify.modify(info, pri.getName());
		} catch (MonitorNotFoundException e) {
			throw e;
		} catch (Exception e) {
			m_log.error("modifyMonitor " + e.getMessage(), e);
			throw new HinemosUnknownException(e.getMessage(), e);
		}
		return flag;
	}
	
	/**
	 * SQL監視情報をマネージャから削除します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="SnmptrapMonitorWrite"
     *     method-intf="Remote"
	 * 
	 * @param monitorId 監視項目ID
	 * @return 削除に成功した場合、true
	 * @throws MonitorNotFoundException 
	 * @throws HinemosUnknownException
	 */
	public boolean deleteMonitor(String monitorId) throws MonitorNotFoundException, HinemosUnknownException{

		// SNMPTRAP監視情報を削除
		DeleteMonitorInfo delete = new DeleteMonitorInfo();

		boolean flag;
		try {
			flag = delete.delete(monitorId);
		} catch (MonitorNotFoundException e) {
			throw e;
		} catch (Exception e) {
			m_log.error("deleteMonitor " + e.getMessage(), e);
			throw new HinemosUnknownException(e.getMessage(), e);
		}
		return flag;
	}
	
	/**
	 * MIB一覧を取得します。<BR>
	 * 
	 * 戻り値はMIBの名前(String)のArrayListです。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="SnmptrapMonitorRead"
     *     method-intf="Remote"
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @return MIB一覧
	 * @throws MonitorNotFoundException
	 * @throws HinemosUnknownException 
	 */
	public ArrayList<String> getMibList() throws MonitorNotFoundException, HinemosUnknownException{

		// MIB一覧を取得
		SelectMonitorInfo select = new SelectMonitorInfo();
		ArrayList<String> list = null;
		try {
			list = select.getMibList();
		} catch (MonitorNotFoundException e) {
			throw e;
		} catch (Exception e) {
			m_log.error("getMibList " + e.getMessage(), e);
			throw new HinemosUnknownException(e.getMessage(), e);
		}

		return list;
	}
	
	/**
	 * 監視マスタ情報一覧を取得します。<BR>
	 * 
	 * 
	 * 戻り値はSnmpTrapMasterInfoのArrayListです。
	 * 
	 * 
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="SnmptrapMonitorRead"
     *     method-intf="Remote"
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @return MIB一覧
	 * @throws MonitorNotFoundException 
	 * @throws HinemosUnknownException
	 * @see com.clustercontrol.snmptrap.bean.SnmpTrapMasterInfo
	 */
	public ArrayList<SnmpTrapMasterInfo> getMasterList(String mib) throws MonitorNotFoundException, HinemosUnknownException{

		// 監視マスタ情報一覧を取得
		SelectMonitorInfo select = new SelectMonitorInfo();
		ArrayList<SnmpTrapMasterInfo> list = null;
		try {
			list = select.getMasterList(mib);
		} catch (MonitorNotFoundException e) {
			throw e;
		} catch (Exception e) {
			m_log.error("getMasterList " + e.getMessage(), e);
			throw new HinemosUnknownException(e.getMessage(), e);
		}

		return list;
	}
	
	/**
	 * SNMPTRAP監視設定情報を取得します。
	 * 
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="SnmptrapMonitorRead"
     *     method-intf="Remote"
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @param monitorId 監視項目ID
	 * @return 監視情報
	 * @throws MonitorNotFoundException 
	 * @throws HinemosUnknownException
	 * @see com.clustercontrol.snmptrap.bean.MonitorSnmpTrapInfo
	 */
	public MonitorSnmpTrapInfo getMonitorInfo(String monitorId) throws MonitorNotFoundException, HinemosUnknownException{

		// SNMPTRAP監視情報を取得
		SelectMonitorInfo select = new SelectMonitorInfo();
		MonitorSnmpTrapInfo info = null;
		try {
			info = select.getMonitorInfo(monitorId);
		} catch (MonitorNotFoundException e) {
			throw e;
		} catch (Exception e) {
			m_log.error("getMonitorInfo " + e.getMessage(), e);
			throw new HinemosUnknownException(e.getMessage(), e);
		}

		return info;
	}

	/**
	 * SNMPTRAP監視一覧リストを取得します。<BR>
	 * 
	 * 戻り値はArrayListのArrayListであり、内部のArrayListは
	 * 下記の様に格納される。
	 * 
	 * <Pre>
	 * ArrayList info = new ArrayList();
	 *  info.add(monitor.getMonitorId());
	 *  info.add(monitor.getDescription());
	 *  info.add(oids.size());
	 *  info.add(facilityPath);
	 *  info.add(monitor.getValidFlg().intValue());
	 *  info.add(monitor.getRegUser());
	 *  info.add(monitor.getRegDate() == null ? null:new Date(monitor.getRegDate().getTime()));
	 *  info.add(monitor.getUpdateUser());
	 *  info.add(monitor.getUpdateDate() == null ? null:new Date(monitor.getUpdateDate().getTime()));
	 *
	 *  list.add(info);
	 * </Pre>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="SnmptrapMonitorRead"
     *     method-intf="Remote"
     *     
     * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @return Objectの2次元配列
	 * @throws MonitorNotFoundException 
	 * @throws HinemosUnknownException
	 */
	public ArrayList<MonitorSnmpTrapInfo> getMonitorList() throws MonitorNotFoundException, HinemosUnknownException{

		// SNMPTRAP監視一覧を取得
		SelectMonitorInfo select = new SelectMonitorInfo();
		ArrayList<MonitorSnmpTrapInfo> list = null;
		try {
			list = select.getMonitorList();
		} catch (MonitorNotFoundException e) {
			throw e;
		} catch (HinemosUnknownException e) {
			throw e;
		} catch (Exception e) {
			m_log.error("getMonitorList " + e.getMessage(), e);
			throw new HinemosUnknownException(e.getMessage(), e);
		}

		return list;
	}
	
	/**
	 * ファシリティIDが使用されているかチェックします。<BR>
	 * <P>
	 * 使用されていたら、UsedFacilityException がスローされる。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="SnmptrapMonitorRead"
	 *     method-intf="Remote"
	 *     
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @param facilityId ファシリティ
	 * @throws UsedFacilityException
	 * 
	 * @see com.clustercontrol.commons.ejb.session.CheckFacility#isUseFacilityId(java.lang.String)
	 * @see com.clustercontrol.bean.PluginConstant;
	 */
	public void isUseFacilityId(String facilityId) throws UsedFacilityException {
		try {
			// ファシリティIDが監視対象になっている監視情報を取得する。
			Collection ct = SnmpTrapInfoUtil.getLocalHome().findByFacilityId(facilityId);
			if(ct != null && ct.size() > 0) {
				throw new UsedFacilityException(PluginConstant.TYPE_SNMPTRAP_MONITOR);
			}
		} catch (FinderException e) {
			m_log.debug("isUseFacilityId() : " + e.getMessage());
		} catch (NamingException e) {
			m_log.debug("isUseFacilityId() : " + e.getMessage());
		}
	}
	
	
	/**
	 * MIBの内容を取得します。<BR>
	 * 
	 * 戻り値はMIBのDTO(SnmpTrapMibMasterData)です。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="SnmptrapMonitorRead"
     *     method-intf="Remote"
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @return MIB一覧
	 * @throws MonitorNotFoundException 
	 * @throws HinemosUnknownException
	 */
	public SnmpTrapMibMasterData getMib(String Mib) throws MonitorNotFoundException, HinemosUnknownException{

		SelectMibMaster  selector = new SelectMibMaster(); 
		
		SnmpTrapMibMasterData data = null;
		try {
			data = selector.getMibMasterData(Mib);
		} catch (MonitorNotFoundException e) {
			throw e;
		} catch (Exception e) {
			m_log.error("SnmpTrapMibMasterData " + e.getMessage(), e);
			throw new HinemosUnknownException(e.getMessage(), e);
		}
		return data;
	}
	
	/**
	 * MIB情報の登録
	 *
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="SnmptrapMonitorWrite"
     *     method-intf="Remote"
	 *
	 * @param MibData
	 * @return
	 * @throws HinemosUnknownException 
	 */
	public boolean addMibMaster(SnmpTrapMibMasterData MibData) throws HinemosUnknownException {
		
		Principal pri = m_context.getCallerPrincipal();
		
		AddMibMaster adder = new AddMibMaster();
		
		boolean flag; 
		try {
			flag = adder.add(MibData , pri.getName());
		} catch (Exception e) {
			m_log.error("addMibMaster " + e.getMessage(), e);
			throw new HinemosUnknownException(e.getMessage(), e);
		}
		return flag;
	}
	
	/**
	 * MIB情報の変更
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="SnmptrapMonitorWrite"
     *     method-intf="Remote"
     *     
	 * @param MibData
	 * @return
	 * @throws MonitorNotFoundException 
	 * @throws HinemosUnknownException
	 */
	public boolean modifyMibMaster(SnmpTrapMibMasterData MibData) throws MonitorNotFoundException, HinemosUnknownException{
		
	Principal pri = m_context.getCallerPrincipal();
		
		ModifyMibMaster modifier = new ModifyMibMaster();
		
		boolean flag;
		try {
			flag = modifier.modify(MibData , pri.getName());
		} catch (MonitorNotFoundException e) {
			throw e;
		} catch (Exception e) {
			m_log.error("modifyMibMaster " + e.getMessage(), e);
			throw new HinemosUnknownException(e.getMessage(), e);
		}
		return flag;
	}
	
	/**
	 * MIB情報の削除
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="SnmptrapMonitorWrite"
     *     method-intf="Remote"
     *     
	 * @param mib
	 * @return
	 * @throws MonitorNotFoundException 
	 * @throws HinemosUnknownException
	 */
	public boolean deleteMibMaster(String Mib) throws MonitorNotFoundException, HinemosUnknownException{
		
		Principal pri = m_context.getCallerPrincipal();
		
		DeleteMibMaster eraser= new DeleteMibMaster();
		
		boolean flag;
		try {
			flag = eraser.delete(Mib);
		} catch (MonitorNotFoundException e) {
			throw e;
		} catch (Exception e) {
			m_log.error("deleteMibMaster " + e.getMessage(), e);
			throw new HinemosUnknownException(e.getMessage(), e);
		}
		return flag;
	}
	
	
	/**
	 * OID情報の登録
	 *
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="SnmptrapMonitorWrite"
     *     method-intf="Remote"
	 *
	 * @param MibData
	 * @return
	 * @throws HinemosUnknownException 
	 */
	public boolean addOidMaster(SnmpTrapMasterInfo Oid) throws HinemosUnknownException {
		
		Principal pri = m_context.getCallerPrincipal();
		
		AddOidMaster adder = new AddOidMaster();
		
		boolean flag;
		try {
			flag = adder.add(Oid , pri.getName());
		} catch (Exception e) {
			m_log.error("addOidMaster " + e.getMessage(), e);
			throw new HinemosUnknownException(e.getMessage(), e);
		}
		return flag;
	}
	
	/**
	 * OID情報の変更
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="SnmptrapMonitorWrite"
     *     method-intf="Remote"
     *     
	 * @param MibData
	 * @return
	 * @throws MonitorNotFoundException 
	 * @throws HinemosUnknownException
	 */
	public boolean modifyOidMaster(SnmpTrapMasterInfo Oid) throws MonitorNotFoundException, HinemosUnknownException{
		
		Principal pri = m_context.getCallerPrincipal();
		
		ModifyOidMaster modifier = new ModifyOidMaster();
		
		boolean flag;
		try {
			flag = modifier.modify(Oid , pri.getName());
		} catch (MonitorNotFoundException e) {
			throw e;
		} catch (Exception e) {
			m_log.error("modifyOidMaster " + e.getMessage(), e);
			throw new HinemosUnknownException(e.getMessage(), e);
		}
		return flag;
	}
	
	/**
	 * OID情報の削除
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
     *     role-name="SnmptrapMonitorWrite"
     *     method-intf="Remote"
     *     
	 * @param mib
	 * @return
	 * @throws MonitorNotFoundException 
	 * @throws HinemosUnknownException
	 */
	public boolean deleteOidMaster(SnmpTrapMasterInfo oid) throws MonitorNotFoundException, HinemosUnknownException{
		
		Principal pri = m_context.getCallerPrincipal();
		
		DeleteOidMaster eraser= new DeleteOidMaster();
		
		boolean flag;
		try {
			flag = eraser.delete(oid);
		} catch (MonitorNotFoundException e) {
			throw e;
		} catch (Exception e) {
			m_log.error("delteOidMaster " + e.getMessage(), e);
			throw new HinemosUnknownException(e.getMessage(), e);
		}
		return flag;
	}
}
