/*
 
 Copyright (C) 2008 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License 
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied 
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 PURPOSE.  See the GNU General Public License for more details.
 
 */

package com.clustercontrol.notify.mail.ejb.session;

import java.rmi.RemoteException;
import java.security.Principal;
import java.util.ArrayList;
import java.util.HashMap;
import javax.ejb.CreateException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;
import javax.ejb.SessionContext;
import javax.naming.NamingException;

import com.clustercontrol.notify.mail.ejb.entity.MailTemplateInfoData;
import com.clustercontrol.notify.mail.factory.AddMailTemplate;
import com.clustercontrol.notify.mail.factory.DeleteMailTemplate;
import com.clustercontrol.notify.mail.factory.ModifyMailTemplate;
import com.clustercontrol.notify.mail.factory.SelectMailTemplate;

/**
 * メールテンプレート機能の管理を行う Session Bean <BR>
 * クライアントからの Entity Bean へのアクセスは、Session Bean を介して行います。
 *
 * @ejb.bean name="MailTemplateController"	
 *           jndi-name="MailTemplateController"
 *           type="Stateless"
 *           transaction-type="Container"
 *           
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=MailTemplateInfoLocal"
 * 
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Home"
 *     
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="LocalHome"
 *     
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Local"
 */
public abstract class MailTemplateControllerBean implements javax.ejb.SessionBean {
	
	/** コンテキスト情報。 */
	@SuppressWarnings("unused")
	private SessionContext m_context;
	
	/**
	 * コンテキスト情報を設定します。<BR>
	 * Session Bean の生成時に行う処理を実装します。
	 * @see javax.ejb.SessionBean#setSessionContext(javax.ejb.SessionContext)
	 */
	public void setSessionContext(SessionContext ctx) throws EJBException, RemoteException {
		m_context = ctx;
	}
	
	/**
	 * メールテンプレート情報をマネージャに登録します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="NotifyWrite"
	 *     method-intf="Remote"
	 * 
	 * @param info 作成対象のメールテンプレート情報
	 * @return 作成に成功した場合、<code> true </code>
	 * @throws CreateException
	 * @throws NamingException
	 * 
	 * @see com.clustercontrol.notify.factory.AddMailTemplate#add(MailTemplateInfoData)
	 */
	public boolean addMailTemplate(MailTemplateInfoData data) throws CreateException, NamingException {
		
		Principal pri = m_context.getCallerPrincipal();
		
		// メールテンプレート情報を登録
		AddMailTemplate mailTemplate = new AddMailTemplate();
		return  mailTemplate.add(data, pri.getName());
	}
	
	/**
	 * マネージャ上のメールテンプレート情報を変更します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="NotifyWrite"
	 *     method-intf="Remote"
	 * 
	 * @param info 変更対象のメールテンプレート情報
	 * @return 変更に成功した場合、<code> true </code>
	 * @throws FinderException
	 * @throws NamingException
	 * 
	 * @see com.clustercontrol.notify.factory.ModifyMailTemplate#modify(MailTemplateInfoData)
	 * @see com.clustercontrol.notify.bean.MailTemplateInfoData
	 */
	public boolean modifyMailTemplate(MailTemplateInfoData data) throws FinderException, NamingException {

		Principal pri = m_context.getCallerPrincipal();
		
		// メールテンプレート情報を更新
		ModifyMailTemplate mailTemplate = new ModifyMailTemplate();
		return  mailTemplate.modify(data, pri.getName());
	}
	
	/**
	 * メールテンプレート情報をマネージャから削除します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="NotifyWrite"
	 *     method-intf="Remote"
	 * 
	 * @param mailTemplateId 削除対象のメールテンプレートID
	 * @return 削除に成功した場合、<code> true </code>
	 * @throws RemoveException
	 * @throws NamingException
	 * 
	 * @see com.clustercontrol.notify.factory.DeleteMailTemplate#delete(String)
	 */
	public boolean deleteMailTemplate(String mailTemplateId) throws RemoveException, NamingException {
		
		// メールテンプレート情報を削除
		DeleteMailTemplate mailTemplate = new DeleteMailTemplate();
		return  mailTemplate.delete(mailTemplateId);
	}
	
	/**
	 * 引数で指定されたメールテンプレート情報を返します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="NotifyRead"
	 *     method-intf="Remote"
	 * 
	 * @param mailTemplateId 取得対象のメールテンプレートID
	 * @return メールテンプレート情報
	 * @throws FinderException
	 * @throws NamingException
	 * 
	 * @see com.clustercontrol.notify.factory.SelectMailTemplate#getMailTemplateInfo(String)
	 */
	public MailTemplateInfoData getMailTemplateInfo(String mailTemplateId) throws FinderException, NamingException {
		
		// メールテンプレート情報を取得
		SelectMailTemplate mailTemplate = new SelectMailTemplate();
		MailTemplateInfoData info = mailTemplate.getMailTemplateInfo(mailTemplateId);
		
		return info;
	}
	
	/**
	 * メールテンプレートID一覧を取得します。<BR>
	 * 
	 * 戻り値のArrayListにはMailTemplateId(String)が順に
	 * 格納されています。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     unchecked="true"
	 *     method-intf="Remote"
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @return メールテンプレートID一覧
	 * @throws CreateException
	 * @throws FinderException
	 * @throws NamingException
	 * 
	 * @see com.clustercontrol.notify.factory.SelectMailTemplate#getMailTemplateIdList()
	 */
	public ArrayList<String> getMailTemplateIdList() throws CreateException, FinderException, NamingException {
		
		// メールテンプレートID一覧を取得
		SelectMailTemplate mailTemplate = new SelectMailTemplate();
		ArrayList<String> list = mailTemplate.getMailTemplateIdList();
		
		return list;
	}
	
	/**
	 * メールテンプレート情報一覧を取得します。<BR>
	 * 
	 * 戻り値はArrayListのArrayListであり以下のように格納される。
	 * 
	 * <Pre>
	 * ArrayList info = new ArrayList();
	 *  info.add(mailTemplate.getMailTemplateId());
	 *  info.add(mailTemplate.getDescription());
	 *	info.add(mailTemplate.getRegUser());
	 *	info.add(mailTemplate.getRegDate() == null ? null:new Date(mailTemplate.getRegDate().getTime()));
	 *	info.add(mailTemplate.getUpdateUser());
	 *	info.add(mailTemplate.getUpdateDate() == null ? null:new Date(mailTemplate.getUpdateDate().getTime()));
	 * 
	 *  list.add(info);
	 * </Pre>
	 * 
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="NotifyRead"
	 *     method-intf="Remote"
	 * 
	 * @return メールテンプレート情報一覧（Objectの2次元配列）
	 * @throws CreateException
	 * @throws FinderException
	 * @throws NamingException
	 * 
	 * @see com.clustercontrol.notify.factory.SelectMailTemplate#getMailTemplateList()
	 */
	public ArrayList<ArrayList> getMailTemplateList() throws CreateException, FinderException, NamingException {
		
		// メールテンプレート一覧を取得
		SelectMailTemplate mailTemplate = new SelectMailTemplate();
		ArrayList<ArrayList> list = mailTemplate.getMailTemplateList();
		
		return list;
	}
	
	/**
	 * メールテンプレート情報のマップを取得します。<BR>
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="NotifyRead"
	 *     method-intf="Remote"
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @return メールテンプレート情報マップ
	 * @throws FinderException
	 * @throws NamingException
	 * 
	 * @see com.clustercontrol.notify.factory.SelectMailTemplate#getMailTemplateMap()
	 */
	public HashMap<String, MailTemplateInfoData> getMailTemplateMap() throws FinderException, NamingException {
		
		// メールテンプレート一覧を取得
		SelectMailTemplate mailTemplate = new SelectMailTemplate();
		HashMap<String, MailTemplateInfoData> map = mailTemplate.getMailTemplateMap();
		
		return map;
	}
}
