/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License 
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied 
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 PURPOSE.  See the GNU General Public License for more details.
 
 */

package com.clustercontrol.notify.ejb.session;

import java.rmi.RemoteException;
import java.security.Principal;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.SessionContext;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.HinemosUnknownException;
import com.clustercontrol.NotifyDuplicateException;
import com.clustercontrol.NotifyNotFoundException;
import com.clustercontrol.notify.bean.NotifyInfo;
import com.clustercontrol.notify.bean.NotifyRelationInfo;
import com.clustercontrol.notify.bean.NotifyTypeMstInfo;
import com.clustercontrol.notify.bean.NotifyTypeMstMap;
import com.clustercontrol.notify.ejb.entity.NotifyTypeMstLocal;
import com.clustercontrol.notify.ejb.entity.NotifyTypeMstUtil;
import com.clustercontrol.notify.factory.AddNotify;
import com.clustercontrol.notify.factory.AddNotifyRelation;
import com.clustercontrol.notify.factory.DeleteNotify;
import com.clustercontrol.notify.factory.DeleteNotifyRelation;
import com.clustercontrol.notify.factory.ModifyNotify;
import com.clustercontrol.notify.factory.ModifyNotifyRelation;
import com.clustercontrol.notify.factory.SelectNotify;
import com.clustercontrol.notify.factory.SelectNotifyRelation;
import com.clustercontrol.notify.queue.NotifyQueueException;

/**
 * 通知機能の管理を行う Session Bean です。<BR>
 * クライアントからの Entity Bean へのアクセスは、Session Bean を介して行います。
 *
 * @ejb.bean name="NotifyController"	
 *           jndi-name="NotifyController"
 *           type="Stateless"
 *           transaction-type="Container"
 * 
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=NotifyCommandInfoLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=NotifyEventInfoLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=NotifyInfoLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=NotifyJobInfoLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=NotifyLogEscalateInfoLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=NotifyMailInfoLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=NotifyRelationInfoLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=NotifyStatusInfoLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=NotifyTypeMstLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=SyslogDuplicationInfoLocal"
 * 
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=EventLogLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=EventMultiInfoLocal"
 * @jboss.depends name="jboss.j2ee:service=EJB,jndiName=StatusInfoLocal"
 * 
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Home"
 *     
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="LocalHome"
 *     
 * @ejb.permission
 *     unchecked="true"
 *     method-intf="Local"
 */
public abstract class NotifyControllerBean implements javax.ejb.SessionBean {
	/** ログ出力のインスタンス。 */
	private static Log m_log = LogFactory.getLog( NotifyControllerBean.class );
	
	/** コンテキスト情報。 */
	@SuppressWarnings("unused")
	private SessionContext m_context;
	
	/**
	 * コンテキスト情報を設定します。<BR>
	 * Session Bean の生成時に行う処理を実装します。
	 * @see javax.ejb.SessionBean#setSessionContext(javax.ejb.SessionContext)
	 */
	public void setSessionContext(SessionContext ctx) throws EJBException, RemoteException {
		m_context = ctx;
	}
	
	/**
	 * 通知情報を作成します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="NotifyWrite"
	 *     method-intf="Remote"
	 * 
	 * @param info 作成対象の通知情報
	 * @return 作成に成功した場合、<code> true </code>
	 * @throws HinemosUnknownException 
	 * @throws NotifyDuplicateException
	 * 
	 * @see com.clustercontrol.notify.factory.AddNotify#add(NotifyInfo)
	 */
	public boolean addNotify(NotifyInfo info) throws HinemosUnknownException, NotifyDuplicateException {
		
		Principal pri = m_context.getCallerPrincipal();
		// 通知情報を登録
		AddNotify notify = new AddNotify();
		boolean flag;
		try {
			flag = notify.add(info,pri.getName());
		} catch (NotifyDuplicateException e) {
			throw e;
		} catch (Exception e){
			m_log.error("addNotify(): " + e.getMessage(), e);
			throw new HinemosUnknownException(e.getMessage(), e);
		}
		
		return  flag;
	}
	
	/**
	 * 通知情報を変更します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="NotifyWrite"
	 *     method-intf="Remote"
	 * 
	 * @param info 変更対象の通知情報
	 * @return 変更に成功した場合、<code> true </code>
	 * @throws NotifyDuplicateException
	 * @throws HinemosUnknownException
	 * 
	 * @see com.clustercontrol.notify.factory.ModifyNotify#modify(NotifyInfo)
	 */
	public boolean modifyNotify(NotifyInfo info) throws NotifyDuplicateException, HinemosUnknownException {
		
		Principal pri = m_context.getCallerPrincipal();
		// 通知情報を更新
		ModifyNotify notify = new ModifyNotify();
		boolean flag;
		try {
			flag = notify.modify(info,pri.getName());
		} catch (NotifyDuplicateException e) {
			throw e;
		} catch (Exception e){
			m_log.error("modifyNotify(): " + e.getMessage(), e);
			throw new HinemosUnknownException(e.getMessage(), e);
		}
		return flag;
	}
	
	/**
	 * 通知情報を削除します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="NotifyWrite"
	 *     method-intf="Remote"
	 * 
	 * @param notifyId 削除対象の通知ID
	 * @return 削除に成功した場合、<code> true </code>
	 * @throws NotifyDuplicateException
	 * @throws HinemosUnknownException
	 * 
	 * @see com.clustercontrol.notify.factory.DeleteNotify#delete(String)
	 */
	public boolean deleteNotify(String notifyId) throws NotifyNotFoundException, HinemosUnknownException {
		
		// 通知情報を削除
		DeleteNotify notify = new DeleteNotify();
		boolean flag;
		try {
			flag = notify.delete(notifyId);
		} catch(NotifyNotFoundException e){
			throw e;
		} catch (Exception e){
			m_log.error("deleteNotify(): " + e.getMessage(), e);
			throw new HinemosUnknownException(e.getMessage(), e);
		}
		return flag;
	}
	
	/**
	 * 引数で指定された通知情報を返します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="NotifyRead"
	 *     method-intf="Remote"
	 * 
	 * @param notifyId 取得対象の通知ID
	 * @return 通知情報
	 * @throws NotifyNotFoundException
	 * @throws HinemosUnknownException
	 * 
	 * @see com.clustercontrol.notify.factory.SelectNotify#getNotify(String)
	 */
	public NotifyInfo getNotify(String notifyId) throws NotifyNotFoundException, HinemosUnknownException {
		
		// 通知情報を取得
		SelectNotify notify = new SelectNotify();
		NotifyInfo info = null;
		try {
			info = notify.getNotify(notifyId);
		} catch (NotifyNotFoundException e){
			throw e;
		} catch (Exception e) {
			m_log.error("getNotify(): " + e.getMessage(), e);
			throw new HinemosUnknownException("getNotify() " + e.getMessage(), e);
		}
		
		return info;
	}
	
	/**
	 * 通知ID一覧を返します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     unchecked="true"
	 *     method-intf="Remote"
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @return 通知ID一覧
	 * @throws NotifyNotFoundException
	 * @throws HinemosUnknownException
	 * 
	 * @see com.clustercontrol.notify.factory.SelectNotify#getNotifyIdList()
	 */
	public ArrayList getNotifyIdList() throws NotifyNotFoundException, HinemosUnknownException {
		
		// 通知ID一覧を取得
		SelectNotify notify = new SelectNotify();
		ArrayList list = null;
		try {
			list = notify.getNotifyIdList();
		} catch (NotifyNotFoundException e){
			throw e;
		} catch (Exception e) {
			m_log.error("getNotifyIdList(): " + e.getMessage(), e);
			throw new HinemosUnknownException("getNotifyIdList() " + e.getMessage(), e);
		}
		return list;
	}
	
	/**
	 * 通知情報一覧を返します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="NotifyRead"
	 *     method-intf="Remote"
	 * 
	 * @return 通知情報一覧（Objectの2次元配列）
	 * @throws NotifyNotFoundException
	 * @throws HinemosUnknownException
	 * 
	 * @see com.clustercontrol.notify.factory.SelectNotify#getNotifyList()
	 */
	public ArrayList getNotifyList() throws NotifyNotFoundException, HinemosUnknownException {
		
		// 通知一覧を取得
		SelectNotify notify = new SelectNotify();
		ArrayList list = null;
		try {
			list = notify.getNotifyList();
		}catch (NotifyNotFoundException e){
			throw e;
		} catch (Exception e) {
			m_log.error("getNotifyList(): " + e.getMessage(), e);
			throw new HinemosUnknownException("getNotifyList() " + e.getMessage(), e);
		}
		
		return list;
	}
		
	/**
	 * 通知情報マップを返します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="NotifyRead"
	 *     method-intf="Remote"
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @return 通知情報マップ
	 * @throws NotifyNotFoundException
	 * @throws HinemosUnknownException 
	 * 
	 * @see com.clustercontrol.notify.factory.SelectNotify#getNotifyMap()
	 */
	public HashMap<String, NotifyInfo> getNotifyMap() throws NotifyNotFoundException, HinemosUnknownException {
		
		// 通知一覧を取得
		SelectNotify notify = new SelectNotify();
		HashMap<String, NotifyInfo> map = null;
		try {
			map = notify.getNotifyMap();
		} catch (NotifyNotFoundException e){
			throw e;
		} catch (Exception e) {
			m_log.error("getNotifyMap(): " + e.getMessage(), e);
			throw new HinemosUnknownException("getNotifyMap(): " + e.getMessage(), e);
		}
		return map;
	}
	
	/**
	 * 障害検知用通知情報一覧を返します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="NotifyRead"
	 *     method-intf="Remote"
	 * 
	 * @return 通知情報一覧（Objectの2次元配列）
	 * @throws NotifyNotFoundException
	 * @throws HinemosUnknownException
	 * 
	 * @see com.clustercontrol.notify.factory.SelectNotify#getNotifyList()
	 */
	public ArrayList getTroubleDetectionNotifyList() throws NotifyNotFoundException, HinemosUnknownException {
		
		// 通知一覧を取得
		SelectNotify notify = new SelectNotify();
		ArrayList list = null;
		try {
			list = notify.getTroubleDetectionNotifyList();
		} catch (NotifyNotFoundException e){
			throw e;
		} catch (Exception e) {
			m_log.error("getTroubleDetectionNotifyList(): " + e.getMessage(), e);
			throw new HinemosUnknownException("getTroubleDetectionNotifyList(): " + e.getMessage(), e);
		}
		return list;
	}
	
	/**
	 * 障害検知用通知情報マップを返します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *     role-name="NotifyRead"
	 *     method-intf="Remote"
	 * 
	 * @jboss.method-attributes
	 *     read-only="true"
	 * 
	 * @return 通知情報マップ
	 * @throws NotifyNotFoundException
	 * @throws HinemosUnknownException
	 * 
	 * @see com.clustercontrol.notify.factory.SelectNotify#getNotifyMap()
	 */
	public HashMap<String, NotifyInfo> getTroubleDetectionNotifyMap() throws NotifyNotFoundException, HinemosUnknownException {
		
		// 通知一覧を取得
		SelectNotify notify = new SelectNotify();
		HashMap<String, NotifyInfo> map = null;
		try {
			map = notify.getTroubleDetectionNotifyMap();
		} catch (NotifyNotFoundException e){
			throw e;
		} catch (Exception e) {
			m_log.error("getTroubleDetectionNotifyMap(): " + e.getMessage(), e);
			throw new HinemosUnknownException("getTroubleDetectionNotifyMap(): " + e.getMessage(), e);
		}
			
		return map;
	}
	
	
	/**
	 * 通知グループに対応する通知を取得します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *  unchecked="true"
	 *     method-intf="both"
	 * 
	 * @param notifyGroupId  通知グループID
	 * @return 通知
	 * @throws NotifyNotFoundException
	 * @throws HinemosUnknownException
	 * 
	 */
	public Collection getNotifyRelation(String notifyGroupId) throws NotifyNotFoundException, HinemosUnknownException {
		
	
		SelectNotifyRelation notify = new SelectNotifyRelation();
		Collection<NotifyRelationInfo> info = null;
		try {
			info = notify.getNotifyRelation(notifyGroupId);
		} catch (NotifyNotFoundException e){
			throw e;
		} catch (Exception e) {
			m_log.error("getNotifyRelation(String): " + e.getMessage(), e);
			throw new HinemosUnknownException("getNotifyRelation(String): " + e.getMessage(), e);
		}
		return info;
	}
	
	
	/**
	 * 通知グループを変更します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *  unchecked="true"
	 *     method-intf="both"
	 * 
	 * @param info 通知のセット
	 * @param notifyGroupId 通知グループID
	 * @return 変更に成功した場合、<code> true </code>
	 * @throws NotifyNotFoundException
	 * @throws HinemosUnknownException
	 * 
	 */
	public boolean modifyNotifyRelation(Collection<NotifyRelationInfo> info, String notifyGroupId) throws NotifyNotFoundException, HinemosUnknownException {
		
		// システム通知情報を更新
		ModifyNotifyRelation notify = new ModifyNotifyRelation();
		boolean flag;
		try {
			flag = notify.modify(info, notifyGroupId);
		} catch (NotifyNotFoundException e){
			throw e;
		} catch (Exception e){
			m_log.error("modifyNotifyRelation(Collection, String): " + e.getMessage(), e);
			throw new HinemosUnknownException(e.getMessage(), e);
		}
		return flag;
	}
	
	/**
	 * 通知グループを削除します。
	 * 
	 * @ejb.interface-method
	 * 
	 * @ejb.permission
	 *  unchecked="true"
	 *     method-intf="both"
	 * 
	 * @param notifyGroupId 通知グループID
	 * @return 削除に成功した場合、<code> true </code>
	 * @throws NotifyNotFoundException
	 * @throws HinemosUnknownException
	 * 
	 */
	public boolean deleteNotifyRelation(String notifyGroupId) throws NotifyNotFoundException, HinemosUnknownException{
		
		// システム通知情報を削除
		DeleteNotifyRelation notify = new DeleteNotifyRelation();
		boolean flag;
		try {
			flag = notify.delete(notifyGroupId);
		} catch (NotifyNotFoundException e){
			throw e;
		} catch (Exception e){
			m_log.error("deleteNotifyRelation(String): " + e.getMessage(), e);
			throw new HinemosUnknownException(e.getMessage(), e);
		}
		return flag;
	}
	

	/**
	 * 通知グループを作成します。
	 * 
	 * @ejb.interface-method
	 *
	 * @ejb.permission
	 *  unchecked="true"
	 *     method-intf="both"
	 * 
	 * @param info 通知グループ
	 * @return 作成に成功した場合、<code> true </code>
	 * @throws HinemosUnknownException 
	 * 
	 */
	public boolean addNotifyRelation(Collection<NotifyRelationInfo> info) throws HinemosUnknownException{
		
		// システム通知情報を登録
		
		if(info != null){
			AddNotifyRelation notify = new AddNotifyRelation();
			boolean flag;
			try {
				flag = notify.add(info);
			} catch (Exception e){
				m_log.error("addNotifyRelation(Collection): " + e.getMessage(), e);
				throw new HinemosUnknownException(e.getMessage(), e);
			}
			return flag;
		}else{
			return true;
		}
	}
	
	/**
	 *　引数で指定した通知IDを利用している通知グループIDを取得する。
	 * 
	 * @ejb.interface-method
	*
	 * @ejb.permission
	 *  unchecked="true"
	 *     method-intf="both"
	 * 
	 * @param notifyId
	 * @return　通知グループIDのリスト
	 * @throws NotifyNotFoundException
	 * @throws HinemosUnknownException
	 */
	public ArrayList<String> checkNotifyId(String notifyId) throws NotifyNotFoundException, HinemosUnknownException{
		ArrayList<String> ret = null;
		
		SelectNotifyRelation notify = new SelectNotifyRelation();
		try {
		ret = notify.getNotifyGroupIdBaseOnNotifyId(notifyId);
		} catch (NotifyNotFoundException e){
			throw e;
		} catch (Exception e){
			m_log.error("checkNotifyId(String): " + e.getMessage(), e);
			throw new HinemosUnknownException(e.getMessage(), e);
		}
		return ret;
	}
	
	/**
	 *　通知定義情報が格納されたマップを取得する。
	 * 
	 * @ejb.interface-method
	 *
	 * @ejb.permission
	 *  unchecked="true"
	 *     method-intf="both"
	 * 
	 * @return　通知定義情報が格納されたマップ
	 * @throws NotifyNotFoundException 
	 * @throws HinemosUnknownException 
	 * @throws NotifyQueueException 定義情報の取得に失敗した場合に投げられる
	 */
	public NotifyTypeMstMap getNotifyTypeMstMap() throws NotifyNotFoundException, HinemosUnknownException {
		try {
			Collection<NotifyTypeMstLocal> mst = NotifyTypeMstUtil.getLocalHome().findAll();
			
			// NotifyParameterMap を生成するために、NotifyTypeMstData のコレクションデータを作成する
			Collection<NotifyTypeMstInfo> params = new ArrayList<NotifyTypeMstInfo>();
			for(NotifyTypeMstLocal typeMst : mst){
				params.add(typeMst.getData());
			}
			
			return new NotifyTypeMstMap(params);
		} catch (FinderException e) {
			m_log.error("getNotifyTypeMstMap(): " + e.getMessage(), e);
			throw new NotifyNotFoundException(e.getMessage(),e);
		} catch (Exception e) {
			m_log.error("getNotifyTypeMstMap(): " + e.getMessage(), e);
			throw new HinemosUnknownException(e.getMessage(), e);
		}
	}
}
