/*

Copyright (C) 2008 NTT DATA Corporation

This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

 */
package com.clustercontrol.commons.quartz;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.Date;
import java.util.Properties;

import javax.naming.CompositeName;
import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.Name;
import javax.naming.NamingException;

import org.jboss.naming.NonSerializableFactory;
import org.jboss.system.ServiceMBeanSupport;
import org.quartz.Scheduler;
import org.quartz.SchedulerException;
import org.quartz.Trigger;
import org.quartz.impl.StdSchedulerFactory;

import com.clustercontrol.jobmanagement.bean.QuartzConstant;

/**
 * JBoss起動時にQuartzのスケジューラを実行するためのJMXサービスを実装したクラス
 * 
 * @jmx.mbean
 *     name="user:service=CustomQuartzService,name=CustomQuartzService"
 *     description="CustomQuartzService MBean"
 *     extends="org.jboss.system.ServiceMBean"
 * 
 * @jboss.service servicefile="custom-quartz"
 */
public class CustomQuartzService extends ServiceMBeanSupport implements CustomQuartzServiceMBean {
	private String jndiName;

	private StdSchedulerFactory schedulerFactory;

	private Properties properties;

	public CustomQuartzService() {
		jndiName = "CustomQuartz";
		properties = new Properties();
	}

	/**
	 * @jmx.managed-attribute
	 */
	public void setJndiName(String jndiName) throws Exception {
		log.info("STATE : " + getState());

		String oldName = this.jndiName;
		this.jndiName = jndiName;

		if (super.getState() == STARTED) {
			try {
				unbind(oldName);
			} catch (NamingException e) {
				NamingException ne = new NamingException("Failed to unbind");
				ne.setRootCause(e);
				log.error(ne.getMessage(), ne);
				throw ne;
			}

			try {
				rebind();
			} catch (NamingException e) {
				NamingException ne = new NamingException("Failed to rebind");
				ne.setRootCause(e);
				log.error(ne);
				log.error(ne.getMessage(), ne);
				throw ne;
			}
		}
	}

	/**
	 * @jmx.managed-attribute
	 */
	public String getJndiName() {
		return this.jndiName;
	}

	@Override
	public String getName() {
		return "CustomQuartzService(" + jndiName + ")";
	}

	/**
	 * @jmx.managed-attribute
	 * @param properties
	 */
	public void setProperties(String pro) {
		ByteArrayInputStream inputStream = new ByteArrayInputStream(pro
				.getBytes());
		try {
			this.properties.load(inputStream);
		} catch (IOException e) {
			IOException ioe = new IOException("Failed to set properties.");
			log.error(ioe.getMessage(), ioe);
		}
	}

	/**
	 * @jmx.managed-attribute
	 * @param properties
	 */
	public String getProperties() {
		try {
			ByteArrayOutputStream outputStream = new ByteArrayOutputStream();
			properties.store(outputStream, "");

			return new String(outputStream.toByteArray());
		} catch (IOException ioe) {
			return "";
		}
	}

	/**
	 * @jmx.managed-attribute
	 */
	public String getSchedulerStatus() {
		try {
			Scheduler scheduler = schedulerFactory.getScheduler();
			if(scheduler.isPaused() == true){
				return "Paused";
			} else if (scheduler.isShutdown() == true){
				return "Shutdown";
			} else {
				return "Started";
			}
		} catch (SchedulerException e) {
			log.error(e.getMessage(), e);
		}
		return "Unknown";
	}

	@Override
	public void createService() throws Exception {
		log.info("Create CustomQuartzService(" + jndiName + ")...");

		try {
			schedulerFactory = new StdSchedulerFactory();
			schedulerFactory.initialize(properties);
		} catch (SchedulerException e) {
			log.error(e.getMessage(), e);
			throw e;
		}

		log.info("CustomQuartzService(" + jndiName + ") created.");
	}

	@Override
	public void startService() throws Exception {
		log.info("Start CustomQuartzService(" + jndiName + ")...");

		try {

			updateStartTime();

			rebind();

		} catch (Exception e) {
			log.error(e.getMessage(), e);
			throw e;
		}

		log.info("CustomQuartzService(" + jndiName + ") started.");
	}

	/**
	 * @jmx.managed-operation
	 */
	public void startScheduler() throws Exception {
		log.info("Start QuartzScheduler(" + jndiName + ")...");

		try {
			Scheduler scheduler = schedulerFactory.getScheduler();
			scheduler.start();
		} catch (SchedulerException e) {
			log.error(e.getMessage(), e);
			throw e;
		}

		log.info("QuartzScheduler(" + jndiName + ") started.");
	}

	/**
	 * @jmx.managed-operation
	 */
	public void pauseScheduler() throws Exception {
		log.info("Pause QuartzScheduler(" + jndiName + ")...");

		try {
			Scheduler scheduler = schedulerFactory.getScheduler();
			scheduler.pause();
		} catch (SchedulerException e) {
			log.error(e.getMessage(), e);
			throw e;
		}

		log.info("QuartzScheduler(" + jndiName + ") paused.");
	}

	@Override
	public void stopService() throws Exception {
		log.info("Stop CustomQuartzService(" + jndiName + ")...");

		try {
			Scheduler scheduler = schedulerFactory.getScheduler();
			scheduler.shutdown();
			unbind(jndiName);
		} catch (SchedulerException e) {
			log.error(e.getMessage(), e);
			throw e;
		} catch (NamingException e) {
			log.error(e.getMessage(), e);
			throw e;
		}

		log.info("CustomQuartzService(" + jndiName + ") stopped.");
	}

	@Override
	public void destroyService() throws Exception {
		log.info("Destroy CustomQuartzService(" + jndiName + ")...");

		schedulerFactory = null;

		log.info("CustomQuartzService(" + jndiName + ") destroyed.");
	}

	/**
	 * 
	 * @param rootCtx
	 * @param name
	 * @return
	 * @throws NamingException
	 */
	private static Context createContext(Context rootCtx, Name name) throws NamingException {
		Context subctx = rootCtx;

		for (int n = 0; n < name.size(); n++) {
			String atom = name.get(n);

			try {
				Object obj = subctx.lookup(atom);
				subctx = (Context) obj;
			} catch (NamingException e) {
				// 存在しない場合は、サブコンテキストを生成
				subctx = subctx.createSubcontext(atom);
			}
		}

		return subctx;
	}

	/**
	 * JNDIサービスにPerformanceRestartManagerクラスのインスタンスをバインドします。
	 * @throws NamingException
	 */
	private void rebind() throws Exception {
		InitialContext rootCtx = new InitialContext();

		Name fullName = rootCtx.getNameParser("").parse(jndiName);
		Name parentName = fullName;
		if(fullName.size() > 1){
			parentName = fullName.getPrefix(fullName.size()-1);
		} else {
			parentName = new CompositeName();
		}

		Context parentCtx = createContext(rootCtx, parentName);
		Name atomName = fullName.getSuffix(fullName.size()-1);
		String atomStirng = atomName.get(0);

		NonSerializableFactory.rebind(parentCtx, atomStirng, schedulerFactory.getScheduler());
	}

	/**
	 * JNDIサービスにバインドされているインスタンスをアンバインドします。
	 * @param jndiName JNDI名
	 * @throws NamingException
	 */
	private void unbind(String jndiName) throws NamingException {
		InitialContext rootCtx = null;

		try {
			rootCtx = new InitialContext();

			Name fullName = rootCtx.getNameParser("").parse(jndiName);
			Name atomName = fullName.getSuffix(fullName.size() - 1);
			String atom = atomName.get(0);

			rootCtx.unbind(jndiName);
			NonSerializableFactory.unbind(atom);
		} finally {
			if(rootCtx != null) { 
				rootCtx.close(); 
			}
		}
	}

	/**
	 * 
	 * トリガの開始時刻（startDate）を現在時刻に更新するためのメソッド。
	 * JBoss起動時（スケジューラ起動時）にcallされる。
	 * 
	 * スケジューラが一定時間停止してから起動した際に、トリガが複数回実行されるのを回避するために、
	 * スケジューラ起動時にcallする。
	 * 
	 * @throws NamingException
	 * @throws SchedulerException
	 */
	private void updateStartTime() throws NamingException, SchedulerException {

		//現在時刻を取得
		Date now = new Date();
		log.debug("set next_fire_time : " + now.getTime());

		try {
			//QuartzのSchedulerを取得
			Scheduler scheduler = schedulerFactory.getScheduler();
			String[] triggerGroupNames = scheduler.getTriggerGroupNames();

			//triggerGroup, triggerNameからトリガを特定
			for (String triggerGroupName : triggerGroupNames) {
				String[] triggerNames = scheduler.getTriggerNames(triggerGroupName);

				for (String triggerName : triggerNames) {

					log.debug("triggerGroupName / triggerName : " + triggerGroupName + " / " + triggerName);

					Trigger trigger = scheduler.getTrigger(triggerName, triggerGroupName);

					//TriggerStateの状態確認 for debug
					if (log.isDebugEnabled() && trigger != null){
						switch (scheduler.getTriggerState(triggerName, triggerGroupName)) {
						case Trigger.STATE_BLOCKED:
							log.debug("TriggerState triggerGroupName / triggerName : " + triggerGroupName + " / " + triggerName + " is STATE_BLOCKED");							
							break;
						case Trigger.STATE_COMPLETE:
							log.debug("TriggerState triggerGroupName / triggerName : " + triggerGroupName + " / " + triggerName + " is STATE_COMPLETE");							
							break;
						case Trigger.STATE_ERROR:
							log.debug("TriggerState triggerGroupName / triggerName : " + triggerGroupName + " / " + triggerName + " is STATE_ERROR");							
							break;
						case Trigger.STATE_NONE:
							log.debug("TriggerState triggerGroupName / triggerName : " + triggerGroupName + " / " + triggerName + " is STATE_NONE");							
							break;
						case Trigger.STATE_NORMAL:
							log.debug("TriggerState triggerGroupName / triggerName : " + triggerGroupName + " / " + triggerName + " is STATE_NORMAL");							
							break;
						case Trigger.STATE_PAUSED:
							log.debug("TriggerState triggerGroupName / triggerName : " + triggerGroupName + " / " + triggerName + " is STATE_PAUSED");							
							break;
						default:
							break;
						}
					}

					// ジョブ実行のためのスケジュール登録の場合は、JBoss再起動前の
					// 許容時間（org.quartz.jobStore.misfireThresholdで指定した時間 （デフォルト1時間））以内の
					// ジョブに関しては実行対象とする必要があるため、ここでStartTimeのリセットは行わない。
					// ここで、ジョブのStartTimeもリセットしてしまうと、HA構成の場合に、
					// フェイルオーバ中に実行するようにスケジューリングされていたジョブがフェイルオーバ後に実行されなくなる。
					if(!QuartzConstant.GROUP_NAME.equals(triggerGroupName)){
						// トリガがnullで無い、かつ「有効(STATE_NORMAL)」の場合
						if (trigger != null 
								&& scheduler.getTriggerState(triggerName, triggerGroupName) == Trigger.STATE_NORMAL) {
							log.debug("triggerName : " + triggerName + " is not PAUSED");

							//トリガに開始時刻をセットして再登録する
							trigger.setStartTime(now);
							scheduler.rescheduleJob(triggerName, triggerGroupName, trigger);
						}
					}

				}
			}
		} catch (SchedulerException e) {
			throw e;
		}
	}

	/**
	 * 
	 * トリガの開始時刻（startDate）を現在時刻に更新するためのメソッド。
	 * ジョブスケジュールも含めて全てのスケジュールを更新する。
	/*
	 * @jmx.managed-operation
	 *
	 * @throws NamingException
	 * @throws SchedulerException
	 */
	public void resetAllSchedule() throws Exception {

		//現在時刻を取得
		Date now = new Date();
		log.debug("set next_fire_time : " + now.getTime());

		try {
			//QuartzのSchedulerを取得
			Scheduler scheduler = schedulerFactory.getScheduler();
			String[] triggerGroupNames = scheduler.getTriggerGroupNames();

			//triggerGroup, triggerNameからトリガを特定
			for (String triggerGroupName : triggerGroupNames) {
				String[] triggerNames = scheduler.getTriggerNames(triggerGroupName);

				for (String triggerName : triggerNames) {

					log.debug("triggerGroupName / triggerName : " + triggerGroupName + " / " + triggerName);

					Trigger trigger = scheduler.getTrigger(triggerName, triggerGroupName);
					
					if (trigger != null) {
						log.debug("setStartTime : " + now.getTime());
						
						//トリガの開始時刻をセットして再登録する
						trigger.setStartTime(now);
						scheduler.rescheduleJob(triggerName, triggerGroupName, trigger);
					}
				}
			}
		} catch (SchedulerException e) {
			throw e;
		}
	}
}
