/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.sql.dialog;

import java.util.ArrayList;

import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableItem;

import com.clustercontrol.bean.YesNoConstant;
import com.clustercontrol.dialog.CommonDialog;
import com.clustercontrol.dialog.ValidateResult;
import com.clustercontrol.monitor.run.bean.MonitorInfo;
import com.clustercontrol.monitor.run.bean.MonitorTypeConstant;
import com.clustercontrol.sql.action.DeleteSql;
import com.clustercontrol.sql.action.GetSql;
import com.clustercontrol.sql.action.ModifySql;
import com.clustercontrol.sql.bean.SqlTabelDefine;
import com.clustercontrol.sql.composite.SqlListComposite;
import com.clustercontrol.sql.composite.SqlListComposite.SelectItem;
import com.clustercontrol.util.Messages;

/**
 * SQL監視一覧ダイアログクラス<BR>
 * 
 * @version 2.2.0
 * @since 2.0.0
 */
public class SqlListDialog extends CommonDialog {

	// 後でpackするためsizeXはダミーの値。
    private final int sizeX = 800;
    private final int sizeY = 440;
    
    // ----- instance フィールド ----- //

    /** SQL一覧 */
    private SqlListComposite sqlList = null;

    /** 追加ボタン（数値） */
    private Button buttonAddNumeric = null;
    
    /** 追加ボタン（文字列） */
    private Button buttonAddString = null;

    /** 変更ボタン */
    private Button buttonModify = null;

    /** 削除ボタン */
    private Button buttonDelete = null;

    /** 有効ボタン */
    private Button buttonValid = null;
    
    /** 無効ボタン */
    private Button buttonInvalid = null;

    // ----- コンストラクタ ----- //

    /**
     * ダイアログのインスタンスを返します。
     * 
     * @param parent
     *            親のシェルオブジェクト
     */
    public SqlListDialog(Shell parent) {
        super(parent);
        setShellStyle(getShellStyle() | SWT.RESIZE | SWT.MAX);
    }

    // ----- instance メソッド ----- //

    /**
     * ダイアログの初期サイズを返します。
     * 
     * @return 初期サイズ
     */
    @Override
	protected Point getInitialSize() {
        return new Point(sizeX, sizeY);
    }

    /**
     * ダイアログエリアを生成します。
     * 
     * @param parent
     *            親のインスタンス
     */
    @Override
	protected void customizeDialog(Composite parent) {
        Shell shell = this.getShell();

        // タイトル
        shell.setText(Messages.getString("dialog.sql.list"));

        // レイアウト
        GridLayout layout = new GridLayout(7, true);
        layout.marginWidth = 10;
        layout.marginHeight = 10;
        layout.numColumns = 7;
        parent.setLayout(layout);

        /*
         * SQL監視一覧
         */

        this.sqlList = new SqlListComposite(parent, SWT.BORDER);
        GridData gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.verticalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        gridData.grabExcessVerticalSpace = true;
        gridData.horizontalSpan = 6;
        this.sqlList.setLayoutData(gridData);

        /*
         * 操作ボタン
         */

        Composite composite = new Composite(parent, SWT.NONE);
        layout = new GridLayout(1, true);
        layout.numColumns = 1;
        composite.setLayout(layout);
        gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.verticalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        gridData.grabExcessVerticalSpace = true;
        gridData.horizontalSpan = 1;
        composite.setLayoutData(gridData);

        // 追加ボタン（数値）
        this.buttonAddNumeric = this.createButton(composite, Messages.getString("add.numeric"));
        this.buttonAddNumeric.addSelectionListener(new AddSelectionListener(MonitorTypeConstant.TYPE_NUMERIC));
        
        // 追加ボタン（文字列）
        @SuppressWarnings("unused") Label dummy = new Label(composite, SWT.NONE);
        this.buttonAddString = this.createButton(composite, Messages.getString("add.string"));
        this.buttonAddString.addSelectionListener(new AddSelectionListener(MonitorTypeConstant.TYPE_STRING));

        // 変更ボタン
        dummy = new Label(composite, SWT.NONE);
        this.buttonModify = this.createButton(composite, Messages.getString("modify"));
        this.buttonModify.addSelectionListener(new SelectionAdapter() {
            @Override
			public void widgetSelected(SelectionEvent e) {
            	
            	String monitorId = null;
            	int monitorType = Integer.MAX_VALUE;
            	
            	Table table = sqlList.getTableViewer().getTable();
            	TableItem[] item = table.getSelection();
            	
            	if(item != null && item.length>0){
            		try{
            			monitorId = (String)((ArrayList)item[0].getData()).get(SqlTabelDefine.MONITOR_ID);
            			monitorType = MonitorTypeConstant.stringToType((String)((ArrayList)item[0].getData()).get(SqlTabelDefine.MONITOR_TYPE));
            		}
            		catch(Exception ex){
            		}
            	}

                if (monitorId != null) {
                	SqlCreateDialog dialog = new SqlCreateDialog(getParentShell(), monitorId, monitorType);
                	if (dialog.open() == IDialogConstants.OK_ID) {
                		int selectIndex = table.getSelectionIndex();
                		sqlList.update();
                		table.setSelection(selectIndex);
                	}
                }
                else{
                	MessageDialog.openWarning(
                			null, 
                			Messages.getString("warning"), 
                			Messages.getString("message.monitor.1"));
                }
            }
        });

        // 削除ボタン
        dummy = new Label(composite, SWT.NONE);
        this.buttonDelete = this.createButton(composite, Messages
                .getString("delete"));
        this.buttonDelete.addSelectionListener(new SelectionAdapter() {
            @Override
			public void widgetSelected(SelectionEvent e) {
            	
            	String monitorId = null;
            	
            	Table table = sqlList.getTableViewer().getTable();
            	TableItem[] item = table.getSelection();
            	
            	if(item != null && item.length>0){
            		try{
            			monitorId = (String)((ArrayList)item[0].getData()).get(SqlTabelDefine.MONITOR_ID);
            		}
            		catch(Exception ex){
            		}
            	}

                if (monitorId != null) {
                	
                  String[] args = { monitorId };
                  if (MessageDialog.openConfirm(
              			null, 
              			Messages.getString("confirmed"), 
              			Messages.getString("message.monitor.39", args))) {
                	  
		        	  boolean result = new DeleteSql().delete(monitorId);
                      if(result){
                        	MessageDialog.openInformation(
                        			null, 
                        			Messages.getString("successful"), 
                        			Messages.getString("message.monitor.37", args));
                          
                          sqlList.update();
                      }
                      else{
                      	MessageDialog.openError(
                    			null, 
                    			Messages.getString("failed"), 
                    			Messages.getString("message.monitor.38", args));
                      }
		          }
                }
                else{
                	MessageDialog.openWarning(
                			null, 
                			Messages.getString("warning"), 
                			Messages.getString("message.monitor.1"));
                }
            }
        });
        
        // 有効ボタン
        dummy = new Label(composite, SWT.NONE);
        dummy = new Label(composite, SWT.NONE);
        this.buttonValid = this.createButton(composite, Messages
                .getString("valid"));
        this.buttonValid.addSelectionListener(new SelectionAdapter() {
            @Override
			public void widgetSelected(SelectionEvent e) {
            	//一括で有効に変更
            	setValid(YesNoConstant.TYPE_YES);
            }
        });
        
        // 無効ボタン
        this.buttonInvalid = this.createButton(composite, Messages
                .getString("invalid"));
        this.buttonInvalid.addSelectionListener(new SelectionAdapter() {
            @Override
			public void widgetSelected(SelectionEvent e) {
            	//一括で無効に変更
            	setValid(YesNoConstant.TYPE_NO);
            }
        });

        // ラインを引く
        Label line = new Label(parent, SWT.SEPARATOR | SWT.HORIZONTAL);
        gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        gridData.horizontalSpan = 8;
        line.setLayoutData(gridData);

        // 画面中央に
        Display display = shell.getDisplay();
        shell.setLocation((display.getBounds().width - shell.getSize().x) / 2,
                (display.getBounds().height - shell.getSize().y) / 2);
        
        //ダイアログのサイズ調整（pack:resize to be its preferred size）
        shell.pack();
        shell.setSize(new Point(sizeX, sizeY));
    }

    /**
     * 入力値チェックをします。
     * 
     * @return 検証結果
     */
    @Override
	protected ValidateResult validate() {
        return super.validate();
    }

    /**
     * 共通のボタンを生成します。
     * 
     * @param parent
     *            親のコンポジット
     * @param label
     *            ボタンのラベル
     * @return 生成されたボタン
     */
    private Button createButton(Composite parent, String label) {
        Button button = new Button(parent, SWT.NONE);

        GridData gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        button.setLayoutData(gridData);

        button.setText(label);

        return button;
    }
    
    /**
     * 閉じるボタンを作成します。
     * 
     * @param parent ボタンバーコンポジット
     */
    @Override
	protected void createButtonsForButtonBar(Composite parent) {
    	
    	// 閉じるボタン
    	this.createButton(parent, IDialogConstants.CANCEL_ID, Messages.getString("close"), false);
    }
    
    /**
     * 追加ボタンSelectionAdapterクラス
     * 
     * @version 2.1.0
     * @since 2.1.0
     */
    class AddSelectionListener extends SelectionAdapter {
    	
    	/** 監視判定タイプ */
    	private int m_monitorType;

		/**
		 * コンストラクタ
		 * @param monitorType 監視判定タイプ
		 */
		public AddSelectionListener(int monitorType) {
			m_monitorType = monitorType;
		}

		/* (非 Javadoc)
		 * @see org.eclipse.swt.events.SelectionAdapter#widgetSelected(org.eclipse.swt.events.SelectionEvent)
		 */
		@Override
		public void widgetSelected(SelectionEvent e) {
			
			SqlCreateDialog dialog = new SqlCreateDialog(getParentShell(), m_monitorType);
			if (dialog.open() == IDialogConstants.OK_ID) {
				sqlList.update();
			}
		}
    }
    
    /**
     * 有効・無効変更処理
     * 
     * @param valid
     */
    public void setValid(int valid) {

		//選択された監視項目IDを取得
    	ArrayList<SelectItem> list = sqlList.getSelectionData();
		if (list != null && list.size() > 0) {

			StringBuilder monitorIds = new StringBuilder();
			for(int i = 0; i < list.size(); i++){
				if(i > 0){
					monitorIds.append(", ");
				}
				SelectItem item = list.get(i);
				monitorIds.append(item.getMonitorId());
			}

            String[] confirmArgs = { monitorIds.toString() };
            String message;
            if(valid == YesNoConstant.TYPE_YES)
            	message = Messages.getString("message.monitor.47",confirmArgs);
            else
            	message = Messages.getString("message.monitor.48",confirmArgs);
            if (!MessageDialog.openConfirm(
        			null, 
        			Messages.getString("confirmed"), 
        			message)) {
            	return;
            }
			
			for(int i = 0; i < list.size(); i++){
				SelectItem item = list.get(i);
				
				if(item != null && !item.getMonitorId().equals("")){
			        //監視情報を取得
			        MonitorInfo info = new GetSql().getSql(item.getMonitorId());
			        
			        //有効・無効を設定
			        info.setValid(valid);
			        
			        //監視情報を更新
			        boolean result = new ModifySql().modify(info);
                    if(!result){
                        String[] args = { item.getMonitorId() };
                    	MessageDialog.openError(
                    			null, 
                    			Messages.getString("failed"), 
                    			Messages.getString("message.monitor..36", args));
                    }
				}
			}
    			
            int selectIndex = sqlList.getTableViewer().getTable().getSelectionIndex();
            sqlList.update();
            sqlList.getTableViewer().getTable().setSelection(selectIndex);
		}
		else{
        	MessageDialog.openWarning(
        			null, 
        			Messages.getString("warning"), 
        			Messages.getString("message.monitor.1"));
        }
    }
}