/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.maintenance.dialog;

import java.util.ArrayList;
import java.util.Date;

import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.MessageBox;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableItem;

import com.clustercontrol.bean.Schedule;
import com.clustercontrol.bean.YesNoConstant;
import com.clustercontrol.dialog.CommonDialog;
import com.clustercontrol.dialog.ValidateResult;
import com.clustercontrol.maintenance.action.DeleteMaintenance;
import com.clustercontrol.maintenance.action.GetMaintenance;
import com.clustercontrol.maintenance.action.GetSchedule;
import com.clustercontrol.maintenance.action.ModifyMaintenance;
import com.clustercontrol.maintenance.bean.MaintenanceInfoDataImp;
import com.clustercontrol.maintenance.bean.MaintenanceTableDefine;
import com.clustercontrol.maintenance.bean.ScheduleInfo;
import com.clustercontrol.maintenance.bean.ScheduleTableDefine;
import com.clustercontrol.maintenance.composite.MaintenanceListComposite;
import com.clustercontrol.util.Messages;

/**
 * メンテナンス機能一覧ダイアログクラスです。
 * 
 * @version 3.0.0
 * @since 2.0.0
 */
public class MaintenanceListDialog extends CommonDialog {

    // 後でpackするためsizeXはダミーの値。
    private final int sizeX = 800;
    private final int sizeY = 390;
	
    // ----- instance フィールド ----- //

    /** メンテナンス一覧 */
//    private NotifyListComposite notifyList = null;
	private MaintenanceListComposite maintenanceList = null;

    /** 追加ボタン */
    private Button buttonAdd = null;

    /** 変更ボタン */
    private Button buttonModify = null;

    /** 削除ボタン */
    private Button buttonDelete = null;
    
    /** 有効ボタン */
    private Button buttonValid = null;
    
    /** 無効ボタン */
    private Button buttonInvalid = null;


    // ----- コンストラクタ ----- //

    /**
     * ダイアログのインスタンスを返します。
     * 
     * @param parent
     *            親のシェルオブジェクト
     */
    public MaintenanceListDialog(Shell parent) {
        super(parent);
        setShellStyle(getShellStyle() | SWT.RESIZE | SWT.MAX);
    }

    // ----- instance メソッド ----- //

    /**
     * ダイアログの初期サイズを返します。
     * 
     * @return 初期サイズ
     */
    @Override
	protected Point getInitialSize() {
        return new Point(sizeX, sizeY);
    }

    /**
     * ダイアログエリアを生成します。
     * 
     * @param parent
     *            親のインスタンス
     */
    @Override
	protected void customizeDialog(Composite parent) {
        Shell shell = this.getShell();

        // タイトル
        shell.setText(Messages.getString("dialog.maintenance.list"));

        // レイアウト
        GridLayout layout = new GridLayout(8, true);
        layout.marginWidth = 10;
        layout.marginHeight = 10;
        layout.numColumns = 8;
        parent.setLayout(layout);

        /*
         * メンテナンス一覧
         */

        this.maintenanceList = new MaintenanceListComposite(parent, SWT.BORDER);
        GridData gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.verticalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        gridData.grabExcessVerticalSpace = true;
        gridData.horizontalSpan = 7;
        this.maintenanceList.setLayoutData(gridData);

        /*
         * 操作ボタン
         */

        Composite composite = new Composite(parent, SWT.NONE);
        layout = new GridLayout(1, true);
        layout.numColumns = 1;
        composite.setLayout(layout);
        gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.verticalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        gridData.grabExcessVerticalSpace = true;
        gridData.horizontalSpan = 1;
        composite.setLayoutData(gridData);

        // 追加ボタン
        this.buttonAdd = this
                .createButton(composite, Messages.getString("add"));
        this.buttonAdd.addSelectionListener(new SelectionAdapter() {
            @Override
			public void widgetSelected(SelectionEvent e) {
            	
            	//作成ダイアログを開く
            	
            	MaintenanceDialog dialog = new MaintenanceDialog(getParentShell());

            	if (dialog.open() == IDialogConstants.OK_ID) {
            		maintenanceList.update();
            	}
            	
            }
        });

        // 変更ボタン
        @SuppressWarnings("unused") Label dummy = new Label(composite, SWT.NONE);
        this.buttonModify = this.createButton(composite, Messages.getString("modify"));
        this.buttonModify.addSelectionListener(new SelectionAdapter() {
            @Override
			public void widgetSelected(SelectionEvent e) {
            	
            	String maintenanceId = null;
            	
            	Table table = maintenanceList.getTableViewer().getTable();
            	TableItem[] item = table.getSelection();
            	
            	if(item != null && item.length>0){
            		try{
            			maintenanceId = (String)((ArrayList)item[0].getData()).get(MaintenanceTableDefine.MAINTENANCE_ID);
            		}
            		catch(Exception ex){
            		}
            	}

                if (maintenanceId != null) {
                	
                	//ダイアログを開く
                	MaintenanceDialog dialog = new MaintenanceDialog(getParentShell(), maintenanceId, false);
                	//scheduleを取得してsetする
//                	ArrayList scheduleList = new GetSchedule().getScheduleList();
                	ArrayList scheduleList = new GetSchedule().getSchedule(maintenanceId);

                	dialog.setSchedule(scheduleList);
                	
                    if (dialog.open() == IDialogConstants.OK_ID) {
                        int selectIndex = table.getSelectionIndex();
                        maintenanceList.update();
                        table.setSelection(selectIndex);
                    }
                }
                else{
                    MessageBox box = new MessageBox(getShell(), SWT.OK | SWT.ICON_WARNING);
                    box.setText(Messages.getString("warning"));
                    box.setMessage(Messages.getString("message.maintenance.8"));
                    box.open();
                  }
            }
        });

        // 削除ボタン
        dummy = new Label(composite, SWT.NONE);
        this.buttonDelete = this.createButton(composite, Messages
                .getString("delete"));
        this.buttonDelete.addSelectionListener(new SelectionAdapter() {
            @Override
			public void widgetSelected(SelectionEvent e) {
            	
            	//削除処理
              	String maintenanceId = null;
//              	String scheduleId = null;
            	
            	Table table = maintenanceList.getTableViewer().getTable();
            	TableItem[] item = table.getSelection();
            	
            	if(item != null && item.length>0){
            		try{
            			maintenanceId = (String)((ArrayList)item[0].getData()).get(MaintenanceTableDefine.MAINTENANCE_ID);
//            			scheduleId = (String)((ArrayList)item[0].getData()).get(MaintenanceTableDefine.SCHEDULE);
            		}
            		catch(Exception ex){
            		}
            	}

                if (maintenanceId != null) {
                	
                  MessageBox confirmBox = new MessageBox(getShell(), SWT.OK | SWT.CANCEL | SWT.ICON_WARNING);
                  confirmBox.setText(Messages.getString("confirmed"));
                  String[] args = { maintenanceId };
                  String message = Messages.getString("message.maintenance.7",args);
                  confirmBox.setMessage(message);

		          if (confirmBox.open() == SWT.OK) {
		        	  boolean result = new DeleteMaintenance().delete(maintenanceId);
                      if(result){
                          MessageBox box = new MessageBox(getShell(), SWT.OK | SWT.ICON_INFORMATION);
                          box.setText(Messages.getString("successful"));
                          box.setMessage(Messages.getString("message.maintenance.5", args));
                          box.open();
                          
                          maintenanceList.update();
                      }
                      else{
                          MessageBox box = new MessageBox(getShell(), SWT.OK | SWT.ICON_ERROR);
                          box.setText(Messages.getString("failed"));
                          box.setMessage(Messages.getString("message.maintenance.6", args));
                          box.open();
                      }
		          }
                }
                else{
                  MessageBox box = new MessageBox(getShell(), SWT.OK | SWT.ICON_WARNING);
                  box.setText(Messages.getString("warning"));
                  box.setMessage(Messages.getString("message.maintenance.8"));
                  box.open();
                }
            }
        });
        
        // 有効ボタン
        dummy = new Label(composite, SWT.NONE);
        dummy = new Label(composite, SWT.NONE);
        this.buttonValid = this.createButton(composite, Messages
                .getString("valid"));
        this.buttonValid.addSelectionListener(new SelectionAdapter() {
            @Override
			public void widgetSelected(SelectionEvent e) {
            	//一括で有効に変更
            	setValid(YesNoConstant.TYPE_YES);
            }
        });
        
        // 無効ボタン
        this.buttonInvalid = this.createButton(composite, Messages
                .getString("invalid"));
        this.buttonInvalid.addSelectionListener(new SelectionAdapter() {
            @Override
			public void widgetSelected(SelectionEvent e) {
            	//一括で無効に変更
            	setValid(YesNoConstant.TYPE_NO);
            }
        });

        // ラインを引く
        Label line = new Label(parent, SWT.SEPARATOR | SWT.HORIZONTAL);
        gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        gridData.horizontalSpan = 8;
        line.setLayoutData(gridData);

        // 画面中央に
        Display display = shell.getDisplay();
        shell.setLocation((display.getBounds().width - shell.getSize().x) / 2,
                (display.getBounds().height - shell.getSize().y) / 2);
        
        //ダイアログのサイズ調整（pack:resize to be its preferred size）
        shell.pack();
        shell.setSize(new Point(sizeX, sizeY));
    }

    /**
     * 入力値チェックをします。
     * 
     * @return 検証結果
     */
    @Override
	protected ValidateResult validate() {
        return super.validate();
    }

    /**
     * 共通のボタンを生成します。
     * 
     * @param parent
     *            親のコンポジット
     * @param label
     *            ボタンのラベル
     * @return 生成されたボタン
     */
    private Button createButton(Composite parent, String label) {
        Button button = new Button(parent, SWT.NONE);

        GridData gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        button.setLayoutData(gridData);

        button.setText(label);

        return button;
    }
    
    /**
     * 閉じるボタンを作成します。
     * 
     * @param parent ボタンバーコンポジット
     */
    @Override
	protected void createButtonsForButtonBar(Composite parent) {
    	
        // 閉じるボタン
        this.createButton(parent, IDialogConstants.CANCEL_ID, IDialogConstants.CLOSE_LABEL, false);
    }

    
    /**
     * 有効・無効変更処理
     * 
     * @param valid
     */
    public void setValid(int valid) {

		//選択されたメンテナンスIDを取得
		ArrayList<String> list = maintenanceList.getSelectionData();
		if (list != null && list.size() > 0) {
			
			StringBuilder maintenanceIds = new StringBuilder();
			for(int i = 0; i < list.size(); i++){
				if(i > 0){
					maintenanceIds.append(", ");
				}
				maintenanceIds.append(list.get(i));
			}

            String[] confirmArgs = { maintenanceIds.toString() };
            String message;
            if(valid == YesNoConstant.TYPE_YES)
            	message = Messages.getString("message.maintenance.14",confirmArgs);
            else
            	message = Messages.getString("message.maintenance.15",confirmArgs);
            
            if (!MessageDialog.openConfirm(
        			null, 
        			Messages.getString("confirmed"), 
        			message)) {
            	return;
            }
			
			for(int i = 0; i < list.size(); i++){
				String maintenanceId = list.get(i);
				
				if(maintenanceId != null && !maintenanceId.equals("")){
			        //メンテナンス情報を取得
					MaintenanceInfoDataImp info = new GetMaintenance().getMaintenanceInfo(maintenanceId);

					ArrayList schedule = new GetSchedule().getSchedule(maintenanceId);
					ScheduleInfo scheduleInfo = new ScheduleInfo();
					
					scheduleInfo.setId(maintenanceId);
					scheduleInfo.setName((String)schedule.get(ScheduleTableDefine.SCHE_NAME));
					scheduleInfo.setJobId((String)schedule.get(ScheduleTableDefine.JOB_ID));
					scheduleInfo.setJobName((String)schedule.get(ScheduleTableDefine.JOB_NAME));
					scheduleInfo.setCalendarId((String)schedule.get(ScheduleTableDefine.CALENDAR_ID));
					scheduleInfo.setSchedule((Schedule)schedule.get(ScheduleTableDefine.SCHEDULE));
					scheduleInfo.setValid(valid);
					scheduleInfo.setCreateUser((String)schedule.get(ScheduleTableDefine.CREATE_USER));
					scheduleInfo.setCreateTime((Date)schedule.get(ScheduleTableDefine.CREATE_TIME));
					scheduleInfo.setUpdateUser((String)schedule.get(ScheduleTableDefine.UPDATE_USER));
					scheduleInfo.setUpdateTime((Date)schedule.get(ScheduleTableDefine.UPDATE_TIME));
			        
			        //有効・無効を設定
			        info.setValid_flg(valid);
			        
			        //メンテナンス情報を更新
			        boolean result = new ModifyMaintenance().modify(info, scheduleInfo);
                    if(!result){
                        String[] args = { maintenanceId };
                    	MessageDialog.openError(
                    			null, 
                    			Messages.getString("failed"), 
                    			Messages.getString("message.maintenance.4", args));
                    }
				}
			}
    			
            int selectIndex = maintenanceList.getTableViewer().getTable().getSelectionIndex();
            maintenanceList.update();
            maintenanceList.getTableViewer().getTable().setSelection(selectIndex);
		}
		else{
        	MessageDialog.openWarning(
        			null, 
        			Messages.getString("warning"), 
        			Messages.getString("message.maintenance.9"));
        }
    }
}