/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License 
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied 
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 PURPOSE.  See the GNU General Public License for more details.
 
 */

package com.clustercontrol.logtransfer.dialog;

import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.PlatformUI;

import com.clustercontrol.ClusterControlPlugin;
import com.clustercontrol.bean.DataRangeConstant;
import com.clustercontrol.bean.FacilityConstant;
import com.clustercontrol.bean.FacilityPath;
import com.clustercontrol.bean.FacilityTreeItem;
import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.composite.StringVerifyListener;
import com.clustercontrol.dialog.CommonDialog;
import com.clustercontrol.dialog.ScopeTreeDialog;
import com.clustercontrol.dialog.ValidateResult;
import com.clustercontrol.logtransfer.action.AddLogTransfer;
import com.clustercontrol.logtransfer.action.GetLogTransfer;
import com.clustercontrol.logtransfer.action.ModifyLogTransfer;
import com.clustercontrol.logtransfer.bean.LogTransferInfo;
import com.clustercontrol.logtransfer.composite.LogTransferFileListComposite;
import com.clustercontrol.repository.bean.FacilityInfo;
import com.clustercontrol.util.Messages;

/**
 * ログ転送作成・変更ダイアログクラス<BR>
 * 
 * @version 2.2.0
 * @since 2.1.0
 */
public class LogTransferCreateDialog extends CommonDialog {
	
	public static final int WIDTH_TITLE = 3;
	public static final int WIDTH_VALUE = 8;
	
	
	// ----- instance フィールド ----- //
	
	/** 入力値を保持するオブジェクト */
	private LogTransferInfo m_inputData = null;
	
	/** 入力値の正当性を保持するオブジェクト */
	private ValidateResult m_validateResult = null;
	
	/** 変更対象の転送設定ID */
	private String m_transferId = null;
	
	/** 転送設定ID */
	private Text m_textTransferId = null;
	
	/** 説明 */
	private Text m_textDescription = null;
	
	/** スコープ */
	protected Text m_textScope = null;
	
	/** スコープ参照ボタン */
	protected Button m_buttonScope = null;
	
	/** 選択されたスコープのファシリティID */
	protected String m_facilityId = null;
	
	/** 転送対象ログファイル一覧情報 */
    LogTransferFileListComposite m_fileList = null;
	
	/** この設定を有効にする */
	private Button m_valid = null;
	
	
	// ----- コンストラクタ ----- //
	
	/**
	 * 作成用ダイアログのインスタンスを返します。
	 * 
	 * @param parent 親のシェルオブジェクト
	 */
	public LogTransferCreateDialog(Shell parent) {
		super(parent);
		setShellStyle(getShellStyle() | SWT.RESIZE | SWT.MAX);
	}
	
	/**
	 * 変更用ダイアログのインスタンスを返します。
	 * 
	 * @param parent 親のシェルオブジェクト
	 * @param transferId 変更する転送設定ID
	 */
	public LogTransferCreateDialog(Shell parent, String transferId) {
		super(parent);
		setShellStyle(getShellStyle() | SWT.RESIZE | SWT.MAX);
		
		this.m_transferId = transferId;
	}
	
	/**
	 * ダイアログの初期サイズを返します。
	 * 
	 * @return 初期サイズ
	 */
	@Override
	protected Point getInitialSize() {
		return new Point(700, 500);
	}
	
	// ----- instance メソッド ----- //
	
	/**
	 * ダイアログエリアを生成します。
	 * 
	 * @param parent 親のインスタンス
	 */
	@Override
	protected void customizeDialog(Composite parent) {
		Shell shell = this.getShell();
		
		// タイトル
		shell.setText(Messages.getString("dialog.logtransfer.create.modify"));
		
		// 変数として利用されるラベル
		Label label = null;
		// 変数として利用されるグリッドデータ
		GridData gridData = null;
		
		// レイアウト
		GridLayout layout = new GridLayout(1, true);
		layout.marginWidth = 10;
		layout.marginHeight = 10;
		layout.numColumns = 15;
		parent.setLayout(layout);
		
		/*
		 * 転送設定ID
		 */
		// ラベル
		label = new Label(parent, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_TITLE;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(Messages.getString("logtransfer.id") + " : ");
		// テキスト
		this.m_textTransferId = new Text(parent, SWT.BORDER | SWT.LEFT);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_VALUE;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		this.m_textTransferId.setLayoutData(gridData);
		if(this.m_transferId != null && !"".equals(this.m_transferId.trim())){
			this.m_textTransferId.setEnabled(false);
		}
		this.m_textTransferId.addVerifyListener(
				new StringVerifyListener(DataRangeConstant.VARCHAR_64));
		
		// 空白
		label = new Label(parent, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 4;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		
		/*
		 * 説明
		 */
		// ラベル
		label = new Label(parent, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_TITLE;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(Messages.getString("description") + " : ");
		// テキスト
		this.m_textDescription = new Text(parent, SWT.BORDER | SWT.LEFT);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_VALUE;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		this.m_textDescription.setLayoutData(gridData);
		this.m_textDescription.addVerifyListener(
				new StringVerifyListener(DataRangeConstant.VARCHAR_256));
		
		// 空白
		label = new Label(parent, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 4;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		
		/*
		 * スコープ
		 */
		// ラベル
		label = new Label(parent, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = WIDTH_TITLE;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		label.setText(Messages.getString("scope") + " : ");
		// テキスト
		this.m_textScope = new Text(parent, SWT.BORDER | SWT.LEFT | SWT.READ_ONLY);
		gridData = new GridData();
		gridData.horizontalSpan = 6;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		this.m_textScope.setLayoutData(gridData);
		// 参照ボタン
		m_buttonScope = new Button(parent, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 2;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		m_buttonScope.setLayoutData(gridData);
		m_buttonScope.setText(Messages.getString("refer"));
		m_buttonScope.addSelectionListener(new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				// シェルを取得
				Shell shell = PlatformUI.getWorkbench().getActiveWorkbenchWindow().getShell();
				
				ScopeTreeDialog dialog = new ScopeTreeDialog(shell);
				if (dialog.open() == IDialogConstants.OK_ID) {
					FacilityTreeItem item = dialog.getSelectItem();
					FacilityInfo info = item.getData();
					m_facilityId = info.getFacilityId();
					if (info.getFacilityType() == FacilityConstant.TYPE_NODE) {
						m_textScope.setText(info.getFacilityName());
					} else {
						FacilityPath path = new FacilityPath(
								ClusterControlPlugin.getDefault()
								.getSeparator());
						m_textScope.setText(path.getPath(item));
					}
				}
			}
		});
		// 空白
		label = new Label(parent, SWT.NONE);
		gridData = new GridData();
		gridData.horizontalSpan = 4;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		label.setLayoutData(gridData);
		
		// グループ
		Composite composite = new Composite(parent, SWT.NONE);
		layout = new GridLayout(1, true);
		layout.marginWidth = 0;
		layout.marginHeight = 0;
		layout.numColumns = 15;
		composite.setLayout(layout);
		gridData = new GridData();
		gridData.horizontalSpan = 15;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.verticalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.grabExcessVerticalSpace = true;
		composite.setLayoutData(gridData);
		
		/*
		 * 転送対象ログファイル一覧
		 */
        m_fileList = new LogTransferFileListComposite(composite, SWT.NONE);
        gridData = new GridData();
		gridData.horizontalSpan = 15;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.verticalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		gridData.grabExcessVerticalSpace = true;
		m_fileList.setLayoutData(gridData);
		
		/*
		 * 有効／無効
		 */
		this.m_valid = new Button(parent, SWT.CHECK);
		gridData = new GridData();
		gridData.horizontalSpan = 15;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		this.m_valid.setLayoutData(gridData);
		this.m_valid.setText(Messages.getString("setting.valid.confirmed"));
		
		// ラインを引く
		Label line = new Label(parent, SWT.SEPARATOR | SWT.HORIZONTAL);
		gridData = new GridData();
		gridData.horizontalSpan = 15;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		line.setLayoutData(gridData);
		
		// 画面中央に
		Display display = shell.getDisplay();
		shell.setLocation((display.getBounds().width - shell.getSize().x) / 2,
				(display.getBounds().height - shell.getSize().y) / 2);
		
		// 転送設定IDが指定されている場合、その情報を初期表示する。
		LogTransferInfo info = null; 
		if(this.m_transferId != null){
			info = new GetLogTransfer().getLogTransfer(this.m_transferId);        	
		}
		else{
			info = new LogTransferInfo();
		}
		this.setInputData(info);
		
	}
	
	/**
	 * 入力内容を返します。
	 * 
	 * @return 入力内容を保持したログ転送情報
	 */
	public LogTransferInfo getInputData() {
		return this.m_inputData;
	}
	
	/**
	 * 各項目に入力値を設定します。
	 * 
	 * @param info 設定値として用いるログ転送情報
	 */
	protected void setInputData(LogTransferInfo info) {
		
		this.m_inputData = info;
		
		// 各項目に反映
		if (info.getTransferId() != null) {
			this.m_textTransferId.setText(info.getTransferId());
			this.m_textTransferId.setEnabled(false);
		}
		if (info.getDescription() != null) {
			this.m_textDescription.setText(info.getDescription());
		}
		if (info.getScope() != null) {
			this.m_textScope.setText(info.getScope());
		}
		if (info.getFacilityId() != null) {
			this.m_facilityId = info.getFacilityId();
		}
		if (info.getFileInfo() != null) {
			this.m_fileList.setInputData(info);
		}
		
		// 有効／無効
		if (info.getValid() == ValidConstant.TYPE_VALID) {
			this.m_valid.setSelection(true);
		}
	}
	
	/**
	 * 入力値を用いてログ転送情報を生成します。
	 * 
	 * @return 入力値を保持したログ転送情報
	 */
	protected LogTransferInfo createInputData() {
		LogTransferInfo info = new LogTransferInfo();
		
		if (this.m_textTransferId.getText() != null
				&& !"".equals((this.m_textTransferId.getText()).trim())) {
			info.setTransferId(this.m_textTransferId.getText());
		} else {
			this.setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.logtransfer.8"));
			return null;
		}
		if (this.m_textDescription.getText() != null
				&& !"".equals((this.m_textDescription.getText()).trim())) {
			info.setDescription(this.m_textDescription.getText());
		}
		if (this.m_textScope.getText() != null
				&& !"".equals((this.m_textScope.getText()).trim())) {
			info.setFacilityId(this.m_facilityId);
		}
		else{
			this.setValidateResult(Messages.getString("message.hinemos.1"), Messages.getString("message.logtransfer.10"));
			return null;
		}
		
		this.m_validateResult = this.m_fileList.createInputData(info);
		if(this.m_validateResult != null){
			return null;
		}
		
		// 有効/無効
		if (this.m_valid.getSelection()) {
			info.setValid(ValidConstant.TYPE_VALID);
		} else {
			info.setValid(ValidConstant.TYPE_INVALID);
		}
		
		return info;
	}
	
	/**
	 * 入力値チェックをします。
	 * 
	 * @return 検証結果
	 */
	@Override
	protected ValidateResult validate() {
		// 入力値生成
		this.m_inputData = this.createInputData();
		
		if (this.m_inputData != null) {
			return super.validate();
		} else {
			return m_validateResult;
		}
	}
	
    /**
     * 入力値をマネージャに登録します。
     * 
     * @return true：正常、false：異常
     * 
     * @see com.clustercontrol.dialog.CommonDialog#action()
     */
    @Override
	protected boolean action() {
    	boolean result = false;
    	
    	LogTransferInfo info = this.getInputData();
    	if(info != null){
            if(this.m_transferId == null){
            	// 作成の場合
            	result = new AddLogTransfer().add(info);
            }
            else{
            	// 変更の場合
            	result = new ModifyLogTransfer().modify(info);
            }
    	}
    	
    	return result;
    }
	
	/**
	 * ＯＫボタンのテキストを返します。
	 * 
	 * @return ＯＫボタンのテキスト
	 */
	@Override
	protected String getOkButtonText() {
		return Messages.getString("ok");
	}
	
	/**
	 * キャンセルボタンのテキストを返します。
	 * 
	 * @return キャンセルボタンのテキスト
	 */
	@Override
	protected String getCancelButtonText() {
		return Messages.getString("cancel");
	}
	
	/**
	 * 無効な入力値の情報を設定します
	 *  
	 */
	private void setValidateResult(String id, String message) {
		
		this.m_validateResult = new ValidateResult();
		this.m_validateResult.setValid(false);
		this.m_validateResult.setID(id);
		this.m_validateResult.setMessage(message);
	}
}