/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.jobmanagement.dialog;

import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Shell;

import com.clustercontrol.bean.JobConstant;
import com.clustercontrol.dialog.CommonDialog;
import com.clustercontrol.dialog.ValidateResult;
import com.clustercontrol.jobmanagement.bean.JobTreeItem;
import com.clustercontrol.jobmanagement.composite.JobTreeComposite;
import com.clustercontrol.util.Messages;

/**
 * ジョブ選択ダイアログクラスです。
 * 
 * @version 1.0.0
 * @since 1.0.0
 */
public class JobTreeDialog extends CommonDialog {

    /** ジョブツリー用のコンポジット */
    private JobTreeComposite treeComposite = null;
    /** ツリーのみフラグ */
    private boolean m_treeOnly = false;
    /** 親ジョブID */
    private String m_parentJobId = null;
    /** ジョブID */
    private String m_jobId = null;
    /** ジョブユニットのジョブID */
    private String m_jobunitId = null;
    /**
     * コンストラクタ
     * 
     * @param parent 親シェル
     */
    public JobTreeDialog(Shell parent) {
        super(parent);

        m_treeOnly = false;
    }

    /**
     * コンストラクタ
     * 
     * @param parent 親シェル
     * @param treeOnly true：ツリーのみ、false：ジョブ情報を含む
     */
    public JobTreeDialog(Shell parent, boolean treeOnly) {
        super(parent);

        m_treeOnly = treeOnly;
    }

    /**
     * コンストラクタ
     * 
     * @param parent 親シェル
     * @param parentJobId 親ジョブID
     * @param jobId ジョブID
     */
    public JobTreeDialog(Shell parent, String parentJobId, String jobId, String jobunitId) {
        super(parent);

        m_treeOnly = true;
        m_parentJobId = parentJobId;
        m_jobId = jobId;
        m_jobunitId = jobunitId;
    }
    
    /**
     * ダイアログの初期サイズを返します。
     * 
     * @return 初期サイズ
     * 
     * @see org.eclipse.jface.window.Window#getInitialSize()
     */
    @Override
	protected Point getInitialSize() {
        return new Point(400, 400);
    }

    /**
     * ダイアログエリアを生成します。
     * 
     * @param parent 親コンポジット
     */
    @Override
	protected void customizeDialog(Composite parent) {
        // タイトル
        parent.getShell().setText(Messages.getString("select.job"));

        GridLayout layout = new GridLayout(5, true);
        parent.setLayout(layout);
        layout.marginHeight = 0;
        layout.marginWidth = 0;

        if (m_jobId == null) {
            treeComposite = new JobTreeComposite(parent, SWT.NONE, m_treeOnly);
        } else {
            treeComposite = new JobTreeComposite(parent, SWT.NONE,
                    m_parentJobId, m_jobId, m_jobunitId);
        }

        GridData gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.verticalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        gridData.grabExcessVerticalSpace = true;
        gridData.horizontalSpan = 5;
        treeComposite.setLayoutData(gridData);

        // アイテムをダブルクリックした場合、それを選択したこととする。
        treeComposite.getTreeViewer().addDoubleClickListener(
                new IDoubleClickListener() {
                    public void doubleClick(DoubleClickEvent event) {
                        okPressed();
                    }
                });
    }

    /**
     * 選択されたジョブツリーアイテムを返します。
     * 
     * @return ジョブツリーアイテム
     */
    public JobTreeItem getSelectItem() {
        return this.treeComposite.getSelectItem();
    }

	/**
	 * 入力値チェックをします。
	 * 
	 * @return 検証結果
	 * 
	 * @see com.clustercontrol.dialog.CommonDialog#validate()
	 */
    @Override
	protected ValidateResult validate() {
        ValidateResult result = null;

        JobTreeItem item = this.getSelectItem();
        if (item instanceof JobTreeItem) {
            if (item.getData().getType() == JobConstant.TYPE_COMPOSITE) {
                result = new ValidateResult();
                result.setValid(false);
                result.setID(Messages.getString("message.hinemos.1"));
                result.setMessage(Messages.getString("message.job.1"));
            }
        } else {
            result = new ValidateResult();
            result.setValid(false);
            result.setID(Messages.getString("message.hinemos.1"));
            result.setMessage(Messages.getString("message.job.1"));
        }

        return result;
    }
    
    /**
     * ＯＫボタンのテキストを返します。
     * 
     * @return ＯＫボタンのテキスト
     */
    @Override
	protected String getOkButtonText() {
        return Messages.getString("ok");
    }
    
    /**
     * キャンセルボタンのテキストを返します。
     * 
     * @return キャンセルボタンのテキスト
     */
    @Override
	protected String getCancelButtonText() {
        return Messages.getString("cancel");
    }
}