/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.jobmanagement.composite;

import java.util.ArrayList;

import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Table;

import com.clustercontrol.jobmanagement.action.GetRunScheduleList;
import com.clustercontrol.jobmanagement.action.GetRunScheduleListTableDefine;
import com.clustercontrol.jobmanagement.bean.RunScheduleListTableDefine;
import com.clustercontrol.jobmanagement.composite.action.RunScheduleSelectionChangedListener;
import com.clustercontrol.jobmanagement.composite.action.VerticalBarSelectionListener;
import com.clustercontrol.util.Messages;

/**
 * ジョブ[実行予定]ビューの左側用のコンポジットクラスです。
 * 
 * @version 2.0.0
 * @since 2.0.0
 */
public class RunScheduleListComposite extends Composite {
	/** テーブルビューア */
    protected RunScheduleTableViewer m_viewer = null;
    /** テーブル */
    protected Table m_table = null;
    /** 件数用ラベル */
    protected Label m_labelCount = null;
    /** スケジュールID */
    protected String m_scheduleId = null;

    /**
     * コンストラクタ
     * 
     * @param parent 親のコンポジット
     * @param style スタイル
     * 
     * @see org.eclipse.swt.SWT
     * @see org.eclipse.swt.widgets.Composite#Composite(Composite parent, int style)
     * @see #initialize()
     */
    public RunScheduleListComposite(Composite parent, int style) {
        super(parent, style);
        initialize();
    }

    /**
     * コンポジットを構築します。
     */
    private void initialize() {
        GridLayout layout = new GridLayout(1, true);
        this.setLayout(layout);
        layout.marginHeight = 0;
        layout.marginWidth = 0;
        
		//カレンダ一覧テーブル作成
        m_table = new Table(this, SWT.H_SCROLL | SWT.V_SCROLL
                | SWT.FULL_SELECTION | SWT.SINGLE | SWT.BORDER);
        m_table.setHeaderVisible(true);
        m_table.setLinesVisible(true);
        
        GridData gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.verticalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        gridData.grabExcessVerticalSpace = true;
        m_table.setLayoutData(gridData);
        
        m_labelCount = new Label(this, SWT.RIGHT);
        gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.verticalAlignment = GridData.FILL;
        m_labelCount.setLayoutData(gridData);

        m_viewer = new RunScheduleTableViewer(m_table);
        m_viewer.createTableColumn(GetRunScheduleListTableDefine.get(),
        		RunScheduleListTableDefine.SORT_COLUMN_INDEX,
        		RunScheduleListTableDefine.SORT_ORDER);

        m_viewer.addSelectionChangedListener(new RunScheduleSelectionChangedListener(this));
    
        m_viewer.getTable().getVerticalBar().addSelectionListener(
        		new VerticalBarSelectionListener(this));
    }

    /**
     * テーブルビューアーを更新します。<BR>
     * 実行予定一覧情報を取得し、テーブルビューアーにセットします。
     * <p>
	 * <ol>
	 * <li>実行予定一覧情報を取得します。</li>
	 * <li>テーブルビューアーに実行予定一覧情報をセットします。</li>
	 * </ol>
	 * 
     * @see com.clustercontrol.jobmanagement.action.GetRunScheduleList#get()
     */
    @Override
	public void update() {
        ArrayList<ArrayList<Object>> info = null;

        //予定一覧情報取得
        GetRunScheduleList getList = new GetRunScheduleList();
        info = getList.get();
        if (info == null) {
        	info = new ArrayList<ArrayList<Object>>();
        }
        
        m_viewer.setInput(info);
        
        selectSchedule(info);

        Object[] args = { new Integer(info.size()) };
        m_labelCount.setText(Messages.getString("records", args));
    }
    
    /**
     * 実行予定一覧の行を選択します。<BR>
     * 前回選択したスケジュールIDと同じジスケジュールIDの行を選択します。
     * 
     * @param info 実行予定情報
     */
    public void selectSchedule(ArrayList info) {
        if (m_scheduleId != null && m_scheduleId.length() > 0) {
            int index = -1;
            for (int i = 0; i < info.size(); i++) {
                ArrayList line = (ArrayList) info.get(i);
                String scheduleId = (String) line
                        .get(RunScheduleListTableDefine.SCHE_ID);
                if (m_scheduleId.compareTo(scheduleId) == 0) {
                    index = i;
                    break;
                }
            }
            if (index == -1) {
            	m_scheduleId = null;
            } else {
                m_viewer.setSelection(
                		new StructuredSelection(info.get(index)), true);
            }
        }
    }

    /**
     * このコンポジットが利用するテーブルビューアを返します。
     * 
     * @return テーブルビューア
     */
    public TableViewer getTableViewer() {
        return m_viewer;
    }

    /**
     * このコンポジットが利用するテーブルを返します。
     * 
     * @return テーブル
     */
    public Table getTable() {
        return m_viewer.getTable();
    }
    
	/**
	 * スケジュールIDを返します。
	 * 
	 * @return スケジュールID
	 */
	public String getScheduleId() {
		return m_scheduleId;
	}
	
	/**
	 * スケジュールIDを設定します。
	 * 
	 * @param scheduleId スケジュールID
	 */
	public void setScheduleId(String scheduleId) {
		m_scheduleId = scheduleId;
	}
}