/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.jobmanagement.composite;

import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Tree;

import com.clustercontrol.jobmanagement.action.GetJobTree;
import com.clustercontrol.jobmanagement.action.GetJobTreeOneLevel;
import com.clustercontrol.jobmanagement.bean.JobTreeItem;

/**
 * ジョブツリー用のコンポジットクラスです。
 *  
 * @version 1.0.0
 * @since 1.0.0
 */
public class JobTreeComposite extends Composite {
	/** ツリービューア */
    private TreeViewer m_viewer = null;
    /** 選択ジョブツリーアイテム */
    private JobTreeItem m_selectItem = null;
    /** ツリーのみ */
    private boolean m_treeOnly = false;
    /** 親ジョブID */
    private String m_parentJobId = null;
    /** ジョブID */
    private String m_jobId = null;
    /** ジョブユニットのジョブID */
    private String m_jobunitId = null;
    /**
     * コンストラクタ
     * 
     * @param parent 親コンポジット
     * @param style スタイル
     * 
     * @see org.eclipse.swt.SWT
     * @see org.eclipse.swt.widgets.Composite#Composite(Composite parent, int style)
     * @see #initialize()
     */
    public JobTreeComposite(Composite parent, int style) {
        super(parent, style);

        m_treeOnly = false;

        initialize();
    }

    /**
     * コンストラクタ
     * 
     * @param parent 親コンポジット
     * @param style スタイル
     * @param treeOnly true：ツリーのみ、false：ジョブ情報を含む
     * 
     * @see org.eclipse.swt.SWT
     * @see org.eclipse.swt.widgets.Composite#Composite(Composite parent, int style)
     * @see #initialize()
     */
    public JobTreeComposite(Composite parent, int style, boolean treeOnly) {
        super(parent, style);

        m_treeOnly = treeOnly;

        initialize();
    }

    /**
     * コンストラクタ
     * 
     * @param parent 親コンポジット
     * @param style スタイル
     * @param parentJobId 親ジョブID
     * @param jobId ジョブID
     * 
     * @see org.eclipse.swt.SWT
     * @see org.eclipse.swt.widgets.Composite#Composite(Composite parent, int style)
     * @see #initialize()
     */
    public JobTreeComposite(Composite parent, int style, String parentJobId,
            String jobId, String jobunitId) {
        super(parent, style);

        m_treeOnly = true;
        m_parentJobId = parentJobId;
        m_jobId = jobId;
        m_jobunitId = jobunitId;

        initialize();
    }

    /**
     * コンポジットを構築します。
     */
    private void initialize() {
        GridLayout layout = new GridLayout(1, true);
        this.setLayout(layout);
        layout.marginHeight = 0;
        layout.marginWidth = 0;

        Tree tree = new Tree(this, SWT.SINGLE | SWT.BORDER);
        GridData gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.verticalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        gridData.grabExcessVerticalSpace = true;
        tree.setLayoutData(gridData);

        m_viewer = new TreeViewer(tree);
        m_viewer.setContentProvider(new JobTreeContentProvider());
        m_viewer.setLabelProvider(new JobTreeLabelProvider());

        // 選択アイテム取得イベント定義
        m_viewer.addSelectionChangedListener(new ISelectionChangedListener() {
            public void selectionChanged(SelectionChangedEvent event) {
                StructuredSelection selection = (StructuredSelection) event
                        .getSelection();

                m_selectItem = (JobTreeItem) selection.getFirstElement();
            }
        });

        update();
    }

    /**
     * このコンポジットが利用するツリービューアを返します。
     * 
     * @return ツリービューア
     */
    public TreeViewer getTreeViewer() {
        return m_viewer;
    }

    /**
     * このコンポジットが利用するツリーを返します。
     * 
     * @return ツリー
     */
    public Tree getTree() {
        return m_viewer.getTree();
    }

    /**
     * ツリービューアーを更新します。<BR>
     * ジョブツリー情報を取得し、ツリービューアーにセットします。
     * <p>
	 * <ol>
	 * <li>ジョブツリー情報を取得します。</li>
	 * <li>ツリービューアーにジョブツリー情報をセットします。</li>
	 * </ol>
     *  
     * @see com.clustercontrol.jobmanagement.action.GetJobTree#getJobTree(boolean)
     */
    @Override
	public void update() {
        JobTreeItem jobTree = null;
        m_selectItem = null;

        //ジョブ一覧情報取得
        if (m_jobId == null) {
            GetJobTree getJobTree = new GetJobTree();
            jobTree = getJobTree.getJobTree(m_treeOnly);
        } else {
            GetJobTreeOneLevel getJobTree = new GetJobTreeOneLevel();
            jobTree = getJobTree.getJobTreeOneLevel(m_parentJobId, m_jobId, m_jobunitId);
        }

        m_viewer.setInput(jobTree);

        //ジョブユニットのレベルまで展開
        m_viewer.expandToLevel(2);
    }

    /**
     * 選択ジョブツリーアイテムを返します。
     * 
     * @return ジョブツリーアイテム
     */
    public JobTreeItem getSelectItem() {
        return m_selectItem;
    }

    /**
     * 選択ジョブツリーアイテムを設定
     * 
     * @param item ジョブツリーアイテム
     */
    public void setSelectItem(JobTreeItem item) {
        m_selectItem = item;
    }
}