/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.http.dialog;

import java.util.ArrayList;

import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;

import com.clustercontrol.bean.DataRangeConstant;
import com.clustercontrol.bean.HinemosModuleConstant;
import com.clustercontrol.bean.MonitorBlockConstant;
import com.clustercontrol.bean.TableColumnInfo;
import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.composite.PositiveNumberVerifyListener;
import com.clustercontrol.composite.StringVerifyListener;
import com.clustercontrol.dialog.CommonDialog;
import com.clustercontrol.dialog.ValidateResult;
import com.clustercontrol.http.action.AddHttp;
import com.clustercontrol.http.action.GetHttp;
import com.clustercontrol.http.action.GetHttpStringValueListTableDefine;
import com.clustercontrol.http.action.ModifyHttp;
import com.clustercontrol.http.bean.HttpCheckInfo;
import com.clustercontrol.monitor.run.bean.MonitorInfo;
import com.clustercontrol.monitor.run.bean.MonitorTypeConstant;
import com.clustercontrol.monitor.run.composite.MonitorBasicScopeComposite;
import com.clustercontrol.monitor.run.composite.MonitorRuleComposite;
import com.clustercontrol.monitor.run.composite.NumericValueInfoComposite;
import com.clustercontrol.monitor.run.composite.StringValueInfoComposite;
import com.clustercontrol.notify.bean.NotifyRelationInfo;
import com.clustercontrol.notify.composite.NotifyInfoComposite;
import com.clustercontrol.util.Messages;

/**
 * HTTP監視作成・変更ダイアログクラス<BR>
 * 
 * @version 3.0.0
 * @since 2.1.0
 */
public class HttpCreateDialog extends CommonDialog {
	
	public static final int WIDTH_TITLE = 5;
	public static final int WIDTH_VALUE = 2;

    // ----- instance フィールド ----- //

    /** 入力値を保持するオブジェクト */
    private MonitorInfo inputData = null;

    /** 入力値の正当性を保持するオブジェクト */
    private ValidateResult validateResult = null;
    
    /** 変更対象の監視項目ID */
    private String monitorId = null;
    
    /** 監視基本情報 */
    MonitorBasicScopeComposite m_monitorBasic = null;
    
    /** 監視条件 共通部分 */
    private MonitorRuleComposite m_monitorRule = null;
    
    /** URL */
    private Text m_textRequestUrl = null;
    
    /** タイムアウト（秒） */
	protected Text m_textTimeout = null;
    
    /** 数値監視判定情報 */
    private NumericValueInfoComposite m_numericValueInfo= null;
    
    /** 文字列監視判定情報 */
    private StringValueInfoComposite m_stringValueInfo= null;
    
    /** 通知情報 */
    private NotifyInfoComposite m_notifyInfo = null;
    
    /** この設定を有効にする */
    private Button confirmValid = null;
    
    /** 監視判定タイプ @version 2.1.0 */
	private int m_monitorType;

    // ----- コンストラクタ ----- //

    /**
     * 作成用ダイアログのインスタンスを返します。
     * 
     * @param parent
     *            親のシェルオブジェクト
     */
    public HttpCreateDialog(Shell parent) {
        super(parent);
    }
    
    /**
     * 作成用ダイアログのインスタンスを返します。
     * 
     * @param parent 親のシェルオブジェクト
     * @param monitorType 監視判定タイプ
     */
    public HttpCreateDialog(Shell parent, int monitorType) {
        super(parent);

        this.m_monitorType = monitorType;
    }
    
    /**
     * 変更用ダイアログのインスタンスを返します。
     * 
     * @param parent 親のシェルオブジェクト
     * @param monitorId 変更する監視項目ID
     * @param monitorType 監視判定タイプ
     */
    public HttpCreateDialog(Shell parent, String monitorId, int monitorType) {
        super(parent);

        this.monitorId = monitorId;
        this.m_monitorType = monitorType;
    }

    // ----- instance メソッド ----- //

    /**
     * ダイアログエリアを生成します。
     * 
     * @param parent 親のインスタンス
     */
    @Override
	protected void customizeDialog(Composite parent) {
        Shell shell = this.getShell();

        // タイトル
        shell.setText(Messages.getString("dialog.http.create.modify"));

        // 変数として利用されるラベル
        Label label = null;
        // 変数として利用されるグリッドデータ
        GridData gridData = null;

        // レイアウト
        GridLayout layout = new GridLayout(1, true);
        layout.marginWidth = 10;
        layout.marginHeight = 10;
        layout.numColumns = 15;
        parent.setLayout(layout);
        
        // 監視基本情報
        m_monitorBasic = new MonitorBasicScopeComposite(parent, SWT.NONE);
        gridData = new GridData();
		gridData.horizontalSpan = 15;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		m_monitorBasic.setLayoutData(gridData);
        
        /*
         * 監視条件グループ
         */
        // ラベル
        label = new Label(parent, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = 15;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        label.setText(Messages.getString("monitor.rule") + " : ");
        
        // グループ
        Group groupRule = new Group(parent, SWT.NONE);
        layout = new GridLayout(1, true);
        layout.marginWidth = 5;
        layout.marginHeight = 5;
        layout.numColumns = 1;
        groupRule.setLayout(layout);
        gridData = new GridData();
        gridData.horizontalSpan = 15;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        groupRule.setLayoutData(gridData);

        m_monitorRule = new MonitorRuleComposite(groupRule, SWT.NONE);
        gridData = new GridData();
		gridData.horizontalSpan = 15;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		m_monitorRule.setLayoutData(gridData);
        
        /*
         * チェック条件グループ
         */
        // ラベル
        label = new Label(groupRule, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = 15;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        label.setText(Messages.getString("check.rule") + " : ");
        
        // グループ
        Group groupCheckRule = new Group(groupRule, SWT.NONE);
        layout = new GridLayout(1, true);
        layout.marginWidth = 5;
        layout.marginHeight = 5;
        layout.numColumns = 15;
        groupCheckRule.setLayout(layout);
        gridData = new GridData();
        gridData.horizontalSpan = 15;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        groupCheckRule.setLayoutData(gridData);
        
        /*
         * URL
         */
        // ラベル
        label = new Label(groupCheckRule, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = WIDTH_TITLE;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        label.setText(Messages.getString("request.url") + " : ");
        // テキスト
        this.m_textRequestUrl = new Text(groupCheckRule, SWT.BORDER | SWT.LEFT);
        gridData = new GridData();
        gridData.horizontalSpan = 10;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        this.m_textRequestUrl.addVerifyListener(
        		new StringVerifyListener(2083));
        this.m_textRequestUrl.setLayoutData(gridData);
        this.m_textRequestUrl.setText("http://");
        
        /*
         * タイムアウト(ミリ秒)
         */
        // ラベル
        label = new Label(groupCheckRule, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = WIDTH_TITLE;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        label.setText(Messages.getString("time.out.sec") + " : ");
        // コンボボックス
        this.m_textTimeout = new Text(groupCheckRule, SWT.BORDER);
        gridData = new GridData();
        gridData.horizontalSpan = WIDTH_VALUE;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        this.m_textTimeout.addVerifyListener(
        		new PositiveNumberVerifyListener(1, DataRangeConstant.SMALLINT_HIGH));
        this.m_textTimeout.setLayoutData(gridData);
        
        // 空白
        label = new Label(groupCheckRule, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = 8;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        
        // 数値監視の場合     
		if(m_monitorType == MonitorTypeConstant.TYPE_NUMERIC){
        
			/*
	         * ノード単位の重要度グループ
	         */
	        // ラベル
	        label = new Label(parent, SWT.NONE);
	        gridData = new GridData();
	        gridData.horizontalSpan = 15;
	        gridData.horizontalAlignment = GridData.FILL;
	        gridData.grabExcessHorizontalSpace = true;
	        label.setLayoutData(gridData);
	        label.setText(Messages.getString("def.result.node") + " : ");
	        
	        // グループ
	        Group groupDefResultNode = new Group(parent, SWT.NONE);
	        layout = new GridLayout(1, true);
	        layout.marginWidth = 5;
	        layout.marginHeight = 5;
	        layout.numColumns = 1;
	        groupDefResultNode.setLayout(layout);
	        gridData = new GridData();
	        gridData.horizontalSpan = 15;
	        gridData.horizontalAlignment = GridData.FILL;
	        gridData.grabExcessHorizontalSpace = true;
	        groupDefResultNode.setLayoutData(gridData);
	        
	        // 数値判定定義情報
	        m_numericValueInfo = new NumericValueInfoComposite(groupDefResultNode, 
	        		SWT.NONE,
	        		true,
	        		Messages.getString("time.period.sec"),
	        		Messages.getString("time.period.sec"),
	        		NumericValueInfoComposite.INPUT_VERIFICATION_POSITIVE_INTEGER);
	        m_numericValueInfo.setInputRange1(0, DataRangeConstant.SMALLINT_HIGH);
	        m_numericValueInfo.setInputRange2(0, DataRangeConstant.SMALLINT_HIGH);
	        gridData = new GridData();
			gridData.horizontalSpan = 15;
			gridData.horizontalAlignment = GridData.FILL;
			gridData.grabExcessHorizontalSpace = true;
			m_numericValueInfo.setLayoutData(gridData);
		}
		
		/*
         * 通知情報の属性グループ
         */
        // ラベル
        label = new Label(parent, SWT.NONE);
        gridData = new GridData();
        gridData.horizontalSpan = 15;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        label.setLayoutData(gridData);
        label.setText(Messages.getString("notify.attribute") + " : ");
        
        // グループ
        Group groupNotifyAttribute = new Group(parent, SWT.NONE);
        layout = new GridLayout(1, true);
        layout.marginWidth = 5;
        layout.marginHeight = 5;
        layout.numColumns = 1;
        groupNotifyAttribute.setLayout(layout);
        gridData = new GridData();
        gridData.horizontalSpan = 15;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        groupNotifyAttribute.setLayoutData(gridData);
        
        // 通知情報（通知ID，アプリケーションID，実行）
        if(m_monitorType == MonitorTypeConstant.TYPE_NUMERIC){
        	this.m_notifyInfo = new NotifyInfoComposite(groupNotifyAttribute, SWT.NONE);
        }
        else if(m_monitorType == MonitorTypeConstant.TYPE_STRING){
        	this.m_notifyInfo = new NotifyInfoComposite(groupNotifyAttribute, SWT.NONE, Messages.getString("notify.id.default"));
        }
		gridData = new GridData();
		gridData.horizontalSpan = 1;
		gridData.horizontalAlignment = GridData.FILL;
		gridData.grabExcessHorizontalSpace = true;
		m_notifyInfo.setLayoutData(gridData);
		
		// 数値監視の場合     
		if(m_monitorType == MonitorTypeConstant.TYPE_NUMERIC){
			
		
		}
		// 文字列監視の場合
		else if(m_monitorType == MonitorTypeConstant.TYPE_STRING){
			
			/*
	         * ノード単位の重要度の定義グループ
	         */
			// ラベル
	        label = new Label(parent, SWT.NONE);
	        gridData = new GridData();
	        gridData.horizontalSpan = 15;
	        gridData.horizontalAlignment = GridData.FILL;
	        gridData.grabExcessHorizontalSpace = true;
	        label.setLayoutData(gridData);
	        label.setText(Messages.getString("def.result.node") + " : ");
	        
	        // グループ
	        Group groupDefResult = new Group(parent, SWT.NONE);
	        layout = new GridLayout(1, true);
	        layout.marginWidth = 5;
	        layout.marginHeight = 5;
	        layout.numColumns = 1;
	        groupDefResult.setLayout(layout);
	        gridData = new GridData();
	        gridData.horizontalSpan = 15;
	        gridData.horizontalAlignment = GridData.FILL;
	        gridData.grabExcessHorizontalSpace = true;
	        groupDefResult.setLayoutData(gridData);
			
	        // ノード単位の重要度
	        ArrayList<TableColumnInfo> tableDefine = GetHttpStringValueListTableDefine.get();
			this.m_stringValueInfo = new StringValueInfoComposite(groupDefResult, SWT.NONE, tableDefine);
			gridData = new GridData();
			gridData.horizontalSpan = 1;
			gridData.horizontalAlignment = GridData.FILL;
			gridData.grabExcessHorizontalSpace = true;
			gridData.heightHint = 250;
			m_stringValueInfo.setLayoutData(gridData);
		}
		
        /*
         * 有効／無効
         */
        this.confirmValid = new Button(parent, SWT.CHECK);
        gridData = new GridData();
        gridData.horizontalSpan = 15;
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        this.confirmValid.setLayoutData(gridData);
        this.confirmValid.setText(Messages.getString("setting.valid.confirmed"));

        // ラインを引く
        Label line = new Label(parent, SWT.SEPARATOR | SWT.HORIZONTAL);
        gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        gridData.horizontalSpan = 15;
        line.setLayoutData(gridData);


        // サイズを最適化
        // グリッドレイアウトを用いた場合、こうしないと横幅が画面いっぱいになります。
        shell.pack();
        shell.setSize(new Point(550, shell.getSize().y));

        // 画面中央に
        Display display = shell.getDisplay();
        shell.setLocation((display.getBounds().width - shell.getSize().x) / 2,
                (display.getBounds().height - shell.getSize().y) / 2);

        // 初期表示
        MonitorInfo info = null; 
        if(this.monitorId == null){
        	// 作成の場合
        	info = new MonitorInfo();
        }
        else{
        	// 変更の場合、情報取得
        	info = new GetHttp().getHttp(this.monitorId);
        }
        this.setInputData(info);
        
    }

    /**
     * 入力内容を返します。
     * 
     * @return 入力内容を保持した通知情報
     */
    public MonitorInfo getInputData() {
        return this.inputData;
    }

    /**
     * 各項目に入力値を設定します。
     * 
     * @param monitor 設定値として用いる監視情報
     */
    protected void setInputData(MonitorInfo monitor) {

    	// 監視基本情報
    	m_monitorBasic.setInputData(monitor);
    	
    	// 監視条件
    	m_monitorRule.setInputData(monitor);
    	
        // 監視条件 HTTP監視情報
    	HttpCheckInfo httpInfo = (HttpCheckInfo)monitor.getHttpCheckInfo();
        if(httpInfo == null){
        	httpInfo = new HttpCheckInfo();
        }
        if(httpInfo != null){
            if (httpInfo.getRequestUrl() != null) {
                this.m_textRequestUrl.setText(httpInfo.getRequestUrl());
            }
            this.m_textTimeout.setText(Integer.toString(httpInfo.getTimeout()));
        }
        
    	
        //通知情報の設定
        if(monitor.getNotifyId() != null)
        this.m_notifyInfo.setNotify(monitor.getNotifyId());
        
        if (monitor.getApplication() != null) {
        	this.m_notifyInfo.setApplication(monitor.getApplication());
        }
        
        // 数値監視の場合     
		if(m_monitorType == MonitorTypeConstant.TYPE_NUMERIC){
			
			// ノード単位の重要度の定義
	        m_numericValueInfo.setInputData(monitor);
	    	
		}
		// 文字列監視の場合
		else if(m_monitorType == MonitorTypeConstant.TYPE_STRING){
			
			// 重要度の結果判定の定義
	        m_stringValueInfo.setInputData(monitor);
		}
        
        // 有効／無効
        if (monitor.getValid() == ValidConstant.TYPE_VALID) {
            this.confirmValid.setSelection(true);
        }
    }

    /**
     * 入力値を用いて通知情報を生成します。
     * 
     * @return 入力値を保持した通知情報
     */
    protected MonitorInfo createInputData() {
    	MonitorInfo info = new MonitorInfo();

    	info.setMonitorTypeId(HinemosModuleConstant.MONITOR_HTTP);
    	info.setMonitorType(m_monitorType);
    	
    	// 監視基本情報
    	validateResult = m_monitorBasic.createInputData(info);
    	if(validateResult != null){
        	return null;
        }
        
        // 監視条件 共通部分
        validateResult = m_monitorRule.createInputData(info);
        if(validateResult != null){
        	return null;
        }
        
        // 監視単位
        info.setMonitorBlock(MonitorBlockConstant.TYPE_SCOPE);
        
        // 監視条件 HTTP監視情報
        HttpCheckInfo httpInfo = new HttpCheckInfo();
        httpInfo.setMonitorTypeId(HinemosModuleConstant.MONITOR_HTTP);
        httpInfo.setMonitorId(info.getMonitorId());
        
        if (this.m_textRequestUrl.getText() != null
                && !"".equals((this.m_textRequestUrl.getText()).trim())) {
    		//テキストボックスから文字列を取得
    		String url = this.m_textRequestUrl.getText();
    		if (url.length() > 0 && (!url.startsWith("http://") && !url.startsWith("https://"))) {
                this.setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.http.5"));
                return null;
            }
        	httpInfo.setRequestUrl(this.m_textRequestUrl.getText());
        } else {
            this.setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.http.1"));
            return null;
        }
        
        // 数値チェック（入力値が数値かどうか）　
        if (this.m_textTimeout.getText() != null
                && !"".equals((this.m_textTimeout.getText()).trim())) {
        	
        	try{
        		httpInfo.setTimeout(Integer.valueOf(this.m_textTimeout.getText().trim()));
        	}
        	catch(NumberFormatException e){
        		this.setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.monitor.42"));
                return null;
        	}
        } else {
            this.setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.monitor.42"));
            return null;
        }        
        info.setHttpCheckInfo(httpInfo);
        
        // 通知関連情報とアプリケーションの設定
		// 通知グループIDの設定
		info.setNotifyGroupId(NotifyRelationInfo.createNotifyGroupId(HinemosModuleConstant.MONITOR_HTTP, httpInfo.getMonitorId()));
		validateResult = m_notifyInfo.createInputData(info);
		if (validateResult != null) {
			if(validateResult.getID() == null){	// 通知ID警告用出力
				if(!displayQuestion(validateResult)){
					validateResult = null;
					return null;
				}
			}
			else{	// アプリケーション未入力チェック
				return null;
			}
		} 
        
        //　数値監視の場合     
		if(m_monitorType == MonitorTypeConstant.TYPE_NUMERIC){
			// ノード単位の結果判定の定義
	        validateResult = m_numericValueInfo.createInputData(info);
	        if(validateResult != null){
	        	return null;
	        }
	        
	        /* 入力値判定 */
	        int runInterval = info.getRunInterval();
	        int timeout = httpInfo.getTimeout();
            int infoUpper = (int) (info.getNumericValueInfo().get(0)).getThresholdUpperLimit();
            int warnUpper = (int) (info.getNumericValueInfo().get(1)).getThresholdUpperLimit();
            
            // 間隔よりタイムアウトが大きい場合
            if (runInterval <= timeout) {
            	this.setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.monitor.43"));
                return null;
            }
            // タイムアウトより通知の上限が大きい場合
            if (timeout <= infoUpper) {
            	this.setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.monitor.50"));
                return null;
            }
            // タイムアウトより警告の上限が大きい場合
            if (timeout <= warnUpper) {
            	this.setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.monitor.51"));
                return null;
            }
            
		}
		// 文字列監視の場合
		else if(m_monitorType == MonitorTypeConstant.TYPE_STRING){
			
			// 結果判定の定義
	        validateResult = m_stringValueInfo.createInputData(info);
	        if(validateResult != null){
	        	return null;
	        }
	        
	        // 結果判定の定義に通知IDが設定されていた場合
	        if(m_stringValueInfo.isNotify()){
	        	if (this.m_notifyInfo.getApplication() == null
	                    || "".equals((this.m_notifyInfo.getApplication()).trim())) {
	            	this.setValidateResult(Messages.getString("message.hinemos.1"),Messages.getString("message.monitor.10"));
	                return null;
	            }
	        }
		}
        
        // 有効/無効
        if (this.confirmValid.getSelection()) {
        	info.setValid(ValidConstant.TYPE_VALID);
        } else {
        	info.setValid(ValidConstant.TYPE_INVALID);
        }

        return info;
    }

    /**
     * 入力値チェックをします。
     * 
     * @return 検証結果
     * 
     * @see com.clustercontrol.dialog.CommonDialog#validate()
     */
    @Override
	protected ValidateResult validate() {
        // 入力値生成
        this.inputData = this.createInputData();

        if (this.inputData != null) {
            return super.validate();
        } else {
            return validateResult;
        }
    }
    
    /**
     * 入力値をマネージャに登録します。
     * 
     * @return true：正常、false：異常
     * 
     * @see com.clustercontrol.dialog.CommonDialog#action()
     */
    @Override
	protected boolean action() {
    	boolean result = false;
    	
    	MonitorInfo info = this.inputData;
    	if(info != null){
            if(this.monitorId == null){
            	// 作成の場合
            	result = new AddHttp().add(info);
            }
            else{
            	// 変更の場合
            	result = new ModifyHttp().modify(info);
            }
    	}
    	
    	return result;
    }

    /**
     * ＯＫボタンのテキストを返します。
     * 
     * @return ＯＫボタンのテキスト
     */
    @Override
	protected String getOkButtonText() {
        return Messages.getString("ok");
    }

    /**
     * キャンセルボタンのテキストを返します。
     * 
     * @return キャンセルボタンのテキスト
     */
    @Override
	protected String getCancelButtonText() {
        return Messages.getString("cancel");
    }
    
    /**
     * 無効な入力値の情報を設定します
     *  
     */
    private void setValidateResult(String id, String message) {

        this.validateResult = new ValidateResult();
        this.validateResult.setValid(false);
        this.validateResult.setID(id);
        this.validateResult.setMessage(message);
    }
}