/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.collectiverun.composite;

import java.rmi.AccessException;
import java.util.HashMap;
import java.util.List;

import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.TableTree;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.layout.RowData;
import org.eclipse.swt.layout.RowLayout;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;

import com.clustercontrol.bean.Property;
import com.clustercontrol.bean.SizeConstant;
import com.clustercontrol.collectiverun.action.GetParameterProperty;
import com.clustercontrol.collectiverun.action.GetType;
import com.clustercontrol.composite.PropertySheet;
import com.clustercontrol.util.Messages;

/**
 * 実行結果詳細コンポジット
 * 
 * @version 2.1.0
 * @since 1.0.0
 *  
 */
public class ParameterViewComposite extends Composite {

    /** プロパティシート */
    protected PropertySheet propertySheet = null;

    /** コンボボックス */
    protected Combo type = null;
    
    /** 種別ハッシュマップ */
    protected HashMap<String, String> typeMap = null;
    
    /** 種別文字列 */
    protected List<String> typeList = null;
    
    private boolean openAccessInformation = false;
    
    /**
     * コンストラクタ
     * 
     * @param parent
     * @param style
     * @since 1.0.0
     */
    public ParameterViewComposite(Composite parent, int style) {
        super(parent, style);
        initialize();
    }

    /**
     * 初期化処理
     * 
     * @since 2.1.0
     */
    private void initialize() {
        GridLayout layout = new GridLayout(1, true);
        this.setLayout(layout);
        layout.marginHeight = 0;
        layout.marginWidth = 0;

        Composite composite = new Composite(this, SWT.NONE);
        RowLayout compositeLayout = new RowLayout(SWT.HORIZONTAL);
        composite.setLayout(compositeLayout);
        compositeLayout.marginHeight = 0;
        compositeLayout.marginWidth = 0;

        GridData gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.verticalAlignment = GridData.FILL;
        composite.setLayoutData(gridData);

        Label management = new Label(composite, SWT.LEFT);
        management.setText(Messages.getString("type") + " : ");
        RowData rowData = new RowData();
        management.setLayoutData(rowData);

        this.type = new Combo(composite, SWT.READ_ONLY);
        rowData = new RowData(160, SizeConstant.SIZE_COMBO_HEIGHT);
        this.type.setLayoutData(rowData);
        
        //種別情報を取得
        GetType getType = new GetType();
        try {
            typeMap = getType.getTypeHashMap();
        } catch (AccessException e) {
        	if(!openAccessInformation){
        		openAccessInformation = true;
        		// アクセス権なしの場合、エラーダイアログを表示する
        		MessageDialog.openInformation(null, Messages.getString("message"),
	                    Messages.getString("message.accesscontrol.16"));
        	}
        }
        try {
        	typeList = getType.getTypeStringList();
        } catch (AccessException e) {
        	if(!openAccessInformation){
        		openAccessInformation = true;
        		// アクセス権なしの場合、エラーダイアログを表示する
        		MessageDialog.openInformation(null, Messages.getString("message"),
	                    Messages.getString("message.accesscontrol.16"));
        	}
        }
        
        for(int i = 0; typeList != null && i < typeList.size(); i++){
        	this.type.add(typeList.get(i));
        }

        //管理者種別
        this.type.addSelectionListener(new SelectionListener() {
            /*
             * (non-Javadoc)
             * 
             * @see org.eclipse.swt.events.SelectionListener#widgetSelected(org.eclipse.swt.events.SelectionEvent)
             */
            public void widgetSelected(SelectionEvent e) {
                update();
            }

            /*
             * (non-Javadoc)
             * 
             * @see org.eclipse.swt.events.SelectionListener#widgetDefaultSelected(org.eclipse.swt.events.SelectionEvent)
             */
            public void widgetDefaultSelected(SelectionEvent e) {

            }
        });
        this.type.select(0);

        TableTree table = new TableTree(this, SWT.H_SCROLL | SWT.V_SCROLL
                | SWT.FULL_SELECTION | SWT.MULTI);
        gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.verticalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        gridData.grabExcessVerticalSpace = true;
        gridData.horizontalSpan = 1;
        table.setLayoutData(gridData);

        this.propertySheet = new PropertySheet(table);

        //プロパティシート表示
        update();
    }

    /**
     * ビュー更新処理
     * 
     * @since 2.1.0
     */
    @Override
	public void update() {
        String typeId = getTypeId();

        Property data = null;
        try {
        	data = new GetParameterProperty().getProperty(typeId);
        } catch (AccessException e) {
        	if(!openAccessInformation){
        		openAccessInformation = true;
        		// アクセス権なしの場合、エラーダイアログを表示する
        		MessageDialog.openInformation(null, Messages.getString("message"),
	                    Messages.getString("message.accesscontrol.16"));
        	}
        }
        
        if(data == null){
        	data = new Property(null, null, "");
        }
        
        this.propertySheet.setInput(data);

        this.propertySheet.expandAll();
    }

    /**
     * ビュー更新処理
     * 
     * @since 2.1.0
     */
    public void update(String sessionId) {
        GetParameterProperty parameter = new GetParameterProperty();
        if (sessionId != null && sessionId.length() > 0) {
            String typeId = null;
            try {
            	typeId = parameter.getTypeId(sessionId);
            } catch (AccessException e) {
            	if(!openAccessInformation){
            		openAccessInformation = true;
            		// アクセス権なしの場合、エラーダイアログを表示する
            		MessageDialog.openInformation(null, Messages.getString("message"),
    	                    Messages.getString("message.accesscontrol.16"));
            	}
            }
            
            if (typeId != null) {
                Property data = null;
                try {
                	data = parameter.getPropertyBySessionId(sessionId);
                } catch (AccessException e) {
                	if(!openAccessInformation){
                		openAccessInformation = true;
                		// アクセス権なしの場合、エラーダイアログを表示する
                		MessageDialog.openInformation(null, Messages.getString("message"),
        	                    Messages.getString("message.accesscontrol.16"));
                	}
                }

                if(data == null){
                	data = new Property(null, null, "");
                }
                
                this.propertySheet.setInput(data);

                this.propertySheet.expandAll();

                type.setText(getTypeString(typeId));
            }
        }
    }

    /**
     * プロパティを取得
     * 
     * @return データモデル
     */
    public Property getInputData() {
        return (Property) this.propertySheet.getInput();
    }

    /**
     * プロパティを設定
     * 
     * @param data
     */
    public void setInputData(Property data) {
    	this.propertySheet.setInput(data);

        this.propertySheet.expandAll();
    }

    /**
     * 管理種別ID取得
     * 
     * @return
     */
    public String getTypeId() {
    	if(typeMap != null){
    		return typeMap.get(this.type.getText());
    	}
    	else{
    		return null;
    	}
    }
    
    /**
     * 管理種別文字列取得
     * 
     * @return
     */
    public String getTypeString(String typeId) {
    	if(typeMap != null){
    		return typeMap.get(typeId);
    	}
    	else{
    		return null;
    	}
    }
    
    /**
     * このコンポジットが利用するテーブルを取得。
     * 
     * @return テーブル
     */
	public TableTree getTableTree() {
		return this.propertySheet.getTableTree();
	}
}