/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License 
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied 
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 PURPOSE.  See the GNU General Public License for more details.
 
 */

package com.clustercontrol.jobmanagement.dao;

import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.text.DateFormat;
import java.util.ArrayList;
import java.util.Collection;

import javax.ejb.CreateException;
import javax.ejb.DuplicateKeyException;
import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.ejb.RemoveException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.JobParamTypeConstant;
import com.clustercontrol.bean.OutputNotifyGroupInfo;
import com.clustercontrol.commons.util.ConnectionManager;
import com.clustercontrol.jobmanagement.bean.SystemParameterConstant;
import com.clustercontrol.jobmanagement.ejb.entity.JobParamInfoBean;
import com.clustercontrol.jobmanagement.ejb.entity.JobParamInfoPK;
import com.clustercontrol.jobmanagement.ejb.entity.JobRelationMasterUtil;
import com.clustercontrol.jobmanagement.ejb.entity.JobSessionLocal;
import com.clustercontrol.jobmanagement.ejb.entity.JobSessionPK;
import com.clustercontrol.jobmanagement.ejb.entity.JobSessionUtil;
import com.clustercontrol.jobmanagement.util.ParameterUtil;

/**
 * ジョブパラメータ情報のDAOインターフェースを実装するクラス<BR>
 * 
 * @version 3.0.0
 * @since 2.1.0
 * 
 * @see com.clustercontrol.jobmanagement.ejb.entity.JobParamInfoBean
 * @see com.clustercontrol.jobmanagement.dao.JobParamInfoDAO
 */
public class JobParamInfoDAOImpl implements JobParamInfoDAO {
	
	/** ログ出力のインスタンス。 */
	protected static Log m_log = LogFactory.getLog(JobParamInfoDAOImpl.class);
	
	public void init() {
	}
	
	/** 
	 * SELECT文を発行します。
	 * 引数で指定されたプライマリキーで検索し、取得したデータベースの内容をEntity Beanに反映します。
	 * 
	 * @see com.clustercontrol.jobmanagement.dao.JobParamInfoDAO#load(com.clustercontrol.jobmanagement.ejb.entity.JobParamInfoPK, com.clustercontrol.jobmanagement.ejb.entity.JobParamInfoBean)
	 */
	public void load(JobParamInfoPK pk, JobParamInfoBean ejb) throws EJBException {
		m_log.debug("load() start : " + pk.toString());
		
		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			
			//SQL文の定義
			String sql = "SELECT * FROM cc_job_param_info WHERE session_id = ? AND job_id = ? AND param_id = ?";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			
			stmt.setString(1, pk.getSession_id());
			stmt.setString(2, pk.getJob_id());
			stmt.setString(3, pk.getParam_id());
			
			res = stmt.executeQuery();
			
			if (res.next()) {
				//取得した値をBeanにセット
				ejb.setDescription(res.getString("description"));
				ejb.setJob_id(res.getString("job_id"));
				ejb.setParam_id(res.getString("param_id"));
				ejb.setParam_type(res.getInt("param_type"));
				ejb.setSession_id(res.getString("session_id"));
				ejb.setValue(res.getString("value"));
			} else {
				String msg = "JobParamInfo data is not found.";
				m_log.error("load() error : " + pk.toString() + " SQLException " + msg);
				throw new EJBException(msg);
			}
		} catch (SQLException e) {
			m_log.error("load() error : " + pk.toString() + " SQLException");
			throw new EJBException(e.getMessage());
		} finally {
			//コネクション、結果セット、プリペアドステートメントのクローズ
			try {
				if (stmt != null) {
					stmt.close();
				}
				if (res != null) {
					res.close();
				}
				if (conn != null) {
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("load() error : " + pk.toString() + " SQLException " + e1.getMessage());
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("load() end : " + pk.toString());
	}
	
	/**
	 * UPDATE文を発行します。
	 * 引数で指定されたEntity Beanの内容でデータベースを更新します。
	 * 
	 * @see com.clustercontrol.jobmanagement.dao.JobParamInfoDAO#store(com.clustercontrol.jobmanagement.ejb.entity.JobParamInfoBean)
	 */
	public void store(JobParamInfoBean ejb) throws EJBException {
		m_log.debug("store() start : " + ejb.getSession_id() + ", " + ejb.getJob_id() + ", " + ejb.getParam_id());
		
		Connection conn = null;
		PreparedStatement stmt = null;
		
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			//SQL文の定義
			StringBuilder sql = new StringBuilder();
			sql.append("UPDATE cc_job_param_info SET ");
			sql.append("param_type = ?, ");
			sql.append("description = ?, ");
			sql.append("value = ? ");
			sql.append("WHERE session_id = ? ");
			sql.append("AND job_id = ? ");
			sql.append("AND param_id = ?");
			
			//SQL文のセット
			stmt = conn.prepareStatement(sql.toString());
			
			stmt.setInt(1, ejb.getParam_type());
			stmt.setString(2, ejb.getDescription());
			stmt.setString(3, ejb.getValue());
			stmt.setString(4, ejb.getSession_id());
			stmt.setString(5, ejb.getJob_id());
			stmt.setString(6, ejb.getParam_id());
			
			int row = stmt.executeUpdate();
			
			if (row != 1) {
				String msg = "result row is not 1";
				m_log.error("store() error : " + ejb.getSession_id() + ", "
						+ ejb.getJob_id() + ", " + ejb.getParam_id() + " SQLException" + msg);
				throw new EJBException(msg);
			}
		} catch (SQLException e) {
			m_log.error("store() error : " + ejb.getSession_id() + ", "
					+ ejb.getJob_id() + ", " + ejb.getParam_id() + " SQLException");
			throw new EJBException(e.getMessage());
		} finally {
			// コネクション、結果セット、プリペアドステートメントのクローズ
			try {
				if (stmt != null) {
					stmt.close();
				}
				if (conn != null) {
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("store() error : " + ejb.getSession_id() + ", "
						+ ejb.getJob_id() + ", " + ejb.getParam_id() + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("store() end : " + ejb.getSession_id() + ", " + ejb.getJob_id() + ", " + ejb.getParam_id());
	}
	
	/**
	 * DELETE文を発行します。
	 * 引数で指定されたプライマリキーでデータベースから削除します。
	 * 
	 * @see com.clustercontrol.jobmanagement.dao.JobParamInfoDAO#remove(com.clustercontrol.jobmanagement.ejb.entity.JobParamInfoPK)
	 */
	public void remove(JobParamInfoPK pk) throws RemoveException, EJBException {
		m_log.debug("remove() start : " + pk.toString());
		
		Connection conn = null;
		PreparedStatement stmt = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			
			//SQL文の定義
			String sql = "DELETE FROM cc_job_param_info WHERE session_id = ? AND job_id = ? AND param_id = ?";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			
			stmt.setString(1, pk.getSession_id());
			stmt.setString(2, pk.getJob_id());
			stmt.setString(3, pk.getParam_id());
			
			int row = stmt.executeUpdate();
			
			if (row != 1 ) {
				String msg = "result row is not 1";
				m_log.error("remove() error : " + pk.toString() + " SQLException" + msg);
				throw new EJBException(msg);	
			}
		} catch (SQLException e) {
			m_log.error("remove() error : " + pk.toString() + " SQLException");
			throw new EJBException(e.getMessage());
			
		} finally {
			try {
				if(stmt != null){
					stmt.close();
				}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("remove() error : " + pk.toString() + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("remove() end : " + pk.toString());
	}
	
	/**
	 * INSERT文を発行します。
	 * 引数で指定されたEntity Beanの内容をデータベースに挿入し、プライマリキーを返します。
	 * 
	 * @see com.clustercontrol.jobmanagement.dao.JobParamInfoDAO#create(com.clustercontrol.jobmanagement.ejb.entity.JobParamInfoBean)
	 */
	public JobParamInfoPK create(JobParamInfoBean ejb) throws CreateException, EJBException {
		m_log.debug("create() start : " + ejb.getSession_id() + ", " + ejb.getJob_id() + ", " + ejb.getParam_id());
		
		Connection conn = null;
		PreparedStatement stmt = null;
		JobParamInfoPK pk;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			//SQL文の定義
			StringBuilder sql = new StringBuilder();
			sql.append("INSERT INTO cc_job_param_info (");
			sql.append("session_id, ");
			sql.append("job_id, ");
			sql.append("param_id, ");
			sql.append("param_type, ");
			sql.append("description, ");
			sql.append("value) ");
			sql.append("VALUES (?,?,?,?,?,?)");
			
			//SQL文のセット
			stmt = conn.prepareStatement(sql.toString());
			
			stmt.setString(1, ejb.getSession_id());
			stmt.setString(2, ejb.getJob_id());
			stmt.setString(3, ejb.getParam_id());
			stmt.setInt(4, ejb.getParam_type());
			stmt.setString(5, ejb.getDescription());
			stmt.setString(6, ejb.getValue());
			int row =	stmt.executeUpdate();
			pk = new JobParamInfoPK(
					ejb.getSession_id(),
					ejb.getJob_id(), 
					ejb.getParam_id());
			
			if (row != 1) {
				String msg = "result row is not 1";
				m_log.error("create() error : " + ejb.getSession_id() + ", " + ejb.getJob_id() + ", " + ejb.getParam_id() + " SQLException");
				throw new EJBException(msg);
			}
		} catch (SQLException e) {
			if(e.getSQLState().equals("23505")){
				m_log.error("create() error : " + ejb.getSession_id() + ", " + ejb.getJob_id() + ", " + ejb.getParam_id() + " SQLException" + "DuplicateKeyException ");
				throw new DuplicateKeyException(e.getMessage());
			}
			throw new CreateException(e.getMessage());
		}finally{
			try {
				if(stmt != null){
					stmt.close();
				}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("create() error : " + ejb.getSession_id() + ", " + ejb.getJob_id() + ", " + ejb.getParam_id() + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("create() end : " + ejb.getSession_id() + ", " + ejb.getJob_id() + ", " + ejb.getParam_id());
		return pk;
	}
	
	/**
	 * 全件取得するSELECT文を発行します。
	 * 取得したデータのプライマリキーをコレクションに格納し返します。
	 * 
	 * @see com.clustercontrol.jobmanagement.ejb.entity.JobParamInfoBean#ejbFindAll()
	 * @see com.clustercontrol.jobmanagement.dao.JobParamInfoDAO#findAll()
	 */
	public Collection findAll() throws FinderException {
		m_log.debug("findAll() start : ");
		
		ArrayList<JobParamInfoPK> ret = new ArrayList<JobParamInfoPK>();
		
		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();	
			//SQL文の定義
			String sql = "SELECT * FROM cc_job_param_info";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			res = stmt.executeQuery();
			
			while(res.next()) {
				JobParamInfoPK pk= new JobParamInfoPK(
						res.getString("session_id"),
						res.getString("job_id"), 
						res.getString("param_id")
				);
				ret.add(pk);
			}
			
		} catch (SQLException e) {
			m_log.error("findAll() error :  SQLException");
			throw new EJBException(e.getMessage());
		}  finally{
			try {
				if(stmt != null){
					stmt.close();}
				if(res != null){
					res.close();}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("findAll() error :  SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("findAll() end : ");
		return ret;
	}
	
	/**
	 * 1件取得するSELECT文を発行します。
	 * 引数で指定されたプライマリキーで、データベースを検索します。
	 * 
	 * @see com.clustercontrol.jobmanagement.ejb.entity.JobParamInfoBean#ejbFindByPrimaryKey(com.clustercontrol.jobmanagement.ejb.entity.JobParamInfoPK)
	 * @see com.clustercontrol.jobmanagement.dao.JobParamInfoDAO#findByPrimaryKey(com.clustercontrol.jobmanagement.ejb.entity.JobParamInfoPK)
	 */
	public JobParamInfoPK findByPrimaryKey(JobParamInfoPK pk) throws FinderException {
		m_log.debug("findByPrimaryKey() start : " + pk.toString());
		
		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			
			//SQL文の定義
			String sql = "SELECT * FROM cc_job_param_info WHERE session_id = ? AND job_id = ? AND param_id = ?";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			
			stmt.setString(1, pk.getSession_id());
			stmt.setString(2, pk.getJob_id());
			stmt.setString(3, pk.getParam_id());
			
			res = stmt.executeQuery();
			
			if (res.next()) {
				return pk;
			} else {
				String msg = "id " + pk.toString() + " not found.";
				throw new FinderException(msg);
			}
		} catch (SQLException e) {
			m_log.error("find() error : " + pk.toString() + " SQLException");
			throw new EJBException(e.getMessage());
		}  finally{
			try {
				if(stmt != null){
					stmt.close();}
				if(res != null){
					res.close();}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("find() error : " + pk.toString() + " SQLException");
				throw new EJBException(e1.getMessage());
			}
			m_log.debug("findByPrimaryKey() end : " + pk.toString());
		}
	}
	
	/**
	 * 引数で指定されたセッションIDに該当するジョブ変数一覧を取得します。<br>
	 * 
	 * @see com.clustercontrol.jobmanagement.ejb.entity.JobParamInfoBean#ejbFindBySessionId(java.lang.String)
	 * @see com.clustercontrol.jobmanagement.dao.JobParamInfoDAO#findBySessionId(java.lang.String)
	 */
	public Collection findBySessionId(String session_id) throws FinderException {
		
		/** Local Variables */
		ArrayList<JobParamInfoPK> ret = new ArrayList<JobParamInfoPK>();
		
		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		
		/** Main */
		if (m_log.isDebugEnabled()) m_log.debug("finding parameters of job session... (session_id = " + session_id + ")");
		
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();	
			
			String sql = "SELECT session_id, job_id, param_id FROM cc_job_param_info WHERE session_id = ?";
			stmt = conn.prepareStatement(sql);
			stmt.setString(1, session_id);
			
			res = stmt.executeQuery();
			
			while(res.next()) {
				JobParamInfoPK pk= new JobParamInfoPK(
						res.getString("session_id"),
						res.getString("job_id"), 
						res.getString("param_id")
				);
				ret.add(pk);
			}
			
			if (m_log.isDebugEnabled()) m_log.debug("found " + ret.size() + "parameters of job session... (session_id = " + session_id + ")");
			
		} catch (SQLException e) {
			if (m_log.isWarnEnabled()) m_log.warn("failed to find parameters of job session... (session_id = " + session_id + ")", e);
			throw new EJBException("failed to find parameters of job session... (session_id = " + session_id + ")");
		}  finally{
			try {
				if(stmt != null){
					stmt.close();}
				if(res != null){
					res.close();}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e) {
				if (m_log.isWarnEnabled()) m_log.warn("failed to find parameters of job session... (session_id = " + session_id + ")", e);
				throw new EJBException("failed to find parameters of job session... (session_id = " + session_id + ")");
			}
		}
		if (m_log.isDebugEnabled()) m_log.debug("successful in finding parameters of job session. (session_id = " + session_id + ")");
		
		return ret;
	}
	
	/**
	 * 複数件取得するSELECT文を発行します。
	 * 引数で指定されたセッションIDとジョブIDで、データベースを検索します。
	 * 
	 * @see com.clustercontrol.jobmanagement.ejb.entity.JobParamInfoBean#ejbFindBySessionIdAndJobId(java.lang.String, java.lang.String)
	 * @see com.clustercontrol.jobmanagement.dao.JobParamInfoDAO#findBySessionIdAndJobId(java.lang.String, java.lang.String)
	 */
	public Collection findBySessionIdAndJobId(String session_id, String job_id) throws FinderException {
		m_log.debug("findBySessionIdAndJobId() start : " + session_id + ", " + job_id);
		
		ArrayList<JobParamInfoPK> ret = new ArrayList<JobParamInfoPK>();
		
		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();	
			//SQL文の定義
			String sql = "SELECT * FROM cc_job_param_info WHERE session_id = ? AND job_id = ?";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			stmt.setString(1, session_id);
			stmt.setString(2, job_id);
			
			res = stmt.executeQuery();
			
			while(res.next()) {
				JobParamInfoPK pk= new JobParamInfoPK(
						res.getString("session_id"),
						res.getString("job_id"), 
						res.getString("param_id")
				);
				ret.add(pk);
			}
			
		} catch (SQLException e) {
			m_log.error("findBySessionIdAndJobId() error : " + session_id + ", " + job_id + " SQLException");
			throw new EJBException(e.getMessage());
		}  finally{
			try {
				if(stmt != null){
					stmt.close();}
				if(res != null){
					res.close();}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("findBySessionIdAndJobId() error : " + session_id + ", " + job_id + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("findBySessionIdAndJobId() end : " + session_id + ", " + job_id);
		return ret;
	}
	
	/**
	 * 複数件取得するSELECT文を発行します。
	 * 引数で指定されたセッションIDとパラメータIDで、データベースを検索します。
	 * 
	 * @see com.clustercontrol.jobmanagement.ejb.entity.JobParamInfoBean#ejbFindBySessionIdAndParamId(java.lang.String, java.lang.String)
	 * @see com.clustercontrol.jobmanagement.dao.JobParamInfoDAO#findBySessionIdAndParamId(java.lang.String, java.lang.String)
	 */
	public Collection findBySessionIdAndParamId(String session_id, String param_id) throws FinderException {
		m_log.debug("findBySessionIdAndParamId() start : " + session_id + ", " + param_id);
		
		ArrayList<JobParamInfoPK> ret = new ArrayList<JobParamInfoPK>();
		
		Connection conn = null;
		PreparedStatement stmt = null;
		ResultSet res = null;
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();	
			//SQL文の定義
			String sql = "SELECT * FROM cc_job_param_info WHERE session_id = ? AND param_id = ?";
			//SQL文のセット
			stmt = conn.prepareStatement(sql);
			stmt.setString(1, session_id);
			stmt.setString(2, param_id);
			
			res = stmt.executeQuery();
			
			while(res.next()) {
				JobParamInfoPK pk= new JobParamInfoPK(
						res.getString("session_id"),
						res.getString("job_id"), 
						res.getString("param_id")
				);
				ret.add(pk);
			}
			
		} catch (SQLException e) {
			m_log.error("findBySessionIdAndParamId() error : " + session_id + ", " + param_id + " SQLException");
			throw new EJBException(e.getMessage());
		}  finally{
			try {
				if(stmt != null){
					stmt.close();}
				if(res != null){
					res.close();}
				if(conn != null){
					conn.close();
				}
			} catch (SQLException e1) {
				m_log.error("findBySessionIdAndParamId() error : " + session_id + ", " + param_id + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("findBySessionIdAndParamId() end : " + session_id + ", " + param_id);
		return ret;
	}
	
	/**
	 * ジョブパラメータ情報をジョブパラメータマスタから作成します。
	 * 
	 * @param sessionId セッションID
	 * @param jobId ジョブID
	 * @param info イベント情報
	 * @throws CreateException
	 * @throws FinderException
	 * @throws NamingException
	 * 
	 * @see com.clustercontrol.jobmanagement.ejb.entity.JobParamInfoBean#ejbHomeMakeSession(String, String, OutputNotifyGroupInfo)
	 * @see com.clustercontrol.notify.message.OutputNotifyGroupInfo
	 */
	public void makeSession(String sessionId, String jobId, OutputNotifyGroupInfo info) throws CreateException, FinderException, NamingException{
		m_log.debug("makeSession() start : " + sessionId );
		
		Connection conn = null;
		Connection connMaster = null;
		PreparedStatement stmt = null;
		PreparedStatement stmtMaster = null;
		
		try {
			conn = ConnectionManager.getConnectionManager().getConnection();
			connMaster = ConnectionManager.getConnectionManager().getConnection();
			
			//SQL文の定義(INSERT)
			StringBuilder sql = new StringBuilder();
			sql.append("INSERT INTO cc_job_param_info (");
			sql.append("session_id, ");
			sql.append("job_id, ");
			sql.append("param_id, ");
			sql.append("param_type, ");
			sql.append("description, ");
			sql.append("value) ");
			sql.append("VALUES (?,?,?,?,?,?)");
			
			//SQL文のセット(INSERT)
			stmt = conn.prepareStatement(sql.toString());
			stmt.setString(1, sessionId);
			
			//最上位のジョブIDを取得
			String topJobId = JobRelationMasterUtil.getLocalHome().getTopJobId(jobId);	
			
			//SQL文の定義(マスタ検索)
			String sqlMaster = "SELECT * FROM cc_job_param_mst WHERE job_id = ?";
			//SQLSQL文のセット(マスタ検索)
			stmtMaster = connMaster.prepareStatement(sqlMaster);
			//SQL文の定義(マスタ検索)
			stmtMaster.setString(1, topJobId);
			
			//SQL実行(マスタ検索)
			ResultSet resMaster = stmtMaster.executeQuery();
			
			while(resMaster.next()) {
				//パラメータ種別取得
				int paramType = resMaster.getInt("param_type");
				//パラメータID取得
				String paramId = resMaster.getString("param_id");
				
				//パラメータ値を取得
				String value = null;
				if(paramType == JobParamTypeConstant.TYPE_SYSTEM){
					//システム変数
					// システム変数に格納する情報が、どの機能のものかチェック
					int functionType = ParameterUtil.checkFunctionType(paramId);
					
					if (functionType == ParameterUtil.TYPE_JOB){
						// ジョブ管理の情報を取得する場合
						value = ParameterUtil.getJobParameterValue(sessionId, paramId);
						
					} else if (functionType == ParameterUtil.TYPE_MONITOR){
						// 監視管理の情報を取得する場合
						value = ParameterUtil.getParameterValue(paramId, info);
					}
					
//					if(paramId.equals(SystemParameterConstant.START_DATE)){
//						//セッション開始日時
//						try {
//							JobSessionLocal session = JobSessionUtil.getLocalHome().findByPrimaryKey(new JobSessionPK(sessionId));
//							value = DateFormat.getDateTimeInstance().format(session.getSchedule_date());
//						} catch (FinderException e) {
//							throw e;
//						} catch (NamingException e) {
//							throw e;
//						}
//					}
//					else{
//						//その他
//						value = ParameterUtil.getParameterValue(paramId, info);
//					}
				}
				else{
					//ユーザ変数
					value = resMaster.getString("value");;
				}
				
				//SQL文のセット(INSERT)
				stmt.setString(2, jobId);
				stmt.setString(3, resMaster.getString("param_id"));
				stmt.setInt(4, resMaster.getInt("param_type"));
				stmt.setString(5, resMaster.getString("description"));
				stmt.setString(6, value);
				
				//INSERTを実行
				int row =	stmt.executeUpdate();
				if (row != 1) {
					String msg = "result row is not 1";
					m_log.error("makeSession() error : " + sessionId + ", " + jobId + " SQLException");
					throw new EJBException(msg);
				}
			}
			resMaster.close();
			
		} catch (SQLException e) {
			if(e.getSQLState().equals("23505")){
				m_log.error("makeSession() error : " + sessionId + ", " + jobId + " SQLException" + "DuplicateKeyException ");
				throw new DuplicateKeyException(e.getMessage());
			}
			throw new CreateException(e.getMessage());
		} finally {
			try {
				if(stmt != null){
					stmt.close();
				}
				if(stmtMaster != null){
					stmtMaster.close();
				}
				if(conn != null){
					conn.close();
				}
				if(connMaster != null){
					connMaster.close();
				}
			} catch (SQLException e1) {
				m_log.error("makeSession() error : " + sessionId + ", " + jobId + " SQLException");
				throw new EJBException(e1.getMessage());
			}
		}
		m_log.debug("makeSession() end : " + sessionId );
	}
}
