/*
                                                                                                
Copyright (C) 2008 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.
 
*/

package com.clustercontrol.poller.snmp;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.net.InetAddress;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.concurrent.ConcurrentHashMap;

import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.naming.NameNotFoundException;

import com.clustercontrol.bean.SnmpVersionConstant;
import com.clustercontrol.poller.FacilityNotFoundException;
import com.clustercontrol.poller.NotInitializedException;
import com.clustercontrol.repository.bean.FacilityAttributeConstant;
import com.clustercontrol.repository.ejb.session.RepositoryControllerLocal;
import com.clustercontrol.repository.ejb.session.RepositoryControllerUtil;
import com.clustercontrol.sharedtable.DataTableNotFoundException;

public class SnmpPollerManager {
	private static Log m_log = LogFactory.getLog( SnmpPollerManager.class );
	
	private String m_quartzJndiName = "QuartzRAM";
	
	private String m_managerName;
	
	/**
	 * Key   : ポーラ識別情報
	 * Value : ポーラ
	 */
	private ConcurrentHashMap<PollerInfo, SnmpPoller> m_pollerTable;
	
	public SnmpPollerManager(String managerName){
		m_managerName = managerName;
		m_pollerTable = new ConcurrentHashMap<PollerInfo, SnmpPoller>();
	}

	public SnmpPoller createPoller(
			String group, 
			String facilityId, 
			boolean indexCheckFlg,
			String tableGroup, 
			String tableName)
	throws NotInitializedException, DataTableNotFoundException, FacilityNotFoundException{
		if(m_log.isDebugEnabled()){
			m_log.debug(
					"create poller    " + 
					"PollerGroup : " + group + ", " +
					"FacilityId : " + facilityId + ", " +
					"IndexCheckFlg : " + indexCheckFlg + ", " +
					"TableGroup : " + tableGroup + ", " +
					"TableName : " + tableName
			);
		}
		
		// リポジトリの情報からSNMPポーリング設定を取得する
		SnmpConfig snmpConfig;
		try {
			snmpConfig = createSnmpConfig(facilityId);
		} catch (FacilityNotFoundException e) {
			throw e;
		}
		
		// 収集情報、情報書き込み対象テーブル情報を格納したポーラを生成する
		// この時点ではインスタンスを生成するだけでポーリングはスケジューリングしないため、
		// ポーリングは実行されない
		SnmpPoller poller = new SnmpPoller(
						group, 
						facilityId, 
						snmpConfig, 
						indexCheckFlg,
						tableGroup, 
						tableName);
		
		// ジョブスケジューラに利用するQuartzのJNDI名を設定
		poller.setQuartzJndiName(m_quartzJndiName);
		
		// マップに登録する
		m_pollerTable.put(new PollerInfo(group, facilityId), poller);
		return poller;
	}
	
	/**
	 * 登録されているポーラを削除します。ポーラが実行中の場合は削除できません。
	 * @param group ポーラグループ
	 * @param facilityId ファシリティID
	 */
	public void removePoller(String group, String facilityId){
		PollerInfo info = new PollerInfo(group, facilityId);
		
		SnmpPoller poller = m_pollerTable.get(info);
		
		if(poller != null){
			// 全ての収集名の収集を終了
			poller.stopPollingAll();
		}
		
		m_pollerTable.remove(new PollerInfo(group, facilityId));
	}

	
	private SnmpConfig createSnmpConfig(String facilityId) throws FacilityNotFoundException{
		m_log.debug("createSnmpConfig() start " + facilityId);
		
		// SNMPの設定を取得
		try {
			RepositoryControllerLocal bean = RepositoryControllerUtil.getLocalHome().create();
			ArrayList<String> attributes = new ArrayList<String>();
			attributes.add(FacilityAttributeConstant.IPPROTOCOLNUMBER);
			attributes.add(FacilityAttributeConstant.IPNETWORKNUMBER);
			attributes.add(FacilityAttributeConstant.IPNETWORKNUMBERV6);
			attributes.add(FacilityAttributeConstant.SNMPCOMMUNITY); // コミュニティ名
			attributes.add(FacilityAttributeConstant.SNMPPORT); // ポート
			attributes.add(FacilityAttributeConstant.SNMPVERSION); // バージョン
			attributes.add(FacilityAttributeConstant.SNMPTIMEOUT); // タイムアウト
			attributes.add(FacilityAttributeConstant.SNMPRETRIES); // リトライ回数
			
			attributes.add(FacilityAttributeConstant.MODIFYTIMESTAMP); // 最終更新日時
			HashMap nodeInfo = bean.getNodeDetail(facilityId, attributes);
			bean.remove();

			// 「IPアドレスのバージョン」により指定されたIPアドレスを設定する。
			Integer ipProtNum = (Integer)nodeInfo.get(FacilityAttributeConstant.IPPROTOCOLNUMBER);
			String ipAddress = null;
			if(ipProtNum != null && ipProtNum == 6){
				ipAddress = (String)nodeInfo.get(FacilityAttributeConstant.IPNETWORKNUMBERV6);
			} else {
				ipAddress = (String)nodeInfo.get(FacilityAttributeConstant.IPNETWORKNUMBER);
			}

			SnmpConfig snmpConfig = new SnmpConfig(InetAddress.getByName(ipAddress));
			
			String community = (String)nodeInfo.get(FacilityAttributeConstant.SNMPCOMMUNITY);		
			m_log.debug("community : " + community);
			if(community != null){
				snmpConfig.setComunity(community);
			}

			Integer port = (Integer)nodeInfo.get(FacilityAttributeConstant.SNMPPORT);
			m_log.debug("port      : " + port);
			if(port != null){
				snmpConfig.setPort(port);
			}

			String versionStr = (String)nodeInfo.get(FacilityAttributeConstant.SNMPVERSION);
			m_log.debug("version   : " + versionStr);
			if(versionStr != null && versionStr != ""){
				Integer version = SnmpVersionConstant.stringToType(versionStr);
				if(versionStr != null){
					// 未定義(-1)の場合はv2cに設定
					if(version == -1){
						version = 1;
					}
					
					snmpConfig.setVersion(version);
				}
			}

			Integer timeout = (Integer)nodeInfo.get(FacilityAttributeConstant.SNMPTIMEOUT);
			if(timeout != null){
				m_log.debug("timeout   : " + timeout);
				snmpConfig.setTimeout(timeout);
			}
			
			Integer retries = (Integer)nodeInfo.get(FacilityAttributeConstant.SNMPRETRIES);
			m_log.debug("retries   : " + retries);
			if(retries != null){
				snmpConfig.setRetries(retries);
			}
			
			m_log.debug("createSnmpConfig() end " + facilityId);
			return snmpConfig;
		} catch (FinderException e) {
			throw new FacilityNotFoundException(facilityId);
		} catch (EJBException e) {
			m_log.error(e.getMessage(), e);
			if (e.getCausedByException() instanceof NameNotFoundException) {
				throw new FacilityNotFoundException(facilityId);
			}
			throw e;
		} catch (Exception e) {
			String message= "Couldn't get Snmp Config : " + facilityId;
			throw new EJBException(message);
		}
	}

	/**
	 * 登録されている全てのポーラの設定を再設定する
	 */
	public void refreshSnmpConfig(){
		// 全てのノードに対してチェックを行う
		Iterator<PollerInfo> itr = m_pollerTable.keySet().iterator();		
		
		while(itr.hasNext()){
			PollerInfo key = itr.next();
			
			// ポーラ名にファシリティＩＤが設定されている
			String facilityId = key.getPollerName();
			
			SnmpConfig config;
			try {
				config = createSnmpConfig(facilityId);
			} catch (FacilityNotFoundException e) {
				m_log.error(e.getMessage(), e);
				continue;
			}
			
			// ポーラの設定を新規のSNMP設定で上書き
			m_pollerTable.get(key).getPollingConfig().setSnmpConfig(config);
		}
	}
	
	/**
	 * 
	 * @param pollerGroup
	 * @param pollerName
	 */
	public SnmpPoller getPoller(String pollerGroup, String pollerName){
		return m_pollerTable.get(new PollerInfo(pollerGroup, pollerName));
	}
	
	/**
	 * このインスタンスのマネージャ名を返します。
	 * @return マネージャ名
	 */
	public String getManagerName() {
		return m_managerName;
	}
	
	/**
	 * ポーラが利用するQuartzのJNDI名を取得します。
	 * @return ポーラが利用するQuartzのJNDI名
	 */
	public String getQuartzJndiName() {
		return m_quartzJndiName;
	}

	/**
	 * ポーラが利用するQuartzのJNDI名を設定します。
	 * @param jndiName ポーラが利用するQuartzのJNDI名
	 */
	public void setQuartzJndiName(String jndiName) {
		m_quartzJndiName = jndiName;
	}

	
	/**
	 * 管理しているポーラの情報を文字列で返します。
	 * @return 管理しているテーブルの情報
	 */
	public String getPollerDebugInfo(){
		String debugStr = "";
		
		Iterator<PollerInfo> itr = m_pollerTable.keySet().iterator();
		
		while(itr.hasNext()){
			PollerInfo info = itr.next();
			int minInterval = m_pollerTable.get(info).getPollingConfig().getMinPollingInterval();
			debugStr = debugStr + info.getPollerGroup() + ", " + info.getPollerName() + ", " + minInterval+ "\n";
			debugStr = debugStr + m_pollerTable.get(info).getPollingConfig().getDebugInfo();
		}
		
		return debugStr;
	}
	
	/**
	 * 管理しているポーラを一意に識別するための情報を保持するクラス
	 * HashMapのキーとして利用できるように equalsメソッドと hashCodeメソッドを実装
	 */
	private class PollerInfo{
		private String m_pollerGroup;
		private String m_pollerName;
		
		public PollerInfo(String pollerGroup, String pollerName){
			m_pollerGroup = pollerGroup;
			m_pollerName = pollerName;
		}

		public String getPollerGroup() {
			return m_pollerGroup;
		}

		public String getPollerName() {
			return m_pollerName;
		}
		
		public boolean equals(Object other) {
			if (other instanceof PollerInfo) {
				PollerInfo info = (PollerInfo)other;
				
				if (this.m_pollerGroup == null && this.m_pollerName == null){
					if (info.m_pollerGroup == null && info.m_pollerName == null){
						return true;
					}
				} else if (this.m_pollerGroup == null && this.m_pollerName != null){
					if (info.m_pollerGroup == null && this.m_pollerName.equals(info.m_pollerName)){
						return true;
					}
				} else if (this.m_pollerGroup != null && this.m_pollerName == null){
					if (this.m_pollerGroup.equals(info.m_pollerGroup) && info.m_pollerName == null){
						return true;
					}
				} else {
					if (this.m_pollerGroup.equals(info.m_pollerGroup)){
						return this.m_pollerName.equals(info.m_pollerName);
					}
				}
				return false;
			} else {
				return false;
			}
		}

		public int hashCode() {
			int result = 17;

			result = 37 * result + ((this.m_pollerGroup != null) ? this.m_pollerGroup.hashCode() : 0);

			result = 37 * result + ((this.m_pollerName != null) ? this.m_pollerName.hashCode() : 0);

			return result;
		}
	}
}
