package org.postgresforest.mng;

import java.util.*;
import java.util.concurrent.*;

import org.postgresforest.constant.ErrorStr;
import org.postgresforest.exception.ForestException;
import org.postgresforest.exception.ForestInitFailedException;
import org.postgresforest.util.*;

import net.jcip.annotations.*;

/**
 * 各管理データベースから情報を読み出し、管理を行うためのクラス<br>
 * <br>
 * このクラスのインスタンスは、各ユーザデータベースの管理情報と１：１
 * に対応している。どのユーザデータベースに対する管理情報なのかは、
 * ForestUrlオブジェクト（本JDBCドライバへの接続文字列）で特定できる。<br>
 * <br>
 * 本クラスのクラスメソッドは、複数ある管理情報の全てを管理し、
 * インスタンスの生存期間などを決定するためのもの
 * <br>
 * このリソースマネージャの内部に持たせる各機能ごとのリソースは、
 * リソースマネージャインスタンスの解放時に同時に解放されなくてはならない。
 * そのため、「ResourceManager#destroy()」関数内に各機能クラスごとの
 * 後処理を登録する必要がある。
 */ @ThreadSafe
public final class ResourceManager {
    
    /**
     * staticフィールド。全リソースマネージャを共通管理するためのマップと、
     * リソースマネージャ取得のための関数で構成される
     */
    
    /** リソースマネージャのインスタンスを格納しておくマップ。接続文字列クラスと１：１に対応する */
    private static final ConcurrentHashMap<ForestUrl, ResourceManager> resourceManagerMap = new ConcurrentHashMap<ForestUrl, ResourceManager>();
    
    /** リソースマネージャの状態を表す列挙型 */
    private static enum EnumManagerState {
        /** 生成直後～初期化最中 */
        INITIAL, 
        /** 初期化完了（正常完了） */
        RUNNING,
        /** 初期化完了（異常完了・即座にマップから取り除くべき） */
        INVALID, 
        /** 被参照数が0で今後動作しない（即座にマップから取り除くべき） */
        DISPOSED; 
    }
    
    /**
     * リソースマネージャのインスタンスを取得する関数<br>
     * <b>注：インスタンスの取得に成功した場合、取得したインスタンスごとに必ず1回、
     * 対になるResourceManager#releaseResourceManager()を呼び出す必要がある</b><br>
     * <br>
     * この関数を呼び出すと、引数で与えられた接続文字列に該当するリソースマネージャが
     * 既に存在する場合には、そのインスタンスを返却する。存在しない場合には、リソース
     * マネージャを新規に作成・登録する。<br>
     * リソースマネージャが初期化中の場合、リソースマネージャの初期化が完了するまで
     * 全ての呼び出しスレッドは停止し、初期化が成功した段階で全ての呼び出しスレッドに
     * リソースマネージャのインスタンスが返る。<br>
     * <br>
     * リソースマネージャの初期化は、内部的にMngInfoManager#init()を呼び出す。
     * MngInfoManager#init()の呼び出しが成功すると、初期化処理は正常に完了したことになり、
     * 失敗した場合にはリソースマネージャの初期化に失敗し、全てのResourceManager#
     * getResourceManagerを呼び出しているスレッドに例外が返る。
     * 
     * @param targetUrl
     * @return 取得したリソースマネージャ
     * @throws ForestException 管理情報の初期ロードに失敗した場合
     */
    public static ResourceManager getResourceManager(final ForestUrl targetUrl, final Properties prop) throws ForestException {
        while (true) {
            ResourceManager resourceManager = resourceManagerMap.get(targetUrl);
            if (resourceManager != null) {
                // インスタンスがマップ上にあるなら、そのステータスをチェック。
                synchronized (resourceManager.lockState) {
                    
                    // 状態がDISPOSEDの場合は、マップからインスタンスを取得してから
                    // このsynchronizedブロックに入るまでの間に、他の参照中スレッドが
                    // 参照カウントを減らしたことによってDISPOSEDになっている。
                    // この場合は再度インスタンスの取得を試みる
                    if (resourceManager.status == EnumManagerState.DISPOSED) {
                        continue;
                    }
                    // それ以外の場合は同じsynchronizedブロックの中で参照カウントを
                    // インクリメントしてしまうことで、ステータスのチェックと参照
                    // カウントの増加をアトミックに実行する
                    resourceManager.refCount++;
                }
                // 続いて初期化完了のラッチを待つ
                // 初期化が完了するまでラッチのawaitはスレッドを停止させる。
                // 既に初期化が完了してラッチが開放されていれば、ラッチのawaitは
                // 即座に返ってくる。
                try {
                    resourceManager.initCompleteLatch.await();
                } catch (InterruptedException e) {
                    // この関数の呼び出しはユーザスレッドなので、interruptされた場合
                    // ユーザ側にinterruptの処理を任せるため、interruptを再セットした
                    // うえで、取得失敗の例外を投げる
                    Thread.currentThread().interrupt();
                    ForestInitFailedException newException =
                        new ForestInitFailedException(ErrorStr.MNGINIT_INTERRUPT.toString());
                    newException.setMultipleCause(Collections.<Exception>singletonList(e));
                    throw newException;
                }
                // この時点でのステータスを確認し、RUNNINGでないならば初期化に失敗して
                // いるため、初期化スレッドが発行したExceptionを読み取り、投げる。
                synchronized (resourceManager.lockState) {
                    if (resourceManager.status != EnumManagerState.RUNNING) {
                        throw resourceManager.initException;
                    }
                }
                return resourceManager;
            } else {
                // インスタンスがマップ上に存在しない場合、新規に生成して登録する
                // ConcurrentHashMapのputIfAbsentはnullが返ればputできたことを示す
                resourceManager = new ResourceManager(targetUrl, prop);
                if (resourceManagerMap.putIfAbsent(targetUrl, resourceManager) == null) {
                    // 登録成功 = ここで新規に生成した管理情報マネージャを活性化させる
                    synchronized (resourceManager.lockState) {
                        resourceManager.refCount++;
                    }
                    // 管理情報の初期化をし、ステータスをRUNNING/INVALIDに設定する
                    try {
                        resourceManager.mngInfoManager.init();
                        synchronized(resourceManager.lockState) {
                            resourceManager.setStatus(EnumManagerState.RUNNING);
                        }
                        return resourceManager;
                    } catch (ForestInitFailedException e) {
                        // この例外を受けた場合は、他の初期化完了を待っている
                        // スレッドにも例外を伝える必要がある
                        synchronized(resourceManager.lockState) {
                            resourceManager.setStatus(EnumManagerState.INVALID);
                        }
                        resourceManager.initException = e;
                        throw e;
                    } catch (InterruptedException e) {
                        // interruptされた場合、このスレッドはアプリケーションスレッド
                        // なので、interruptをセットし直したうえで、結論としては初期化完了に
                        // 失敗したということで例外を生成し、他のスレッドに例外を伝える
                        synchronized(resourceManager.lockState) {
                            resourceManager.setStatus(EnumManagerState.INVALID);
                        }
                        ForestInitFailedException newException =
                            new ForestInitFailedException(ErrorStr.MNGINIT_INTERRUPT.toString());
                        newException.setMultipleCause(Collections.<Exception>singletonList(e));
                        resourceManager.initException = newException;
                        throw newException;
                    } finally {
                        // ラッチを開放して初期化完了を待つ他のスレッドに知らせる
                        resourceManager.initCompleteLatch.countDown();
                    }
                } else {
                    // 登録失敗 = 他のスレッドによってほぼ同時刻に別のマップを登録された
                    // リトライして再取得を試みる
                    continue;
                }
            }
        }
    }
    
    
    /**
     * 以降、インスタンスメソッド（各接続文字列ごとのリソースを管理するためのメソッド群）
     */
    
    /** statusと参照カウントを変更する場合に取得するロック このロック取得無しに変更してはならない */
    private final Object lockState = new Object();
    
    /** 
     * このインスタンスの状態を表す変数<br>
     * <b>値の変更をする際は必ず lockState のロックを握った上で、setStatus() 関数を用いる</b>
     */
    @GuardedBy("lockState") private volatile EnumManagerState status = EnumManagerState.INITIAL;
    
    /**
     * ステータスを変更するための関数<br>
     * 注：この関数は、lockStateのロックを取得して呼び出さなくてはならない<br>
     * <br>
     * INITIAL -> RUNNING -> DISPOSED<br>
     * または<br>
     * INITIAL -> INVALID<br>
     * 以外の状態遷移はあってはならない<br>
     * <br>
     * 状態変化のタイミングは以下の通り<br>
     * INITIAL -> RUNNING  : 初期化が完了した段階<br>
     *                       （initCompleteラッチが開放される）<br>
     * INITIAL -> INVALID  : 初期化が完了したが失敗している<br>
     *                       （ラッチは開放されるがその後getMngInfoはnullが返る）<br>
     * RUNNING -> DISPOSED : 参照カウントが0となった段階<br>
     *                       （リソースを解放する）<br>
     * @param newStatus 新規に反映させるステータス値
     * @throws IllegalStateException 正常でないステータス遷移の場合（プログラム的なミス）
     */
    @GuardedBy("lockState")
    private void setStatus(EnumManagerState newStatus) throws IllegalStateException {
        switch (newStatus) {
            case INITIAL:
                // INITIALへの変更（インスタンス初期化時を除き）はいかなる状況でもNG
                break;
                
            case RUNNING:
                // INITIAL -> RUNNING はOK
                if (status == EnumManagerState.INITIAL) {
                    status = newStatus;
                    return;
                }
                break;
                
            case DISPOSED:
                // RUNNING -> DISPOSED はOK
                if (status == EnumManagerState.RUNNING) {
                    status = newStatus;
                    // マップからの登録解除とリソースの解放を行う
                    resourceManagerMap.remove(this.targetUrl, this);
                    destroy();
                    return;
                }
                break;
                
            case INVALID:
                // INITIAL -> INVALID はOK
                if (status == EnumManagerState.INITIAL) {
                    status = newStatus;
                    // マップからの登録解除とリソースの解放を行う
                    resourceManagerMap.remove(this.targetUrl, this);
                    destroy();
                    return;
                }
                break;
                
            default:
                break;
            
        }
        // 変更ができなかった場合はプログラム的なミス
        // 変更できる状況で呼び出すべき
        throw new IllegalStateException(ErrorStr.ILLEGAL_STATEMENT.toString() + " status change " + status.toString() + " -> " + newStatus.toString());
    }
    
    /** このインスタンスの参照カウント 必ずlockStateのロックを取得すること */
    @GuardedBy("lockState") private int refCount = 0;
    
    /** 管理情報マネージャのinitが完了すると開放されるラッチ（initの成功・失敗は問わず開放される）*/
    private final CountDownLatch initCompleteLatch = new CountDownLatch(1);
    
    /** 管理情報マネージャのinitが失敗した場合に、その例外がセットされる */
    private volatile ForestInitFailedException initException = null;
    
    /** このリソースマネージャに紐づく接続文字列（リソースマネージャ解放の際に使用する） */
    private final ForestUrl targetUrl;
    
    /** 管理情報DBへアクセスするためのリソース */
    private final MngInfoManager mngInfoManager;
    
    /**
     * 各ユーザコネクションのAPIを実行するためのExecutor
     * （全てのEntrypointCommonResourceで共用する）
     */
    private final ExecutorService apiExecutor;
    
    /** コンストラクタ */
    private ResourceManager(ForestUrl targetUrl, Properties prop) {
       this.targetUrl = targetUrl;
       this.mngInfoManager = new MngInfoManager(targetUrl, prop);
       this.apiExecutor = Executors.newCachedThreadPool(new ForestThreadFactory(targetUrl, "JdbcApiExecThreadPool"));
    }
    
    /**
     * デストラクタ（のようなもの）<br>
     * このインスタンスを破棄する際（ステータスがINVALID/DISPOSEDになる場合）
     * に呼ばれ、各種リソースの後処理を行うための関数<br>
     * <b>注：本関数は１つのインスタンスに対して複数呼ばれる可能性もあり、
     * 各種リソースはそれを考慮した作りにする必要がある</b>
     */
    private void destroy() {
        mngInfoManager.destroy();
        apiExecutor.shutdownNow();
    }
    
    /**
     * このリソースマネージャを参照しなくなったことを宣言するために呼び出す関数<br>
     * 本関数を呼ぶことでリソースマネージャインスタンスの参照カウントが減り、
     * 0となった時点でそのリソースマネージャのリソースを解放する。<br>
     * <br>
     * <b>注：getResourceManagerを呼び出した側が、必ず1度だけ本関数を呼ぶ必要がある。
     * 複数回呼び出した場合の動作は未定義。逆に１度も呼ばないと、リソースリークする</b>
     */
    public void releaseResourceManager() {
        synchronized (lockState) {
            refCount--;
            if (refCount < 1 && status == EnumManagerState.RUNNING) {
                setStatus(EnumManagerState.DISPOSED);
            }
        }
    }
    
    /**
     * 各種内部メソッドの一部公開
     */
    
    /**
     * ユーザに公開するJDBC-APIの各タスクを実行するために使用する。内部的には
     * ExecutorService.invokeAll(Callable<T>, long, TimeUnit) を呼び出しているため、
     * 関数の詳細な仕様はExecutorService.invokeAllを参照。
     * （但し、タイムアウトの単位は「秒」で固定としている）<br>
     * <br>
     * ここで使用するExecutorServiceはリソースマネージャに管理されており、
     * リソースマネージャが不要になった（そのリソースマネージャにアクセスする
     * ForestConnectionが存在しなくなった）段階で、executorは自動的に開放される。
     * @param <T>
     * @param apiTasks
     * @param timeout 実行制限時間（秒）
     * @return
     * @throws InterruptedException
     */
    public <T> List<Future<T>> execJdbcApiTask(final List<Callable<T>> apiTasks, final int timeout)
    throws InterruptedException {
        return apiExecutor.invokeAll(apiTasks, timeout, TimeUnit.SECONDS);
    }
    
    /** 現在の管理情報のスナップショットを取得する関数 */
    public MngInfo getMngInfo() {
        return mngInfoManager.getMngInfo();
    }
    
    /**
     * 指定されたidのユーザデータベースを縮退させる<br>
     * 詳細はMngInfoManager#setuUdbInvalid()を参照
     * @param oldMngInfo 縮退と判断した時点でのMngInfoのスナップショット
     * @param serverId 縮退状態へステータスを変更する対象のサーバID（0 or 1）
     * @param exception 縮退と判断することになったExceptionオブジェクト
     * @param taskClassName 縮退が発生した時に実行していたタスクのクラス名
     * @return 置き換えが成功した場合はtrue、置き換えまでに他スレッドにより
     * 置き換えが発生していた場合はfalse
     */
    public boolean setUdbInvalid(final MngInfo oldMngInfo, final int serverId, final Exception exception, final String taskClassName) {
        return mngInfoManager.setUdbInvalid(oldMngInfo, serverId, exception, taskClassName);
    }
    
    /**
     * リカバリのために更新抑制フェーズであるならば、リカバリが完了
     * （成功・失敗問わず）するまでこの関数は待ち状態となり、抑制解除に
     * なった段階で戻ってくる。抑制がかかっていない状態であれば何もせず
     * 即座に返る。
     * @throws InterruptedException ラッチを待つ間にinterruptされた場合
     */
    public void waitRecovery() throws InterruptedException {
        mngInfoManager.waitRecovery();
    }
    
    /**
     * リカバリが正常完了した時に、RecoveryCompletedListenerで定義された
     * コールバック関数を呼ぶように登録するための関数。
     * （内部的にはMngInfoManagerのsetRecoveryCompletedListenerを呼び出している）
     * @param listener 登録したいコールバック関数を実装したオブジェクト
     */
    public void setRecoveryCompletedListener(RecoveryCompletedListener listener) {
        mngInfoManager.setRecoveryCompletedListener(listener);
    }
    
    /** currentPrefServerIdをガードするためのロック */
    private final Object lockPrefServerId = new Object();
    /** 現在払いだされた優先実行系のサーバID。0または1のみを取る */
    private int currentPrefServerId = 1;
    /**
     * 優先実行サーバのIDを払いだす。払いだすIDは0または1。
     * 試験容易化のため、サーバIDは各リソースマネージャごとに
     * 0 -> 1 -> 0 -> 1 -> 0 ...
     * の順で払いだされることとする。
     * @return
     */
    @GuardedBy("lockPrefServerId")
    public int getNextPreferentialServerId() {
        synchronized (lockPrefServerId) {
            currentPrefServerId = (currentPrefServerId == 0) ? 1 : 0;
            return currentPrefServerId;
        }
    }
}
