/*
                                                                                                
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.
 
*/

package com.clustercontrol.performance.jmx;

import javax.naming.CompositeName;
import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.Name;
import javax.naming.NamingException;

import org.jboss.naming.NonSerializableFactory;
import org.jboss.system.ServiceMBeanSupport;

import com.clustercontrol.performance.bean.QuartzConstant;

/**
 * JBoss起動時に能管理機能の実績収集、リソース監視機能用性能値収集を再実行するためのJMXサービスを実装したクラス
 * 
 * @jmx.mbean
 *     name="user:service=PerformanceService,name=PerformanceService"
 *     description="PerformanceService MBean"
 *     extends="org.jboss.system.ServiceMBean"
 * 
 * @jboss.service servicefile="snmppoller"
 */
public class PerformanceService extends ServiceMBeanSupport implements PerformanceServiceMBean{
	private String jndiName = "PerformanceService";  // JNDIに登録する際の名前
	
	private PerformanceRestartManager manager;
	
	// リポジトリ変更時にメッセージを受け取りノード情報を変更するクラス
	private ReceiveRepositoryTopic m_receiveRepositoryTopic;
	
	/**
	 * PerformanceServiceクラスのインスタンスを生成します。
	 *
	 */
	public PerformanceService(){
	}
	
    /**
     * 自身が登録されているJNDI名を取得します。
     * 
     * @jmx.managed-attribute
     * @return JNDI名
     */
    public String getJndiName() {
        return jndiName;
    }
    
    /**
     * JNDI名を設定します。
     * 
     * @jmx.managed-attribute
     * 
     * @param jndiName JNDI名
     * @throws NamingException 
     * @throws NamingException
     */
    public void setJndiName(String jndiName) throws NamingException {
        log.info("setJndiName() SnmpService : " + jndiName);
    	
        String oldName = this.jndiName;
        this.jndiName = jndiName;
        
        if (super.getState() == STARTED) {
        	try {
        		unbind(oldName);
        	} catch (NamingException e){
        		NamingException ne = new NamingException("Failed to unbind");
        		ne.setRootCause(e);
        		log.error(ne);
        	}

        	try{
        		rebind();
        	} catch (NamingException e){
        		NamingException ne = new NamingException("Failed to rebind");
        		ne.setRootCause(e);
        		log.error(ne);
        	}
        }
    }
	
    /**
     * サービス名を取得します。
     * 
     * @jmx.managed-attribute
     */
    public String getName() {
        return "PerformanceService(" + jndiName + ")";
    }
    
    /**
     * DBに登録されている収集で、ステータスが実行中の収集を再開します。
     * 
     * @jmx.managed-operation
     */
    public void restartCollector(){
    	manager.restartAll();
    }
	
    /**
     * サービス生成時にJMXマイクロカーネルから呼ばれます。
     */
    public void createService() {
        log.info("Create PerformanceService(" + jndiName + ")");
 
        manager = new PerformanceRestartManager();

        log.info("Created PerformanceService(" + jndiName + ")");
    }
    
    /**
     * サービス開始時にJMXマイクロカーネルから呼ばれます。
     */
    public void startService() throws NamingException {
        log.info("Start PerformanceService(" + jndiName + ")");
    	// 性能管理機能の全ての収集を再開する       
        manager.restartAll();
        // リポジトリがアップデートされた場合に処理する
		m_receiveRepositoryTopic = new ReceiveRepositoryTopic(manager);
        
    	rebind();
        log.info("Started PerformanceService(" + jndiName + ")");
    }
    
    /**
     * サービスを停止時にJMXマイクロカーネルから呼ばれます。
     */
    public void stopService() throws NamingException {
        log.info("Stop PerformanceService(" + jndiName + ")");
    	unbind(jndiName);
        
    	m_receiveRepositoryTopic.terminate();
    	
        // テーブルを全て削除する
//        snmpSharedTable.deleteTableAll();
        
        log.info("Stoped PerformanceService(" + jndiName + ")");
    }
    
    /**
     * サービスを削除時にJMXマイクロカーネルから呼ばれます。
     */
    public void destroyService() throws Exception {
        log.info("Destroy PerformanceService(" + jndiName + ")");
        
        manager = null;

        log.info("Destroyed PerformanceService(" + jndiName + ")");
    }
    
    /**
     * 
     * @param rootCtx
     * @param name
     * @return
     * @throws NamingException
     */
    private static Context createContext(Context rootCtx, Name name) throws NamingException {
    	Context subctx = rootCtx;
    	
    	for (int n = 0; n < name.size(); n++) {
    		String atom = name.get(n);
    		
    		try {
    			Object obj = subctx.lookup(atom);
    			subctx = (Context) obj;
    		} catch (NamingException e) {
    			// 存在しない場合は、サブコンテキストを生成
    			subctx = subctx.createSubcontext(atom);
    		}
    	}
    	
    	return subctx;
    }
    
    /**
     * JNDIサービスにPerformanceRestartManagerクラスのインスタンスをバインドします。
     * @throws NamingException
     */
    private void rebind() throws NamingException {
    	InitialContext rootCtx = new InitialContext();
    	
    	Name fullName = rootCtx.getNameParser("").parse(jndiName);
    	Name parentName = fullName;
    	if(fullName.size() > 1){
    		parentName = fullName.getPrefix(fullName.size()-1);
    	} else {
    		parentName = new CompositeName();
    	}
    	
    	Context parentCtx = createContext(rootCtx, parentName);
    	Name atomName = fullName.getSuffix(fullName.size()-1);
    	String atomStirng = atomName.get(0);
    	
    	NonSerializableFactory.rebind(parentCtx, atomStirng, manager);
    }

    /**
     * JNDIサービスにバインドされているインスタンスをアンバインドします。
     * @param jndiName JNDI名
     * @throws NamingException
     */
    private void unbind(String jndiName) throws NamingException {
    	InitialContext rootCtx = null;
    	
    	try {
    		rootCtx = new InitialContext();
    		
    		Name fullName = rootCtx.getNameParser("").parse(jndiName);
    		Name atomName = fullName.getSuffix(fullName.size() - 1);
    		String atom = atomName.get(0);
    		
    		rootCtx.unbind(jndiName);
    		NonSerializableFactory.unbind(atom);
    	} finally {
    		if(rootCtx != null) { 
    			rootCtx.close(); 
    		}
    	}
    }
    
    /**
	 * 性能管理機能で利用するQuartzのJNID名を返します。
     * @jmx.managed-attribute
     */
    public String getQuartzName() {
        return QuartzConstant.getQuartzName();
    }
    
    /**
	 * 性能管理機能で利用するQuartzのJNID名を設定します。
     * @jmx.managed-attribute
     */
    public void setQuartzName(String quartzName) {
    	QuartzConstant.setQuartzName(quartzName);
    }
    
    /**
	 * 計算処理実行開始を設定するための、SNMPポーリングからのディレイタイムを返します。
	 * 対象は、収集間隔1分未満のもの
	 * 
     * @jmx.managed-attribute
     */
    public int getDelayTimeUnder1min() {
        return QuartzConstant.getDelayTimeUnder1min();
    }
    
    /**
	 * 計算処理実行開始を設定するための、SNMPポーリングからのディレイタイムを設定します。
	 * 対象は、収集間隔1分未満のもの
	 * 
     * @jmx.managed-attribute
     */
    public void setDelayTimeUnder1min(int delayTimeUnder1min) {
    	QuartzConstant.setDelayTimeUnder1min(delayTimeUnder1min);
    }
    
	/**
	 * 計算処理実行開始を設定するための、SNMPポーリングからのディレイタイムを返します。
	 * 対象は、収集間隔1分以上のもの
	 * 
     * @jmx.managed-attribute
     */
	public int getDelayTimeOver1min() {
		return QuartzConstant.getDelayTimeOver1min();
	}
	
    /**
	 * 計算処理実行開始を設定するための、SNMPポーリングからのディレイタイムを設定します。
	 * 対象は、収集間隔1分以上のもの
	 *  
     * @jmx.managed-attribute
     */	
	public void setDelayTimeOver1min(int delayTimeOver1min) {
		QuartzConstant.setDelayTimeOver1min(delayTimeOver1min);
	}
}
