/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.monitor.run.factory;

import java.rmi.RemoteException;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.Iterator;
import java.util.Locale;

import javax.ejb.CreateException;
import javax.ejb.FinderException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.swt.SWT;
import org.quartz.SchedulerException;

import com.clustercontrol.bean.RunIntervalConstant;
import com.clustercontrol.bean.TableColumnInfo;
import com.clustercontrol.monitor.run.bean.MonitorCheckInfo;
import com.clustercontrol.monitor.run.bean.MonitorInfo;
import com.clustercontrol.monitor.run.bean.MonitorTabelDefine;
import com.clustercontrol.monitor.run.bean.MonitorTypeConstant;
import com.clustercontrol.monitor.run.ejb.entity.MonitorInfoLocal;
import com.clustercontrol.monitor.run.ejb.entity.MonitorInfoPK;
import com.clustercontrol.monitor.run.ejb.entity.MonitorInfoUtil;
import com.clustercontrol.notify.ejb.session.NotifyController;
import com.clustercontrol.notify.ejb.session.NotifyControllerLocal;
import com.clustercontrol.notify.ejb.session.NotifyControllerUtil;
import com.clustercontrol.repository.ejb.session.RepositoryControllerLocal;
import com.clustercontrol.repository.ejb.session.RepositoryControllerUtil;
import com.clustercontrol.util.Messages;
import com.clustercontrol.util.apllog.AplLogger;

/**
 * 監視情報を検索する抽象クラス<BR>
 * <p>
 * 監視種別（真偽値，数値，文字列）の各クラスで継承してください。
 *
 * @version 3.0.0
 * @since 2.0.0
 */
abstract public class SelectMonitor {
	
	/** ログ出力のインスタンス。 */
	protected static Log m_log = LogFactory.getLog( SelectMonitor.class );
	
	/** 監視情報のローカルコンポーネント。 */
	protected MonitorInfoLocal m_monitor;
	
	/** 監視情報。 */
	protected MonitorInfo m_monitorInfo;
	
	/** 監視対象ID。 */
	protected String m_monitorTypeId;
	
	/** 監視項目ID。 */
	protected String m_monitorId;
	
	/**
	 * 引数で指定された監視情報を返します。
	 * <p>
	 * <ol>
	 * <li>引数で指定された監視情報を取得します。</li>
	 * <li>Quartzより、有効/無効を取得します。</li>
	 * <li>監視情報より判定情報を取得します。各監視種別（真偽値，数値，文字列）のサブクラスで実装します（{@link #getJudgementInfo()}）。</li>
	 * <li>監視情報よりチェック条件を取得します。各監視管理のサブクラスで実装します（{@link #getCheckInfo()}）。</li>
	 * </ol>
	 * 
	 * @param monitorTypeId 監視対象ID
	 * @param monitorId 監視項目ID
	 * @return 監視情報
	 * @throws CreateException
	 * @throws FinderException
	 * @throws SchedulerException
	 * @throws NamingException
	 * 
	 * @see com.clustercontrol.monitor.run.ejb.entity.MonitorInfoBean
	 * @see com.clustercontrol.monitor.run.factory.SelectSchedule#getValid(String, String)
	 * @see #getJudgementInfo()
	 * @see #getCheckInfo()
	 */
	public MonitorInfo getMonitor(String monitorTypeId, String monitorId) throws CreateException, FinderException, SchedulerException, NamingException {
		
		m_monitorTypeId = monitorTypeId;
		m_monitorId = monitorId;
		
		MonitorInfo bean = null;
		try 
		{
			// 監視情報を取得
			MonitorInfoPK pk = new MonitorInfoPK(m_monitorId, m_monitorTypeId);
			m_monitor = MonitorInfoUtil.getLocalHome().findByPrimaryKey(pk);
			
            // スコープの取得
            RepositoryControllerLocal repository = RepositoryControllerUtil.getLocalHome().create();
			String facilityPath = repository.getFacilityPath(m_monitor.getFacilityId(), null);
			
			NotifyControllerLocal nc = NotifyControllerUtil.getLocalHome().create();
			Collection notifyId = nc.getNotifyRelation(m_monitor.getNotifyGroupId());
			
			
			// Quartzより有効/無効の取得
			int valid = new SelectSchedule().getValid(m_monitorTypeId, m_monitorId);
					
			bean = new MonitorInfo(
					m_monitor.getApplication(),
					m_monitor.getCalendarId(),
					m_monitor.getDescription(),
					m_monitor.getDeterminationId(),
					facilityPath,
					m_monitor.getFacilityId(),
					m_monitor.getFailurePriority(),
					m_monitor.getMonitorBlock(),
					m_monitorId,
					m_monitorTypeId,
					m_monitor.getMonitorType(),
					m_monitor.getRegDate(),
					m_monitor.getRegUser(),
					m_monitor.getRunInterval(),
					m_monitor.getTimeout(),
					m_monitor.getFailureMessageId(),
					m_monitor.getFailureMessage(),
					m_monitor.getNotifyGroupId(),
					notifyId,
					m_monitor.getUpdateDate(),
					m_monitor.getUpdateUser(),
					valid,
					getJudgementInfo(),
					getCheckInfo());
			
		} catch (CreateException e) {
			outputLog(e, "getMonitor()", "010");
			throw e;
		} catch (FinderException e) {
			outputLog(e, "getMonitor()", "010");
			throw e;
		} catch (SchedulerException e) {
			outputLog(e, "getMonitor()", "010");
			throw e;
		} catch (NamingException e) {
			outputLog(e, "getMonitor()", "010");
			throw e;
		} 
		
		return bean;
	}
	
	/**
	 * 判定情報を返します。
	 * <p>
	 * 各監視種別（真偽値，数値，文字列）のサブクラスで実装します。
	 * 
	 * @return 判定情報（{@link com.clustercontrol.monitor.run.bean.MonitorJudgementInfo}のリスト）
	 * @throws FinderException
	 * @throws NamingException
	 */
	public abstract ArrayList getJudgementInfo() throws FinderException, NamingException;
	
	/**
	 * チェック条件情報を返します。
	 * <p>
	 * 各監視管理のサブクラスで実装します。
	 * 
	 * @return チェック条件情報
	 * @throws FinderException
	 * @throws NamingException
	 */
	public abstract MonitorCheckInfo getCheckInfo() throws FinderException, NamingException;

	/**
	 * 監視情報一覧を返します。
	 * <p>
	 * <ol>
	 * <li>引数で指定された監視対象の監視情報を取得します。</li>
	 * <li>１監視情報をテーブルのカラム順（{@link com.clustercontrol.monitor.run.bean.MonitorTabelDefine}）に、リスト（{@link ArrayList}）にセットします。</li>
	 * <li>この１監視情報を保持するリストを、監視情報一覧を保持するリスト（{@link ArrayList}）に格納し返します。<BR>
	 *  <dl>
	 *  <dt>監視情報一覧（Objectの2次元配列）</dt>
	 *  <dd>{ 監視情報1 {カラム1の値, カラム2の値, … }, 監視情報2{カラム1の値, カラム2の値, …}, … }</dd>
	 *  </dl>
	 * </li>
	 * </ol>
	 * 
	 * @param monitorTypeId 監視対象ID
	 * @return 監視情報一覧（Objectの2次元配列）
	 * @throws CreateException
	 * @throws FinderException
	 * @throws SchedulerException
	 * @throws NamingException
	 * 
	 * @see com.clustercontrol.monitor.run.bean.MonitorTabelDefine
	 * @see #collectionToArray(Collection)
	 */
	public ArrayList getMonitorList(String monitorTypeId) throws CreateException, FinderException, SchedulerException, NamingException {

		ArrayList list = null;
		try 
		{
			// 監視情報一覧を取得
			Collection ct = MonitorInfoUtil.getLocalHome().findAll(monitorTypeId);
			
			// 2次元配列に変換
			list = this.collectionToArray(ct);
			
		} catch (CreateException e) {
			outputLog(e, "getMonitorList()", "011");
			throw e;
		} catch (FinderException e) {
			outputLog(e, "getMonitorList()", "011");
			throw e;
		} catch (SchedulerException e) {
			outputLog(e, "getMonitorList()", "011");
			throw e;
		} catch (NamingException e) {
			outputLog(e, "getMonitorList()", "011");
			throw e;
		}
		return list;
	}
	
	/**
	 * 監視一覧のテーブル定義情報を返します。
	 * リストに、カラム毎にテーブルカラム情報をセットします。
	 * 
	 * @param locale ロケール情報
	 * @return テーブル定義情報（{@link com.clustercontrol.bean.TableColumnInfo}のリスト）
	 * @throws CreateException
	 * @throws FinderException
	 * @throws SchedulerException
	 * @throws NamingException
	 * 
	 * @see com.clustercontrol.bean.TableColumnInfo#TableColumnInfo(java.lang.String, int, int, int)
	 * @see com.clustercontrol.monitor.run.bean.MonitorTabelDefine
	 */
	@SuppressWarnings("unchecked")
	public ArrayList getMonitorListTableDefine(Locale locale) {

		/** テーブル情報定義配列 */
		ArrayList tableDefine = new ArrayList();

		tableDefine.add(MonitorTabelDefine.MONITOR_ID, 
				new TableColumnInfo(Messages.getString("monitor.id", locale), TableColumnInfo.NONE, 100, SWT.LEFT));
		tableDefine.add(MonitorTabelDefine.DESCRIPTION, 
				new TableColumnInfo(Messages.getString("description", locale), TableColumnInfo.NONE, 200, SWT.LEFT));
		tableDefine.add(MonitorTabelDefine.FACILITY_ID, 
				new TableColumnInfo(Messages.getString("scope", locale), TableColumnInfo.FACILITY, 200, SWT.LEFT));
		tableDefine.add(MonitorTabelDefine.MONITOR_TYPE, 
				new TableColumnInfo(Messages.getString("monitor.type", locale), TableColumnInfo.NONE, 80, SWT.LEFT));
		tableDefine.add(MonitorTabelDefine.CALENDAR_ID, 
				new TableColumnInfo(Messages.getString("calendar", locale), TableColumnInfo.NONE, 100, SWT.LEFT));
		tableDefine.add(MonitorTabelDefine.RUN_INTERVAL, 
				new TableColumnInfo(Messages.getString("run.interval", locale), TableColumnInfo.NONE, 60, SWT.LEFT));
		tableDefine.add(MonitorTabelDefine.VALID_FLG, 
				new TableColumnInfo(Messages.getString("valid", locale) + "/" + Messages.getString("invalid", locale), TableColumnInfo.VALID, 80, SWT.LEFT));
		tableDefine.add(MonitorTabelDefine.CREATE_USER, 
				new TableColumnInfo(Messages.getString("creator.name", locale), TableColumnInfo.NONE, 80, SWT.LEFT));
		tableDefine.add(MonitorTabelDefine.CREATE_TIME, 
				new TableColumnInfo(Messages.getString("create.time", locale), TableColumnInfo.NONE, 130, SWT.LEFT));
		tableDefine.add(MonitorTabelDefine.UPDATE_USER, 
				new TableColumnInfo(Messages.getString("modifier.name", locale), TableColumnInfo.NONE, 80, SWT.LEFT));
		tableDefine.add(MonitorTabelDefine.UPDATE_TIME, 
				new TableColumnInfo(Messages.getString("update.time", locale), TableColumnInfo.NONE, 130, SWT.LEFT));

		return tableDefine;
	}
	
	/**
	 * DBより取得した監視情報のObjectの2次元配列を返します。
	 * <p>
	 * <ol>
	 * <li>監視情報をテーブルのカラム順（{@link com.clustercontrol.monitor.run.bean.MonitorTabelDefine}）に、リスト（{@link ArrayList}）にセットします。</li>
	 * <li>1つの監視情報を保持するリストを、監視情報一覧を保持するリスト（{@link ArrayList}）に格納します。
	 * <p>
	 *  <dl>
	 *  <dt>監視情報一覧（Objectの2次元配列）</dt>
	 *  <dd>{ 監視情報1 {カラム1の値, カラム2の値, … }, 監視情報2{カラム1の値, カラム2の値, …}, … }</dd>
	 *  </dl>
	 * </li>
	 * </ol>
	 * 
	 * @param ct 監視情報のコレクション
	 * @return 監視情報一覧（Objectの2次元配列）
	 * 
	 * @see com.clustercontrol.monitor.run.bean.MonitorTabelDefine
	 */
	@SuppressWarnings("unchecked")
	public ArrayList collectionToArray(Collection ct) throws CreateException, FinderException, SchedulerException, NamingException{
		
		RepositoryControllerLocal repository = RepositoryControllerUtil.getLocalHome().create();
		SelectSchedule schedule = new SelectSchedule();
		
		ArrayList list = new ArrayList();
		Iterator itr = ct.iterator();
		while(itr.hasNext())
		{
			MonitorInfoLocal monitor = (MonitorInfoLocal)itr.next();
			
			// スコープの取得
			String facilityPath = repository.getFacilityPath(monitor.getFacilityId(), null);
			
			ArrayList info = new ArrayList();
			info.add(monitor.getMonitorId());
			info.add(monitor.getDescription());
			info.add(facilityPath);
			info.add(MonitorTypeConstant.typeToString(monitor.getMonitorType().intValue()));
			info.add(monitor.getCalendarId());
			info.add(RunIntervalConstant.typeToString(monitor.getRunInterval().intValue()));
			info.add(schedule.getValid(monitor.getMonitorTypeId(), monitor.getMonitorId()));
			info.add(monitor.getRegUser());
			info.add(monitor.getRegDate() == null ? null:new Date(monitor.getRegDate().getTime()));
			info.add(monitor.getUpdateUser());
			info.add(monitor.getUpdateDate() == null ? null:new Date(monitor.getUpdateDate().getTime()));
			list.add(info);
		}
		return list;
	}
	
	/**
	 * アプリケーションログにログを出力します。
	 * 
	 * @param e 例外
	 * @param method メソッド名
	 * @param index アプリケーションログのインデックス
	 */
	private void outputLog(Exception e, String method, String index) {
		AplLogger apllog = new AplLogger("MON", "mon");
		String[] args = {m_monitorTypeId, m_monitorId };
        apllog.put("SYS", index, args);
        m_log.debug(method + ":" + e.getMessage());
	}
}
