/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.calendar.factory;

import java.sql.Time;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collection;
import java.util.Date;
import java.util.Iterator;

import javax.ejb.FinderException;
import javax.naming.NamingException;

import com.clustercontrol.calendar.bean.RunConstant;
import com.clustercontrol.calendar.bean.ScheduleTableConstant;
import com.clustercontrol.calendar.ejb.entity.CalendarAddInfoLocal;
import com.clustercontrol.calendar.ejb.entity.CalendarAddInfoUtil;
import com.clustercontrol.calendar.ejb.entity.CalendarInfoLocal;
import com.clustercontrol.calendar.ejb.entity.CalendarInfoUtil;
import com.clustercontrol.calendar.ejb.entity.CalendarWeekdayInfoLocal;
import com.clustercontrol.calendar.ejb.entity.CalendarWeekdayInfoUtil;

/**
 * カレンダを検索するファクトリークラス<BR>
 *
 * @version 2.0.0
 * @since 2.0.0
 */
public class SelectCalendar {

    /**
     * カレンダ一覧を取得します。<BR>
     * 
     * オブジェクトの2次元配列として返す
     * 
     * @return
     * @throws FinderException
     * @throws NamingException
     */
    @SuppressWarnings("unchecked")
	public ArrayList getCalendarList() throws FinderException, NamingException {
        ArrayList list = new ArrayList();
  
        try {
			//全カレンダを取得
			Collection ct = CalendarInfoUtil.getLocalHome().findAll();
			
			if(ct != null){
			    Iterator itr = ct.iterator();
			    while(itr.hasNext()){
			    	CalendarInfoLocal cal = (CalendarInfoLocal)itr.next();
			        
			        ArrayList info = new ArrayList();
			        info.add(cal.getCalendar_id());
			        info.add(cal.getCalendar_name());
			        info.add(cal.getValid_time_from());
			        info.add(cal.getValid_time_to());
			        info.add(cal.getDescription());
			        info.add(cal.getReg_user());
			        info.add(cal.getReg_date());
			        info.add(cal.getUpdate_user());
			        info.add(cal.getUpdate_date());
			        info.add(Boolean.valueOf(true));
			        list.add(info);
			    }
			}
		} catch (FinderException e) {
			throw new FinderException("SelectCalendar.getCalendarList() find error : " + e.getMessage());
		} catch (NamingException e) {
			throw e;
		}
	    
        return list;
    }
    
    /**
     * カレンダ曜日別情報一覧を取得します。<BR>
     * 
     * オブジェクトの2次元配列として返す
     * 
     * @return
     * @throws FinderException
     * @throws NamingException
     */
    @SuppressWarnings("unchecked")
	public ArrayList getWeekdayList(String id) throws FinderException, NamingException {
        ArrayList list = new ArrayList();
  
        try {
			//カレンダIDの曜日別情報を取得
			Collection ct = CalendarWeekdayInfoUtil.getLocalHome().findByCalendarId(id);
			
			if(ct != null){
			    Iterator itr = ct.iterator();
			    while(itr.hasNext()){
			    	CalendarWeekdayInfoLocal cal = (CalendarWeekdayInfoLocal)itr.next();
			        
			        ArrayList info = new ArrayList();
			        info.add(cal.getWeekday());
			        info.add(cal.getTime_from());
			        info.add(cal.getTime_to());
			        info.add(cal.getDescription());
			        info.add(cal.getReg_user());
			        info.add(cal.getReg_date());
			        info.add(cal.getUpdate_user());
			        info.add(cal.getUpdate_date());
			        list.add(info);
			    }
			}
		} catch (FinderException e) {
			throw new FinderException("SelectCalendar.getWeekdayList() find error : calendarId=" + id);
		} catch (NamingException e) {
			throw e;
		}
	    
        return list;
    }
    
    /**
     * カレンダ追加情報(例外実行情報) 一覧を取得します。<BR>
     * 
     * オブジェクトの2次元配列として返す
     * 
     * @return
     * @throws FinderException
     * @throws NamingException
     */
    @SuppressWarnings("unchecked")
	public ArrayList getAddList(String id) throws FinderException, NamingException {
        ArrayList list = new ArrayList();
  
        try {
			//カレンダIDの追加情報を取得
			Collection ct = CalendarAddInfoUtil.getLocalHome().findByCalendarId(id);
			
			if(ct != null){
			    Iterator itr = ct.iterator();
			    while(itr.hasNext()){
			    	CalendarAddInfoLocal cal = (CalendarAddInfoLocal)itr.next();
			        
			        ArrayList info = new ArrayList();
			        info.add(Boolean.valueOf(RunConstant.typeToBoolean(cal.getExecute_flg().intValue())));
			        info.add(cal.getTime_from());
			        info.add(cal.getTime_to());
			        info.add(cal.getDescription());
			        info.add(cal.getReg_user());
			        info.add(cal.getReg_date());
			        info.add(cal.getUpdate_user());
			        info.add(cal.getUpdate_date());
			        list.add(info);
			    }
			}
		} catch (FinderException e) {
			throw new FinderException("SelectCalendar.getAddList() find error : calendarId=" + id);
		} catch (NamingException e) {
			throw e;
		}
	    
        return list;
    }
    
    /**
     * カレンダ追加情報(例外実行情報)一覧を取得します。<BR>
     * 
     * オブジェクトの2次元配列として返す
     * 
     * @param base
     * @return
     * @throws FinderException
     * @throws NamingException
     */
    @SuppressWarnings("unchecked")
	public ArrayList getSchedule(Date base) throws FinderException, NamingException {
        ArrayList list = new ArrayList();
  
        //全カレンダを取得
        Collection ct = CalendarInfoUtil.getLocalHome().findAll();
        
	    if(ct != null){
            Iterator itr = ct.iterator();
            while(itr.hasNext()){
            	CalendarInfoLocal cal = (CalendarInfoLocal)itr.next();
                
                ArrayList info = new ArrayList();
                info.add(cal.getCalendar_id());
                
        		for(int i = 0; i < ScheduleTableConstant.DAYS; i++){
            		Calendar baseCalendar = Calendar.getInstance();
            		baseCalendar.setTime(base);
        			baseCalendar.add(Calendar.DAY_OF_MONTH, i);
        			Date work = baseCalendar.getTime();
        			if(isSchedule(cal.getCalendar_id(), work)){
        				info.add(work);
        			}
        			else{
        				info.add(null);
        			}
        		}
        		list.add(info);
            }
	    }
	    
        return list;
    }
    
    /**
     * 実行可能かをチェックします。<BR>
     * 
     * 指定カレンダにて、指定した日時が実行可能かチェックし、Bool値を返します。
     * 
     * @param id
     * @param checkTimestamp
     * @return 
     * @throws FinderException
     * @throws NamingException
     */
	public Boolean isRun(String id, Date checkTimestamp) throws FinderException, NamingException {
    	Boolean run = Boolean.valueOf(false);
  
        try {
			//カレンダ情報を取得
			CalendarInfoLocal cal = CalendarInfoUtil.getLocalHome().findByPrimaryKey(id);
			if(cal != null){
				if(checkTimestamp.after(cal.getValid_time_from()) && 
						checkTimestamp.before(cal.getValid_time_to())){

			    	//追加情報を取得
			    	Collection ct = cal.getCalendarAddInfo();
			        Iterator itr = ct.iterator();
			        while(itr.hasNext()){
			        	CalendarAddInfoLocal add = (CalendarAddInfoLocal)itr.next();
			        	Boolean check = isRunByAddInfo(add, checkTimestamp);
			            if(check != null){
			            	if(check.booleanValue()){
			            		run = Boolean.valueOf(true);
			            	}
			            	else{
			            		run = Boolean.valueOf(false);
			            	}
			            	return run;
			            }
			        }
			        
			    	//曜日別情報を取得
			    	ct = cal.getCalendarWeekdayInfo();
			        itr = ct.iterator();
			        while(itr.hasNext()){
			        	CalendarWeekdayInfoLocal weekday = (CalendarWeekdayInfoLocal)itr.next();
			            if(isRunByWeekdayInfo(weekday, checkTimestamp)){
			            	run = Boolean.valueOf(true);
			            	return run;
			            }
			        }
				}
			}
		} catch (FinderException e) {
			throw new FinderException("SelectCalendar.isRun() find error : calendarId=" + id);
		} catch (NamingException e) {
			throw e;
		}
	    
        return run;
    }
    
    /**
     * 実行チェック（追加情報）
     * 
     * 指定カレンダの追加情報にて、指定した日時が実行可能かチェックし、Bool値を返す。
     * 
     * @param id
     * @param checkTimestamp
     * @return 
     * @throws FinderException
     * @throws NamingException
     */
	protected Boolean isRunByAddInfo(CalendarAddInfoLocal add, Date checkTimestamp) {
    	Boolean run = null;
  
        if(add != null){
        	if(checkTimestamp.after(add.getTime_from()) && 
        			checkTimestamp.before(add.getTime_to())){
        		run = Boolean.valueOf(RunConstant.typeToBoolean(add.getExecute_flg().intValue()));
        	}
	    }
	    
        return run;
    }
	
    /**
     * 実行チェック（曜日別情報）
     * 
     * 指定カレンダの曜日別情報にて、指定した日時が実行可能かチェックし、Bool値を返す。
     * 
     * @param id
     * @param checkTimestamp
     * @return 
     * @throws FinderException
     * @throws NamingException
     */
	protected Boolean isRunByWeekdayInfo(CalendarWeekdayInfoLocal weekday, Date checkTimestamp) {
    	Boolean run = Boolean.valueOf(false);
    	
    	Calendar work = Calendar.getInstance();
    	work.setTime(checkTimestamp);
    	int dayOfWeek = work.get(Calendar.DAY_OF_WEEK);
    	
    	Calendar checkCalendar = Calendar.getInstance();
    	checkCalendar.set(Calendar.YEAR, 1970);
    	checkCalendar.set(Calendar.MONTH, 0);
    	checkCalendar.set(Calendar.DAY_OF_MONTH, 1);
    	checkCalendar.set(Calendar.HOUR_OF_DAY, work.get(Calendar.HOUR_OF_DAY));
    	checkCalendar.set(Calendar.MINUTE, work.get(Calendar.MINUTE));
    	checkCalendar.set(Calendar.SECOND, work.get(Calendar.SECOND));
        Time checkTime = new Time(checkCalendar.getTime().getTime());

        if(weekday != null && dayOfWeek == weekday.getWeekday().intValue()){
        	if(checkTime.after(weekday.getTime_from()) && 
        			checkTime.before(weekday.getTime_to())){
        		run = Boolean.valueOf(true);
        	}
	    }
	    
        return run;
    }
	
    /**
     * 予定チェック
     * 
     * 指定カレンダにて、指定した日付に実行可能な時間帯があるかチェックし、Bool値を返す。
     * 
	 * @param id
	 * @param checkTimestamp
	 * @return
	 * @throws FinderException
	 * @throws NamingException
	 */
	public Boolean isSchedule(String id, Date checkTimestamp) throws FinderException, NamingException {
    	Boolean run = Boolean.valueOf(false);
  
        try {
			//カレンダ情報を取得
			CalendarInfoLocal cal = CalendarInfoUtil.getLocalHome().findByPrimaryKey(id);
			if(cal != null){
				if(checkTimestamp.after(cal.getValid_time_from()) && 
						checkTimestamp.before(cal.getValid_time_to())){
			    	//追加情報を取得
			        if(isScheduleByAddInfo(id, checkTimestamp)){
			        	run = Boolean.valueOf(true);
			        	return run;
			        }
			        
			    	//曜日別情報を取得
			        if(isScheduleByWeekdayInfo(id, checkTimestamp)){
			        	run = Boolean.valueOf(true);
			        	return run;
			        }
				}
			}
		} catch (FinderException e) {
			throw new FinderException("SelectCalendar.isSchedule() find error : calendarId=" + id);
		} catch (NamingException e) {
			throw e;
		}
	    
        return run;
    }
	
    /**
     * 予定チェック（追加情報）
     * 
     * 指定カレンダの追加情報にて、指定した日付に実行可能な時間帯があるかチェックし、Bool値を返す。
     * 
     * @param id
     * @param checkTimestamp
     * @return 
     * @throws FinderException 
     * @throws FinderException
     * @throws NamingException 
     * @throws NamingException
     */
	protected Boolean isScheduleByAddInfo(String id, Date checkTimestamp) throws FinderException, NamingException {
    	Boolean run = Boolean.valueOf(false);

		try {
			if(id != null && id.length() > 0){
				Collection ct = CalendarAddInfoUtil.getLocalHome().selectSchdule(id, checkTimestamp);
				if(ct.size() > 0){
					run = Boolean.valueOf(true);
				}
			}
		} catch (FinderException e) {
			throw new FinderException("SelectCalendar.isScheduleByAddInfo() find error : calendarId=" + id + ", checkTimestamp=" + checkTimestamp);
		} catch (NamingException e) {
			throw e;
		}

        return run;
    }
	
    /**
     * 予定チェック（曜日別情報）
     * 
     * 指定カレンダの曜日別情報にて、指定した日付に実行可能な時間帯があるかチェックし、Bool値を返す。
     * 
     * @param id
     * @param checkTimestamp
     * @return 
     * @throws FinderException 
     * @throws FinderException
     * @throws NamingException 
     * @throws NamingException
     */
	protected Boolean isScheduleByWeekdayInfo(String id, Date checkTimestamp) throws FinderException, NamingException {
    	Boolean run = Boolean.valueOf(false);
    	
    	Calendar work = Calendar.getInstance();
    	work.setTime(checkTimestamp);
    	int dayOfWeek = work.get(Calendar.DAY_OF_WEEK);

		try {
			if(id != null && id.length() > 0){
				Collection ct = CalendarWeekdayInfoUtil.getLocalHome().findByWeekday(id, Integer.valueOf(dayOfWeek));
				if(ct.size() > 0){
					
					Iterator itr = ct.iterator();
					while(itr.hasNext()){
						// 曜日情報を取得
						CalendarWeekdayInfoLocal weekday = (CalendarWeekdayInfoLocal)itr.next();
						
						long timeFrom = weekday.getTime_from().getTime();
						long timeTo = weekday.getTime_to().getTime();
						
						// 指定カレンダの追加情報にて、指定した日付に実行不可能な時間帯があるかチェック
						Calendar baseCalendar = Calendar.getInstance();
						baseCalendar.setTimeInMillis(timeFrom);
						work.set(Calendar.HOUR_OF_DAY, baseCalendar.get(Calendar.HOUR_OF_DAY));
						work.set(Calendar.MINUTE, baseCalendar.get(Calendar.MINUTE));
						work.set(Calendar.SECOND,baseCalendar.get(Calendar.SECOND));
						work.set(Calendar.MILLISECOND,0);
				        Date start = work.getTime();
				        
				        baseCalendar.setTimeInMillis(timeTo);
				        work.set(Calendar.HOUR_OF_DAY, baseCalendar.get(Calendar.HOUR_OF_DAY));
				        work.set(Calendar.MINUTE, baseCalendar.get(Calendar.MINUTE));
				        work.set(Calendar.SECOND, baseCalendar.get(Calendar.SECOND));
				        work.set(Calendar.MILLISECOND,0);
				        Date end = work.getTime();
						
				        Collection ctNo = CalendarAddInfoUtil.getLocalHome().selectNoSchdule(id, start, end);
						if(ctNo.size() <= 0){
							run = Boolean.valueOf(true);
							return run;
						}
					}
				}
			}
		} catch (FinderException e) {
			throw new FinderException("SelectCalendar.isScheduleByWeekdayInfo() find error : calendarId=" + id + ", checkTimestamp=" + checkTimestamp);
		} catch (NamingException e) {
			throw e;
		}
		
        return run;
    }
	
    /**
     * カレンダID一覧を取得します。<BR>
     * 
     * @return
     * @throws FinderException
     * @throws NamingException
     */
	@SuppressWarnings("unchecked")
	public ArrayList getCalendarIdList() throws FinderException, NamingException {
        ArrayList list = new ArrayList();
  
        try {
			//全カレンダを取得
			Collection ct = CalendarInfoUtil.getLocalHome().findAll();
			
			if(ct != null){
			    Iterator itr = ct.iterator();
			    while(itr.hasNext()){
			    	CalendarInfoLocal cal = (CalendarInfoLocal)itr.next();

			        list.add(cal.getCalendar_id());
			    }
			}
		} catch (FinderException e) {
			throw new FinderException("SelectCalendar.getCalendarIdList() find error " + e.getMessage());
		} catch (NamingException e) {
			throw e;
		}
	    
        return list;
    }
}
