/*
 
Copyright (C) 2009 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.vm.factory;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Locale;

import javax.ejb.CreateException;
import javax.ejb.FinderException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.bean.Property;
import com.clustercontrol.bean.PropertyConstant;
import com.clustercontrol.repository.bean.FacilityAttributeConstant;
import com.clustercontrol.repository.bean.FacilityTreeAttributeConstant;
import com.clustercontrol.repository.ejb.session.RepositoryControllerLocal;
import com.clustercontrol.repository.ejb.session.RepositoryControllerUtil;
import com.clustercontrol.repository.factory.ScopeProperty;
import com.clustercontrol.util.Messages;
import com.clustercontrol.util.PropertyUtil;
import com.clustercontrol.vm.VmException;

/**
 * 仮想化のリポジトリ割当更新用の処理クラス
 * 
 * @version 3.1.0
 * @since 3.1.0
 *
 */
public class UpdateVmNodeAllocation {

	protected static Log m_log = LogFactory.getLog(UpdateVmNodeAllocation.class);

	public static final int VM_SORT_ORDER = 9000;
	public static final int UNALLOCATED_SORT_ORDER = 9000;
	public static final int HOST_NODE_SCOPE_SORT_ORDER = 200;
	
	// リポジトリEJBのセッションBean
	RepositoryControllerLocal repository = null;
	
	/**
	 * 仮想化の割当情報最新化
	 */
	public void update(Locale locale) throws VmException, FinderException, CreateException, NamingException{
		
		updateVmNodeAllocation(locale);
	}
	
	
	/**
	 * ホストノード用スコープのfacilityIdを作成する
	 * @param hostId
	 * @return
	 */
	public String getHostNodeScopeFacilityId(String hostId){
		return hostId + "scope";
	}
	
	
	/**
	 * repository変数の初期化
	 * @throws VmException
	 */
	private void setRepositoryController() throws VmException{
	
		// クラス変数に設定されていない場合は設定する
        if(repository == null){
        	try {
        		repository = RepositoryControllerUtil.getLocalHome().create();
        	} catch (Exception e) {
				m_log.error("setRepositoryController() : ",e);
				VmException ve = new VmException("リポジトリへのアクセスに失敗しました。");
				ve.setStackTrace(e.getStackTrace());
				throw ve;
			}
        }
	}
	
	/**
	 * Vmの最新割当のマップ(hostFacilityIdと、guestFacilityIdのArrayList)を作成する。<BR>
	 * <BR>
	 * 
	 * @param locale
	 * @return
	 * @throws FinderException
	 * @throws NamingException
	 * @throws CreateException
	 * @throws VmException
	 */
	public void updateVmNodeAllocation(Locale locale) 
			throws FinderException, NamingException, CreateException, VmException {
		
		m_log.debug("updateManagementNode() : start");

		//戻り値
		HashMap<String, ArrayList<String>> newMap = new HashMap<String, ArrayList<String>>();
		HashMap<String, ArrayList<String>> releaseMap = new HashMap<String, ArrayList<String>>();
		
		////////////////////////////////////////////////////////
		// 初期化
		////////////////////////////////////////////////////////
		m_log.debug("updateManagementNode() : init");
		
		//共通変数
		ArrayList<String> hostNodeList = null;	//管理ノード=hostId
		ArrayList<String> guestNodeList = null;	//管理ノード=hostIdであるguestノード一覧
		ArrayList<String> unknownNodeList = null;	//現在の管理ノードが
        HashMap<String, String> condition = new HashMap<String, String>(); // ノード検索用
        HashMap<String, String> attribute = new HashMap<String, String>(); // ノードプロパティ更新用
		
        //リポジトリ
        setRepositoryController();
        
		//共通変数（外部API操作用）
        VmNodeController vmNodeController = new VmNodeController();

        
		//ホストノード一覧(hostNodeList)を取得（プラットフォームは意識しない）
		condition.clear();
		condition.put(FacilityAttributeConstant.VIRTNODETYPE, "host");
		condition.put(FacilityAttributeConstant.VALID, "TRUE");
		try{
			hostNodeList = repository.findByCondition(condition);
			
			//for debug
			Iterator<String> strItr = hostNodeList.iterator();
			while (strItr.hasNext()) {
				String string = (String) strItr.next();
				m_log.debug("updateManagementNode() : hostNodeList -> " + string);
			}
		} catch (FinderException e) {
			m_log.error("searchNode() : ",e);
			throw e;
		} catch (NamingException e) {
			m_log.error("searchNode() : ",e);
			throw e;
		}

		//unknownNodeListの初期値は「ノード種別=guest」
		condition.clear();
		condition.put(FacilityAttributeConstant.VIRTNODETYPE, "guest"); //Guestノードであること
		try{
			unknownNodeList = repository.findByCondition(condition);
			
			//for debug
			Iterator<String> strItr = unknownNodeList.iterator();
			while (strItr.hasNext()) {
				String string = (String) strItr.next();
				m_log.debug("updateManagementNode() : unknownNodeList -> " + string);
			}
		} catch (FinderException e) {
			m_log.error("searchNode() : ",e);
			throw e;
		} catch (NamingException e) {
			m_log.error("searchNode() : ",e);
			throw e;
		}
		newMap.put(FacilityTreeAttributeConstant.UNALLOCATED_SCOPE, new ArrayList<String>());
		releaseMap.put(FacilityTreeAttributeConstant.UNALLOCATED_SCOPE, new ArrayList<String>());
		
		////////////////////////////////////////////////////////
		//ホストノード単位で走査を行う
		////////////////////////////////////////////////////////
		m_log.debug("updateManagementNode() : search host");


		// リポジトリに登録済みのホストノードに対して
		Iterator<String> hostItr = hostNodeList.iterator();
		while (hostItr.hasNext()) {
			String hostId = (String) hostItr.next();
			m_log.debug("updateManagementNode() : search host hostId = " + hostId);

			newMap.put(hostId, new ArrayList<String>());
			releaseMap.put(hostId, new ArrayList<String>());
			
			////////////////////////////////////////////////////////
			//管理ノード=hostIdとなるゲストノード一覧を取得
			try{
				m_log.debug("updateManagementNode() : get guest list at vmManagementNode = " + hostId);
				
				condition.clear();
				condition.put(FacilityAttributeConstant.VIRTNODETYPE, "guest"); //Guestノードであること
				condition.put(FacilityAttributeConstant.VMMANAGEMENTNODE, hostId); //管理ノードがターゲットノード
				guestNodeList = repository.findByCondition(condition);
				
			} catch (FinderException e) {
				m_log.error("updateManagementNode() : ",e);
				throw e;
			} catch (NamingException e) {
				m_log.error("updateManagementNode() : ",e);
				throw e;
			}
			
			////////////////////////////////////////////////////////
			//個々のゲストノードが今の場所にいるかをチェック
			try {
				Iterator<String> guestItr = guestNodeList.iterator();
				while (guestItr.hasNext()) {
					String guestId = (String) guestItr.next();
					
					//現在もそこに居る場合
					if(vmNodeController.isAllocated(guestId, hostId, Locale.getDefault())){
						m_log.debug("updateManagementNode() : guestId = " + guestId + " is in hostId = " + hostId);
						newMap.get(hostId).add(guestId);
						if(!unknownNodeList.remove(guestId)){
							m_log.warn("updateManagementNode() : guestId = " + guestId + " is not exists?");							
						}
					}
					//現在は異なる場所に居る場合
					else{
						m_log.debug("updateManagementNode() : guestId = " + guestId + " is not in hostId = " + hostId);
						releaseMap.get(hostId).add(guestId);
					}
				}
			} catch (VmException e) {
				throw e;
			} catch (FinderException e) {
				m_log.error("updateManagementNode() : ",e);
				throw e;
			} catch (CreateException e) {
				m_log.error("updateManagementNode() : ",e);
				throw e;
			} catch (NamingException e) {
				m_log.error("updateManagementNode() : ",e);
				throw e;
			}	
		}

		////////////////////////////////////////////////////////
		//unknownListなゲストノードが今何処にいるかを逆に走査する
		////////////////////////////////////////////////////////
		m_log.debug("updateManagementNode() : search unknown guest");

		// 最新の配置が不明なノード一覧に対して
		Iterator<String> unknownItr = unknownNodeList.iterator();
		while (unknownItr.hasNext()) {
			try {
				String unknownId = (String) unknownItr.next();
				m_log.debug("updateManagementNode() : search unknown guest guestId = " + unknownId);

				////////////////////////////////////////////////////////
				//全ホストOSを探索しに行く
				boolean exists = false;

				hostItr = hostNodeList.iterator();
				while (hostItr.hasNext()) {
					String hostId = (String) hostItr.next();
					m_log.debug("updateManagementNode() : search unknown guest guestId = " + unknownId + " : search hostId " + hostId);
					
					//リポジトリに登録済みのホストノードの該当ノードが見つかれば管理ノードを最新化する
					if(vmNodeController.isAllocated(unknownId, hostId, Locale.getDefault())){
						m_log.debug("updateManagementNode() : search unknown guest guestId = " + unknownId + " : search hostId " + hostId + " exists!");

						//リポジトリの更新を行う
						attribute.clear();
						attribute.put(FacilityAttributeConstant.VMMANAGEMENTNODE, hostId);
						repository.setNodeDetail(unknownId, attribute);
						newMap.get(hostId).add(unknownId);
						
						// UNALLOCATED_SCOPEより削除（）
						releaseMap.get(FacilityTreeAttributeConstant.UNALLOCATED_SCOPE).add(unknownId);
						
						exists = true;
						break;
					}
					else{
						m_log.debug("updateManagementNode() : search unknown guest guestId = " + unknownId + " : search hostId " + hostId + " not exists!");
					}
				}
				
				////////////////////////////////////////////////////////
				//もし見つからなければ管理ノードをUNALLOCATIONにする
				if(!exists){

					m_log.debug("updateManagementNode() : search unknown guest guestId = " + unknownId + " : UNALLOCATION");

					//リポジトリの更新を行う
					attribute.clear();
					attribute.put(FacilityAttributeConstant.VMMANAGEMENTNODE,FacilityTreeAttributeConstant.UNALLOCATED_SCOPE);
					repository.setNodeDetail(unknownId, attribute);
					
					newMap.get(FacilityTreeAttributeConstant.UNALLOCATED_SCOPE).add(unknownId);
				}
			} catch (FinderException e) {
				m_log.error("updateManagementNode() : ",e);
				throw e;
			} catch (NamingException e) {
				m_log.error("updateManagementNode() : ",e);
				throw e;
			} catch (CreateException e) {
				m_log.error("updateManagementNode() : ",e);
				throw e;
			} catch (VmException e) {
				throw e;
			}
		}

		try{
			//更新処理実行
			updateScope(newMap, releaseMap, locale);
		}finally{
			//最後にリポジトリ更新のTOPICを送信する
			repository.sendRepositoryUpdateTopic();
		}
	}


	/**
	 * VMスコープ配下の割当情報を最新化する。<BR>
	 * <BR>
	 * 
	 * @param newMap 新しい割当定義情報。必要となるスコープが無い場合は作成し、定義内のノードを割り当てる。<BR>
	 * @param releaseMap スコープ割当解除のリスト。対象ホストノードスコープが無い場合は何もしない。<BR>
	 * @param locale
	 * @throws VmException
	 */
	public void updateScope(HashMap<String, ArrayList<String>> newMap, HashMap<String, ArrayList<String>> releaseMap, Locale locale)
		throws VmException{
		
		
		////////////////////////////////////////////////////////
		// 初期化
		////////////////////////////////////////////////////////
		m_log.debug("updateVmNodeAllocation() : init");
		
		ArrayList<String> guestNodeList = null;	//管理ノード=hostIdであるguestノード一覧
		ArrayList<String> list = new ArrayList<String>();
		
        //リポジトリ
        setRepositoryController();
		
		
		////////////////////////////////////////////////////////
		//基本環境の確認
		////////////////////////////////////////////////////////
		m_log.debug("updateVmNodeAllocation() : init enviroment");
		
		// VMスコープの存在確認と無い場合は作成する
		m_log.debug("updateVmNodeAllocation() : init VM scope");
        makeVmRelatedScope("", 
        		FacilityTreeAttributeConstant.VM_SCOPE, 
        		Messages.getString("vm.scope.vm"), "Auto Create", VM_SORT_ORDER, locale);
		
		// UNALLOCATIONスコープの存在確認と無い場合は作成する
		m_log.debug("updateVmNodeAllocation() : init UNALLOCATED scope");
        makeVmRelatedScope(FacilityTreeAttributeConstant.VM_SCOPE, 
        		FacilityTreeAttributeConstant.UNALLOCATED_SCOPE, 
        		Messages.getString("vm.scope.unallocated"), "Auto Create", UNALLOCATED_SORT_ORDER, locale);
		
		////////////////////////////////////////////////////////
		//ホストノード単位で走査を行う
		////////////////////////////////////////////////////////
		m_log.debug("updateVmNodeAllocation() : main");
        
        // ホストノードに対して
		Iterator<String> hostItr = newMap.keySet().iterator();
		while (hostItr.hasNext()) {
			try {
				String hostId = (String) hostItr.next();
				m_log.debug("updateVmNodeAllocation() : host node scope check : hostId = " + hostId);
				
				//ホストノード用のスコープの存在確認と無い場合は作成
				list.clear();
				list.add(FacilityAttributeConstant.CN);
				HashMap<String, String> map = repository.getNodeDetail(hostId, list);
				String facilityName = null;
				if(map != null && map.size() > 0)
					facilityName = (String)map.get(FacilityAttributeConstant.CN);
				else
					facilityName = hostId;
				m_log.debug("updateVmNodeAllocation() : add host node scope facilityName = " + facilityName);
				// ホストノード用スコープの作成
				if(hostId.equals(FacilityTreeAttributeConstant.UNALLOCATED_SCOPE)){
			        makeVmRelatedScope(FacilityTreeAttributeConstant.VM_SCOPE, 
			        		FacilityTreeAttributeConstant.UNALLOCATED_SCOPE, 
			        		facilityName, "Auto Create at " + (new Date()).toString(), HOST_NODE_SCOPE_SORT_ORDER, locale);					
				}else{
					makeVmRelatedScope(FacilityTreeAttributeConstant.VM_SCOPE, 
			        		getHostNodeScopeFacilityId(hostId), 
			        		facilityName, "Auto Create at " + (new Date()).toString(), HOST_NODE_SCOPE_SORT_ORDER, locale);
				}

				
				//必要の無いゲストノードをホストノード用スコープから削除する
				guestNodeList = releaseMap.get(hostId);
				if(guestNodeList != null && guestNodeList.size() > 0){
					m_log.debug("release host node scope " + hostId);
					if(hostId.equals(FacilityTreeAttributeConstant.UNALLOCATED_SCOPE)){
						releaseVmNodeAtHostNodeScope(FacilityTreeAttributeConstant.UNALLOCATED_SCOPE, guestNodeList);
					}
					else{
						releaseVmNodeAtHostNodeScope(getHostNodeScopeFacilityId(hostId), guestNodeList);
					}
				}
		        
				//個々のゲストノードをホストノード用スコープに追加する
				guestNodeList = newMap.get(hostId);
				if(guestNodeList != null && guestNodeList.size() > 0){
					m_log.debug("add host node scope " + hostId);
					if(hostId.equals(FacilityTreeAttributeConstant.UNALLOCATED_SCOPE)){
						assignVmNodeAtHostNodeScope(FacilityTreeAttributeConstant.UNALLOCATED_SCOPE, guestNodeList);
					}
					else{
						assignVmNodeAtHostNodeScope(getHostNodeScopeFacilityId(hostId), guestNodeList);
					}
				}
				
			} catch (VmException e) {
				m_log.error("updateVmNodeAllocation() : ",e);
				throw e;
			} catch (Exception e) {
				m_log.error("updateVmNodeAllocation() : ",e);
				VmException ve = new VmException("ホストノード用スコープの更新に失敗しました。");
				ve.setStackTrace(e.getStackTrace());
				throw ve;
			}
		}
		
	}

	
	/**
	 * 指定スコープが存在しないなら作成する。<BR>
	 * <BR>
	 * 
	 * @param parentFacilityId
	 * @param targetFacilityId
	 * @param targetFacilityName
	 * @param targetDescription
	 * @param sortOrder
	 * @param locale
	 * @throws VmException
	 */
	public void makeVmRelatedScope(
			String parentFacilityId,
			String targetFacilityId,
			String targetFacilityName,
			String targetDescription,
			int sortOrder,
			Locale locale) throws VmException{
		
		m_log.debug("makeVmRelatedScope() : start targetFacilityId " + targetFacilityId + " on " + parentFacilityId);

		
		////////////////////////////////////////////////////////
		// 初期化
		////////////////////////////////////////////////////////
		m_log.debug("makeVmRelatedScope() : init");
		
		String facilityId = null;
		ArrayList scopeList = null;

		setRepositoryController();
        //ScopeProperty scopeProperty = repository.getScopeProperty();
		
		////////////////////////////////////////////////////////
		// メイン処理
		////////////////////////////////////////////////////////
		m_log.debug("makeVmRelatedScope() : main");
		
		try {
			// 対象スコープの存在チェックフラグ
			boolean exists = false;
			
			// 直下のスコープに対して
			scopeList = repository.getScopeList(parentFacilityId);
			Iterator<ArrayList> scopeItr = scopeList.iterator();
			while (scopeItr.hasNext()) {
				ArrayList arrayList = (ArrayList) scopeItr.next();
				facilityId = (String)arrayList.get(0);
				m_log.debug("makeVmRelatedScope() : search scope : facilityId = " + facilityId);
				
				// 存在するか？
				if(facilityId != null && 
						facilityId.equals(targetFacilityId)){
					m_log.debug("makeVmRelatedScope() : Scope " + targetFacilityId + " is in " + parentFacilityId);
					exists = true;
				}
			}
			
			//存在しないならVMスコープを作成する
			if(!exists){
				m_log.debug("makeVmRelatedScope() : Scope " + targetFacilityId + " is not in " + parentFacilityId);

				// 空のプロパティオブジェクトを作成
				Property property = repository.getScopeProperty(PropertyConstant.MODE_MODIFY, locale);
				
				// ファシリティID
				ArrayList propertyList = PropertyUtil.getProperty(property, ScopeProperty.FACILITY_ID);
				((Property)propertyList.get(0)).setValue(targetFacilityId);
				// ファシリティ名
				propertyList = PropertyUtil.getProperty(property, ScopeProperty.FACILITY_NAME);
				((Property)propertyList.get(0)).setValue(targetFacilityName);
				// 説明
				propertyList = PropertyUtil.getProperty(property, ScopeProperty.DESCRIPTION);
				((Property)propertyList.get(0)).setValue(targetDescription);

				// 作成（ここではリポジトリ更新TOPICを送信しない）
				repository.addScope(parentFacilityId, property, sortOrder,false);//TOPIC未送信
				m_log.debug("makeVmRelatedScope() : Scope " + targetFacilityId + " make succeeded ! ");
			}
		} catch (Exception e) {
			m_log.error("makeVmRelatedScope() : ",e);
			VmException ve = new VmException(targetFacilityId + "スコープの作成に失敗しました。");
			ve.setStackTrace(e.getStackTrace());
			throw ve;
		} 

	}	
	
	

	/**
	 * 指定ホストノード用スコープにノードを割り当てる。<BR>
	 * <BR>
	 * 
	 * @param hostNodeScopeId
	 * @param guestNodeList
	 * @throws VmException
	 */
	public void assignVmNodeAtHostNodeScope(String hostNodeScopeId, ArrayList<String> guestNodeList)
		throws VmException{
		
		m_log.debug("assignVmNodeAtHostNodeScope() : start hostNodeScopeId = " + hostNodeScopeId);

		////////////////////////////////////////////////////////
		// 初期化
		////////////////////////////////////////////////////////
		m_log.debug("assignVmNodeAtHostNodeScope() : init");
		
		ArrayList scopeList = null;
		String facilityId = null;
		
		////////////////////////////////////////////////////////
		// メイン処理
		////////////////////////////////////////////////////////
		m_log.debug("assignVmNodeAtHostNodeScope() : main");
		try {
			scopeList = repository.getScopeList(hostNodeScopeId);

			// ゲストノードが既に割り当てられているか
			Iterator<String> guestItr = guestNodeList.iterator();
			while (guestItr.hasNext()) {
				String guestId = (String) guestItr.next();
				m_log.debug("assignVmNodeAtHostNodeScope() : loop guestId = " + guestId);

				
				boolean exists = false;
				
				// 存在確認
				Iterator scopeItr = scopeList.iterator();
				while (scopeItr.hasNext()) {
					ArrayList arrayList = (ArrayList) scopeItr.next();
					facilityId = (String)arrayList.get(0);
					
					if(guestId.equals(facilityId)){
						exists = true;
						break;
					}
				}
				
				// 存在しなければ登録する（ここではリポジトリ更新TOPICを送信しない）
				if(!exists){
					m_log.debug("assignVmNodeAtHostNodeScope() : add guestId = " + guestId);
					String[] glist = {guestId};
					repository.assignNodeScope(hostNodeScopeId, glist, false);//TOPIC未送信
					m_log.debug("assignVmNodeAtHostNodeScope() : add guestId = " + guestId + " succeeded!!");
				}else{
					m_log.debug("assignVmNodeAtHostNodeScope() : already exists : guestId = " + guestId);
				}
					
			}
		} catch (Exception e) {
			m_log.error("assignVmNodeAtHostNodeScope() : ",e);
			VmException ve = new VmException(hostNodeScopeId + "スコープへのゲストノード割当に失敗しました。");
			ve.setStackTrace(e.getStackTrace());
			throw ve;
		} 
	}
	
	
	/**
	 * 指定ホストノード用スコープからノード割当を解除する。<BR>
	 * <BR>
	 * @param hostNodeScopeId
	 * @param guestNodeList
	 * @throws VmException
	 */
	public void releaseVmNodeAtHostNodeScope(String hostNodeScopeId, ArrayList<String> guestNodeList)
		throws VmException{
		
		m_log.debug("releaseVmNodeAtHostNodeScope() : start hostNodeScopeId = " + hostNodeScopeId);

		////////////////////////////////////////////////////////
		// 初期化
		////////////////////////////////////////////////////////
		m_log.debug("releaseVmNodeAtHostNodeScope() : init");
		
		ArrayList scopeList = null;
		String facilityId = null;
		
		////////////////////////////////////////////////////////
		// メイン処理
		////////////////////////////////////////////////////////
		m_log.debug("releaseVmNodeAtHostNodeScope() : main");
		try {
			scopeList = repository.getScopeList(hostNodeScopeId);

			// ゲストノードが既に割り当てられているか
			Iterator<String> guestItr = guestNodeList.iterator();
			while (guestItr.hasNext()) {
				String guestId = (String) guestItr.next();
				m_log.debug("releaseVmNodeAtHostNodeScope() : loop guestId = " + guestId);

				// 存在確認
				Iterator scopeItr = scopeList.iterator();
				while (scopeItr.hasNext()) {
					ArrayList arrayList = (ArrayList) scopeItr.next();
					facilityId = (String)arrayList.get(0);
					
					if(guestId.equals(facilityId)){
						m_log.debug("releaseVmNodeAtHostNodeScope() : release guestId = " + guestId);
						String[] glist = {guestId};
						repository.releaseNodeScope(hostNodeScopeId, glist, false);
						break;
					}
				}
			}
		} catch (Exception e) {
			m_log.error("releaseVmNodeAtHostNodeScope() : ",e);
			VmException ve = new VmException(hostNodeScopeId + "スコープへのゲストノード割当解除に失敗しました。");
			ve.setStackTrace(e.getStackTrace());
			throw ve;
		} 
	}
}
