/*

 Copyright (C) 2008 NTT DATA Corporation

 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation, version 2.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 */

package com.clustercontrol.poller.impl;

import java.math.BigInteger;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.Vector;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.sblim.wbem.cim.CIMDataType;
import org.sblim.wbem.cim.CIMInstance;
import org.sblim.wbem.cim.CIMNameSpace;
import org.sblim.wbem.cim.CIMObjectPath;
import org.sblim.wbem.cim.CIMTransportException;
import org.sblim.wbem.cim.CIMValue;
import org.sblim.wbem.cim.UnsignedInt16;
import org.sblim.wbem.cim.UnsignedInt32;
import org.sblim.wbem.cim.UnsignedInt64;
import org.sblim.wbem.cim.UnsignedInt8;
import org.sblim.wbem.client.CIMClient;
import org.sblim.wbem.client.PasswordCredential;
import org.sblim.wbem.client.UserPrincipal;
import org.sblim.wbem.util.SessionProperties;

import com.clustercontrol.bean.PollerProtocolConstant;
import com.clustercontrol.sharedtable.DataTable;

/**
 * WBEMのポーリングを実行するクラス
 * 
 * @version 3.1.0
 * @since 3.1.0
 */
public class WbemPollerImpl {
	protected static Log m_log = LogFactory.getLog(WbemPollerImpl.class);

	// IPアドレス（ポーリングの度に更新）
	private String m_ipAddress;

	// 収集のターゲットとなるCIMクラスとプロパティ（全てのポーリングが終了するまで値を保持）
	private String[] m_cimText;

	// 取得したデータを格納するテーブル（ポーリングの度に更新）
	private DataTable m_dataTable;	
	
	// CIMサーバアクセス用アドレス(例: http://xx.xx.xx.xx:5988)
	private String m_cimAgentAddress = null;
	
	// 名前空間(デフォルト: root/cimv2)
	private String m_nameSpace = "root/cimv2";
	
	// 結果を格納するHashMap（Key:CIMクラス.プロパティ名）
	private HashMap<String, CIMValue> m_retMap = new HashMap<String, CIMValue>();
		
	// クラス変数の初期化
	private void init(){

	}

	/**
	 * メインルーチン　
	 * IPアドレスと　DataTableを受け取り、
	 * ポーリングした結果をDataTableに代入する
	 *  
	 * @param ipAddress IPアドレス
	 * @param port ポート番号
	 * @param protocol 通信プロトコル
	 * @param user 接続ユーザ
	 * @param password 接続パスワード
	 * @param nameSpace 名前空間
	 * @param retries １回のポーリングでのリトライ回数
	 * @param timeout ポーリングのタイムアウト
	 * @param cimList 対象CIMクラス、プロパティのリスト
	 * @param indexCheckFlg ポーリング結果のインデックスが揃っているかのチェック
	 * @param table 取得したMIB値を格納するテーブル
	 */
	public void polling(
			InetAddress ipAddress, 
			int port, 
			String protocol,
			String user,
			String password,
			String nameSpace,
			int retries, 
			int timeout, 
			List<String> cimList, // cimクラスとプロパティ名のリスト
			DataTable table) {
		// クラス変数を初期化
		init();

		m_dataTable = table;
		m_ipAddress = ipAddress.getHostAddress();
		m_cimAgentAddress = protocol + "://" + m_ipAddress + ":" + port;
		
		if(nameSpace != null) {
			m_nameSpace = nameSpace;
		}
		
		// デバッグ出力
		if (m_log.isDebugEnabled()) {
			m_log.debug("polling() start : " + m_ipAddress.toString());
			m_log.debug("Port            : " + port);
			m_log.debug("Protocol        : " + protocol);
			m_log.debug("User            : " + user);
			m_log.debug("Password        : " + password);
			m_log.debug("Retries         : " + retries);
			m_log.debug("Timeout         : " + timeout);
		}	
		
		long enumerationStart = System.currentTimeMillis();
		
		CIMClient cimClient = null;
		
		try {
		
			// *****************************
			// 1. Create user credentials
			// *****************************
			UserPrincipal userPr = new UserPrincipal(user);
			PasswordCredential pwCred = new PasswordCredential(password);	
			
			// *****************************
			// 2. Set NameSpace
			// - URL is set like: http(s)://<IP>:Port
			// - Namespace does not need to be specified in COPs if set in this constuctor
			// - There is no server authentication being done. Thus: No need for a truststore
			// *****************************
			CIMNameSpace ns = new CIMNameSpace(m_cimAgentAddress, m_nameSpace);
			
			// *****************************
			// 3. Create CIM Client
			// *****************************
			cimClient = new CIMClient(ns, userPr, pwCred);
			
			// *****************************
			// 4. Create Session Properties
			// *****************************
			SessionProperties properties = cimClient.getSessionProperties();
			if(properties == null){
				properties = new SessionProperties();
				cimClient.setSessionProperties(properties);
			}
			properties.setHttpTimeOut(timeout);
			
			m_cimText = new String[cimList.size()];
			
			
			// 問い合わせ用のHashMapを作成
			HashMap<String, ArrayList<String>> requestMap = new HashMap<String, ArrayList<String>>();
			Iterator<String> itr = cimList.iterator();
			String[] targetValue = new String[2];
			String cimClass = "";
			String cimProperty = "";
			ArrayList<String> propertyList = null;
			
			int i = 0;
			while(itr.hasNext()) {
				
				m_cimText[i] = itr.next();
				
				targetValue = m_cimText[i].split("\\.");
				cimClass = targetValue[0];
				cimProperty = targetValue[1];
				
				propertyList = requestMap.get(cimClass);
				
				// 既に存在する場合は、プロパティ名を追加する
				if(propertyList != null && propertyList.size() != 0){
					propertyList.add(cimProperty);
				}
				// 存在しない場合は、新しく作成する
				else {
					propertyList = new ArrayList<String>();
					propertyList.add(cimProperty);
					requestMap.put(cimClass, propertyList);
				}
				
				i++;
			}
			
			
			CIMObjectPath cop = null;
			CIMInstance ci = null;
			CIMValue value = null;
			Enumeration<CIMInstance> enm = null;
			Iterator<String> itr2 = null;
			
			// 設定したリトライ回数分リトライする
			for(int j = 0; j < retries; j++) {
				boolean errorFlg = false;
				try {
					Set<String> keySet = requestMap.keySet();
					itr = keySet.iterator();
					
					// 値の取得
					while (itr.hasNext()) {
						
						cimClass = itr.next();
						propertyList = requestMap.get(cimClass);
						
						m_log.debug("CIMClass : " + cimClass);
						
						cop = new CIMObjectPath(cimClass);
						enm = cimClient.enumInstances(cop, true);
						
						i = 0;
						while(enm.hasMoreElements()) {
							
							ci = enm.nextElement();
							itr2 = propertyList.iterator();
							
							while(itr2.hasNext()) {
								cimProperty = itr2.next();
								
								if(ci.getProperty(cimProperty) != null){
									
									value = ci.getProperty(cimProperty).getValue();
									m_retMap.put(cimClass+"."+cimProperty+"."+i, value);
									
								}
							}
							i++;
						}
						
					}
				}
				catch(CIMTransportException e){
					errorFlg = true;
					m_log.warn("polling() warning :" + m_ipAddress.toString() + " " + e.getID());
				}
				catch(Exception e) {
					errorFlg = true;
					m_log.warn("polling() warning :" + m_ipAddress.toString() + " unforeseen error");
				}
				finally {
					if(!errorFlg){
						break;
					}
				}
			}
		}
		catch(Exception e){
			m_log.warn("polling() warning :" + m_ipAddress.toString() + " unforeseen error");
		}
		finally {
			if(cimClient != null) {
				try {
					cimClient.close();
				}
				catch (Exception e) {
					m_log.warn("polling():" + m_ipAddress.toString() + " Session close failed");
				}
			}
		}
		
		long enumerationStop = System.currentTimeMillis();
		
		// デバッグ出力
		if (m_log.isDebugEnabled()) {
			m_log.debug("polling() end : time : " + (enumerationStop - enumerationStart));
		}
		
		
		
		// ***************
		// 結果を格納
		// ***************
		try {
			// 結果に何も格納されていない場合は、何もせず終了
			if (m_retMap == null || m_retMap.size() == 0) {
				m_log.debug("wbemReceived() : " + m_ipAddress.toString() + " result is empty");
				return;
			}
			
			long time = System.currentTimeMillis(); // 取得時刻
			
			Set<String> keySet = m_retMap.keySet();
			Iterator<String> itr = keySet.iterator();
			
			while(itr.hasNext()) {
				
				String cimString = itr.next();
				CIMValue value = m_retMap.get(cimString);
				
				if(value.getType().getType() == CIMDataType.UINT8){
					long ret = ((UnsignedInt8)value.getValue()).longValue();
					m_dataTable.putValue(getEntryKey(cimString), time, ret);
					
					m_log.debug("polling() dataTable put : " +
							"entryKey : " + getEntryKey(cimString) + 
							", time : " + time +
							", value : " + ret);
				}
				else if(value.getType().getType() == CIMDataType.UINT16){
					long ret = ((UnsignedInt16)value.getValue()).longValue();
					m_dataTable.putValue(getEntryKey(cimString), time, ret);
					
					m_log.debug("polling() dataTable put : " +
							"entryKey : " + getEntryKey(cimString) + 
							", time : " + time +
							", value : " + ret);
				}
				else if(value.getType().getType() == CIMDataType.UINT32){
					long ret = ((UnsignedInt32)value.getValue()).longValue();
					m_dataTable.putValue(getEntryKey(cimString), time, ret);
					
					m_log.debug("polling() dataTable put : " +
							"entryKey : " + getEntryKey(cimString) + 
							", time : " + time +
							", value : " + ret);
				}
				else if(value.getType().getType() == CIMDataType.UINT64) {
					BigInteger bigInt = ((UnsignedInt64)value.getValue()).bigIntValue();
					long ret = bigInt.longValue();
					m_dataTable.putValue(getEntryKey(cimString), time, ret);
					
					m_log.debug("polling() dataTable put : " +
							"entryKey : " + getEntryKey(cimString) + 
							", time : " + time +
							", value : " + ret);
				}
				else if(value.getType().getType() == CIMDataType.STRING) {
					String ret = (String)value.getValue();
					m_dataTable.putValue(getEntryKey(cimString), time, ret);
					
					m_log.debug("polling() dataTable put : " +
							"entryKey : " + getEntryKey(cimString) + 
							", time : " + time +
							", value : " + ret);
					
				}
				else if(value.getType().getType() == CIMDataType.STRING_ARRAY){
					
					Vector<String> ret = (Vector<String>)value.getValue();
					m_dataTable.putValue(getEntryKey(cimString), time, ret);
					
					m_log.debug("polling() dataTable put : " +
							"entryKey : " + getEntryKey(cimString) + 
							", time : " + time +
							", value : " + ret);
				}
				else {
					m_log.debug("polling() data type is nothing");
				}
				
			}
		} catch (Exception e) { // 何か例外が生じた場合は収集を停止する
			m_log.error("InternalError ", e);

			// 収集したデータを全てクリアする
			m_dataTable.clear();
		}
		
	}
	
	
	
	/**
	 * DataTableに格納するためのEntryKeyを返すメソッド
	 * 
	 * @param cimString CIMクラスとプロパティの文字列
	 */
	private String getEntryKey(String cimString){
		
		return PollerProtocolConstant.PROTOCOL_WBEM + "." + cimString;
	}
	
	/**
	 * 単体試験用
	 * 
	 * @param args
	 */
	public static void main(String[] args) {
		// パラメータを設定
		WbemPollerImpl poller = new WbemPollerImpl();
		
		InetAddress ipAddress = null;
		try {
			ipAddress = InetAddress.getByName(args[0]);
		} catch (UnknownHostException e) {
			m_log.error(e, e);
			return;
		}
		int port = Integer.parseInt(args[1]);
		String protocol = args[2];
		String user = args[3];
		String password = args[4];
		String nameSpace = args[5];  // 設定不可
		int retries = Integer.parseInt(args[6]);
		int timeout = Integer.parseInt(args[7]);
		ArrayList<String> cims = new ArrayList<String>();

		for(int i=9; i<args.length; i++){
			cims.add(args[i]);
		}

		// ポーリングを行い値を収集
		DataTable dataTable = new DataTable();
		
		poller.polling(
				ipAddress, 
				port,
				protocol,
				user,
				password,
				nameSpace,
				retries,
				timeout,
				cims, 
				dataTable);
		
	}
}
