/*
                                                                                                
Copyright (C) 2008 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.
 
*/

package com.clustercontrol.poller;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.net.InetAddress;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.concurrent.ConcurrentHashMap;

import javax.ejb.EJBException;
import javax.ejb.FinderException;
import javax.naming.NameNotFoundException;

import com.clustercontrol.bean.SnmpVersionConstant;
import com.clustercontrol.poller.cfg.PollerConfig;
import com.clustercontrol.poller.cfg.SnmpPollerConfig;
import com.clustercontrol.poller.cfg.WbemPollerConfig;
import com.clustercontrol.poller.cfg.VmPollerConfig;
import com.clustercontrol.repository.bean.FacilityAttributeConstant;
import com.clustercontrol.repository.ejb.session.RepositoryControllerLocal;
import com.clustercontrol.repository.ejb.session.RepositoryControllerUtil;
import com.clustercontrol.sharedtable.DataTableNotFoundException;

public class PollerManager {
	private static Log m_log = LogFactory.getLog( PollerManager.class );
	
	private String m_quartzJndiName = "QuartzRAM";
	
	private String m_managerName;
	
	/**
	 * Key   : ポーラ識別情報
	 * Value : ポーラ
	 */
	private ConcurrentHashMap<PollerInfo, PollingController> m_pollerTable;
	
	public PollerManager(String managerName){
		m_managerName = managerName;
		m_pollerTable = new ConcurrentHashMap<PollerInfo, PollingController>();
	}

	public PollingController createPoller(
			String group, 
			String facilityId, 
			boolean indexCheckFlg,
			String tableGroup, 
			String tableName)
	throws NotInitializedException, DataTableNotFoundException, FacilityNotFoundException{
		if(m_log.isDebugEnabled()){
			m_log.debug(
					"create poller    " + 
					"PollerGroup : " + group + ", " +
					"FacilityId : " + facilityId + ", " +
					"IndexCheckFlg : " + indexCheckFlg + ", " +
					"TableGroup : " + tableGroup + ", " +
					"TableName : " + tableName
			);
		}
		
		// リポジトリの情報から各種ポーリング設定を取得する
		PollerConfig pollerConfig;
		try {
			pollerConfig = createPollerConfig(facilityId);
		} catch (FacilityNotFoundException e) {
			throw e;
		}
		
		// 収集情報、情報書き込み対象テーブル情報を格納したポーラを生成する
		// この時点ではインスタンスを生成するだけでポーリングはスケジューリングしないため、
		// ポーリングは実行されない
		PollingController poller = new PollingController(
						group, 
						facilityId, 
						pollerConfig, 
						indexCheckFlg,
						tableGroup, 
						tableName);
		
		// ジョブスケジューラに利用するQuartzのJNDI名を設定
		poller.setQuartzJndiName(m_quartzJndiName);
		
		// マップに登録する
		m_pollerTable.put(new PollerInfo(group, facilityId), poller);
		return poller;
	}
	
	/**
	 * 登録されているポーラを削除します。ポーラが実行中の場合は削除できません。
	 * @param group ポーラグループ
	 * @param facilityId ファシリティID
	 */
	public void removePoller(String group, String facilityId){
		PollerInfo info = new PollerInfo(group, facilityId);
		
		PollingController poller = m_pollerTable.get(info);
		
		if(poller != null){
			// 全ての収集名の収集を終了
			poller.stopPollingAll();
		}
		
		m_pollerTable.remove(new PollerInfo(group, facilityId));
	}

	
	private PollerConfig createPollerConfig(String facilityId) throws FacilityNotFoundException{
		m_log.debug("createPollerConfig() start " + facilityId);
		
		try {
			RepositoryControllerLocal bean = RepositoryControllerUtil.getLocalHome().create();
			ArrayList<String> attributes = new ArrayList<String>();
			
			// *****************************
			// Pollerに共通の設定を取得
			// *****************************
			attributes.add(FacilityAttributeConstant.IPPROTOCOLNUMBER);
			attributes.add(FacilityAttributeConstant.IPNETWORKNUMBER);
			attributes.add(FacilityAttributeConstant.IPNETWORKNUMBERV6);
			attributes.add(FacilityAttributeConstant.MODIFYTIMESTAMP); // 最終更新日時
			
			
			// *****************************
			// SNMPの設定を取得
			// *****************************
			attributes.add(FacilityAttributeConstant.SNMPCOMMUNITY); // コミュニティ名
			attributes.add(FacilityAttributeConstant.SNMPPORT); // ポート
			attributes.add(FacilityAttributeConstant.SNMPVERSION); // バージョン
			attributes.add(FacilityAttributeConstant.SNMPTIMEOUT); // タイムアウト
			attributes.add(FacilityAttributeConstant.SNMPRETRIES); // リトライ回数
			
			
			// *****************************
			// WBEMの設定を取得
			// *****************************
			attributes.add(FacilityAttributeConstant.WBEMUSER); // Wbem接続ユーザ名
			attributes.add(FacilityAttributeConstant.WBEMUSERPASSWORD); // Wbem接続ユーザパスワード
			attributes.add(FacilityAttributeConstant.WBEMPORT); // Wbem接続ポート
			attributes.add(FacilityAttributeConstant.WBEMPROTOCOL); // Wbem接続プロトコル
			attributes.add(FacilityAttributeConstant.WBEMTIMEOUT); // Wbem接続タイムアウト
			attributes.add(FacilityAttributeConstant.WBEMRETRIES); // Wbem接続リトライ回数
			
			
			// *****************************
			// 仮想化の設定を取得
			// *****************************
			attributes.add(FacilityAttributeConstant.VMNAME);
			attributes.add(FacilityAttributeConstant.VMUSER);
			attributes.add(FacilityAttributeConstant.VMUSERPASSWORD);
			attributes.add(FacilityAttributeConstant.VMPROTOCOL);
			attributes.add(FacilityAttributeConstant.VIRTNODETYPE);
			attributes.add(FacilityAttributeConstant.VMMANAGEMENTNODE);
			
			
			// リポジトリから値を取得する
			HashMap nodeInfo = bean.getNodeDetail(facilityId, attributes);
			bean.remove();

			// 「IPアドレスのバージョン」により指定されたIPアドレスを設定する。
			Integer ipProtNum = (Integer)nodeInfo.get(FacilityAttributeConstant.IPPROTOCOLNUMBER);
			String ipAddress = null;
			if(ipProtNum != null && ipProtNum == 6){
				ipAddress = (String)nodeInfo.get(FacilityAttributeConstant.IPNETWORKNUMBERV6);
			} else {
				ipAddress = (String)nodeInfo.get(FacilityAttributeConstant.IPNETWORKNUMBER);
			}

			PollerConfig pollerConfig = new PollerConfig(InetAddress.getByName(ipAddress));
			
			
			// *****************************
			// SNMPの関連の設定をpollerConfigに追加
			// *****************************
			SnmpPollerConfig snmpConfig = pollerConfig.getSnmpConfig();
			
			String community = (String)nodeInfo.get(FacilityAttributeConstant.SNMPCOMMUNITY);		
			m_log.debug("community : " + community);
			if(community != null){
				snmpConfig.setComunity(community);
			}

			Integer port = (Integer)nodeInfo.get(FacilityAttributeConstant.SNMPPORT);
			m_log.debug("port      : " + port);
			if(port != null){
				snmpConfig.setPort(port);
			}

			String versionStr = (String)nodeInfo.get(FacilityAttributeConstant.SNMPVERSION);
			m_log.debug("version   : " + versionStr);
			if(versionStr != null && versionStr != ""){
				Integer version = SnmpVersionConstant.stringToType(versionStr);
				if(versionStr != null){
					// 未定義(-1)の場合はv2cに設定
					if(version == -1){
						version = 1;
					}
					
					snmpConfig.setVersion(version);
				}
			}

			Integer timeout = (Integer)nodeInfo.get(FacilityAttributeConstant.SNMPTIMEOUT);
			if(timeout != null){
				m_log.debug("timeout   : " + timeout);
				snmpConfig.setTimeout(timeout);
			}
			
			Integer retries = (Integer)nodeInfo.get(FacilityAttributeConstant.SNMPRETRIES);
			m_log.debug("retries   : " + retries);
			if(retries != null){
				snmpConfig.setRetries(retries);
			}
			
			// *****************************
			// WBEM関連の設定をpollerConfigに追加
			// *****************************
			WbemPollerConfig wbemConfig = pollerConfig.getWbemConfig();
			String wbemUser = (String)nodeInfo.get(FacilityAttributeConstant.WBEMUSER);		
			m_log.debug("wbemuser : " + wbemUser);
			if(wbemUser != null && !wbemUser.equals("")){
				wbemConfig.setUserName(wbemUser);
			}
			
			String wbemPassword = (String)nodeInfo.get(FacilityAttributeConstant.WBEMUSERPASSWORD);		
			m_log.debug("wbempassword : " + wbemPassword);
			if(wbemPassword != null && !wbemPassword.equals("")){
				wbemConfig.setPassword(wbemPassword);
			}
			
			Integer wbemPort = (Integer)nodeInfo.get(FacilityAttributeConstant.WBEMPORT);		
			m_log.debug("wbemport : " + wbemPort);
			if(wbemPort != null){
				wbemConfig.setPort(wbemPort.intValue());
			}
			
			String wbemProtocol = (String)nodeInfo.get(FacilityAttributeConstant.WBEMPROTOCOL);		
			m_log.debug("wbemprotocol : " + wbemProtocol);
			if(wbemProtocol != null && !wbemProtocol.equals("")){
				wbemConfig.setProtocol(wbemProtocol);
			}
			
			Integer wbemTimeout = (Integer)nodeInfo.get(FacilityAttributeConstant.WBEMTIMEOUT);		
			m_log.debug("wbemtimeout : " + wbemTimeout);
			if(wbemTimeout != null){
				wbemConfig.setTimeout(wbemTimeout.intValue());
			}
			
			Integer wbemRetries = (Integer)nodeInfo.get(FacilityAttributeConstant.WBEMRETRIES);		
			m_log.debug("wbemretries : " + wbemRetries);
			if(wbemRetries != null){
				wbemConfig.setRetries(wbemRetries.intValue());
			}
			
			// *****************************
			// 仮想化関連の設定をpollerConfigに追加
			// *****************************
			VmPollerConfig vmConfig = pollerConfig.getVmConfig();
			
			String vmName = (String)nodeInfo.get(FacilityAttributeConstant.VMNAME);
			m_log.debug("vmname : " + vmName);
			if(vmName != null){
				vmConfig.setVmName(vmName);
			}
			
			String vmNodeType = (String)nodeInfo.get(FacilityAttributeConstant.VIRTNODETYPE);
			m_log.debug("vmNodeType : " + vmNodeType);
			if(vmNodeType != null) {
				vmConfig.setVmNodeType(vmNodeType);
			}
			
			// ホストノードから値を取得する
			String vmManagementNode = null;
			if(vmNodeType != null) {
				if(vmNodeType.equals("host")) {
					vmManagementNode = facilityId;
				} else if(vmNodeType.equals("guest")){
					vmManagementNode = (String)nodeInfo.get(FacilityAttributeConstant.VMMANAGEMENTNODE);
				}
			}
			if (vmManagementNode != null) {
				HashMap vmManagementNodeInfo = bean.getNodeDetail(vmManagementNode, attributes);
				bean.remove();
				String vmManagementNodeIp = null;
				// 「IPアドレスのバージョン」により指定されたIPアドレスを設定する。
				Integer ipProtNumHost = (Integer)vmManagementNodeInfo.get(FacilityAttributeConstant.IPPROTOCOLNUMBER);
				if(ipProtNumHost != null && ipProtNumHost == 6){
					vmManagementNodeIp = (String)vmManagementNodeInfo.get(FacilityAttributeConstant.IPNETWORKNUMBERV6);
				} else {
					vmManagementNodeIp = (String)vmManagementNodeInfo.get(FacilityAttributeConstant.IPNETWORKNUMBER);
				}
				if(vmManagementNodeIp != null) {
					vmConfig.setVmManagementNodeIp(vmManagementNodeIp);
				}
				
				String vmUser = (String)vmManagementNodeInfo.get(FacilityAttributeConstant.VMUSER);
				m_log.debug("vmUser : " + vmUser);
				if(vmUser != null){
					vmConfig.setVmUser(vmUser);
				}
				
				String vmUserPassword = (String)vmManagementNodeInfo.get(FacilityAttributeConstant.VMUSERPASSWORD);
				m_log.debug("vmUserPassword : " + vmUserPassword);
				if(vmUserPassword != null){
					vmConfig.setVmUserPassword(vmUserPassword);
				}
				
				String vmProtocol = (String)vmManagementNodeInfo.get(FacilityAttributeConstant.VMPROTOCOL);
				m_log.debug("vmProtocol : " + vmProtocol);
				if(vmProtocol != null){
					vmConfig.setVmProtocol(vmProtocol);
				}
			}
			
			m_log.debug("createPollerConfig() end " + facilityId);
			return pollerConfig;
		} catch (FinderException e) {
			throw new FacilityNotFoundException(facilityId);
		} catch (EJBException e) {
			m_log.error(e.getMessage(), e);
			if (e.getCausedByException() instanceof NameNotFoundException) {
				throw new FacilityNotFoundException(facilityId);
			}
			throw e;
		} catch (Exception e) {
			String message= "Couldn't get Poller Config : " + facilityId;
			throw new EJBException(message);
		}
	}

	/**
	 * 登録されている全てのポーラの設定を再設定する
	 */
	public void refreshSnmpConfig(){
		// 全てのノードに対してチェックを行う
		Iterator<PollerInfo> itr = m_pollerTable.keySet().iterator();		
		
		while(itr.hasNext()){
			PollerInfo key = itr.next();
			
			// ポーラ名にファシリティＩＤが設定されている
			String facilityId = key.getPollerName();
			
			PollerConfig config;
			try {
				config = createPollerConfig(facilityId);
			} catch (FacilityNotFoundException e) {
				m_log.error(e.getMessage(), e);
				continue;
			}
			
			// ポーラの設定を新規の設定で上書き
			m_pollerTable.get(key).getPollingConfig().setPollerConfig(config);
		}
	}
	
	/**
	 * 
	 * @param pollerGroup
	 * @param pollerName
	 */
	public PollingController getPoller(String pollerGroup, String pollerName){
		return m_pollerTable.get(new PollerInfo(pollerGroup, pollerName));
	}
	
	/**
	 * このインスタンスのマネージャ名を返します。
	 * @return マネージャ名
	 */
	public String getManagerName() {
		return m_managerName;
	}
	
	/**
	 * ポーラが利用するQuartzのJNDI名を取得します。
	 * @return ポーラが利用するQuartzのJNDI名
	 */
	public String getQuartzJndiName() {
		return m_quartzJndiName;
	}

	/**
	 * ポーラが利用するQuartzのJNDI名を設定します。
	 * @param jndiName ポーラが利用するQuartzのJNDI名
	 */
	public void setQuartzJndiName(String jndiName) {
		m_quartzJndiName = jndiName;
	}

	
	/**
	 * 管理しているポーラの情報を文字列で返します。
	 * @return 管理しているテーブルの情報
	 */
	public String getPollerDebugInfo(){
		String debugStr = "";
		
		Iterator<PollerInfo> itr = m_pollerTable.keySet().iterator();
		
		while(itr.hasNext()){
			PollerInfo info = itr.next();
			int minInterval = m_pollerTable.get(info).getPollingConfig().getMinPollingInterval();
			debugStr = debugStr + info.getPollerGroup() + ", " + info.getPollerName() + ", " + minInterval+ "\n";
			debugStr = debugStr + m_pollerTable.get(info).getPollingConfig().getDebugInfo();
		}
		
		return debugStr;
	}
	
	/**
	 * 管理しているポーラを一意に識別するための情報を保持するクラス
	 * HashMapのキーとして利用できるように equalsメソッドと hashCodeメソッドを実装
	 */
	private class PollerInfo{
		private String m_pollerGroup;
		private String m_pollerName;
		
		public PollerInfo(String pollerGroup, String pollerName){
			m_pollerGroup = pollerGroup;
			m_pollerName = pollerName;
		}

		public String getPollerGroup() {
			return m_pollerGroup;
		}

		public String getPollerName() {
			return m_pollerName;
		}
		
		public boolean equals(Object other) {
			if (other instanceof PollerInfo) {
				PollerInfo info = (PollerInfo)other;
				
				if (this.m_pollerGroup == null && this.m_pollerName == null){
					if (info.m_pollerGroup == null && info.m_pollerName == null){
						return true;
					}
				} else if (this.m_pollerGroup == null && this.m_pollerName != null){
					if (info.m_pollerGroup == null && this.m_pollerName.equals(info.m_pollerName)){
						return true;
					}
				} else if (this.m_pollerGroup != null && this.m_pollerName == null){
					if (this.m_pollerGroup.equals(info.m_pollerGroup) && info.m_pollerName == null){
						return true;
					}
				} else {
					if (this.m_pollerGroup.equals(info.m_pollerGroup)){
						return this.m_pollerName.equals(info.m_pollerName);
					}
				}
				return false;
			} else {
				return false;
			}
		}

		public int hashCode() {
			int result = 17;

			result = 37 * result + ((this.m_pollerGroup != null) ? this.m_pollerGroup.hashCode() : 0);

			result = 37 * result + ((this.m_pollerName != null) ? this.m_pollerName.hashCode() : 0);

			return result;
		}
	}
}
