/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.repository.dialog;

import java.util.ArrayList;

import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;

import com.clustercontrol.dialog.CommonDialog;
import com.clustercontrol.dialog.ValidateResult;
import com.clustercontrol.repository.composite.NodeFilterComposite;
import com.clustercontrol.util.Messages;

/**
 * スコープへのノード割り当て解除ダイアログクラス<BR>
 * 
 * @version 1.0.0
 * @since 1.0.0
 */
public class NodeReleaseDialog extends CommonDialog {

    // ----- instance フィールド ----- //

    /** 割り当て対象スコープのファシリティID */
    private String facilityId = "";

    /** ノード一覧テーブル */
    private NodeFilterComposite nodeList = null;

    /** 選択されたアイテム */
    private String[] filterItems = null;

    // ----- コンストラクタ ----- //

    /**
     * 指定した形式のダイアログのインスタンスを返します。
     * 
     * @param parent
     *            親のシェルオブジェクト
     * @param facilityId
     *            初期表示するスコープのファシリティID
     */
    public NodeReleaseDialog(Shell parent, String facilityId) {
        super(parent);

        this.facilityId = facilityId;
    }

    // ----- instance メソッド ----- //

    /**
     * ダイアログの初期サイズを返します。
     * 
     * @return 初期サイズ
     */
    protected Point getInitialSize() {
        return new Point(450, 500);
    }

    /**
     * ダイアログエリアを生成します。
     * 
     * @param parent
     *            親のインスタンス
     */
    protected void customizeDialog(Composite parent) {
        Shell shell = this.getShell();

        // タイトル
        shell.setText(Messages.getString("dialog.repository.select.nodes"));

        // レイアウト
        GridLayout layout = new GridLayout(1, true);
        layout.marginWidth = 10;
        layout.marginHeight = 10;
        parent.setLayout(layout);

        /*
         * ノード一覧
         */

        // テーブル
        this.nodeList = new NodeFilterComposite(parent, SWT.NONE,
                this.facilityId);
        GridData gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.verticalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        gridData.grabExcessVerticalSpace = true;
        gridData.horizontalSpan = 1;
        this.nodeList.setLayoutData(gridData);

        this.nodeList.update();

        // ラインを引く
        Label line = new Label(parent, SWT.SEPARATOR | SWT.HORIZONTAL);
        gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        gridData.horizontalSpan = 1;
        line.setLayoutData(gridData);

        // 画面中央に
        Display display = shell.getDisplay();
        shell.setLocation((display.getBounds().width - shell.getSize().x) / 2,
                (display.getBounds().height - shell.getSize().y) / 2);
    }

    /**
     * 入力値チェックをします。
     * 
     * @return 検証結果
     */
    protected ValidateResult validate() {
        ValidateResult result = null;

        StructuredSelection selection = (StructuredSelection) this.nodeList
                .getTableViewer().getSelection();

        Object[] items = selection.toArray();
        if (items == null || items.length == 0) {
            result = new ValidateResult();
            result.setID(Messages.getString("message.hinemos.2"));
            result.setMessage(Messages.getString("message.repository.2"));
            result.setValid(false);
        } else {
            int size = items.length;
            this.filterItems = new String[size];
            for (int i = 0; i < size; i++) {
                this.filterItems[i] = (String) ((ArrayList) items[i]).get(0);
            }
        }

        return result;
    }

    /**
     * 割り当て対象スコープのファシリティIDを返します。
     * 
     * @return 割り当て対象スコープのファシリティID
     */
    public String getScopeId() {
        return this.facilityId;
    }

    /**
     * 選択されたノードを返します。
     * 
     * @return ノードのファシリティID
     */
    public String[] getFilterItems() {
        return this.filterItems;
    }

    /**
     * ＯＫボタンのテキストを返します。
     * 
     * @return ＯＫボタンのテキスト
     */
    protected String getOkButtonText() {
        return Messages.getString("ok");
    }

    /**
     * キャンセルボタンのテキストを返します。
     * 
     * @return キャンセルボタンのテキスト
     */
    protected String getCancelButtonText() {
        return Messages.getString("cancel");
    }
}