/*
 
Copyright (C) 2006 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.repository.dialog;

import java.net.Inet4Address;
import java.net.Inet6Address;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Hashtable;
import java.util.Iterator;

import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.TableTree;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.layout.RowData;
import org.eclipse.swt.layout.RowLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Text;

import com.clustercontrol.ClusterControlPlugin;
import com.clustercontrol.bean.DeviceTypeConstant;
import com.clustercontrol.bean.FacilityConstant;
import com.clustercontrol.bean.FacilityPath;
import com.clustercontrol.bean.FacilityTreeItem;
import com.clustercontrol.bean.Property;
import com.clustercontrol.bean.PropertyConstant;
import com.clustercontrol.bean.SizeConstant;
import com.clustercontrol.composite.PropertySheet;
import com.clustercontrol.dialog.CommonDialog;
import com.clustercontrol.dialog.ScopeTreeDialog;
import com.clustercontrol.dialog.ValidateResult;
import com.clustercontrol.repository.action.AddNodeProperty;
import com.clustercontrol.repository.action.GetNodeProperty;
import com.clustercontrol.repository.action.GetNodePropertyBySNMP;
import com.clustercontrol.repository.action.ModifyNodeProperty;
import com.clustercontrol.repository.factory.NodeProperty;
import com.clustercontrol.util.Messages;
import com.clustercontrol.util.PropertyUtil;

/**
 * ノードの作成・変更ダイアログクラス<BR>
 * 
 * @version 3.0.1
 * @since 1.0.0
 */
public class NodeCreateDialog extends CommonDialog {

    // ----- instance フィールド ----- //

    /** 初期表示ノードのファシリティID */
    private String facilityId = "";

    /** 一括変更対象スコープのファシリティID */
    private String scopeId = "";

    /** 変更用ダイアログ判別フラグ */
    private boolean isModifyDialog = false;

    /** 「登録方法の選択」でのノード指定フラグ */
    private boolean isNodeSelected = true;

    /** ノード単位選択ラジオボタン */
    private Button radioCreateTypeNode = null;

    /** スコープ単位選択ラジオボタン */
    private Button radioCreateTypeScope = null;

    /** スコープ文字列表示テキスト */
    private Text textScope = null;

    /** スコープ指定ダイアログ表示ボタン */
    private Button buttonScope = null;

    /** ノード属性プロパティシート */
    private PropertySheet propertySheet = null;
    
    /** クリア実行フラグ */
    private boolean isClearDone = false;
    
    /** クリアボタン */
    private Button clearButtonScope = null;
    
    //SNMPで検出する機能のための追加コンポーネント
    private Button buttonAuto = null;
    
    private Label    ipAddressText = null; //"IP Address";
    private Text     ipAddressBox  = null;
    private Label    communityText = null; //"community";
    private Text     communityBox  = null;
    private Label    portText      = null; //"port";
    private Text     portBox       = null;
    private Label    versionText      = null; //"version";
    private Combo    versionBox       = null;
    
    // ----- コンストラクタ ----- //

    /**
     * 指定した形式のダイアログのインスタンスを返します。
     * 
     * @param parent
     *            親のシェルオブジェクト
     * @param facilityId
     *            初期表示するノードのファシリティID
     * @param isModifyDialog
     *            変更用ダイアログとして利用する場合は、true
     */
    public NodeCreateDialog(Shell parent, String facilityId,
            boolean isModifyDialog) {
        super(parent);

        this.facilityId = facilityId;
        this.isModifyDialog = isModifyDialog;
    }

    // ----- instance メソッド ----- //

    /**
     * ダイアログの初期サイズを返します。
     * 
     * @return 初期サイズ
     */
    protected Point getInitialSize() {
        return new Point(500, 600);
    }

    /**
     * ダイアログエリアを生成します。
     * 
     * @param parent
     *            親のインスタンス
     */
    protected void customizeDialog(Composite parent) {
        Shell shell = this.getShell();

        // タイトル
        shell.setText(Messages
                .getString("dialog.repository.node.create.modify"));

        // レイアウト
        GridLayout layout = new GridLayout(1, true);
        layout.marginWidth = 10;
        layout.marginHeight = 10;
        parent.setLayout(layout);

 
        // 登録方法の選択(グループ)
        Group group = new Group(parent, SWT.SHADOW_NONE);
        layout = new GridLayout(1, true);
        group.setLayout(layout);
        GridData gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        gridData.horizontalSpan = 1;
        group.setLayoutData(gridData);
        group.setText(Messages.getString("registration.method"));

        // ノード単位
        this.radioCreateTypeNode = new Button(group, SWT.RADIO);
        gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        gridData.horizontalSpan = 1;
        this.radioCreateTypeNode.setLayoutData(gridData);
        this.radioCreateTypeNode.setText(Messages
                .getString("registration.method.node.base"));
        
        
        //SNMPで検索するためのコンポジット
        Composite compositeSNMP = new Composite(group, SWT.NONE);
        RowLayout rowLayoutAuto = new RowLayout();
        rowLayoutAuto.marginLeft=20;
        rowLayoutAuto.type = SWT.HORIZONTAL;
        compositeSNMP.setLayout(rowLayoutAuto);
        
        //SNMPによる検出のグループ(SNMP)
        Group groupAuto = new Group(compositeSNMP, SWT.RIGHT);
        layout = new GridLayout(5, false);
        groupAuto.setLayout(layout);   
        groupAuto.setText("Find By SNMP");
        
        //検索条件のコンポジット
       // Composite compositeIP = new Composite(groupAuto,SWT.NONE);
       // GridLayout layoutIP= new GridLayout(4, false);
       // groupAuto.setLayout(layoutIP);
        
        
        //IPアドレス
        this.ipAddressText = new Label(groupAuto, SWT.NONE);
        this.ipAddressText.setText("IP Address : ");
        
        this.ipAddressBox = new Text(groupAuto, SWT.BORDER | SWT.SINGLE);
        GridData grid = new GridData(GridData.FILL_HORIZONTAL);
        grid.minimumWidth = 80;
        this.ipAddressBox.setLayoutData(grid);
        this.ipAddressBox.setText("192.168.0.1");
        
        
        
        //ポート名
        this.portText = new Label(groupAuto, SWT.NONE);
        this.portText.setText("Port : ");
        
        this.portBox = new Text(groupAuto, SWT.BORDER | SWT.SINGLE);
         grid = new GridData(GridData.FILL_HORIZONTAL);
        grid.minimumWidth = 50;
        this.portBox.setLayoutData(grid);
        this.portBox.setText("161");
        
        Label dummy2 = new Label(groupAuto, SWT.NONE);
        //dummy2.setLayoutData(new RowData(12, SizeConstant.SIZE_LABEL_HEIGHT));
        
        //コミュニティ名
        this.communityText = new Label(groupAuto, SWT.NONE);
        this.communityText.setText("Community Name : ");
        
        this.communityBox = new Text(groupAuto, SWT.BORDER | SWT.SINGLE);
        this.communityBox.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
        this.communityBox.setText("public");
        
        
        //バージョン
        this.versionText = new Label(groupAuto, SWT.NONE);
        this.versionText.setText("version : ");
        
        this.versionBox = new Combo(groupAuto, SWT.RADIO);
        this.versionBox.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
        this.versionBox.add("1",0);
        this.versionBox.add("2c",1);
    	// デフォルトをv2cとする
        this.versionBox.select(1);
        
        this.buttonAuto = new Button(groupAuto, SWT.PUSH | SWT.RIGHT);
        this.buttonAuto.setText(" Find ");
        gridData = new GridData();
        gridData.horizontalIndent= 30; 
        this.buttonAuto.setLayoutData(gridData);
        this.buttonAuto.addSelectionListener(new SelectionAdapter() {
                @SuppressWarnings("unchecked")
                public void widgetSelected(SelectionEvent e) {
                        try{
                                int portnum = Integer.parseInt(portBox.getText());
                                String ipAddressString = (String)ipAddressBox.getText();
                                
                                // IPアドレスチェック
                                InetAddress address = InetAddress.getByName(ipAddressString);
                                
                                if (address instanceof Inet4Address){

                                	//IPv4の場合はさらにStringをチェック
                                	if (!ipAddressString.matches(".{1,3}?\\..{1,3}?\\..{1,3}?\\..{1,3}?")){
                                		MessageDialog.openWarning(getShell(), "Warning",  Messages.getString("message.repository.37"));
                                		return;
                                	}
                                	
                                } else if (address instanceof Inet6Address){
                                	//IPv6の場合は特にStringチェックは無し
                                } else {
                                	MessageDialog.openWarning(getShell(), "Warning",  Messages.getString("message.repository.37"));
                                	return;
                                }
                                
                                Hashtable pollingData = new Hashtable();
                                pollingData.put("IPAddress",  ipAddressBox.getText());
                                pollingData.put("port",       portBox.getText());
                                pollingData.put("community",  communityBox.getText());
                                pollingData.put("version",    new Integer(versionBox.getSelectionIndex()));
                                
                                /*
                                System.out.println("IP Address : " + (String) pollingData.get("IPAddress") );
                                System.out.println("port : " + (String)pollingData.get("port"));
                                System.out.println("community : " + (String)pollingData.get("community"));
                                System.out.println("Version : " + (Integer)pollingData.get("version"));
                                */
                                
                                Property propertySNMP = null;
                                GetNodePropertyBySNMP poller = new GetNodePropertyBySNMP();
                                
                                if (isModifyDialog) {
                                        propertySNMP = poller.search(pollingData,
                                                        PropertyConstant.MODE_MODIFY);
                                } else {
                                        propertySNMP = poller.search(pollingData,
                                                        PropertyConstant.MODE_ADD);
                                }                    
                                propertySheet.setInput(propertySNMP);
                                
                                //プロパティシートの展開
                                //レベル1までの展開（デバイス情報、ファイルシステム情報は展開しない。）
                                propertySheet.expandToLevel(1);
                                //ネットワークのインデックスは6
                                Object element =  propertySheet.getElementAt(6);
                                propertySheet.expandToLevel(element, 2);
                                //OSのインデックスは7
                                 element =   propertySheet.getElementAt(7);
                                propertySheet.expandToLevel(element, 2);

                        } catch (NumberFormatException e1){
                        	// port番号が不正な場合
                        	MessageDialog.openWarning(getShell(), "Warning", Messages.getString("message.repository.38"));
                            
                        } catch (UnknownHostException e2) {
                        	// IPアドレスが不正な場合
                            MessageDialog.openWarning(getShell(), "Warning",  Messages.getString("message.repository.37"));
						}
                }
        });
        

        // スコープ単位
        this.radioCreateTypeScope = new Button(group, SWT.RADIO);
        gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        gridData.horizontalSpan = 1;
        this.radioCreateTypeScope.setLayoutData(gridData);
        this.radioCreateTypeScope.setText(Messages
                .getString("registration.method.scope.base"));

        // スコープ指定
        Composite composite = new Composite(group, SWT.NONE);
        RowLayout rowLayout = new RowLayout();
        rowLayout.type = SWT.HORIZONTAL;
        composite.setLayout(rowLayout);
        gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        gridData.horizontalSpan = 1;
        composite.setLayoutData(gridData);
        // ダミーラベル
        Label dummy = new Label(composite, SWT.NONE);
        dummy.setLayoutData(new RowData(12, SizeConstant.SIZE_LABEL_HEIGHT));
        // スコープ文字列テキスト
        this.textScope = new Text(composite, SWT.BORDER | SWT.LEFT
                | SWT.READ_ONLY);
        this.textScope.setText("");
        this.textScope.setLayoutData(new RowData(300,
                SizeConstant.SIZE_TEXT_HEIGHT));
        this.buttonScope = new Button(composite, SWT.PUSH);
        this.buttonScope.setText(Messages.getString("refer"));
        this.buttonScope.pack();
        if (!this.isModifyDialog) {
            this.buttonScope.setEnabled(false);
        }
        this.buttonScope.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(SelectionEvent e) {
                ScopeTreeDialog dialog = new ScopeTreeDialog(getParentShell(),
                        true);
                if (dialog.open() == IDialogConstants.OK_ID) {
                    FacilityTreeItem selectItem = dialog.getSelectItem();
                    if (selectItem.getData().getType() == FacilityConstant.TYPE_COMPOSITE) {
                        textScope.setText("");
                        scopeId = "";
                    } else {
                        FacilityPath facilityPath = new FacilityPath(
                                ClusterControlPlugin.getDefault()
                                        .getSeparator());
                        textScope.setText(facilityPath.getPath(selectItem));
                        scopeId = selectItem.getData().getFacilityId();
                    }
                }
            }
        }

        );

        // ディフォルトではノード単位を選択
        this.radioCreateTypeNode.setSelection(true);
        this.textScope.setEnabled(false);
        this.buttonScope.setEnabled(false);

        // ラジオボタンのイベント
        this.radioCreateTypeNode.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(SelectionEvent e) {
                textScope.setEnabled(false);
                buttonScope.setEnabled(false);
                clearButtonScope.setEnabled(false);
            }
        });
        this.radioCreateTypeScope.addSelectionListener(new SelectionAdapter() {
            public void widgetSelected(SelectionEvent e) {
                textScope.setEnabled(true);
                buttonScope.setEnabled(true);
                clearButtonScope.setEnabled(true);
            }
        });

        // 登録処理の場合、スコープ関連は使用不可とする。
        if (!this.isModifyDialog) {
            this.radioCreateTypeScope.setEnabled(false);
            this.clearButtonScope.setEnabled(false);
        }else{
        	//変更処理の場合には、SNMPによる検出を使えなくする。
        	  this.ipAddressBox.setText("           ");
        	  this.communityBox.setText("      ");
        	  this.portBox.setText("   ");
        	  this.versionBox.select(0);
        	  this.ipAddressBox.setEnabled(false);
        	  this.communityBox.setEnabled(false);
        	  this.portBox.setEnabled(false);
        	  this.versionBox.setEnabled(false);
        	  this.buttonAuto.setEnabled(false);
              this.clearButtonScope.setEnabled(false);
        }

        /*
         * 属性プロパティシート
         */

        // ラベル
        Label label = new Label(parent, SWT.LEFT);
        gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        gridData.horizontalSpan = 1;
        label.setLayoutData(gridData);
        label.setText(Messages.getString("attribute") + " : ");

        // プロパティシート
        TableTree table = new TableTree(parent, SWT.H_SCROLL | SWT.V_SCROLL
                | SWT.FULL_SELECTION | SWT.MULTI | SWT.BORDER);
        gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.verticalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        gridData.grabExcessVerticalSpace = true;
        gridData.horizontalSpan = 1;
        table.setLayoutData(gridData);

        this.propertySheet = new PropertySheet(table);
        this.propertySheet.setCopy(true);

        // プロパティ取得及び設定
        Property property = null;
        if (this.isModifyDialog) {
            property = new GetNodeProperty().getProperty(this.facilityId,
                    PropertyConstant.MODE_MODIFY);
        } else {
            property = new GetNodeProperty().getProperty(this.facilityId,
                    PropertyConstant.MODE_ADD);
        }
        this.propertySheet.setInput(property);
        
        //プロパティシートの展開
        //レベル1までの展開（デバイス情報、ファイルシステム情報は展開しない。）
        this.propertySheet.expandToLevel(1);
        //ネットワークのインデックスは7
        Object element =  this.propertySheet.getElementAt(7);
        this.propertySheet.expandToLevel(element, 1);
        //OSのインデックスは8
         element =  this.propertySheet.getElementAt(8);
        this.propertySheet.expandToLevel(element, 1);
       

        // ラインを引く
        Label line = new Label(parent, SWT.SEPARATOR | SWT.HORIZONTAL);
        gridData = new GridData();
        gridData.horizontalAlignment = GridData.FILL;
        gridData.grabExcessHorizontalSpace = true;
        gridData.horizontalSpan = 1;
        line.setLayoutData(gridData);

        // 画面中央に
        Display display = shell.getDisplay();
        shell.setLocation((display.getBounds().width - shell.getSize().x) / 2,
                (display.getBounds().height - shell.getSize().y) / 2);
    }

    /**
     * 既存のボタンに加え、クリアボタンを追加します。
     * 
     * @param parent
     *            ボタンバーコンポジット
     */
    protected void createButtonsForButtonBar(Composite parent) {
        // クリアボタン
    	clearButtonScope = 
        this.createButton(parent, IDialogConstants.OPEN_ID, Messages
                .getString("clear"), false);
        this.getButton(IDialogConstants.OPEN_ID).addSelectionListener(
                new SelectionAdapter() {
                    public void widgetSelected(SelectionEvent e) {
                        // プロパティ取得及び設定
                        Property property = null;
                        if (isModifyDialog()) {
                            property = new GetNodeProperty().getProperty(null,
                                    PropertyConstant.MODE_MODIFY);

                            //ファシリティID設定
                            ArrayList list = PropertyUtil.getProperty(property,
                                    NodeProperty.FACILITY_ID);
                            ((Property) list.get(0)).setValue(getFacilityId());
                        } else {
                            property = new GetNodeProperty().getProperty(null,
                                    PropertyConstant.MODE_ADD);
                        }

                        //Booleanエディタのプロパティにnullを設定する
                        PropertyUtil.setPropertyValue(property,
                                PropertyConstant.EDITOR_BOOL, null);

                        setInputData(property);
                        
                        // クリア実行フラグ 
                        isClearDone = true;
                    }
                });

        super.createButtonsForButtonBar(parent);
    }

    /**
     * 入力値チェックをします。
     * 
     * @return 検証結果
     */
    protected ValidateResult validate() {
    	ValidateResult result = null;
    	
        this.isNodeSelected = this.radioCreateTypeNode.getSelection();

        if(this.isNodeSelected){
			//ファシリティIDの入力チェック
	        ArrayList values = PropertyUtil.getPropertyValue(getInputData(), NodeProperty.FACILITY_ID);
			String value = (String)values.get(0);
			if(value.compareTo("") == 0){
	            result = new ValidateResult();
	            result.setID(Messages.getString("message.hinemos.1"));
	            result.setMessage(Messages.getString("message.repository.21"));
	            result.setValid(false);
	            return result;
			}
			
			//ファシリティ名の入力チェック
	        values = PropertyUtil.getPropertyValue(getInputData(), NodeProperty.FACILITY_NAME);
			value = (String)values.get(0);
			if(value.compareTo("") == 0){
	            result = new ValidateResult();
	            result.setID(Messages.getString("message.hinemos.1"));
	            result.setMessage(Messages.getString("message.repository.22"));
	            result.setValid(false);
	            return result;
			}
			
			//プラットフォームの入力チェック
	        values = PropertyUtil.getPropertyValue(getInputData(), NodeProperty.PLATFORM_FAMILY_NAME);
			value = (String)values.get(0);
			if(value.compareTo("") == 0){
	            result = new ValidateResult();
	            result.setID(Messages.getString("message.hinemos.1"));
	            result.setMessage(Messages.getString("message.repository.36"));
	            result.setValid(false);
	            return result;
			}
			
			//IPアドレスの入力チェック
	        values = PropertyUtil.getPropertyValue(getInputData(), NodeProperty.IP_ADDRESS_VERSION);
	        Object obj = values.get(0);
	     
	        
	        if(obj.getClass() == Integer.class){
	        
	        	value = ((Integer)obj).toString();
	        	
	        }else{
	        	
	        	value = (String)values.get(0);
	        }
	        
			if(value.compareTo("") == 0 || value.compareTo("4") == 0  ){
				//versionが空か4の場合には、
				ArrayList ipValues =  PropertyUtil.getPropertyValue(getInputData(), NodeProperty.IP_ADDRESS_V4);
				String ipvValue =  (String)ipValues.get(0);
				if(ipvValue.compareTo("") ==0){
					result = new ValidateResult();
					result.setID(Messages.getString("message.hinemos.1"));
					result.setMessage(Messages.getString("message.repository.24"));
					result.setValid(false);
					return result;
				}
			}else if(value.compareTo("6") == 0 ){
				//	versionが6の場合には、
				ArrayList ipValues =  PropertyUtil.getPropertyValue(getInputData(), NodeProperty.IP_ADDRESS_V6);
				String ipvValue =  (String)ipValues.get(0);
				if(ipvValue.compareTo("") ==0){
					result = new ValidateResult();
					result.setID(Messages.getString("message.hinemos.1"));
					result.setMessage(Messages.getString("message.repository.25"));
					result.setValid(false);
					return result;
				}
				
			}
			
			//ノード名の入力チェック
	        values = PropertyUtil.getPropertyValue(getInputData(), NodeProperty.NODE_NAME);
			value = (String)values.get(0);
			if(value.compareTo("") == 0){
	            result = new ValidateResult();
	            result.setID(Messages.getString("message.hinemos.1"));
	            result.setMessage(Messages.getString("message.repository.23"));
	            result.setValid(false);
	            return result;
			}
			
	        //ノード種別指定の必須条件
	        values = PropertyUtil.getPropertyValue(getInputData(), NodeProperty.VIRTNODETYPE);
			value = (String)values.get(0);
			//host指定の場合
			if(value != null && value.equals("host")){
				
		        values = PropertyUtil.getPropertyValue(getInputData(), NodeProperty.VIRTSOLUTION);
				String virtSolution = (String)values.get(0);
		        values = PropertyUtil.getPropertyValue(getInputData(), NodeProperty.VMUSER);
				String vmUser = (String)values.get(0);
		        values = PropertyUtil.getPropertyValue(getInputData(), NodeProperty.VMUSERPASSWORD);
				String vmUserpassword = (String)values.get(0);
		        values = PropertyUtil.getPropertyValue(getInputData(), NodeProperty.VMPROTOCOL);
				String vmProtocol = (String)values.get(0);

				//必須項目を全て入力すること
				if(virtSolution == null || virtSolution.compareTo("") == 0
						|| vmUser == null || vmUser.compareTo("") == 0 
						|| vmUserpassword == null || vmUserpassword.compareTo("") == 0
						|| vmProtocol == null || vmProtocol.compareTo("") == 0){
					
					//
		            result = new ValidateResult();
		            result.setID(Messages.getString("message.hinemos.1"));
		            result.setMessage(Messages.getString("message.repository.42"));
		            result.setValid(false);
		            return result;
				}
				
	        //guest指定の場合
			}else if(value != null && value.equals("guest")){
				
		        values = PropertyUtil.getPropertyValue(getInputData(), NodeProperty.VIRTSOLUTION);
				String virtSolution = (String)values.get(0);
		        values = PropertyUtil.getPropertyValue(getInputData(), NodeProperty.VMMANAGEMENTNODE);
				String vmManagementNode = (String)values.get(0);
		        values = PropertyUtil.getPropertyValue(getInputData(), NodeProperty.VMNAME);
				String vmName = (String)values.get(0);
		        values = PropertyUtil.getPropertyValue(getInputData(), NodeProperty.VMID);
				String vmId = (String)values.get(0);
				
				//必須項目を全て入力すること
				if(virtSolution == null || virtSolution.compareTo("") == 0
						|| vmManagementNode == null || vmManagementNode.compareTo("") == 0 
						|| vmName == null || vmName.compareTo("") == 0
						|| vmId == null || vmId.compareTo("") == 0){
					
					//
		            result = new ValidateResult();
		            result.setID(Messages.getString("message.hinemos.1"));
		            result.setMessage(Messages.getString("message.repository.43"));
		            result.setValid(false);
		            return result;
				}
			}
			
			// LDAPのdnをデバイス名から、デバイスTYPE＋デバイスINDEXに変更したため、
			// 名前入力チェックをコメントアウトした　Ver2.4.0 uchiyama
			/*
			//デバイス名のチェック
			values = PropertyUtil.getPropertyValue(getInputData(), NodeProperty.DEVICE_NAME);
			Iterator itr = values.iterator();
			while(itr.hasNext()){
				
				String judg_str = itr.next().toString();
				
				if(judg_str.matches(".*[\\\\/;,<>+\"]+.*")) {
		            result = new ValidateResult();
		            result.setID(Messages.getString("message.hinemos.1"));
		            result.setMessage(Messages.getString("message.repository.29"));
		            result.setValid(false);
		            return result;
				}
				else if(judg_str.matches("#.*")) {
		            result = new ValidateResult();
		            result.setID(Messages.getString("message.hinemos.1"));
		            result.setMessage(Messages.getString("message.repository.30"));
		            result.setValid(false);
		            return result;
				}
			}
			*/
			
			//デバイスの入力チェック
			//ルール１：
			//デバイスINDEXとデバイスTYPEが入力してある、かつ、デバイス名、表示名が未入力の場合は、エラーを表示
			//ルール２：
			//(ver3.0.1より廃止)デバイスTYPEに「disk」「nic」以外の文字が入力してある場合は、エラをー表示
			//ルール３：
			//デバイス名に重複がある場合は、エラーを表示
			//ルール４：
			//表示名に重複がある場合は、エラーを表示
			values = PropertyUtil.getPropertyValue(getInputData(), NodeProperty.DEVICE_INDEX);
			Iterator itr = values.iterator();
			
			ArrayList values2 = PropertyUtil.getPropertyValue(getInputData(), NodeProperty.DEVICE_TYPE);
			Iterator itr2 = values2.iterator();
			
			ArrayList values3 = PropertyUtil.getPropertyValue(getInputData(), NodeProperty.DEVICE_NAME);
			Iterator itr3 = values3.iterator();
			
			ArrayList values4 = PropertyUtil.getPropertyValue(getInputData(), NodeProperty.DEVICE_DISPLAY_NAME);
			Iterator itr4 = values4.iterator();
			
			String deviceIndex;
			String deviceType;
			String deviceName;
			String deviceDisplayName;
			ArrayList<String> checkDn = new ArrayList<String>();
			ArrayList<String> checkName = new ArrayList<String>();
			ArrayList<String> checkDisplayName = new ArrayList<String>();
			//デバイスの登録数だけチェックする
			while(itr.hasNext()){
				deviceIndex = itr.next().toString();
				deviceType = itr2.next().toString();
				deviceName = itr3.next().toString();
				deviceDisplayName = itr4.next().toString();
				
				//デバイスINDEXとデバイスTYPEが未入力でない
				if(deviceIndex != null && !deviceIndex.equals("") && 
						deviceType != null && !deviceType.equals("")) {
					
					//デバイス名が未入力
					if(deviceName == null || deviceName.equals("")) {
						String[] args = { deviceType,deviceIndex,Messages.getString("device.name") };
						result = new ValidateResult();
			            result.setID(Messages.getString("message.hinemos.1"));
			            result.setMessage(Messages.getString("message.repository.33", args));
			            result.setValid(false);
			            return result;
					}
					
					//デバイス表示名が未入力
					if(deviceDisplayName == null || deviceDisplayName.equals("")) {
						String[] args = { deviceType,deviceIndex,Messages.getString("device.display.name") };
						result = new ValidateResult();
			            result.setID(Messages.getString("message.hinemos.1"));
			            result.setMessage(Messages.getString("message.repository.33", args));
			            result.setValid(false);
			            return result;
					}
					
					//nic、disk以外の文字列の入力も許可するため、以下のチェックをコメントアウトした ver3.0.1 uchiyama
					//デバイスTYPEに「disk」「nic」以外が入力されている
					/*
					if(!deviceType.equals(DeviceTypeConstant.DEVICE_DISK) && !deviceType.equals(DeviceTypeConstant.DEVICE_NIC)) {
						String[] args = { deviceName, DeviceTypeConstant.DEVICE_DISK, DeviceTypeConstant.DEVICE_NIC };
						result = new ValidateResult();
			            result.setID(Messages.getString("message.hinemos.1"));
			            result.setMessage(Messages.getString("message.repository.35", args));
			            result.setValid(false);
			            return result;
					}
					*/
					
					// デバイスタイプにて入力不可文字が使用されていないかをチェック
					String[] args = {deviceName};
					if(deviceType.matches(".*[\\\\/;,<>+\"]+.*")) {
			            result = new ValidateResult();
			            result.setID(Messages.getString("message.hinemos.1"));
			            result.setMessage(Messages.getString("message.repository.29", args));
			            result.setValid(false);
			            return result;
					}
					else if(deviceType.matches("#.*")) {
			            result = new ValidateResult();
			            result.setID(Messages.getString("message.hinemos.1"));
			            result.setMessage(Messages.getString("message.repository.30", args));
			            result.setValid(false);
			            return result;
					}
					
					
					
					checkDn.add(deviceType+"_"+deviceIndex);
					checkName.add(deviceName);
					checkDisplayName.add(deviceDisplayName);
				}
			}
			
			// デバイスINDEX、デバイスTYPEが重複していないか確認する
			// デバイス名が重複していないか確認する
			for(int i = 0; i < checkDn.size(); i++) {
				for(int j = 1 + i; j < checkDn.size(); j++) {
					//DN（TYPE_INDEX）重複チェック
					if (checkDn.get(i).equals(checkDn.get(j))) {
						String[] args = checkDn.get(i).split("_");
						result = new ValidateResult();
			            result.setID(Messages.getString("message.hinemos.1"));
			            result.setMessage(Messages.getString("message.repository.34", args));
			            result.setValid(false);
			            return result;
					}
					// デバイス名重複チェック
					if (checkName.get(i).equals(checkName.get(j))) {
						String[] args = { Messages.getString("device.name"), checkName.get(i) };
						result = new ValidateResult();
			            result.setID(Messages.getString("message.hinemos.1"));
			            result.setMessage(Messages.getString("message.repository.32", args));
			            result.setValid(false);
			            return result;
					}
					// デバイス表示名重複チェック
					if (checkDisplayName.get(i).equals(checkDisplayName.get(j))) {
						String[] args = { Messages.getString("device.display.name"), checkDisplayName.get(i) };
						result = new ValidateResult();
			            result.setID(Messages.getString("message.hinemos.1"));
			            result.setMessage(Messages.getString("message.repository.32", args));
			            result.setValid(false);
			            return result;
					}
				}
			}
			
			
			//ファイルシステムの入力項目チェック
			//ルール１：
			//ファイルシステムINDEXが入力されている、かつ、マウント位置、表示名が未入力の場合は、エラーを表示
			//ルール２：
			//ファイルシステムINDEXが重複している場合は、エラーを表示
			//ルール３：
			//マウント位置が重複している場合は、エラーを表示
			//ルール４：
			//表示名に重複がある場合は、エラーを表示
			values = PropertyUtil.getPropertyValue(getInputData(), NodeProperty.FILE_SYSTEM_INDEX);
			itr = values.iterator();
			
			values2 = PropertyUtil.getPropertyValue(getInputData(), NodeProperty.FILE_SYSTEM_MOUNT_POINT);
			itr2  = values2.iterator();
			
			values3 = PropertyUtil.getPropertyValue(getInputData(), NodeProperty.FILE_SYSTEM_DISPLAY_NAME);
			itr3  = values3.iterator();
			
			String fileSystemIndex;
			String fileSystemMountPoint;
			String fileSystemDisplayName;
			checkDn = new ArrayList<String>();
			checkName = new ArrayList<String>();
			checkDisplayName = new ArrayList<String>();
			
			// ファイルシステムの登録数だけチェックする
			while(itr.hasNext()){
				fileSystemIndex = itr.next().toString();
				fileSystemMountPoint = itr2.next().toString();
				fileSystemDisplayName = itr3.next().toString();
				
				//ファイルシステムINDEXが未入力でない
				if(fileSystemIndex != null && !fileSystemIndex.equals("")){
					//マウント位置が未入力
					if(fileSystemMountPoint == null || fileSystemMountPoint.equals("")){
						String[] args = { fileSystemIndex,Messages.getString("file.system.mount.point") };
						result = new ValidateResult();
			            result.setID(Messages.getString("message.hinemos.1"));
			            result.setMessage(Messages.getString("message.repository.31", args));
			            result.setValid(false);
			            return result;
					}
					//デバイス表示名が未入力
					if(fileSystemDisplayName == null || fileSystemDisplayName.equals("")) {
						String[] args = { fileSystemIndex,Messages.getString("file.system.display.name") };
						result = new ValidateResult();
			            result.setID(Messages.getString("message.hinemos.1"));
			            result.setMessage(Messages.getString("message.repository.31", args));
			            result.setValid(false);
			            return result;
					}
					checkDn.add(fileSystemIndex);
					checkName.add(fileSystemMountPoint);
					checkDisplayName.add(fileSystemDisplayName);
				}
			}
			
			// ファイルシステムINDEXが重複していないか確認する
			// マウント位置が重複していないか確認する
			for(int i = 0; i < checkDn.size(); i++) {
				for(int j = 1 + i; j < checkDn.size(); j++) {
					//ファイルシステムINDEX重複チェック
					if (checkDn.get(i).equals(checkDn.get(j))) {
						String[] args = { Messages.getString("file.system.index"), checkDn.get(i) };
						result = new ValidateResult();
			            result.setID(Messages.getString("message.hinemos.1"));
			            result.setMessage(Messages.getString("message.repository.32", args));
			            result.setValid(false);
			            return result;
					}
					//マウント位置重複チェック
					if (checkName.get(i).equals(checkName.get(j))) {
						String[] args = { Messages.getString("file.system.mount.point"), checkName.get(i) };
						result = new ValidateResult();
			            result.setID(Messages.getString("message.hinemos.1"));
			            result.setMessage(Messages.getString("message.repository.32", args));
			            result.setValid(false);
			            return result;
					}
					//表示名重複チェック
					if (checkDisplayName.get(i).equals(checkDisplayName.get(j))) {
						String[] args = { Messages.getString("file.system.display.name"), checkDisplayName.get(i) };
						result = new ValidateResult();
			            result.setID(Messages.getString("message.hinemos.1"));
			            result.setMessage(Messages.getString("message.repository.32", args));
			            result.setValid(false);
			            return result;
					}
				}
			}
			
			// 備考の重複チェック
			values = PropertyUtil.getPropertyValue(getInputData(), NodeProperty.NOTE);
			String st1;
			String st2;
			for (int i = 0; i < values.size() - 1; i++ ){
				st1 = (String) values.get(i);
				for (int j = i + 1; j < values.size(); j++) {
					st2 = (String) values.get(j);
					if (st1.equals(st2)){
						String[] args = { Messages.getString("note"), st1 };
						result = new ValidateResult();
			            result.setID(Messages.getString("message.hinemos.1"));
			            result.setMessage(Messages.getString("message.repository.32", args));
			            result.setValid(false);
			            return result;
					}
				}
			}
			
			
        }
        else{
			//スコープの入力チェック
			if(scopeId.compareTo("") == 0){
	            result = new ValidateResult();
	            result.setID(Messages.getString("message.hinemos.1"));
	            result.setMessage(Messages.getString("message.hinemos.3"));
	            result.setValid(false);
	            return result;
			}
        }

        return result;
    }
    
    /**
     * 入力値をマネージャに登録します。
     * 
     * @return true：正常、false：異常
     * 
     * @see com.clustercontrol.dialog.CommonDialog#action()
     */
    protected boolean action() {
    	boolean result = false;
    	
    	Property property = this.getInputData();
    	if(property != null){
            if(!this.isModifyDialog()){
            	// 作成の場合
            	result = new AddNodeProperty().add(property);
            }
            else{
            	// 変更の場合
                if (this.isNodeSelected()) {
                	result = new ModifyNodeProperty().edit(property);
                } else {
                	boolean done = false;
                	if(this.isClearDone()){
                		// クリアボタンがクリックされていた場合、更新処理を行う。
                		done = true;
                	}
                	else{
                        // 確認ダイアログにてOKが選択された場合、更新処理を行う。
                        if (MessageDialog.openConfirm(
                    			null, 
                    			Messages.getString("confirmed"), 
                    			Messages.getString("message.repository.20"))) {
                        	done = true;
                        }
                	}
                	
                	// スコープ単位で更新
                	if(done){
                		result = new ModifyNodeProperty().editAllInScope(
                					property, this.getScopeId());
                	}
                }
            }
    	}
    	
    	return result;
    }

    /**
     * 変更用ダイアログなのかを返します。
     * 
     * @return 変更用ダイアログの場合、true
     */
    public boolean isModifyDialog() {
        return this.isModifyDialog;
    }

    /**
     * 「登録方法の選択」にて、ノード単位が選択されているかを返します。
     * 
     * @return ノード単位が選択されている場合、true
     */
    public boolean isNodeSelected() {
        return this.isNodeSelected;
    }

    /**
     * 一括変更対象スコープのファシリティIDを返します。
     * 
     * @return 一括変更対象スコープのファシリティID
     */
    public String getScopeId() {
        return this.scopeId;
    }
    
    /**
     * クリアボタンがクリックされているかを返します。
     * 
     * @return クリアボタンがクリックされている場合、true
     */
    public boolean isClearDone() {
        return this.isClearDone;
    }

    /**
     * 入力値を保持したデータモデルを生成します。
     * 
     * @return データモデル
     */
    public Property getInputData() {
        return (Property) this.propertySheet.getInput();
    }

    /**
     * 入力値を保持したデータモデルを設定します。
     * 
     * @param property
     */
    public void setInputData(Property property) {
        propertySheet.setInput(property);
    }

    /**
     * @return Returns the facilityId.
     */
    public String getFacilityId() {
        return facilityId;
    }

    /**
     * ＯＫボタンのテキストを返します。
     * 
     * @return ＯＫボタンのテキスト
     */
    protected String getOkButtonText() {
        if (isModifyDialog()) {
            return Messages.getString("modify");
        } else {
            return Messages.getString("register");
        }
    }

    /**
     * キャンセルボタンのテキストを返します。
     * 
     * @return キャンセルボタンのテキスト
     */
    protected String getCancelButtonText() {
        return Messages.getString("cancel");
    }
}