/*
 
Copyright (C) 2008 NTT DATA Corporation
 
This program is free software; you can redistribute it and/or
Modify it under the terms of the GNU General Public License 
as published by the Free Software Foundation, version 2.
 
This program is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied 
warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
PURPOSE.  See the GNU General Public License for more details.
 
*/

package com.clustercontrol.notify.mail.factory;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Locale;

import javax.ejb.CreateException;
import javax.ejb.FinderException;
import javax.naming.NamingException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eclipse.swt.SWT;

import com.clustercontrol.bean.TableColumnInfo;
import com.clustercontrol.bean.ValidConstant;
import com.clustercontrol.notify.bean.NotifyEventInfo;
import com.clustercontrol.notify.bean.NotifyInfo;
import com.clustercontrol.notify.ejb.entity.NotifyEventInfoLocal;
import com.clustercontrol.notify.ejb.entity.NotifyInfoLocal;
import com.clustercontrol.notify.ejb.entity.NotifyInfoUtil;
import com.clustercontrol.notify.mail.bean.MailTemplateTableDefine;
import com.clustercontrol.notify.mail.ejb.entity.MailTemplateInfo;
import com.clustercontrol.notify.mail.ejb.entity.MailTemplateInfoData;
import com.clustercontrol.notify.mail.ejb.entity.MailTemplateInfoLocal;
import com.clustercontrol.notify.mail.ejb.entity.MailTemplateInfoUtil;
import com.clustercontrol.util.Messages;
import com.clustercontrol.util.apllog.AplLogger;

/**
 * メールテンプレート情報を検索するクラス<BR>
 *
 * @version 2.4.0
 * @since 2.4.0
 */
public class SelectMailTemplate {
	
	/** ログ出力のインスタンス。 */
	protected static Log m_log = LogFactory.getLog( SelectMailTemplate.class );
	
	/**
	 * メールテンプレート情報を返します。
	 * 
	 * @param mailTemplateId 取得対象のメールテンプレートID
	 * @return メールテンプレート情報
	 * @throws FinderException
	 * @throws NamingException
	 * 
	 * @see com.clustercontrol.notify.ejb.entity.MailTemplateInfoBean
	 */
	public MailTemplateInfoData getMailTemplateInfo(String mailTemplateId) throws FinderException, NamingException {

		MailTemplateInfoData data = null;
		
		try {
			// メールテンプレート情報を取得
			MailTemplateInfoLocal infoLocal = MailTemplateInfoUtil.getLocalHome().findByPrimaryKey(mailTemplateId);
			
			data = new MailTemplateInfoData();
			data.setMailTemplateId(infoLocal.getMailTemplateId());
			data.setDescription(infoLocal.getDescription());
			data.setSubject(infoLocal.getSubject());
			data.setBody(infoLocal.getBody());
			data.setRegDate(infoLocal.getRegDate());
			data.setUpdateDate(infoLocal.getUpdateDate());
			data.setRegUser(infoLocal.getRegUser());
			data.setUpdateUser(infoLocal.getUpdateUser());
			
		} catch (FinderException e) {
			AplLogger apllog = new AplLogger("MAILTEMP", "mailTemplate");
            String[] args = { mailTemplateId };
            apllog.put("SYS", "004", args);
            m_log.warn("getMailTemplate():" + e.getMessage());
			throw e;
		} catch (NamingException e) {
			AplLogger apllog = new AplLogger("MAILTEMP", "mailTemplate");
            String[] args = { mailTemplateId };
            apllog.put("SYS", "004", args);
            m_log.warn("getMailTemplate():" + e.getMessage());
			throw e;
		}
		return data;
	}
	
	/**
	 * テンプレートID一覧を返します。
	 * <p>
	 * テンプレートIDの昇順に並んだメールテンプレートID一覧を返します。
	 * 
	 * @return テンプレートID一覧
	 * @throws FinderException
	 * @throws NamingException
	 * 
	 * @see com.clustercontrol.notify.ejb.entity.MailTemplateInfoBean
	 */
	public ArrayList<String> getMailTemplateIdList() throws FinderException, NamingException {

		ArrayList<String> list = null;
		try 
		{
			// メールテンプレートID一覧を取得
			Collection ct = MailTemplateInfoUtil.getLocalHome().findAllOrderByMailTemplateId();
			
			Iterator itr = ct.iterator();
			while(itr.hasNext())
			{
				if(list == null){
					list = new ArrayList<String>();
				}
				
				MailTemplateInfoLocal mailTemplate = (MailTemplateInfoLocal)itr.next();
				list.add(mailTemplate.getMailTemplateId());
			}
			
		} catch (FinderException e) {
			AplLogger apllog = new AplLogger("MAILTEMP", "mailTemplate");
            apllog.put("SYS", "005");
            m_log.warn("getMailTemplateIdList():" + e.getMessage());
			throw e;
		} catch (NamingException e) {
			AplLogger apllog = new AplLogger("MAILTEMP", "mailTemplate");
            apllog.put("SYS", "005");
            m_log.warn("getMailTemplateIdList():" + e.getMessage());
			throw e;
		}
		return list;
	}

	/**
	 * メールテンプレート情報一覧を返します。
	 * <p>
	 * <ol>
	 * <li>メールテンプレートIDの昇順に並んだ全てのメールテンプレート情報を取得します。</li>
	 * <li>１メールテンプレート情報をテーブルのカラム順（{@link com.clustercontrol.notify.bean.MailTemplateTableDefine}）に、リスト（{@link ArrayList}）にセットします。</li>
	 * <li>この１メールテンプレート情報を保持するリストを、メールテンプレート情報一覧を保持するリスト（{@link ArrayList}）に格納し返します。<BR>
	 *  <dl>
	 *  <dt>メールテンプレート情報一覧（Objectの2次元配列）</dt>
	 *  <dd>{ メールテンプレート情報1 {カラム1の値, カラム2の値, … }, メールテンプレート情報2{カラム1の値, カラム2の値, …}, … }</dd>
	 *  </dl>
	 * </li>
	 * </ol>
	 * 
	 * @return メールテンプレート情報一覧（Objectの2次元配列）
	 * @throws CreateException
	 * @throws FinderException
	 * @throws NamingException
	 * 
	 * @see com.clustercontrol.notify.bean.MailTemplateTableDefine
	 * @see #collectionToArray(Collection)
	 */
	public ArrayList getMailTemplateList() throws CreateException, FinderException, NamingException {

		ArrayList list = null;
		try 
		{
			// メールテンプレート情報一覧を取得
			Collection ct = MailTemplateInfoUtil.getLocalHome().findAllOrderByMailTemplateId();
			
			// 2次元配列に変換
			list = this.collectionToArray(ct);
			
		} catch (CreateException e) {
			AplLogger apllog = new AplLogger("MAILTEMP", "mailTemplate");
            apllog.put("SYS", "006");
            m_log.warn("getMailTemplateList():" + e.getMessage());
			throw e;
		} catch (FinderException e) {
			AplLogger apllog = new AplLogger("MAILTEMP", "mailTemplate");
            apllog.put("SYS", "006");
            m_log.warn("getMailTemplateList():" + e.getMessage());
			throw e;
		} catch (NamingException e) {
			AplLogger apllog = new AplLogger("MAILTEMP", "mailTemplate");
            apllog.put("SYS", "006");
            m_log.warn("getMailTemplateList():" + e.getMessage());
			throw e;
		}
		return list;
	}
	
	/**
	 * メールテンプレート情報マップを返します。
	 * <p>
	 * <ol>
	 * <li>全てのメールテンプレート情報を取得します。</li>
	 * <li>メールテンプレート情報マップに、メールテンプレート情報をセットします。</li>
	 *  <dl>
	 *   <dt>メールテンプレート情報マップ</dt>
	 *   <dd>キー：メールテンプレートID</dd>
	 *   <dd>値　：メールテンプレート情報</dd>
	 *  </dl>
	 * </li>
	 * </ol>
	 * 
	 * @return メールテンプレート情報マップ
	 * @throws FinderException
	 * @throws NamingException
	 */
	public HashMap<String, MailTemplateInfoData> getMailTemplateMap() throws FinderException, NamingException {

		HashMap<String, MailTemplateInfoData> map = new HashMap<String, MailTemplateInfoData>();
		try 
		{
			// メールテンプレート情報一覧を取得
			Collection ct = MailTemplateInfoUtil.getLocalHome().findAll();
			Iterator itr = ct.iterator();
			MailTemplateInfoLocal mailTemplate = null;
			
			while(itr.hasNext()){
				
				mailTemplate = (MailTemplateInfoLocal)itr.next();
		        
				MailTemplateInfoData data = new MailTemplateInfoData();
				data.setMailTemplateId(mailTemplate.getMailTemplateId());
				data.setDescription(mailTemplate.getDescription());
				data.setSubject(mailTemplate.getSubject());
				data.setBody(mailTemplate.getBody());
				data.setRegDate(mailTemplate.getRegDate());
				data.setUpdateDate(mailTemplate.getUpdateDate());
				data.setRegUser(mailTemplate.getRegUser());
				data.setUpdateUser(mailTemplate.getUpdateUser());
				
		        map.put(mailTemplate.getMailTemplateId(), data);
			}
			
		} catch (FinderException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
            apllog.put("SYS", "006");
            m_log.debug("getNotifyMap():" + e.getMessage());
			throw e;
		} catch (NamingException e) {
			AplLogger apllog = new AplLogger("NOTIFY", "notify");
            apllog.put("SYS", "006");
            m_log.debug("getNotifyMap():" + e.getMessage());
			throw e;
		}
		return map;
	}

	/**
	 * メールテンプレート情報一覧のテーブル定義情報を取得します。<BR>
	 * リストに、カラム毎にテーブルカラム情報をセットします。
	 * 
	 * @param locale ロケール情報
	 * @return テーブル定義情報（{@link com.clustercontrol.bean.TableColumnInfo}のリスト）
	 * 
	 * @see com.clustercontrol.bean.TableColumnInfo#TableColumnInfo(java.lang.String, int, int, int)
	 * @see com.clustercontrol.notify.mail.bean.MailTemplateTableDefine
	 */
	public ArrayList<TableColumnInfo> getMailTemplateListTableDefine(Locale locale) {
		
		/** テーブル情報定義配列 */
		ArrayList<TableColumnInfo> tableDefine = new ArrayList<TableColumnInfo>();
		
		tableDefine.add(MailTemplateTableDefine.MAIL_TEMPLATE_ID,
				new TableColumnInfo(Messages.getString("mail.template.id", locale), TableColumnInfo.NONE, 130, SWT.LEFT));
		
		tableDefine.add(MailTemplateTableDefine.DESCRIPTION,
				new TableColumnInfo(Messages.getString("description", locale), TableColumnInfo.NONE, 200, SWT.LEFT));
		
		tableDefine.add(MailTemplateTableDefine.CREATE_USER,
				new TableColumnInfo(Messages.getString("creator.name", locale), TableColumnInfo.NONE, 80, SWT.LEFT));
		
		tableDefine.add(MailTemplateTableDefine.CREATE_TIME,
				new TableColumnInfo(Messages.getString("create.time", locale), TableColumnInfo.NONE, 130, SWT.LEFT));
		
		tableDefine.add(MailTemplateTableDefine.UPDATE_USER,
				new TableColumnInfo(Messages.getString("modifier.name", locale), TableColumnInfo.NONE, 80, SWT.LEFT));
		
		tableDefine.add(MailTemplateTableDefine.UPDATE_TIME,
				new TableColumnInfo(Messages.getString("update.time", locale), TableColumnInfo.NONE, 130, SWT.LEFT));
				
		return tableDefine;
	}
	
	
	/**
	 * DBより取得したメールテンプレート情報のObjectの2次元配列を取得します。<BR>
	 * <p>
	 * <ol>
	 * <li>メールテンプレート情報をテーブルのカラム順（{@link com.clustercontrol.notify.mail.bean.MailTemplateTableDefine}）に、リスト（{@link ArrayList}）にセットします。</li>
	 * <li>1つのメールテンプレート情報を保持するリストを、メールテンプレート情報一覧を保持するリスト（{@link ArrayList}）に格納します。
	 * <p>
	 *  <dl>
	 *  <dt>メールテンプレート情報一覧（Objectの2次元配列）</dt>
	 *  <dd>{ メールテンプレート情報1 {カラム1の値, カラム2の値, … }, メールテンプレート情報2{カラム1の値, カラム2の値, …}, … }</dd>
	 *  </dl>
	 * </li>
	 * </ol>
	 * 
	 * @param ct メールテンプレート情報のコレクション
	 * @return メールテンプレート情報一覧（Objectの2次元配列）
	 * 
	 * @see com.clustercontrol.notify.mail.bean.MailTemplateTableDefine
	 */
	private ArrayList<ArrayList> collectionToArray(Collection ct) throws CreateException, FinderException, NamingException{
		
		ArrayList<ArrayList> list = new ArrayList<ArrayList>();
		Iterator itr = ct.iterator();
		while(itr.hasNext())
		{
			MailTemplateInfoLocal mailTemplate = (MailTemplateInfoLocal)itr.next();
			
			ArrayList<Object> info = new ArrayList<Object>();
			info.add(mailTemplate.getMailTemplateId());
			info.add(mailTemplate.getDescription());
			info.add(mailTemplate.getRegUser());
			info.add(new Date(mailTemplate.getRegDate().getTime()));
			info.add(mailTemplate.getUpdateUser());
			info.add(new Date(mailTemplate.getUpdateDate().getTime()));
			list.add(info);
		}
		return list;
	}
}
