/*
 
 Copyright (C) 2006 NTT DATA Corporation
 
 This program is free software; you can redistribute it and/or
 Modify it under the terms of the GNU General Public License 
 as published by the Free Software Foundation, version 2.
 
 This program is distributed in the hope that it will be
 useful, but WITHOUT ANY WARRANTY; without even the implied 
 warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 PURPOSE.  See the GNU General Public License for more details.
 
 */

package com.clustercontrol.http.util;

import java.io.IOException;
import java.io.InputStreamReader;
import java.io.UnsupportedEncodingException;

import org.apache.commons.httpclient.Header;
import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.HttpException;
import org.apache.commons.httpclient.HttpStatus;
import org.apache.commons.httpclient.methods.GetMethod;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import com.clustercontrol.util.Messages;

/**
 * HTTPアクセスを実施するクライアントクラス<BR>
 *
 * @version 2.1.0
 * @since 2.1.0
 */
public class GetHttpResponse {
	
	protected static Log m_log = LogFactory.getLog( GetHttpResponse.class );
	
	/** ボディ取得対象のContent-Type */
	public static final String TARGET_CONTENT_TYPE = "text";
	
	/** プロキシサーバ　ホスト名 */
	protected String m_proxyHost;
	
	/** プロキシサーバ ポート */
	protected int m_proxyPort;
	
	/** ステータスコード */
	protected int m_statusCode;
	
	/** ヘッダ */
	protected StringBuffer m_header;
	
	/** ボディ */
	protected String m_responseBody;
	
	/** フッタ */
	protected StringBuffer m_footer;
	
	/** 応答時間（ミリ秒） */
	protected long m_responseTime;
	
	/** エラーメッセージ */
	protected String m_errorMessage;
	
	
	/**
	 * コンストラクタ
	 */
	public GetHttpResponse() {
		super();
	}
	
	/**
	 * コンストラクタ
	 * 
	 * @param host プロキシサーバ　ホスト名
	 * @param port プロキシサーバ　ポート
	 */
	public GetHttpResponse(String host, int port) {
		super();
		m_proxyHost = host;
		m_proxyPort = port;
	}
	
	/**
	 * リクエストURLから情報取得
	 * 
	 * @param url URL
	 * @param timeout タイムアウト（ミリ秒）
	 * @return
	 */
	public boolean execute(String url, int timeout) {
		
		boolean result = false;
		
		// Create an instance of HttpClient.
		HttpClient client = new HttpClient();
		
		// proxyの設定
		if(m_proxyHost != null && !"".equals(m_proxyHost.trim())){
			client.getHostConfiguration().setProxy(m_proxyHost, m_proxyPort);
		}
		
		// Create a method instance.
		GetMethod method = new GetMethod(url);
		
		// タイムアウト設定
		if(timeout > 0){
			// establish a connection within n milliseconds
			client.getHttpConnectionManager().getParams().setConnectionTimeout(timeout);
			
			// socket timeout in milliseconds
			method.getParams().setSoTimeout(timeout);
		}
		
		try {
			// Execute the method.
			long start = System.currentTimeMillis();
			m_statusCode = client.executeMethod(method);
			long end = System.currentTimeMillis();
			m_responseTime = end - start;
			
			// Header
			Header[] headers = method.getResponseHeaders();
			if (headers != null && headers.length>0) {
				m_header = new StringBuffer();
				for (int i = 0; i < headers.length; i++) {
					m_header.append(headers[i]);
				}
			}
			
			// Footer
			Header[] footers = method.getResponseFooters();
			if (footers != null && footers.length>0) {
				m_header = new StringBuffer();
				for (int i = 0; i < footers.length; i++) {
					m_footer.append(footers[i]);
				}
			}
			
			if (m_statusCode == HttpStatus.SC_OK) {
				// Content-Typeがtext文書の場合のみ、Bodyを取得
				Header header = method.getResponseHeader("Content-Type");
				if(header != null && header.getValue() != null && header.getValue().indexOf(TARGET_CONTENT_TYPE) != -1){

					InputStreamReader in = new InputStreamReader(method.getResponseBodyAsStream(), "JISAutoDetect");
					StringBuffer stb = new StringBuffer();
					
					int c = 0; 
					while((c = in.read()) != -1){
						stb.append((char)c);
					}

					m_responseBody = stb.toString();
					result = true;
					
				}
				else{
					m_errorMessage = Messages.getString("message.http.2");
				}
			}
			else{
				m_errorMessage = method.getStatusText();
			}
			
		} catch (UnsupportedEncodingException e) {
			m_log.debug("execute(): " + e.getMessage());
			m_errorMessage = "UnsupportedEncodingException : " +  e.getMessage();
		} catch (HttpException e) {
			m_log.debug("execute(): Fatal protocol violation. " + e.getMessage());
			m_errorMessage = "HttpException : " + e.getMessage();
		} catch (IOException e) {
			m_log.debug("execute(): Fatal transport error. "  + e.getMessage());
			m_errorMessage = "IOException : " + e.getMessage();
		} finally {
			// Release the connection.
			if(method != null){
				method.releaseConnection();
			}
		}
		return result;
	}
	
	/**
	 * @return m_footer を戻します。
	 */
	public String getFooter() {
		
		if(m_footer != null){
			return m_footer.toString();
		}
		return null;
	}
	
	/**
	 * @return m_header を戻します。
	 */
	public String getHeader() {
		
		if(m_header != null){
			return m_header.toString();
		}
		return null;
	}
	
	/**
	 * @return m_responseBody を戻します。
	 */
	public String getResponseBody() {
		return m_responseBody;
	}
	
	/**
	 * @return m_responseTime を戻します。
	 */
	public long getResponseTime() {
		return m_responseTime;
	}
	
	/**
	 * @return m_statusCode を戻します。
	 */
	public int getStatusCode() {
		return m_statusCode;
	}
	
	/**
	 * @return m_errorMessage を戻します。
	 */
	public String getErrorMessage() {
		return m_errorMessage;
	}
}
