package org.postgresforest.apibase;

import java.sql.*;
import java.util.concurrent.*;

import org.postgresforest.exception.ForestTaskNotExecutedException;

/**
 * PostgreSQLのJDBCドライバに対してStatementクラスのAPIを実行するための
 * Callableタスクのインプリを集約したクラス。このクラス自体は、メンバや
 * メソッドを持たない（内部クラスを持つのみ）
 * 
 * 各タスクは、コンストラクタで与えられたJDBCリソースについて、nullか否かを
 * 必ずチェックする必要がある。nullの場合、その系は縮退していることを意味し、
 * call内で即座に「ForestTaskNotExecutedException」をスローする。
 */
public final class StatementTask {
    
    public static final class Cancel implements Callable<Void> {
        private final Statement stmt;
        public Cancel(Statement stmt) {
            this.stmt = stmt;
        }
        public Void call() throws Exception {
            if (stmt == null) {
                throw new ForestTaskNotExecutedException();
            }
            stmt.cancel();
            return null;
        }
    }
    
    public static final class Execute implements Callable<Boolean> {
        private final Statement stmt;
        private final String sql;
        public Execute(Statement stmt, String sql) {
            this.stmt = stmt;
            this.sql = sql;
        }
        public Boolean call() throws Exception {
            if (stmt == null) {
                throw new ForestTaskNotExecutedException();
            }
            return Boolean.valueOf(stmt.execute(sql));
        }
    }
    
    public static final class ExecuteUpdate implements Callable<Integer> {
        private final Statement stmt;
        private final String sql;
        public ExecuteUpdate(Statement stmt, String sql) {
            this.stmt = stmt;
            this.sql = sql;
        }
        public Integer call() throws Exception {
            if (stmt == null) {
                throw new ForestTaskNotExecutedException();
            }
            return Integer.valueOf(stmt.executeUpdate(sql));
        }
    }
    
    public static final class ExecuteQuery implements Callable<ResultSet> {
        private final Statement stmt;
        private final String sql;
        public ExecuteQuery(Statement stmt, String sql) {
            this.stmt = stmt;
            this.sql = sql;
        }
        public ResultSet call() throws Exception {
            if (stmt == null) {
                throw new ForestTaskNotExecutedException();
            }
            return stmt.executeQuery(sql);
        }
    }
    
    public static final class ClearWarnings implements Callable<Void> {
        private final Statement stmt;
        public ClearWarnings(Statement stmt) {
            this.stmt = stmt;
        }
        public Void call() throws Exception {
            if (stmt == null) {
                throw new ForestTaskNotExecutedException();
            }
            stmt.clearWarnings();
            return null;
        }
    }
    
    public static final class SetCursorName implements Callable<Void> {
        private final Statement stmt;
        private final String arg0;
        public SetCursorName(Statement stmt, String arg0) {
            this.stmt = stmt;
            this.arg0 = arg0;
        }
        public Void call() throws Exception {
            if (stmt == null) {
                throw new ForestTaskNotExecutedException();
            }
            stmt.setCursorName(arg0);
            return null;
        }
    }
    
    public static final class SetEscapeProcessing implements Callable<Void> {
        private final Statement stmt;
        private final boolean arg0;
        public SetEscapeProcessing(Statement stmt, boolean arg0) {
            this.stmt = stmt;
            this.arg0 = arg0;
        }
        public Void call() throws Exception {
            if (stmt == null) {
                throw new ForestTaskNotExecutedException();
            }
            stmt.setEscapeProcessing(arg0);
            return null;
        }
    }
    
    public static final class SetMaxFieldSize implements Callable<Void> {
        private final Statement stmt;
        private final int arg0;
        public SetMaxFieldSize(Statement stmt, int arg0) {
            this.stmt = stmt;
            this.arg0 = arg0;
        }
        public Void call() throws Exception {
            if (stmt == null) {
                throw new ForestTaskNotExecutedException();
            }
            stmt.setMaxFieldSize(arg0);
            return null;
        }
    }
    
    public static final class SetMaxRows implements Callable<Void> {
        private final Statement stmt;
        private final int arg0;
        public SetMaxRows(Statement stmt, int arg0) {
            this.stmt = stmt;
            this.arg0 = arg0;
        }
        public Void call() throws Exception {
            if (stmt == null) {
                throw new ForestTaskNotExecutedException();
            }
            stmt.setMaxRows(arg0);
            return null;
        }
    }
    
    public static final class SetQueryTimeout implements Callable<Void> {
        private final Statement stmt;
        private final int arg0;
        public SetQueryTimeout(Statement stmt, int arg0) {
            this.stmt = stmt;
            this.arg0 = arg0;
        }
        public Void call() throws Exception {
            if (stmt == null) {
                throw new ForestTaskNotExecutedException();
            }
            stmt.setQueryTimeout(arg0);
            return null;
        }
    }
    
    public static final class GetMaxFieldSize implements Callable<Integer> {
        private final Statement stmt;
        public GetMaxFieldSize(Statement stmt) {
            this.stmt = stmt;
        }
        public Integer call() throws Exception {
            if (stmt == null) {
                throw new ForestTaskNotExecutedException();
            }
            return Integer.valueOf(stmt.getMaxFieldSize());
        }
    }
    
    public static final class GetMaxRows implements Callable<Integer> {
        private final Statement stmt;
        public GetMaxRows(Statement stmt) {
            this.stmt = stmt;
        }
        public Integer call() throws Exception {
            if (stmt == null) {
                throw new ForestTaskNotExecutedException();
            }
            return Integer.valueOf(stmt.getMaxRows());
        }
    }
    
    public static final class GetQueryTimeout implements Callable<Integer> {
        private final Statement stmt;
        public GetQueryTimeout(Statement stmt) {
            this.stmt = stmt;
        }
        public Integer call() throws Exception {
            if (stmt == null) {
                throw new ForestTaskNotExecutedException();
            }
            return Integer.valueOf(stmt.getQueryTimeout());
        }
    }
    
    public static final class GetWarnings implements Callable<SQLWarning> {
        private final Statement stmt;
        public GetWarnings(Statement stmt) {
            this.stmt = stmt;
        }
        public SQLWarning call() throws Exception {
            if (stmt == null) {
                throw new ForestTaskNotExecutedException();
            }
            return stmt.getWarnings();
        }
    }
    
    public static final class GetMoreResults implements Callable<Boolean> {
        private final Statement stmt;
        public GetMoreResults(Statement stmt) {
            this.stmt = stmt;
        }
        public Boolean call() throws Exception {
            if (stmt == null) {
                throw new ForestTaskNotExecutedException();
            }
            return Boolean.valueOf(stmt.getMoreResults());
        }
    }
    
    public static final class GetResultSet implements Callable<ResultSet> {
        private final Statement stmt;
        public GetResultSet(Statement stmt) {
            this.stmt = stmt;
        }
        public ResultSet call() throws Exception {
            if (stmt == null) {
                throw new ForestTaskNotExecutedException();
            }
            return stmt.getResultSet();
        }
    }
    
    public static final class GetUpdateCount implements Callable<Integer> {
        private final Statement stmt;
        public GetUpdateCount(Statement stmt) {
            this.stmt = stmt;
        }
        public Integer call() throws Exception {
            if (stmt == null) {
                throw new ForestTaskNotExecutedException();
            }
            return Integer.valueOf(stmt.getUpdateCount());
        }
    }
    
    public static final class Close implements Callable<Void> {
        private final Statement stmt;
        public Close(Statement stmt) {
            this.stmt = stmt;
        }
        public Void call() throws Exception {
            if (stmt == null) {
                throw new ForestTaskNotExecutedException();
            }
            stmt.close();
            return null;
        }
    }
    
}
